/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* sprite functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#include "vgagames2.h"

extern void vg_sound_attach_hidden(const char *,const char *,int);
extern void vg_sound_copy_hidden(const char *,const char *);
extern int vg_sound_play_hidden(const char *,int,int);
extern void vg_sound_deta_hidden(const char *);

static unsigned int hidden_zhl=0;

sprite * vg_sprite_createnew(void);
sprite * vg_sprite_createfromfile(const char *);
int vg_sprite_add(sprite *,const bitmap *,int,const char *,int);
sprite * vg_sprite_duplicate(const sprite *);
sprite * vg_sprite_rotate(sprite *,int);
sprite * vg_sprite_zoom(sprite *,double,double);
sprite * vg_sprite_mirror(sprite *,int);
bitmap * vg_sprite_getnext(sprite *);
bitmap * vg_sprite_getcurrent(const sprite *);
void vg_sprite_reset(sprite *);
void vg_sprite_free(sprite *);
int vg_nw_dumpsprite(const sprite *);
void vg_nw_undumpsprite(sprite *,int);

static char * strip_white(char *);


/* +++ internal functions +++ */

static char * strip_white(char * buf) {
/* strip comment and leading and following whitespaces */
  char * p1,* p2;
  /* strip comment */
  if ((p1=strchr(buf,'#'))!=NULL) {*p1='\0';}
  /* strip leading and following whitespaces */
  p1=buf+strspn(buf," \f\t\v\r\n");
  if (*p1=='\0') {return(p1);}
  for (p2=p1+strlen(p1)-1;strchr(" \f\t\v\r\n",(int)*p2)!=NULL;p2--) {
    if (p2<p1) {break;}
  }
  p2[1]='\0';
  return(p1);
} /* Ende strip_white */


/* +++ functions +++ */

sprite * vg_sprite_createnew() {
/* create a new empty sprite
** return: sprite
**         or NULL = error
*/
  sprite * spt;
  if ((spt=calloc(1,sizeof(sprite)))==NULL) {fprintf(stderr,"vg_sprite_createnew: calloc: %s.\n",strerror(errno)); return(NULL);}
  spt->max_elem=0;
  return(spt);
} /* Ende vg_sprite_createnew */


sprite * vg_sprite_createfromfile(const char * filename) {
/* load a sprite from a file
** 1.arg: filename with relative path to game-directory
** return: sprite
**         or NULL = error
**
** sprite-file format:
**   "[SPRITE]"
**   [#]<bitmap filename> [LOOP=<loops>] [ZOOM=<x-zoom>,<y-zoom>] [ROTATE=<degrees>] [MIRROR=<1|2>] [SOUND=<sound filename>] [SNDVOL=<0-200>]
**   ...
**
** E.g.:
**   [SPRITE]
**   bitmaps/1.vga LOOP=3 ROTATE=180 SOUND=wave/1.wav SNDVOL=100
**   bitmaps/1.vga LOOP=3
**   #bitmaps/2.vga LOOP=20
** shows for 3 loops 1.vga, rotated for 180 degrees, starting sound 1.wav with volume=100 percent
** shows for 3 loops 1.vga, not rotated
** sprite ends, because 2.vga is commented out
** Note, that bitmaps and sounds are relative to sprite-file, not to game-directory
*/
  char buf[4096],* p1,* p2;
  sprite * spt;
  bitmap * grf;
  FILE * ffp;
  struct {
    char * bmp,* snd;
    int loop,bitflag;
    double zoom1,zoom2;
    int rotate;
    int mirror;
    int sndvol;
  } parm;
  char fsnd[1024],cwkop[512];

  strlcpy(cwkop,_cwdir2,sizeof(cwkop));
  /* read sprite and allocate memory */
  if (filename==NULL) {fprintf(stderr,"vg_sprite_createfromfile: invalid argument (NULL).\n"); return(NULL);}
  if (*filename!='/') {
    snprintf(buf,sizeof(buf),"%s/%s",*_cwdir2=='\0'?CWDIR:_cwdir2,filename);
  } else {snprintf(buf,sizeof(buf),"%s",filename);}
  strlcpy(_cwdir2,buf,sizeof(_cwdir2));
  if ((p1=strrchr(_cwdir2,'/'))!=NULL) {*p1='\0';} else {strlcpy(_cwdir2,".",sizeof(_cwdir2));}
  if ((ffp=fopen(buf,"r"))==NULL) {fprintf(stderr,"vg_sprite_createfromfile: cannot open file \"%s\".\n",filename); goto sc_bad;}
  if (fgets(buf,sizeof(buf),ffp)==NULL) {fprintf(stderr,"vg_sprite_createfromfile: empty file \"%s\".\n",filename); goto sc_bad;}
  if (strncmp(buf,"[SPRITE]",8)!=0) {fprintf(stderr,"vg_sprite_createfromfile: file seems not to be a sprite file: \"%s\".\n",filename); goto sc_bad;}
  if ((spt=vg_sprite_createnew())==NULL) {fprintf(stderr,"vg_sprite_createfromfile: error calling vg_sprite_createnew()\n"); goto sc_bad;}
  while (fgets(buf,sizeof(buf),ffp)!=NULL) {
    p1=strip_white(buf);
    if (*p1=='\0') {continue;}
    /* extract parameters */
    memset(&parm,0,sizeof(parm));
    parm.loop=1;
    parm.sndvol=100;
    /* bitmap */
    if ((p2=strpbrk(p1," \t"))!=NULL) {*p2='\0';}
    parm.bmp=p1;
    /* more parameters */
    while (p2!=NULL) {
      p1=p2+1;
      p1=p1+strspn(p1," \t");
      if (*p1=='\0') {break;}
      if (strncmp(p1,"LOOP=",5)==0) {
        parm.loop=atoi(p1+5);
      } else if (strncmp(p1,"ZOOM=",5)==0) {
        parm.zoom1=atof(p1+5);
        if ((p2=strchr(p1+5,','))!=NULL) {
          parm.zoom2=atof(p2+1);
          parm.bitflag|=1;
        }
      } else if (strncmp(p1,"ROTATE=",7)==0) {
        parm.rotate=atoi(p1+7);
        parm.bitflag|=2;
      } else if (strncmp(p1,"MIRROR=",7)==0) {
        parm.mirror=atoi(p1+7);
        parm.bitflag|=4;
      } else if (strncmp(p1,"SOUND=",6)==0) {
        if ((p2=strpbrk(p1+6," \t"))!=NULL) {*p2='\0';}
        if (p1[6]!='/') {
          snprintf(fsnd,sizeof(fsnd),"%s/%s",_cwdir2,p1+6);
          parm.snd=fsnd;
        } else {parm.snd=p1+6;}
      } else if (strncmp(p1,"SNDVOL=",7)==0) {
        parm.sndvol=atoi(p1+7);
      }
      p2=strpbrk(p1," \t");
    }
    if ((grf=vg_bitmap_createfromfile(parm.bmp))==NULL) {fprintf(stderr,"vg_sprite_createfromfile: error calling vg_bitmap_createfromfile()\n"); goto sc_bad;}
    if (parm.bitflag) {  /* modify bitmap */
      bitmap * grfsave=grf;
      if (parm.bitflag&1) {
        if ((grf=vg_bitmap_zoom(grf,parm.zoom1,parm.zoom2))==NULL) {fprintf(stderr,"vg_sprite_createfromfile: error calling vg_bitmap_zoom()\n"); goto sc_bad;}
      }
      if (parm.bitflag&2) {
        if ((grf=vg_bitmap_rotate(grf,parm.rotate))==NULL) {fprintf(stderr,"vg_sprite_createfromfile: error calling vg_bitmap_rotate()\n"); goto sc_bad;}
      }
      if (parm.bitflag&4) {
        if ((grf=vg_bitmap_mirror(grf,parm.mirror))==NULL) {fprintf(stderr,"vg_sprite_createfromfile: error calling vg_bitmap_mirror()\n"); goto sc_bad;}
      }
      if ((grf=vg_bitmap_duplicate(grf))==NULL) {fprintf(stderr,"vg_sprite_createfromfile: error calling vg_bitmap_duplicate()\n"); goto sc_bad;}
      vg_bitmap_free(grfsave);
    }
    if (vg_sprite_add(spt,grf,parm.loop,parm.snd,parm.sndvol)<0) {fprintf(stderr,"vg_sprite_createfromfile: error calling vg_sprite_add()\n"); goto sc_bad;}
    vg_bitmap_free(grf);
  }
  fclose(ffp);
  strlcpy(_cwdir2,cwkop,sizeof(_cwdir2));
  return(spt);
sc_bad:
  if (ffp!=NULL) {fclose(ffp);}
  strlcpy(_cwdir2,cwkop,sizeof(_cwdir2));
  return(NULL);
} /* Ende vg_sprite_createfromfile */


int vg_sprite_add(sprite * spt,const bitmap * grf,int loop,const char * sound,int sndvol) {
/* add a sprite element to an existing sprite
** 1.arg: sprite
** 2.arg: bitmap to add (will be duplicated)
** 3.arg: how many loops
** 4.arg: sound filename or NULL
** 5.arg: sound volume percent (0-200)
** return: 0=OK or -1=error
*/
  if ((spt==NULL) || (grf==NULL)) {fprintf(stderr,"vg_sprite_add: invalid argument (NULL).\n"); return(-1);}
  if (loop<=0) {return(0);}
  if (spt->max_elem==0) {
    if ((spt->elem=malloc(sizeof(*spt->elem)))==NULL) {fprintf(stderr,"vg_sprite_add: malloc: %s.\n",strerror(errno)); return(-1);}
  } else {
    if ((spt->elem=realloc(spt->elem,(spt->max_elem+1)*sizeof(*spt->elem)))==NULL) {fprintf(stderr,"vg_sprite_add: realloc: %s.\n",strerror(errno)); spt->max_elem=0; return(-1);}
  }
  memset(&spt->elem[spt->max_elem],0,sizeof(*spt->elem));
  spt->elem[spt->max_elem].anz_loop=loop;
  if ((spt->elem[spt->max_elem].grf=vg_bitmap_duplicate(grf))==NULL) {fprintf(stderr,"vg_sprite_add: error calling vg_bitmap_duplicate()\n"); return(-1);}
  if ((sound!=NULL) && (*sound!='\0')) {
    snprintf(spt->elem[spt->max_elem].sname,sizeof(spt->elem[spt->max_elem].sname),"h-a%u",++hidden_zhl);
    vg_sound_attach_hidden(sound,spt->elem[spt->max_elem].sname,sndvol);
  }
  spt->max_elem++;
  return(0);
} /* Ende vg_sprite_add */


sprite * vg_sprite_duplicate(const sprite * sspt) {
/* duplicate a sprite, creating a new one
** 1.arg: sprite
** return: duplicated sprite
**         or NULL = error
*/
  sprite * dspt;
  int i1;
  if (sspt==NULL) {fprintf(stderr,"vg_sprite_duplicate: invalid argument (NULL).\n"); return(NULL);}
  if ((dspt=vg_sprite_createnew())==NULL) {fprintf(stderr,"vg_sprite_duplicate: error calling vg_sprite_createnew()\n"); return(NULL);}
  for (i1=0;i1<sspt->max_elem;i1++) {
    if (vg_sprite_add(dspt,sspt->elem[i1].grf,sspt->elem[i1].anz_loop,NULL,0)<0) {
      fprintf(stderr,"vg_sprite_duplicate: error calling vg_sprite_add()\n");
      return(NULL);
    }
    if (*sspt->elem[i1].sname!='\0') {
      snprintf(dspt->elem[i1].sname,sizeof(dspt->elem[i1].sname),"h-a%u",++hidden_zhl);
      vg_sound_copy_hidden(dspt->elem[i1].sname,sspt->elem[i1].sname);
    }
  }
  return(dspt);
} /* Ende vg_sprite_duplicate */


sprite * vg_sprite_rotate(sprite * spt,int deg) {
/* rotate a sprite
** 1.arg: sprite to rotate
** 2.arg: degrees to rotate clockwise (-360 to 360)
** return: pointer to modified 1.arg
**         or NULL = error
*/
  int i1;
  bitmap * grfsave;
  if (spt==NULL) {fprintf(stderr,"vg_sprite_rotate: invalid argument (NULL).\n"); return(NULL);}
  for (i1=0;i1<spt->max_elem;i1++) {
    grfsave=spt->elem[i1].grf;
    if ((spt->elem[i1].grf=vg_bitmap_duplicate(vg_bitmap_rotate(spt->elem[i1].grf,deg)))==NULL) {
      fprintf(stderr,"vg_sprite_rotate: error calling vg_bitmap_duplicate(vg_bitmap_rotate())\n");
      return(NULL);
    }
    vg_bitmap_free(grfsave);
  }
  return(spt);
} /* Ende vg_sprite_rotate */


sprite * vg_sprite_zoom(sprite * spt,double xmul,double ymul) {
/* zoom a sprite greater or smaller
** 1.arg: sprite to zoom
** 2.arg: horizontal zooming (x axis), where 1.0 is original size
** 3.arg: vertical zooming (y axis), where 1.0 is original size
** return: pointer to modified 1.arg
**         or NULL = error
*/
  int i1;
  bitmap * grfsave;
  if (spt==NULL) {fprintf(stderr,"vg_sprite_zoom: invalid argument (NULL).\n"); return(NULL);}
  for (i1=0;i1<spt->max_elem;i1++) {
    grfsave=spt->elem[i1].grf;
    if ((spt->elem[i1].grf=vg_bitmap_duplicate(vg_bitmap_zoom(spt->elem[i1].grf,xmul,ymul)))==NULL) {
      fprintf(stderr,"vg_sprite_zoom: error calling vg_bitmap_duplicate(vg_bitmap_zoom())\n");
      return(NULL);
    }
    vg_bitmap_free(grfsave);
  }
  return(spt);
} /* Ende vg_sprite_zoom */


sprite * vg_sprite_mirror(sprite * spt,int m_art) {
/* mirror a sprite vertically or horizontally
** 1.arg: sprite to mirror
** 2.arg: MIRROR_VT=vertically or MIRROR_HT=horizontally
** return: pointer to modified 1.arg
**         or NULL = error
*/
  int i1;
  bitmap * grfsave;
  if (spt==NULL) {fprintf(stderr,"vg_sprite_mirror: invalid argument (NULL).\n"); return(NULL);}
  for (i1=0;i1<spt->max_elem;i1++) {
    grfsave=spt->elem[i1].grf;
    if ((spt->elem[i1].grf=vg_bitmap_duplicate(vg_bitmap_mirror(spt->elem[i1].grf,m_art)))==NULL) {
      fprintf(stderr,"vg_sprite_mirror: error calling vg_bitmap_duplicate(vg_bitmap_mirror())\n");
      return(NULL);
    }
    vg_bitmap_free(grfsave);
  }
  return(spt);
} /* Ende vg_sprite_mirror */


bitmap * vg_sprite_getnext(sprite * spt) {
/* return pointer to current bitmap of a sprite after incrementing counter
** 1.arg: sprite
** return: pointer to bitmap of sprite
**         or NULL=sprite ended (next call will start again)
*/
  if (spt==NULL) {fprintf(stderr,"vg_sprite_getnext: invalid argument (NULL).\n"); return(NULL);}
  if (spt->curr_elem==0) {
    if (++spt->curr_elem>spt->max_elem) {spt->curr_elem=spt->curr_loop=0; return(NULL);}
    spt->curr_loop=0;
  }
  while (++spt->curr_loop>spt->elem[spt->curr_elem-1].anz_loop) {
    if (++spt->curr_elem>spt->max_elem) {spt->curr_elem=spt->curr_loop=0; return(NULL);}
    spt->curr_loop=0;
  }
  if ((spt->curr_loop==1) && (*spt->elem[spt->curr_elem-1].sname!='\0')) {
    (void)vg_sound_play_hidden(spt->elem[spt->curr_elem-1].sname,0,1);
  }
  return(spt->elem[spt->curr_elem-1].grf);
} /* Ende vg_sprite_getnext */


bitmap * vg_sprite_getcurrent(const sprite * spt) {
/* return pointer to current bitmap of a sprite without incrementing counter
** 1.arg: sprite
** return: pointer to bitmap of sprite
**         or NULL=sprite ended
*/
  if (spt==NULL) {fprintf(stderr,"vg_sprite_getcurrent: invalid argument (NULL).\n"); return(NULL);}
  if (spt->curr_elem==0) {return(NULL);}
  return(spt->elem[spt->curr_elem-1].grf);
} /* Ende vg_sprite_getcurrent */


void vg_sprite_reset(sprite * spt) {
/* reset a sprite (next call of vg_sprite_getnext() will start at the beginning)
** 1.arg: sprite
*/
  int i1;
  if (spt==NULL) {fprintf(stderr,"vg_sprite_reset: invalid argument (NULL).\n"); return;}
  for (i1=0;i1<spt->max_elem;i1++) {
    if (*spt->elem[i1].sname!='\0') {vg_sound_stop(spt->elem[i1].sname,0);}
  }
  spt->curr_elem=spt->curr_loop=0;
} /* Ende vg_sprite_reset */


void vg_sprite_free(sprite * spt) {
/* free a sprite
** 1.arg: sprite
*/
  int i1;
  if (spt==NULL) {fprintf(stderr,"vg_sprite_free: invalid argument (NULL).\n"); return;}
  for (i1=0;i1<spt->max_elem;i1++) {
    vg_bitmap_free(spt->elem[i1].grf);
    if (*spt->elem[i1].sname!='\0') {vg_sound_deta_hidden(spt->elem[i1].sname);}
  }
  free(spt->elem);
  free(spt);
} /* Ende vg_sprite_free */


int vg_nw_dumpsprite(const sprite * spt) {
/* return dump of sprite
** 1.arg: sprite
** return: dump of sprite
**         it fits into a char, if the sprite has < 128 lines
**         it fits into a short, if the sprite has < 32768 lines
*/
  int retw;
  if (spt==NULL) {fprintf(stderr,"vg_nw_dumpsprite: invalid argument (NULL).\n"); return(0);}
  retw=spt->curr_elem;
  if (retw<0) {return(0);}
  if (spt->curr_loop>0) {retw|=0x8000000;}
  return(retw);
} /* Ende vg_nw_dumpsprite */


void vg_nw_undumpsprite(sprite * spt,int dump) {
/* set sprite counter according to dump
** 1.arg: sprite
** 2.arg: dump (from vg_nw_dumpsprite())
*/
  int loop;
  if (spt==NULL) {fprintf(stderr,"vg_nw_undumpsprite: invalid argument (NULL).\n"); return;}
  if (dump<0) {dump&=0x7fffffff; loop=1;} else {loop=0;}
  if (dump>spt->max_elem) {return;}
  spt->curr_elem=dump;
  spt->curr_loop=loop;
} /* Ende vg_nw_undumpsprite */
