/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* for converting text to selected language-code */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#include "vgagames2.h"
#include "inthash.h"
#include "utf8.h"

void default_textformat(const char *);
void load_textformat(const char *);
const char * vg_get_textformat(const char *);

static char deflkz[4]="";
static char ** translang=NULL;
static int translen=0;
static struct ihash ** transhash=NULL;


/* +++ functions +++ */

void default_textformat(const char * lkz) {
/* sets the default language, passed by vg_init_vgagames() */
  if (lkz!=NULL) {
    strlcpy(deflkz,lkz,sizeof(deflkz));
  } else {*deflkz='\0';}
} /* Ende default_textformat */


void load_textformat(const char * lkz) {
/* load text of selected language
** reads <CWDIR>/share/textformat.<language>
** where <language> is a lowercase language-code with 2 characters
** and <CWDIR>/share/textformat, which is the default file
** (symlink to another textformat.<language>).
** The lines in "textformat" are compared with them in "textformat.<language>"
** to get the translation.
** Lines with a '#' at the begin are considered as comments.
** Empty lines are skipped.
*/
  char buf[4096];
  const char * ptr;
  int i1,zln;
  FILE * ffa,* ffb;
  size_t s1;

  /* clean translation */
  if (transhash!=NULL) {inthash_free(transhash); transhash=NULL;}
  for (i1=0;i1<translen;i1++) {free(translang[i1]);}
  if (translang!=NULL) {free(translang); translang=NULL;}
  translen=0;
  if (lkz==NULL || *lkz=='\0') {lkz=deflkz;}  /* use default */
  if (lkz==NULL || *lkz=='\0') {return;}  /* no translation */

  /* open files */
  snprintf(buf,sizeof(buf),"%s/share/textformat",CWDIR);
  if ((ffa=fopen(buf,"r"))==NULL) {return;}
  snprintf(buf,sizeof(buf),"%s/share/textformat.%s",CWDIR,lkz);
  if ((ffb=fopen(buf,"r"))==NULL) {fclose(ffa); return;}

  /* count lines */
  i1=0;
  while (fgets(buf,sizeof(buf),ffa)!=NULL) {
    if (*buf=='#') {continue;}
    ptr=buf+strspn(buf," \f\t\v\r");
    if (*ptr!='\n') {i1++;}
  }
  rewind(ffa);

  if ((transhash=inthash_new(primzahl(i1*2,1)))==NULL) {fclose(ffa); fclose(ffb); translen=0; return;}
  for (zln=0;;) {
    while ((ptr=fgets(buf,sizeof(buf),ffa))!=NULL) {
      if (*buf=='#') {continue;}
      ptr=buf+strspn(buf," \f\t\v\r");
      if (*ptr!='\n') {break;}
    }
    if (ptr==NULL) {break;}
    zln++;
    s1=strlen(buf);
    if (s1>0 && buf[s1-1]=='\n') {s1--;}
    if (s1>0 && buf[s1-1]=='\r') {s1--;}
    buf[s1]='\0';
    ptr=utf8text(buf);
    if (inthash_add(transhash,ptr,zln)<0) {fclose(ffa); fclose(ffb); translen=0; return;}

    while ((ptr=fgets(buf,sizeof(buf),ffb))!=NULL) {
      if (*buf=='#') {continue;}
      ptr=buf+strspn(buf," \f\t\v\r");
      if (*ptr!='\n') {break;}
    }
    if (ptr==NULL) {*buf='\0';}
    s1=strlen(buf);
    if (s1>0 && buf[s1-1]=='\n') {s1--;}
    if (s1>0 && buf[s1-1]=='\r') {s1--;}
    buf[s1]='\0';
    ptr=utf8text(buf);
    translang=(translen==0?malloc(sizeof(char *)):realloc(translang,sizeof(char *)*(translen+1)));
    if (translang==NULL) {fclose(ffa); fclose(ffb); translen=0; return;}
    if ((translang[translen++]=strdup(ptr))==NULL) {fclose(ffa); fclose(ffb); translen=0; return;}
  }
  fclose(ffa);
  fclose(ffb);
} /* Ende load_textformat */


const char * vg_get_textformat(const char * text) {
/* return translated text */
  const char * ansitext;
  int zln;
  if (translen==0) {return(text);}  /* no translation */
  ansitext=utf8text(text);
  if (inthash_get(transhash,ansitext,&zln)<0) {return(text);}  /* not found */
  return(translang[zln-1]);
} /* Ende vg_get_textformat */
