/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* vga_conv.c: convert-routines from and to vgagames bitmap format */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#include "vgagames2.h"
#include "color.h"

struct f_mem {
  char * data;
  size_t pos,len,size;
};


FILE * convtovga(FILE *);
int convfromvga(FILE *,FILE *,int);

static void fileclose(FILE *);
static struct f_mem * ftomem(FILE *);
static struct f_mem * newmem(void);
static void freemem(struct f_mem **);
static size_t memwrite(void *,size_t,size_t,struct f_mem *);
static FILE * memflush(struct f_mem *);
static size_t memread(void *,size_t,size_t,struct f_mem *);
static int memseek(struct f_mem *,long,int);
static int memgetc(struct f_mem *);
static int memget3int(struct f_mem *,int *,int *,int *);

static FILE * bmptovga(struct f_mem *);
static FILE * ppmtovga(struct f_mem *,int);
static int vgatobmp(struct f_mem *,FILE *);
static int vgatoppm(struct f_mem *,FILE *,int);


FILE * convtovga(FILE * ffin) {
/* convert to vga-format and return (2 bytes read) */
  char buf[4];
  FILE * ffout;
  struct f_mem * mmin;
  if (ffin==NULL) {return(NULL);}
  if (fread(buf,sizeof(char),2,ffin)!=2) {
    fprintf(stderr,"convtovga: empty inputfile.\n");
    fileclose(ffin);
    return(NULL);
  }
  if (strncmp(buf,"VG",2)==0) {
    if (ffin==stdin) {
      char bf[1024];
      size_t sz;
      if ((ffin=tmpfile())==NULL) {
        fprintf(stderr,"convtovga: tmpfile: %s\n",strerror(errno));
        return(NULL);
      }
      fwrite("VG",sizeof(char),2,ffin);
      while ((sz=fread(bf,sizeof(char),sizeof(bf),stdin))>0) {
        fwrite(bf,sizeof(char),sz,ffin);
      }
      fflush(ffin);
      fseek(ffin,2L,SEEK_SET);
    }
    return(ffin);
  }
  if ((mmin=ftomem(ffin))==NULL) {return(NULL);}
  if (strncmp(buf,"BM",2)==0) {
    ffout=bmptovga(mmin);
  } else if (strncmp(buf,"P3",2)==0) {
    ffout=ppmtovga(mmin,3);
  } else if (strncmp(buf,"P6",2)==0) {
    ffout=ppmtovga(mmin,6);
  } else {
    fprintf(stderr,"convtovga: unknown format: \"%.2s\".\n",buf);
    ffout=NULL;
  }
  freemem(&mmin);
  return(ffout);
} /* Ende convtovga */


int convfromvga(FILE * ffin,FILE * ffout,int vfmt) {
/* convert from vga-format into 2.arg (not closed) */
  char buf[1024];
  int retw;
  struct f_mem * mmin;
  if (ffin==NULL) {return(-1);}
  if (ffout==NULL) {fileclose(ffin); return(-1);}
  if (fread(buf,sizeof(char),2,ffin)!=2) {
    fprintf(stderr,"convfromvga: empty inputfile.\n");
    fileclose(ffin);
    return(-1);
  }
  if (strncmp(buf,"VG",2)!=0) {
    fprintf(stderr,"convfromvga: inputfile is no VgaGames bitmap format.\n");
    fileclose(ffin);
    return(-1);
  }
  if ((mmin=ftomem(ffin))==NULL) {return(-1);}
  if (vfmt==VGAFORMAT_VGA) {
    size_t anz;
    fwrite("VG",sizeof(char),2,ffout);
    while ((anz=memread(buf,sizeof(char),sizeof(buf),mmin))>0) {
      fwrite(buf,sizeof(char),anz,ffout);
    }
    fflush(ffout);
    retw=0;
  } else if (vfmt==VGAFORMAT_BMP) {
    retw=vgatobmp(mmin,ffout);
  } else if (vfmt==VGAFORMAT_PPM3) {
    retw=vgatoppm(mmin,ffout,3);
  } else if (vfmt==VGAFORMAT_PPM6) {
    retw=vgatoppm(mmin,ffout,6);
  } else {
    fprintf(stderr,"convfromvga: unknown format: %d.\n",vfmt);
    retw=-1;
  }
  freemem(&mmin);
  return(retw);
} /* Ende convfromvga */


static void fileclose(FILE * ffin) {
  if (fseek(ffin,0L,SEEK_SET)==0) {fclose(ffin);}
} /* Ende fileclose */


static struct f_mem * ftomem(FILE * ffin) {
/* ffin is at pos 2 */
  const size_t block=8192;
  struct f_mem * mmem;
  size_t s1;
  if ((mmem=calloc(1,sizeof(*mmem)))==NULL) {
    fprintf(stderr,"ftomem: calloc: %s\n",strerror(errno));
    fileclose(ffin);
    return(NULL);
  }
  mmem->pos=2;
  mmem->len=2;
  mmem->size=mmem->len+block;
  if ((mmem->data=malloc(sizeof(char)*mmem->size))==NULL) {
    fprintf(stderr,"ftomem: malloc: %s\n",strerror(errno));
    fileclose(ffin);
    return(NULL);
  }
  while ((s1=fread(mmem->data+mmem->len,sizeof(char),block,ffin))>0) {
    mmem->len+=s1;
    mmem->size=mmem->len+block;
    if ((mmem->data=realloc(mmem->data,sizeof(char)*mmem->size))==NULL) {
      fprintf(stderr,"ftomem: realloc: %s\n",strerror(errno));
      fileclose(ffin);
      return(NULL);
    }
  }
  fileclose(ffin);
  return(mmem);
} /* Ende ftomem */


static struct f_mem * newmem() {
  struct f_mem * mmem;
  if ((mmem=calloc(1,sizeof(*mmem)))==NULL) {
    fprintf(stderr,"newmem: calloc: %s\n",strerror(errno));
    return(NULL);
  }
  mmem->pos=0;
  mmem->len=0;
  mmem->size=1;
  if ((mmem->data=malloc(sizeof(char)))==NULL) {
    fprintf(stderr,"newmem: malloc: %s\n",strerror(errno));
    return(NULL);
  }
  return(mmem);
} /* Ende newmem */


static void freemem(struct f_mem ** mmem) {
  if (mmem==NULL || *mmem==NULL) {return;}
  if ((*mmem)->data!=NULL) {free((*mmem)->data);}
  free(*mmem);
  *mmem=NULL;
} /* Ende memfree */


static size_t memwrite(void * ptr,size_t size,size_t nmemb,struct f_mem * mmem) {
  const size_t block=8192;
  size_t pos;
  if (ptr==NULL || size==0 || nmemb==0 || mmem==NULL || mmem->data==NULL) {return(0);}
  pos=mmem->pos+size*nmemb;
  if (pos>mmem->size) {
    if ((mmem->data=realloc(mmem->data,sizeof(char)*(pos+block)))==NULL) {
      fprintf(stderr,"memwrite: realloc: %s\n",strerror(errno));
      return(0);
    }
    mmem->size=pos+block;
  }
  memmove(mmem->data+mmem->pos,ptr,size*nmemb);
  if (mmem->len<pos) {mmem->len=pos;}
  mmem->pos+=size*nmemb;
  return(size*nmemb);
} /* Ende memwrite */


static FILE * memflush(struct f_mem * mmem) {
  FILE * ffout;
  if (mmem==NULL) {errno=EINVAL; return(NULL);}
  if ((ffout=tmpfile())==NULL) {
    fprintf(stderr,"memflush: tmpfile: %s\n",strerror(errno));
    return(NULL);
  }
  fwrite(mmem->data,1,mmem->len,ffout);
  fflush(ffout);
  return(ffout);
} /* Ende memflush */


static size_t memread(void * ptr,size_t size,size_t nmemb,struct f_mem * mmem) {
  size_t rest;
  if (ptr==NULL || size==0 || nmemb==0 || mmem==NULL || mmem->data==NULL) {return(0);}
  if ((rest=(mmem->len-mmem->pos)/size)<nmemb) {nmemb=rest;}
  if (nmemb>0) {
    memmove(ptr,mmem->data+mmem->pos,nmemb*size);
    mmem->pos+=nmemb*size;
  }
  return(nmemb);
} /* Ende memread */


static int memseek(struct f_mem * mmem,long offset,int whence) {
  size_t pos;
  if (mmem==NULL) {errno=EINVAL; return(-1);}
  pos=0;
  if (whence==SEEK_CUR) {pos=mmem->pos;} else if (whence==SEEK_END) {pos=mmem->len;}
  pos+=offset;
  if (pos<0 || pos>mmem->len) {errno=EINVAL; return(-1);}
  mmem->pos=pos;
  return(0);
} /* Ende memseek */


static int memgetc(struct f_mem * mmem) {
  if (mmem==NULL || mmem->data==NULL) {errno=EINVAL; return(EOF);}
  if (mmem->pos==mmem->len) {return(EOF);}
  mmem->pos++;
  return((int)(unsigned char)mmem->data[mmem->pos-1]);
} /* Ende memgetc */


static int memget3int(struct f_mem * mmem,int * v1,int * v2,int * v3) {
  size_t pos1;
  int anz,obzahl;
  if (mmem==NULL || mmem->data==NULL || v1==NULL || v2==NULL || v3==NULL) {errno=EINVAL; return(EOF);}
  if (mmem->pos==mmem->len) {return(EOF);}
  obzahl=0;
  for (anz=0,pos1=mmem->pos;mmem->pos<mmem->len;mmem->pos++) {
    if (mmem->data[mmem->pos]<'0' || mmem->data[mmem->pos]>'9') {
      if (obzahl) {
        anz++;
        switch(anz) {
          case 1: *v1=atoi(mmem->data+pos1); break;
          case 2: *v2=atoi(mmem->data+pos1); break;
          case 3: *v3=atoi(mmem->data+pos1); break;
        }
        if (anz==3) {break;}
        pos1=mmem->pos;
        obzahl=0;
      }
      if ((unsigned char)mmem->data[mmem->pos]>' ') {break;}
    } else {obzahl=1;}
  }
  return(anz);
} /* Ende memget3int */


static FILE * bmptovga(struct f_mem * mmin) {
/* mmin is at pos 2 */
  union {
    short hw;
    char cw[2];
#ifdef ENDIAN_IS_BIG
  } _c2h,_c2ht;
#else
  } _c2h;
#endif
  union {
    int iw;
    char cw[4];
#ifdef ENDIAN_IS_BIG
  } _c2i,_c2it;
#else
  } _c2i;
#endif
  farbkonv fw[256];
  unsigned char kv[256];  /* kv[w]=x */
  char buf[8];
  int i1,i2,i3,i4,i5,ifl,bwid,bhei,anzfarb,bytedata;
  long offdat,hlen,lw;
  short bpp;
  unsigned char c[4];
  struct f_mem * mmout;
  FILE * ffout;

  /* Initialization */
  for (i1=0;i1<256;i1++) {
    fw[i1].r=fw[i1].g=fw[i1].b=0;
    kv[i1]=(unsigned char)i1;
  }

  /* reading windows bitmap */
  memseek(mmin,0x0a,SEEK_SET);  /* 0x0a: offset data */
  if (memread(&_c2i.iw,4,1,mmin)<1) {
    fprintf(stderr,"bmptovga: short read of input-file.\n");
    return(NULL);
  }
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  offdat=(long)_c2i.iw;
  if (memread(&_c2i.iw,4,1,mmin)<1) {  /* 0x0e: info-header size */
    fprintf(stderr,"bmptovga: short read of input-file.\n");
    return(NULL);
  }
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  hlen=(long)_c2i.iw;
  if (memread(&_c2i.iw,4,1,mmin)<1) {  /* 0x12: width of bitmap */
    fprintf(stderr,"bmptovga: short read of input-file.\n");
    return(NULL);
  }
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  bwid=_c2i.iw;
  if (memread(&_c2i.iw,4,1,mmin)<1) {  /* 0x16: height of bitmap */
    fprintf(stderr,"bmptovga: short read of input-file.\n");
    return(NULL);
  }
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  bhei=_c2i.iw;
  memseek(mmin,0x1C,SEEK_SET);  /* 0x1c: bits per pixel */
  if (memread(&_c2h.hw,2,1,mmin)<1) {
    fprintf(stderr,"bmptovga: short read of input-file.\n");
    return(NULL);
  }
#ifdef ENDIAN_IS_BIG
  _c2ht.cw[0]=_c2h.cw[1]; _c2ht.cw[1]=_c2h.cw[0];
  _c2h.hw=_c2ht.hw;
#endif
  bpp=_c2h.hw;
  if (bpp>8) {
    fprintf(stderr,"bmptovga: input-file has too deep color depth (>256).\n");
    return(NULL);
  }
  if (memread(&_c2i.iw,4,1,mmin)<1) {  /* 0x1e: compression */
    fprintf(stderr,"bmptovga: short read of input-file.\n");
    return(NULL);
  }
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  lw=(long)_c2i.iw;
  if (lw!=0L) {
    fprintf(stderr,"bmptovga: input-file uses compression: not supported.\n");
    return(NULL);
  }

  anzfarb=(1<<(int)bpp);
  lw=14L+hlen+(long)(anzfarb*4);
  if (offdat!=lw) {
    fprintf(stderr,"bmptovga: corrupt colormap of input-file: %ld != %ld\n",offdat,lw);
    return(NULL);
  }

  memseek(mmin,0x36,SEEK_SET);  /* 0x36: colormap */
  for (i1=0;i1<anzfarb;i1++) {
    if (memread(c,1,4,mmin)<4) {
      fprintf(stderr,"bmptovga: short read of input-file.\n");
      return(NULL);
    }
    fw[i1].r=(c[2]>>2);
    fw[i1].g=(c[1]>>2);
    fw[i1].b=(c[0]>>2);
  }
  lw=(long)(bwid*(int)bpp/32*32);
  if (bwid*(int)bpp%32>0) {lw+=32;}
  lw*=(long)bhei;
  bytedata=(int)(lw/8L);

  /* write vgagames bitmap */
  if ((mmout=newmem())==NULL) {return(NULL);}
  memwrite("VG",sizeof(char),2,mmout);
  snprintf(buf,sizeof(buf),"%05d",bwid);
  memwrite(&buf,sizeof(char),5,mmout);  /* width */
  snprintf(buf,sizeof(buf),"%05d",bhei);
  memwrite(&buf,sizeof(char),5,mmout);  /* height */
  for (i1=0;i1<256;i1++) {  /* colormap */
    memwrite(&fw[i1].r,sizeof(char),1,mmout);  /* red byte */
    memwrite(&fw[i1].g,sizeof(char),1,mmout);  /* green byte */
    memwrite(&fw[i1].b,sizeof(char),1,mmout);  /* blue byte */
  }
  c[0]=0;
  for (i1=0;i1<bwid;i1++) {
    for (i2=0;i2<bhei;i2++) {memwrite(c,sizeof(char),1,mmout);}
  }

  memseek(mmout,-bwid,SEEK_CUR);
  for (ifl=0,i4=bpp-1,i5=0,i1=0;i1<bytedata;i1+=4) {
    if (memread(c,1,4,mmin)<4) {
      fprintf(stderr,"bmptovga: short read of input-file.\n");
      freemem(&mmout);
      return(NULL);
    }
    for (i3=0;i3<4;i3++) {
      if (bpp==8) {
        if (i5++<bwid) {memwrite(&kv[c[i3]],sizeof(char),1,mmout);}
      } else {
        for (i2=7;i2>=0;i2--) {
          ifl|=(!!((unsigned int)c[i3]&(1<<i2))<<i4);
          if (--i4<0) {
            if (i5++<bwid) {memwrite(&kv[ifl],sizeof(char),1,mmout);}
            i4=bpp-1;
            ifl=0;
          }
        }
      }
    }
    if (i5>=bwid) {i5=0; i4=bpp-1; ifl=0; memseek(mmout,-bwid*2,SEEK_CUR);}
  }
  if ((ffout=memflush(mmout))==NULL) {return(NULL);}
  freemem(&mmout);
  fseek(ffout,2L,SEEK_SET);
  return(ffout);
} /* Ende bmptovga */


static FILE * ppmtovga(struct f_mem * mmin,int ppmart) {
/* mmin is at pos 2 */
  farbkonv fw[256];
  unsigned char kv[256];  /* kv[w]=x */
  char buf[64];
  int ival,i1,i2,i3,i4,bwid=0,bhei=0,maxv=0,zuviel;
  unsigned char c[3];
  struct f_mem * mmout;
  FILE * ffout;

  zuviel=2;
zu_viel:
  /* Initialization */
  for (i1=0;i1<256;i1++) {
    fw[i1].r=fw[i1].g=fw[i1].b=0;
    kv[i1]=(unsigned char)i1;
  }

  /* read width, height, maxval */
  for (i3=0;i3<3;i3++) {
    i2=0;
    while ((ival=memgetc(mmin))!=EOF) {
      if ((ival>='0') && (ival<='9')) {
        buf[i2++]=(char)ival;
        if (i2==(int)sizeof(buf)) {
          fprintf(stderr,"ppmtovga: too big value in input-file.\n");
          return(NULL);
        }
        continue;
      }
      if (i2>0) {break;}
      if (ival=='#') {
        while ((ival=memgetc(mmin))!=EOF) {
          if (ival=='\n') {break;}
        }
      }
    }
    if (ival==EOF) {
      fprintf(stderr,"ppmtovga: short read of input-file.\n");
      return(NULL);
    }
    buf[i2]='\0';
    if (i3==0) {bwid=atoi(buf);}
    else if (i3==1) {bhei=atoi(buf);}
    else if (i3==2) {maxv=atoi(buf);}
  }
  if (maxv>255) {
    fprintf(stderr,"ppmtovga: input-file has too deep color depth (>255).\n");
    fprintf(stderr,"Pre-convert it with \"pnmdepth 255 <input-file>\".\n");
    return(NULL);
  }

  if ((mmout=newmem())==NULL) {return(NULL);}
  memwrite("VG",sizeof(char),2,mmout);
  snprintf(buf,sizeof(buf),"%05d",bwid);
  memwrite(&buf,sizeof(char),5,mmout);  /* width */
  snprintf(buf,sizeof(buf),"%05d",bhei);
  memwrite(&buf,sizeof(char),5,mmout);  /* height */
  buf[0]=buf[1]=buf[2]=0;
  for (i1=0;i1<256;i1++) {
    memwrite(buf,sizeof(char),3,mmout);  /* place for colormap */
  }

  i4=0;
  for (i1=0;i1<bhei;i1++) {
    for (i2=0;i2<bwid;i2++) {
      if (ppmart==3) {  /* plain PPM */
        int v1=0,v2=0,v3=0;
        if (memget3int(mmin,&v1,&v2,&v3)!=3) {
          fprintf(stderr,"ppmtovga: short read of input-file.\n");
          freemem(&mmout);
          return(NULL);
        }
        c[0]=(unsigned char)v1;
        c[1]=(unsigned char)v2;
        c[2]=(unsigned char)v3;
      } else {  /* normal (raw) PPM */
        if (memread(c,1,3,mmin)<3) {
          fprintf(stderr,"ppmtovga: short read of input-file.\n");
          freemem(&mmout);
          return(NULL);
        }
      }
      c[0]=(c[0]>>zuviel)<<zuviel;
      c[1]=(c[1]>>zuviel)<<zuviel;
      c[2]=(c[2]>>zuviel)<<zuviel;
      for (i3=0;i3<i4;i3++) {
        if (((c[0]>>2)==fw[i3].r) \
        && ((c[1]>>2)==fw[i3].g) \
        && ((c[2]>>2)==fw[i3].b)) {
          memwrite(&kv[i3],sizeof(char),1,mmout);
          break;
        }
      }
      if (i3==i4) {
        if (++i4==256) {
          freemem(&mmout);
          if (++zuviel==8) {
            fprintf(stderr,"ppmtovga: too many colors in input-file.\n");
            return(NULL);
          }
          memseek(mmin,2L,SEEK_SET);
          goto zu_viel;
        }
        fw[i3].r=(c[0]>>2);
        fw[i3].g=(c[1]>>2);
        fw[i3].b=(c[2]>>2);
        memwrite(&kv[i3],sizeof(char),1,mmout);
      }
    }
  }

  memseek(mmout,12L,SEEK_SET);
  for (i1=0;i1<256;i1++) {  /* colormap */
    memwrite(&fw[i1].r,sizeof(char),1,mmout);  /* red byte */
    memwrite(&fw[i1].g,sizeof(char),1,mmout);  /* green byte */
    memwrite(&fw[i1].b,sizeof(char),1,mmout);  /* blue byte */
  }
  if ((ffout=memflush(mmout))==NULL) {return(NULL);}
  freemem(&mmout);
  fseek(ffout,2L,SEEK_SET);
  return(ffout);
} /* Ende ppmtovga */


static int vgatobmp(struct f_mem * mmin,FILE * ffout) {
/* mmin is at pos 2, output to ffout */
  union {
    short hw;
    char cw[2];
#ifdef ENDIAN_IS_BIG
  } _c2h,_c2ht;
#else
  } _c2h;
#endif
  union {
    int iw;
    char cw[4];
#ifdef ENDIAN_IS_BIG
  } _c2i,_c2it;
#else
  } _c2i;
#endif
  char buf[128];
  int bw,bh,bz,i1,i2;
  farbkonv fw[256];
  unsigned char px,* vdat;
  struct f_mem * mmout;

  /* width and height */
  if (memread(buf,sizeof(char),5,mmin)<5) {
    fprintf(stderr,"vgatobmp: short read of input-file.\n");
    return(-1);
  }
  buf[5]='\0';
  bw=atoi(buf);
  if (memread(buf,sizeof(char),5,mmin)<5) {
    fprintf(stderr,"vgatobmp: short read of input-file.\n");
    return(-1);
  }
  buf[5]='\0';
  bh=atoi(buf);
  if (bw<=0 || bh<=0 || bw*bh/bh!=bw) {
    fprintf(stderr,"vgatobmp: invalid width/height (%d/%d) size of input-file.\n",bw,bh);
    return(-1);
  }
  if ((vdat=malloc(sizeof(unsigned char)*(bw*bh)))==NULL) {
    fprintf(stderr,"vgatobmp: malloc: %s\n",strerror(errno));
    return(-1);
  }

  if ((mmout=newmem())==NULL) {return(-1);}
  bz=bw; if (bw%4) {bz+=4-bw%4;}
  memwrite("BM",sizeof(char),2,mmout);  /* 0x00 */

  _c2i.iw=0x436+bh*bz;
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x02: filesize in bytes */

  _c2h.hw=0;
#ifdef ENDIAN_IS_BIG
  _c2ht.cw[0]=_c2h.cw[1]; _c2ht.cw[1]=_c2h.cw[0];
  _c2h.hw=_c2ht.hw;
#endif
  memwrite(&_c2h.hw,2,1,mmout);  /* 0x06: must be 0 */
  memwrite(&_c2h.hw,2,1,mmout);  /* 0x08: must be 0 */

  _c2i.iw=0x436;
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x0a: offset to data */

  _c2i.iw=0x28;
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x0e: info-header size */

  _c2i.iw=bw;
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x12: width of bitmap */

  _c2i.iw=bh;
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x16: height of bitmap */

  _c2h.hw=1;
#ifdef ENDIAN_IS_BIG
  _c2ht.cw[0]=_c2h.cw[1]; _c2ht.cw[1]=_c2h.cw[0];
  _c2h.hw=_c2ht.hw;
#endif
  memwrite(&_c2h.hw,2,1,mmout);  /* 0x1a: must be 1 */

  _c2h.hw=8;
#ifdef ENDIAN_IS_BIG
  _c2ht.cw[0]=_c2h.cw[1]; _c2ht.cw[1]=_c2h.cw[0];
  _c2h.hw=_c2ht.hw;
#endif
  memwrite(&_c2h.hw,2,1,mmout);  /* 0x1c: bits per pixel */

  _c2i.iw=0;
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x1e: compression */

  _c2i.iw=bh*bz;
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x22: datasize in bytes */

  _c2i.iw=0;
#ifdef ENDIAN_IS_BIG
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x26: may be 0 */
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x2a: may be 0 */
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x2e: may be 0 */
  memwrite(&_c2i.iw,4,1,mmout);  /* 0x32: may be 0 */


  for (i1=0;i1<256;i1++) {  /* read in colormap */
    memread(&fw[i1].r,sizeof(unsigned char),1,mmin);  /* red byte */
    fw[i1].r<<=2;
    memread(&fw[i1].g,sizeof(unsigned char),1,mmin);  /* green byte */
    fw[i1].g<<=2;
    memread(&fw[i1].b,sizeof(unsigned char),1,mmin);  /* blue byte */
    fw[i1].b<<=2;
  }
  px=0;
  for (i1=0;i1<256;i1++) {  /* write out colormap */
    memwrite(&fw[i1].b,sizeof(unsigned char),1,mmout);  /* blue byte */
    memwrite(&fw[i1].g,sizeof(unsigned char),1,mmout);  /* green byte */
    memwrite(&fw[i1].r,sizeof(unsigned char),1,mmout);  /* red byte */
    memwrite(&px,sizeof(unsigned char),1,mmout);        /* 0 */
  }


  if (memread(vdat,sizeof(unsigned char),bw*bh,mmin)!=bw*bh) {
    fprintf(stderr,"vgatobmp: short read of input-file.\n");
    return(-1);
  }

  px=0;
  for (i1=bh-1;i1>=0;i1--) {
    memwrite(vdat+i1*bw,sizeof(unsigned char),bw,mmout);
    for (i2=bw;i2<bz;i2++) {memwrite(&px,sizeof(unsigned char),1,mmout);}
  }
  free(vdat);

  fwrite(mmout->data,1,mmout->len,ffout);
  fflush(ffout);
  freemem(&mmout);
  return(0);
} /* Ende vgatobmp */


static int vgatoppm(struct f_mem * mmin,FILE * ffout,int ppmart) {
/* mmin is at pos 2, output to ffout */
  char buf[128];
  int bw,bh,i1,i2,zlz;
  farbkonv fw[256];
  unsigned char px,rgb;
  struct f_mem * mmout;

  /* width and height */
  if (memread(buf,sizeof(char),5,mmin)<5) {
    fprintf(stderr,"vgatoppm: short read of input-file.\n");
    return(-1);
  }
  buf[5]='\0';
  bw=atoi(buf);
  if (memread(buf,sizeof(char),5,mmin)<5) {
    fprintf(stderr,"vgatoppm: short read of input-file.\n");
    return(-1);
  }
  buf[5]='\0';
  bh=atoi(buf);

  for (i1=0;i1<256;i1++) {  /* colormap */
    memread(&fw[i1].r,sizeof(char),1,mmin);  /* red byte */
    memread(&fw[i1].g,sizeof(char),1,mmin);  /* green byte */
    memread(&fw[i1].b,sizeof(char),1,mmin);  /* blue byte */
  }

  if ((mmout=newmem())==NULL) {return(-1);}
  snprintf(buf,sizeof(buf),"P%d\n%d %d\n255\n",ppmart,bw,bh);
  memwrite(buf,sizeof(char),strlen(buf),mmout);
  for (i1=0;i1<bh;i1++) {
    zlz=0;
    for (i2=0;i2<bw;i2++) {
      if (memread(&px,sizeof(unsigned char),1,mmin)<1) {
        fprintf(stderr,"vgatoppm: short read of input-file.\n");
        return(-1);
      }
      if (ppmart==3) {  /* plain PPM */
        snprintf(buf,sizeof(buf),"%s%3d %3d %3d",zlz==0?"":"   ",fw[(int)px].r<<2,fw[(int)px].g<<2,fw[(int)px].b<<2);
        memwrite(buf,sizeof(char),strlen(buf),mmout);
        if (++zlz==5) {memwrite("\n",1,1,mmout); zlz=0;}
      } else {  /* normal (raw) PPM */
        rgb=fw[(int)px].r<<2;
        memwrite(&rgb,sizeof(unsigned char),1,mmout);
        rgb=fw[(int)px].g<<2;
        memwrite(&rgb,sizeof(unsigned char),1,mmout);
        rgb=fw[(int)px].b<<2;
        memwrite(&rgb,sizeof(unsigned char),1,mmout);
      }
    }
    if (zlz>0) {memwrite("\n",1,1,mmout);}
  }

  fwrite(mmout->data,1,mmout->len,ffout);
  fflush(ffout);
  freemem(&mmout);
  return(0);
} /* Ende vgatoppm */
