/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* vgag-bitmap.c: convert-program from and to vgagames bitmap format */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "vgagames2.h"
#include "vga_conv.h"


char * arg0,* inpfile;

int vg_colm(FILE *,FILE *);


int main(int argc,char ** argv) {
  int i1,gfmt,showcol,erg=0;
  FILE * ffin,* ffout;
  gfmt=showcol=0;
  ffout=stdout;
  if ((arg0=strrchr(argv[0],'/'))==NULL) {arg0=argv[0];} else {arg0++;}

  opterr=opterr?opterr:0;
  while ((i1=getopt(argc,argv,"+o:f:ch"))!=-1) {
    switch(i1) {
      case 'o':
        if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
        if (strcmp(optarg,"-")!=0) {
          unlink(optarg);
          if ((ffout=fopen(optarg,"w"))==NULL) {
            fprintf(stderr,"%s: Cannot write \"%s\": %s\n",arg0,optarg,strerror(errno));
            exit(1);
          }
        }
        break;
      case 'f':
        if (strcmp(optarg,"vga")==0) {
          gfmt=VGAFORMAT_VGA;
        } else if (strcmp(optarg,"bmp")==0) {
          gfmt=VGAFORMAT_BMP;
        } else if (strcmp(optarg,"ppm3")==0) {
          gfmt=VGAFORMAT_PPM3;
        } else if (strcmp(optarg,"ppm6")==0) {
          gfmt=VGAFORMAT_PPM6;
        }
        break;
      case 'c': showcol=1; break;
      case 'h':
        fprintf(stderr,"\n");
        fprintf(stderr,"1. usage: %s -f <to-format> [-o <outputfile>] <inputfile>\n",arg0);
        fprintf(stderr,"convert bitmap formats:\n");
        fprintf(stderr,"  -o <outputfile>  output filename or \"-\" for stdout\n");
        fprintf(stderr,"  -f <to-format>   to which format to convert bitmap:\n");
        fprintf(stderr,"                     \"vga\":  convert to vgagames bitmap\n");
        fprintf(stderr,"                     \"bmp\":  convert to windows bitmap\n");
        fprintf(stderr,"                     \"ppm3\": convert to plain PPM\n");
        fprintf(stderr,"                     \"ppm6\": convert to normal (raw) PPM\n");
        fprintf(stderr,"  <inputfile> may be of any format mentioned above,\n");
        fprintf(stderr,"              for reading from stdin use: -\n");
        fprintf(stderr,"\n");
        fprintf(stderr,"2. usage: %s -c <inputfile>\n",arg0);
        fprintf(stderr,"give out colormap:\n");
        fprintf(stderr,"  <inputfile> must be of the vgagames format.\n");
        fprintf(stderr,"\n");
        fprintf(stderr,"Example:\n");
        fprintf(stderr,"  Convert windows bitmap g.bmp to g.vga:\n");
        fprintf(stderr,"    %s -o g.vga g.bmp\n",arg0);
        fprintf(stderr,"  Convert g.vga to normal PPM g.ppm:\n");
        fprintf(stderr,"    %s -f ppm6 -o g.ppm g.vga\n",arg0);
        fprintf(stderr,"  Give out colormap of g.vga:\n");
        fprintf(stderr,"    %s -c g.vga\n",arg0);
        fprintf(stderr,"\n");
        if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
        exit(1);
      default:
        fprintf(stderr,"Try -h for help.\n");
        if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
        exit(1);
    }
  }
  if (gfmt && showcol) {
    fprintf(stderr,"%s: use of -f with -c not valid. Try -h for help.\n",arg0);
    if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
    exit(1);
  }
  if (optind>=argc) {
    fprintf(stderr,"%s: no input file specified. Try -h for help.\n",arg0);
    if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
    exit(1);
  }
  inpfile=argv[optind];
  if (strcmp(inpfile,"-")!=0) {
    if ((ffin=fopen(inpfile,"r"))==NULL) {
      fprintf(stderr,"%s: Cannot read input-file \"%s\": %s\n",arg0,inpfile,strerror(errno));
      if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
      exit(1);
    }
  } else {ffin=stdin;}

  if (showcol) {  /* show only colormap */
    char magic[4];
    if (fread(magic,sizeof(char),2,ffin)!=2) {
      fprintf(stderr,"%s: Input-file \"%s\" is no graphic file\n",arg0,inpfile);
      fclose(ffin);
      if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
      exit(1);
    }
    if (strncmp(magic,"VG",2)!=0) {
      fprintf(stderr,"%s: Input-file \"%s\" is no VgaGames format bitmap\n",arg0,inpfile);
      fclose(ffin);
      if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
      exit(1);
    }
    erg=vg_colm(ffin,ffout);
    fclose(ffin);
    if (ffout!=stdout) {fclose(ffout); ffout=stdout;}

  } else {  /* convert */
    FILE * tfp;
    if (ffout==stdout) {
      if ((tfp=tmpfile())==NULL) {
        fprintf(stderr,"%s: creating tmpfile: %s\n",arg0,strerror(errno));
        fclose(ffin);
        if (ffout!=stdout) {fclose(ffout); ffout=stdout;}
        exit(1);
      }
    } else {tfp=ffout;}
    if ((ffin=convtovga(ffin))!=NULL) {rewind(ffin);}
    erg=convfromvga(ffin,tfp,gfmt);
    if (erg==0 && ffout==stdout) {
      char buf[1024];
      size_t anz;
      rewind(tfp);
      while ((anz=fread(buf,sizeof(char),sizeof(buf),tfp))>0) {
        fwrite(buf,sizeof(char),anz,ffout);
      }
      fflush(ffout);
    }
    fclose(tfp); ffout=stdout;
  }
  if (erg<0) {fprintf(stderr,"%s: error input-file \"%s\".\n",arg0,inpfile); exit(1);}
  exit(0);
} /* Ende main */


int vg_colm(FILE * ffin,FILE * ffout) {
/* ffin is at pos 2, output to ffout */
  size_t anz;
  char buf[1024],* p1;
  int w,h;
  /* width */
  if (fread(buf,sizeof(char),5,ffin)!=5) {
    fprintf(stderr,"%s: short read of input-file \"%s\"\n",arg0,inpfile);
    return(-1);
  } else {buf[5]='\0'; w=atoi(buf);}
  /* height */
  if (fread(buf,sizeof(char),5,ffin)!=5) {
    fprintf(stderr,"%s: short read of input-file \"%s\"\n",arg0,inpfile);
    return(-1);
  } else {buf[5]='\0'; h=atoi(buf);}
  if ((anz=fread(buf,sizeof(char),256*3,ffin))==256*3) {
    int i1;
    if ((p1=strrchr(inpfile,'/'))!=NULL) {p1++;} else {p1=inpfile;}
    fprintf(ffout,"# Colormap for %s (%dx%d)\n\n",p1,w,h);
    for (i1=0;i1<256;i1++) {
      fprintf(ffout,"%d=%d %d %d\n",i1,buf[i1*3],buf[i1*3+1],buf[i1*3+2]);
    }
  } else {
    fprintf(stderr,"%s: short read of input-file \"%s\"\n",arg0,inpfile);
    return(-1);
  }
  fflush(ffout);
  return(0);
} /* Ende vg_colm */
