/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* vgag-sound.c: multichannel sound server
** commands are read from stdin and responses are given out to stdout
** stderr is output for verbose output
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include "config.h"
#include "cwdir.h"
#include "inthash.h"
#include "vgag-sound.h"
#include "extsoundserver.h"
#include "ualaw.h"

#define MAX_INSTANCES 5  /* how often the same file may play simultaneously */

#define NO_INR

#ifndef S_ISLNK
# define S_ISLNK(mode) (((mode) & S_IFMT) == S_IFLNK)
#endif

#define GROUP_ALL  "ALL"

extern int fnpatt(const char *,const char *);

char * arg0;
char workpath[512],cwdir[512];
unsigned char * frg_uc;
int * frg_si;
int paused;
uid_t saved_uid;
gid_t saved_gid;

#define BLOCK_LEN  (PIPE_BUFFERSIZE>4096?PIPE_BUFFERSIZE:4096)

struct sound_dev * sdptr;

struct {  /* sound files */
  int volcalc;  /* whether to recalculate volume */
  int sn_max;  /* number of struct *sn */
  int gn_max;  /* number of struct *gn */
  struct {  /* s_name parameters */
    char s_name[16];  /* s_name without instance-number */
#define FEXT_WAV 1
#define FEXT_AU  2
#define FEXT_MID 3
#define FEXT_MP3 4
    int fext;  /* extension (FEXT_*) */
    char * filename;  /* pathname */
    int sndbegin;  /* file position of beginning sound-data */
#define PCMCODE_LINEAR 1
#define PCMCODE_MULAW  2
#define PCMCODE_ALAW   3
    int pcmcode;  /* PCMCODE_* */
    int bitsamp;  /* 8 or 16 */
    int channel;  /* 1=mono or 2=stereo */
    int samrate;  /* sample rate */
    int volpc;  /* volumepercent */
    int volume;  /* volume 0-200 (step 2) */
    int imax;  /* highest number of inr */
    struct s_instc {  /* instances */
      int inr;  /* instance-number */
      int status;  /* 0=stopped, 1=playing, 2=paused-at-end, 3=wait-for-cont */
      int paused;  /* 1=paused, 0=running */
      int cr_pc,cr_val;  /* in-/decreasing percent and actual percent value */
      int loop;  /* how many loops left (0=infinite) */
      int bitsamp;  /* copy of sf.sn[].bitsamp */
      int channel;  /* copy of sf.sn[].channel */
      int samrate;  /* copy of sf.sn[].samrate */
      FILE * ffp;  /* file/pipe pointer */
      unsigned char data[BLOCK_LEN];  /* data buffer */
      int dpos,dend;  /* actual position in data, last position in data */
      struct s_instc * nptr;  /* linked list */
    } * nr;
  } * sn;
  struct {  /* g_name parameters */
    char g_name[16];  /* group-name */
    int volume;  /* volume 0-200 (step 2) */
    char ** s_name;  /* list of s_names belonging to group */
  } * gn;
  struct h_instc {  /* instances for hidden s_name */
    char s_name[16];  /* s_name without instance-number (because always 1) */
    int fext;  /* extension (FEXT_*) */
    char * filename;  /* pathname */
    int sndbegin;  /* file position of beginning sound-data */
    int pcmcode;  /* PCMCODE_* */
    int bitsamp;  /* 8 or 16 */
    int channel;  /* 1=mono or 2=stereo */
    int samrate;  /* sample rate */
    int volpc;  /* volumepercent */
    int status;  /* 0=stopped, 1=playing */
    int howloop;  /* 0=no loop, 1=infinite loops */
    FILE * ffp;  /* file/pipe pointer */
    unsigned char data[BLOCK_LEN];  /* data buffer */
    int dpos,dend;  /* actual position in data, last position in data */
    struct h_instc * nptr;  /* linked list */
  } * hn;
  struct ihash ** ihs;  /* hash for index of s_name and g_name */
} sf;

void do_alrm(int);
FILE * start_sound(const char *,int,int,int *,int *,int *);
void free_sf(void);
void cmd_atta(const char *,const char *,int,int);
void cmd_copy(const char *,const char *);
void cmd_play(const char *,int,int,int);
void cmd_deta(const char *);
void cmd_wrapper(const char *,int,const char *);
void cmd_paus(const char *,int,void *);
void cmd_cont(const char *,int,void *);
void cmd_pate(const char *,int,void *);
void cmd_catb(const char *,int,void *);
void cmd_stop(const char *,int,void *);
void cmd_gclr(const char *);
void cmd_gadd(const char *,const char *);
void cmd_gdel(const char *,const char *);
void cmd_volm(const char *,int);
void cmd_frgs(int,int,int,int);
void output_block(void);
int fill_fragment(double,int,int,int,int,int,FILE *,unsigned char *,int *,int *);
int calc_sfakt(int);
unsigned char * read_data(int,FILE *,unsigned char *,int *,int *);
int read_sounddata(int,void *,int);
double volofsound(int);
void helptopic(const char *);


int main(int argc,char ** argv) {
  int i1,srate,channel;
  char verbosefile[512],* p1,* p2,* p3;
  char comline[512],* pcom,comarg[4][512];

  if (get_cwdir(argv[0],cwdir,sizeof(cwdir))<0) {
    fprintf(stderr,"%s: error calling get_cwdir.\n",argv[0]);
    exit(1);
  }
  if ((arg0=strrchr(argv[0],'/'))==NULL) {arg0=argv[0];} else {arg0++;}
  srate=22050;
  channel=2;
  *workpath=*verbosefile='\0';
  memset(&sf,0,sizeof(sf));
  saved_uid=geteuid();
  saved_gid=getegid();

  opterr=opterr?opterr:0;
  while ((i1=getopt(argc,argv,"+c:r:p:v:h"))!=-1) {
    switch(i1) {
      case 'c':
        channel=atoi(optarg);
        break;
      case 'r':
        srate=atoi(optarg);
        break;
      case 'p':
        snprintf(workpath,sizeof(workpath),"%s",optarg);
        break;
      case 'v':
        snprintf(verbosefile,sizeof(verbosefile),"%s",optarg);
        break;
      case 'h':
        fprintf(stderr,"\nUsage: %s <options>\n",arg0);
        fprintf(stderr,"Sound server for VgaGames2.\n");
        fprintf(stderr,"Options:\n");
        fprintf(stderr,"  -p <path>         path from where sound files are to be loaded\n");
        fprintf(stderr,"                    (if missing \".\" is used).\n");
        fprintf(stderr,"  -r <sample rate>  sample rate (11025, 22050, 44100)\n");
        fprintf(stderr,"                    (if missing \"22050\" is used).\n");
        fprintf(stderr,"  -c <channels>     1=mono or 2=stereo\n");
        fprintf(stderr,"                    (if missing \"2\" (stereo) is used).\n");
        fprintf(stderr,"  -v <file>         file or \"-\" (stderr) for verbose output\n");
        fprintf(stderr,"                    (no verbose output if missing).\n");
        fprintf(stderr,"\n");
        fprintf(stderr,"Supported formats:\n");
        fprintf(stderr,"  - Wave (PCM, mu-law, a-law)\n");
        fprintf(stderr,"  - Sun/NeXT audio (PCM, mu-law, a-law)\n");
        fprintf(stderr,"  - MIDI\n");
        fprintf(stderr,"  - MP3\n\n");
        fprintf(stderr,"Compiled for: < ");
        for (sdptr=s_dev;sdptr->open_dev!=NULL;sdptr++) {
          fprintf(stderr,"%s ",sdptr->libname);
        }
        fprintf(stderr,">\n");
        fprintf(stderr,"One of these can be explicitly set by the environment variable SOUND_DEV,\n");
        if (s_dev[0].open_dev!=NULL) {fprintf(stderr,"e.g. SOUND_DEV=%s\n",s_dev[0].libname);}
        fprintf(stderr,"\n");
        exit(1);
      default:
        fprintf(stderr,"Try -h for help.\n");
        exit(1);
    }
  }
  if ((channel<1) || (channel>2)) {channel=1;}
  if (srate<11025+11025/2) {srate=11025;}
  else if (srate>22050+22050/2) {srate=44100;}
  else {srate=22050;}
  if (*workpath=='\0') {strlcpy(workpath,".",sizeof(workpath));}
  if (*verbosefile!='\0') {
    if (strcmp(verbosefile,"-")!=0) {
      struct stat sbuf;
      int obda;
      if (*verbosefile!='/') {
        char bb[512];
        snprintf(bb,sizeof(bb),"%s/%s",workpath,verbosefile);
        snprintf(verbosefile,sizeof(verbosefile),"%s",bb);
      }
      if (((obda=access(verbosefile,F_OK))<0) || (access(verbosefile,W_OK)==0)) {
        if ((lstat(verbosefile,&sbuf)==0) && (S_ISLNK(sbuf.st_mode))) {unlink(verbosefile);}
        if (freopen(verbosefile,"w",stderr)!=NULL) {
          if (obda<0) {fchown(STDERR_FILENO,getuid(),getgid());}
        }
      } else {
        fprintf(stderr,"Cannot write file \"%s\".\n",verbosefile);
        freopen("/dev/null","w",stderr);
      }
    }
  } else {freopen("/dev/null","w",stderr);}
  fprintf(stderr,"Soundserver for VgaGames2.\n");
  fprintf(stderr,"Working path: \"%s\"\n",workpath);
  fprintf(stderr,"Supported sound devices: (");
  for (sdptr=s_dev;sdptr->open_dev!=NULL;sdptr++) {
    fprintf(stderr," %s ",sdptr->libname);
  }
  fprintf(stderr,")\n");

  {struct sigaction sa;
   memset(&sa,0,sizeof(sa));
   sa.sa_handler=SIG_DFL;
   sigaction(SIGPIPE,&sa,NULL);
   sa.sa_handler=do_alrm;
   sigaction(SIGALRM,&sa,NULL);
  }

  /* open device and fall back to original user */
  extsoundserver(0);  /* stop external sound servers */
  if ((p1=getenv("SOUND_DEV"))!=NULL && *p1=='\0') {p1=NULL;}
  for (sdptr=s_dev;sdptr->open_dev!=NULL;sdptr++) {
    if (p1!=NULL && strcasecmp(p1,sdptr->libname)!=0) {
      fprintf(stderr,"skipping device \"%s\" ...\n",sdptr->libname);
      continue;
    }
    fprintf(stderr,"trying device \"%s\" ...\n",sdptr->libname);
    if (sdptr->open_dev(sdptr,srate,channel)==0) {
      fprintf(stderr,"... device \"%s\" OK\n",sdptr->libname);
      break;
    }
    fprintf(stderr,"... device \"%s\" FAILED\n",sdptr->libname);
  }
  if (sdptr->open_dev==NULL) {
    fprintf(stderr,"No device could be opened\n");
    extsoundserver(1);  /* resume external sound servers */
    fclose(stderr);
    exit(1);
  }
  for (i1=1;(sdptr->frag_size>>i1)>0;i1++) {;}
  if ((sdptr->frag_size=(1<<(i1-1)))<128) {sdptr->frag_size=128;}
  if (((frg_uc=malloc(sizeof(unsigned char)*sdptr->frag_size))==NULL) \
  || ((frg_si=malloc(sizeof(int)*sdptr->frag_size))==NULL)) {
    fprintf(stderr,"%s: malloc: %s\n",arg0,strerror(errno));
    sdptr->mixer_dev(sdptr,-2);
    sdptr->close_dev(sdptr);
    extsoundserver(1);  /* resume external sound servers */
    fclose(stderr);
    exit(1);
  }
  if ((sf.ihs=inthash_new(503))==NULL) {
    sdptr->mixer_dev(sdptr,-2);
    sdptr->close_dev(sdptr);
    extsoundserver(1);  /* resume external sound servers */
    fclose(stderr);
    exit(1);
  }
  if ((i1=sdptr->mixer_dev(sdptr,-1))>=0) {
    sdptr->mixer_dev(sdptr,i1);
  }
  paused=0;
  setegid(getgid());
  seteuid(getuid());

  fcntl(STDIN_FILENO,F_SETFL,fcntl(STDIN_FILENO,F_GETFL,0)|O_NONBLOCK);
  *(pcom=comline)='\0';
  fprintf(stderr,"Sample rate=%d, channels=%d, %d Bit, Fragments=%d, Fragment-Size=%d\n",sdptr->srate,sdptr->channel,sdptr->bitsamp,sdptr->frag_anz,sdptr->frag_size);
  fprintf(stderr,"\n");
  helptopic(NULL);
  fprintf(stderr,"> ");
  fflush(stderr);
  for (i1=0;i1<99;i1++) {  /* close-on-exec for every non default file */
    if ((i1==STDIN_FILENO)||(i1==STDOUT_FILENO)||(i1==STDERR_FILENO)) {continue;}
    fcntl(i1,F_SETFD,fcntl(i1,F_GETFD,0)|FD_CLOEXEC);
  }

  p1=NULL;
  while (1) {  /* server loop */

    if (p1==NULL) {
      /* read command line */
      i1=sizeof(comline)-(pcom-comline);
      while ((i1=read(STDIN_FILENO,pcom,i1))>0) {
        if ((p1=memchr(pcom,'\n',i1))!=NULL) {pcom+=i1; break;}
        pcom+=i1;
        if ((i1=sizeof(comline)-(pcom-comline))<1) {  /* line too long, discard */
          fprintf(stderr,"%s: line discarded: \"%.*s\"\n",arg0,(int)sizeof(comline),comline);
          pcom=comline; i1=sizeof(comline);
        }
      }
      if ((i1==-1) && (errno!=EAGAIN)) {
        fprintf(stderr,"%s: read stdin: %s\n",arg0,strerror(errno));
        sdptr->mixer_dev(sdptr,-2);
        sdptr->close_dev(sdptr);
        extsoundserver(1);  /* resume external sound servers */
        free_sf();
        fclose(stderr);
        exit(1);
      }
      if (i1==0) {break;}  /* stdin closed */
    }

    if (p1!=NULL) {  /* command line ready */
      *p1='\0';
      /* parse line into max. 4 whitespace-separated parts */
      *comarg[0]=*comarg[1]=*comarg[2]=*comarg[3]='\0';
      p2=comline; if ((p3=strchr(p2,' '))!=NULL) {
        *p3='\0'; memmove(comarg[0],p2,p3-p2+1);
        p2=p3+1; if ((p3=strchr(p2,' '))!=NULL) {
          *p3='\0'; memmove(comarg[1],p2,p3-p2+1);
          p2=p3+1; if ((p3=strchr(p2,' '))!=NULL) {
            *p3='\0'; memmove(comarg[2],p2,p3-p2+1);
            memmove(comarg[3],p3+1,strlen(p3+1)+1);
          } else {memmove(comarg[2],p2,strlen(p2)+1);}
        } else {memmove(comarg[1],p2,strlen(p2)+1);}
      } else {memmove(comarg[0],p2,strlen(p2)+1);}
      /* delete line from comline */
      if (p1+1<pcom) {memmove(comline,p1+1,pcom-(p1+1));}
      pcom=comline+(pcom-(p1+1));
      if (pcom>comline) {p1=memchr(comline,'\n',(size_t)(pcom-comline));} else {p1=NULL;}

      if (*comarg[0]!='\0') {
        /* execute command */
        fprintf(stderr,"\nGot command: \"%s <%s> <%s> <%s>\"\n",comarg[0],comarg[1],comarg[2],comarg[3]);
        if (strcmp(comarg[0],"HELP")==0) {
          helptopic(comarg[1]);
        } else if (strcmp(comarg[0],"ATTA")==0) {
          cmd_atta(comarg[1],comarg[2],atoi(comarg[3]),0);
        } else if (strcmp(comarg[0],"_ATTA")==0) {
          cmd_atta(comarg[1],comarg[2],atoi(comarg[3]),1);
        } else if (strcmp(comarg[0],"_COPY")==0) {
          cmd_copy(comarg[1],comarg[2]);
        } else if (strcmp(comarg[0],"PLAY")==0) {
          cmd_play(comarg[1],atoi(comarg[2]),atoi(comarg[3]),0);
        } else if (strcmp(comarg[0],"_PLAY")==0) {
          cmd_play(comarg[1],atoi(comarg[2]),atoi(comarg[3]),1);
        } else if (strcmp(comarg[0],"_DETA")==0) {
          cmd_deta(comarg[1]);
        } else if (strcmp(comarg[0],"PAUS")==0) {
          cmd_wrapper(comarg[1],0,comarg[0]);
        } else if (strcmp(comarg[0],"CONT")==0) {
          cmd_wrapper(comarg[1],0,comarg[0]);
        } else if (strcmp(comarg[0],"PATE")==0) {
          cmd_wrapper(comarg[1],0,comarg[0]);
        } else if (strcmp(comarg[0],"CATB")==0) {
          cmd_wrapper(comarg[1],0,comarg[0]);
        } else if (strcmp(comarg[0],"STOP")==0) {
          cmd_wrapper(comarg[1],atoi(comarg[2]),comarg[0]);
        } else if (strcmp(comarg[0],"GCLR")==0) {
          cmd_gclr(comarg[1]);
        } else if (strcmp(comarg[0],"GADD")==0) {
          cmd_gadd(comarg[1],comarg[2]);
        } else if (strcmp(comarg[0],"GDEL")==0) {
          cmd_gdel(comarg[1],comarg[2]);
        } else if (strcmp(comarg[0],"VOLM")==0) {
          cmd_volm(comarg[1],*comarg[2]=='\0'?-1:atoi(comarg[2]));
        } else if (strcmp(comarg[0],"FRGS")==0) {
          int iw[4];
          char * pv=comarg[1],* pw;
          iw[0]=iw[1]=iw[2]=iw[3]=0;
          if ((pw=strchr(pv,','))!=NULL) {iw[0]=atoi(pv); pv=pw+1;}
          if ((pw=strchr(pv,','))!=NULL) {iw[1]=atoi(pv); pv=pw+1;}
          if ((pw=strchr(pv,','))!=NULL) {iw[2]=atoi(pv); pv=pw+1; iw[3]=atoi(pv);}
          cmd_frgs(iw[0],iw[1],iw[2],iw[3]);
        } else if (strcmp(comarg[0],"????")==0) {
          {struct s_instc * ilm;
           int ii;
           fprintf(stderr,"S_NAME:\n");
           for (ii=0;ii<sf.sn_max;ii++) {
             fprintf(stderr,"- %d: \"%s\" [%s]\n",ii,sf.sn[ii].s_name,sf.sn[ii].filename);
             for (ilm=sf.sn[ii].nr;ilm!=NULL;ilm=ilm->nptr) {
               fprintf(stderr,"    inr=%d status=%d paused=%d ffp=%p\n",ilm->inr,ilm->status,ilm->paused,ilm->ffp);
             }
           }
           fprintf(stderr,"\n");
          }
          {struct h_instc * ilm;
           fprintf(stderr,"H_NAME:\n");
           for (ilm=sf.hn;ilm!=NULL;ilm=ilm->nptr) {
             fprintf(stderr,"- \"%s\" [%s]: status=%d ffp=%p\n",ilm->s_name,ilm->filename,ilm->status,ilm->ffp);
           }
           fprintf(stderr,"\n");
          }
          {char ** pptr;
           int ii;
           fprintf(stderr,"G_NAME:\n");
           for (ii=0;ii<sf.gn_max;ii++) {
             fprintf(stderr,"- %d: \"%s\"\n",ii,sf.gn[ii].g_name);
             for (pptr=sf.gn[ii].s_name;*pptr!=NULL;pptr++) {
               fprintf(stderr,"    s_name=%s\n",*pptr);
             }
           }
           fprintf(stderr,"\n");
          }
        } else {
          fprintf(stderr,"Command not understood. Try \"HELP\".\n");
        }
      }
      fprintf(stderr,"> ");
      fflush(stderr);
    }

    /* read evtl. sound blocks, merge it and write it to the device */
    output_block();
  }  /* end server loop */

  /* end server */
  sdptr->mixer_dev(sdptr,-2);
  sdptr->close_dev(sdptr);
  free(frg_uc);
  free(frg_si);
  free_sf();
  extsoundserver(1);  /* resume external sound servers */
  fprintf(stderr,"Exit server.\n"); fflush(stderr);
  fclose(stderr);
  exit(0);
} /* Ende main */


void do_alrm(int signum) {;}


FILE * start_sound(const char * filename,int fext,int sndbegin,int * bitsamp,int * channel,int * samrate) {
  FILE * ffp=NULL;
  char buf[1024];
  int i1;
  if ((filename==NULL) || (*filename=='\0')) {
    fprintf(stderr,"PLAY: no filename\n");
    return(NULL);
  }
  if ((fext==FEXT_WAV) || (fext==FEXT_AU)) {
    if ((ffp=fopen(filename,"r"))==NULL) {
      fprintf(stderr,"PLAY: fopen \"%s\": %s\n",filename,strerror(errno));
      return(NULL);
    }
    if (fseek(ffp,(long)sndbegin,SEEK_SET)<0) {
      fprintf(stderr,"PLAY: fseek \"%s\": %s\n",filename,strerror(errno));
      return(NULL);
    }
  } else if (fext==FEXT_MID) {
    i1=snprintf(buf,sizeof(buf),"%s/vgag-midi -Or%c -s%d -o - %s 2> /dev/null",cwdir,sdptr->channel==1?'M':'S',sdptr->srate,filename);
    if ((i1==-1) || (i1>=(int)sizeof(buf))) {
      fprintf(stderr,"PLAY: \"%s\": programmpath too long\n",filename);
      return(NULL);
    }
    if ((ffp=popen(buf,"r"))==NULL) {
      fprintf(stderr,"PLAY: popen \"%s\": %s\n",filename,strerror(errno));
      return(NULL);
    }
    *channel=sdptr->channel;
    *samrate=sdptr->srate;
  } else if (fext==FEXT_MP3) {
    i1=snprintf(buf,sizeof(buf),"%s/vgag-mp3 %s %s -r %d -q -s %s 2> /dev/null",cwdir,sdptr->bitsamp==8?"--8bit":"",sdptr->channel==1?"--mono":"--stereo",sdptr->srate,filename);
    if ((i1==-1) || (i1>=(int)sizeof(buf))) {
      fprintf(stderr,"PLAY: \"%s\": programmpath too long\n",filename);
      return(NULL);
    }
    if ((ffp=popen(buf,"r"))==NULL) {
      fprintf(stderr,"PLAY: popen \"%s\": %s\n",filename,strerror(errno));
      return(NULL);
    }
    *bitsamp=sdptr->bitsamp;
    *channel=sdptr->channel;
    *samrate=sdptr->srate;
  } else {
    fprintf(stderr,"PLAY: \"%s\": unknown extension: %d\n",filename,fext);
    return(NULL);
  }
  fcntl(fileno(ffp),F_SETFD,fcntl(fileno(ffp),F_GETFD,0)|FD_CLOEXEC);
  return(ffp);
} /* Ende start_sound */


void free_sf() {
  int idx;
  if (sf.ihs!=NULL) {inthash_free(sf.ihs);}
  {struct s_instc * ielm,* nelm;
   for (idx=0;idx<sf.sn_max;idx++) {
     for (ielm=sf.sn[idx].nr;ielm!=NULL;ielm=nelm) {
       nelm=ielm->nptr;
       free(ielm);
     }
     if (sf.sn[idx].filename!=NULL) {free(sf.sn[idx].filename);}
   }
  }
  {char ** pptr;
   for (idx=0;idx<sf.gn_max;idx++) {
     for (pptr=sf.gn[idx].s_name;*pptr!=NULL;pptr++) {free(*pptr);}
     free(sf.gn[idx].s_name);
   }
  }
  {struct h_instc * ielm,* nelm;
   for (ielm=sf.hn;ielm!=NULL;ielm=nelm) {
     nelm=ielm->nptr;
     if (ielm->filename!=NULL) {free(ielm->filename);}
     free(ielm);
   }
  }
} /* Ende free_sf */


void cmd_atta(const char * filename,const char * s_name,int volpc,int flag) {
  FILE * ffp;
  char pathname[1024];
  const char * kptr;
  int fext,pcmcode,bitsamp,channel,samrate;
  unsigned int sndbegin,sndlen;
  size_t s1;

  if ((filename==NULL) || (*filename=='\0') || (s_name==NULL) || ((int)strlen(s_name)>15)) {
    fprintf(stderr,"ATTA: invalid parameter\n");
    return;
  }
  if (volpc<0) {volpc=0;} else if (volpc>200) {volpc=200;}
  if (strchr(s_name,'#')!=NULL) {
    fprintf(stderr,"ATTA: instance-number not valid here\n");
    return;
  }
  if (strspn(s_name,"#0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-")!=strlen(s_name)) {
    fprintf(stderr,"ATTA: invalid character \"%s\"\n",s_name);
    return;
  }
  if (((flag==0) && (strncmp(s_name,"s-",2)!=0)) || ((flag) && (strncmp(s_name,"h-",2)!=0))) {
    fprintf(stderr,"ATTA: s_name \"%s\" does not begin with \"s-\"\n",s_name);
    return;
  }
  if ((flag==0) && (inthash_get(sf.ihs,s_name,NULL)!=-1)) {
    fprintf(stderr,"ATTA: \"%s\" already attached\n",s_name);
    return;
  }

  /* get file extension */
  fext=0;
  if ((kptr=strrchr(filename,'.'))!=NULL) {
    if ((strcasecmp(kptr,".wav")==0) || (strcasecmp(kptr,".wave")==0)) {
      fext=FEXT_WAV;
    } else if (strcasecmp(kptr,".au")==0) {
      fext=FEXT_AU;
    } else if ((strcasecmp(kptr,".mid")==0) || (strcasecmp(kptr,".midi")==0)) {
      fext=FEXT_MID;
    } else if (strcasecmp(kptr,".mp3")==0) {
      fext=FEXT_MP3;
    }
  }
  if (fext==0) {
    fprintf(stderr,"ATTA: unknown file extension \"%s\"\n",kptr==NULL?"":kptr);
    return;
  }

  /* open file */
  if (*filename!='/') {
    snprintf(pathname,sizeof(pathname),"%s/%s",workpath,filename);
  } else {
    snprintf(pathname,sizeof(pathname),"%s",filename);
  }
  if ((ffp=fopen(pathname,"r"))==NULL) {
    fprintf(stderr,"ATTA: cannot read \"%s\"\n",pathname);
    return;
  }

  if (fext==FEXT_WAV) {  /* check file with current settings */
    /* WAV-File Header:
    ** "RIFF" (char[4])
    **    filelength (uint)
    ** "WAVE" (char[4])
    ** [ "????" (char[4])
    **      blocklength (uint)
    **      data of length blocklength ]
    ** [ ... ]
    ** "fmt " (char[4])
    **    blocklength: "16" (uint)
    **    PCM-code: "1" (ushort)
    **    channels: "1"=mono or "2"=stereo (ushort)
    **    sample rate (uint)
    **    bytes per second (uint)
    **    bytes per sample (ushort)
    **    bits per sample (ushort)
    ** [ "????" (char[4])
    **      blocklength (uint)
    **      data of length blocklength ]
    ** [ ... ]
    ** "data"
    **    blocklength (uint)
    **    wavedata of length blocklength
    ** [ "????" (char[4])
    **      blocklength (uint)
    **      data of length blocklength ]
    ** [ ... ]
    */
    union {
      unsigned int i;
      char c[4];
    } c2i;
    union {
      unsigned short h;
      char c[2];
    } c2h;
    char data[512];
    int bsize,bytsek,bytsamp;
    long l1;

    /* +++ "RIFF" + filelength +++ */
    if (fread(data,sizeof(char),4+4,ffp)!=4+4) {fprintf(stderr,"ATTA: short read \"%s\" (RIFF)\n",pathname); fclose(ffp); return;}
    if (strncmp(data,"RIFF",4)!=0) {fprintf(stderr,"ATTA: \"%s\" is no WAVE-file.\n",pathname); fclose(ffp); return;}
    /* +++ "WAVE" +++ */
    if (fread(data,sizeof(char),4,ffp)!=4) {fprintf(stderr,"ATTA: short read \"%s\" (WAVE)\n",pathname); fclose(ffp); return;}
    if (strncmp(data,"WAVE",4)!=0) {fprintf(stderr,"ATTA: \"%s\" is no WAVE-file.\n",pathname); fclose(ffp); return;}
    /* +++ ???? + blocklength +++ */
    while (1) {
      if (fread(data,sizeof(char),4+4,ffp)!=4+4) {fprintf(stderr,"ATTA: short read \"%s\" (else)\n",pathname); fclose(ffp); return;}
#ifdef ENDIAN_IS_BIG
      c2i.c[0]=data[4+3]; c2i.c[1]=data[4+2]; c2i.c[2]=data[4+1]; c2i.c[3]=data[4+0];
#else
      c2i.c[0]=data[4+0]; c2i.c[1]=data[4+1]; c2i.c[2]=data[4+2]; c2i.c[3]=data[4+3];
#endif
      bsize=c2i.i;
      if (strncmp(data,"fmt ",4)==0) {break;}
      fseek(ffp,(long)bsize,SEEK_CUR);
    }
    /* +++ "fmt " settings +++ */
    if (bsize<16) {fprintf(stderr,"ATTA: \"%s\": fmt-section too short.\n",pathname); fclose(ffp); return;}
    if (fread(data,sizeof(char),16,ffp)!=16) {fprintf(stderr,"ATTA: short read \"%s\" (fmt)\n",pathname); fclose(ffp); return;}
#ifdef ENDIAN_IS_BIG
    c2h.c[0]=data[1]; c2h.c[1]=data[0];
#else
    c2h.c[0]=data[0]; c2h.c[1]=data[1];
#endif
    if (c2h.h==1) {
      pcmcode=PCMCODE_LINEAR;
    } else if (c2h.h==6) {
      pcmcode=PCMCODE_ALAW;
    } else if (c2h.h==7) {
      pcmcode=PCMCODE_MULAW;
    } else {fprintf(stderr,"ATTA: \"%s\": PCM-Code %hu not [1|6|7]\n",pathname,c2h.h); fclose(ffp); return;}
#ifdef ENDIAN_IS_BIG
    c2h.c[0]=data[3]; c2h.c[1]=data[2];
#else
    c2h.c[0]=data[2]; c2h.c[1]=data[3];
#endif
    channel=(int)c2h.h;
    if ((channel!=1) && (channel!=2)) {fprintf(stderr,"ATTA: \"%s\": %d channels found, need 1 or 2\n",pathname,channel); fclose(ffp); return;}
#ifdef ENDIAN_IS_BIG
    c2i.c[0]=data[7]; c2i.c[1]=data[6]; c2i.c[2]=data[5]; c2i.c[3]=data[4];
#else
    c2i.c[0]=data[4]; c2i.c[1]=data[5]; c2i.c[2]=data[6]; c2i.c[3]=data[7];
#endif
    samrate=c2i.i;
    if (samrate!=11025 && samrate!=22050 && samrate!=44100) {fprintf(stderr,"ATTA: \"%s\": warning: sample rate %d found (valid are 11025, 22050, 44100)\n",pathname,samrate);}
#ifdef ENDIAN_IS_BIG
    c2i.c[0]=data[11]; c2i.c[1]=data[10]; c2i.c[2]=data[9]; c2i.c[3]=data[8];
#else
    c2i.c[0]=data[8]; c2i.c[1]=data[9]; c2i.c[2]=data[10]; c2i.c[3]=data[11];
#endif
    bytsek=c2i.i;
#ifdef ENDIAN_IS_BIG
    c2h.c[0]=data[13]; c2h.c[1]=data[12];
#else
    c2h.c[0]=data[12]; c2h.c[1]=data[13];
#endif
    bytsamp=(int)c2h.h;
#ifdef ENDIAN_IS_BIG
    c2h.c[0]=data[15]; c2h.c[1]=data[14];
#else
    c2h.c[0]=data[14]; c2h.c[1]=data[15];
#endif
    bitsamp=(int)c2h.h;
    if ((bitsamp!=8) && (bitsamp!=16)) {fprintf(stderr,"ATTA: \"%s\": bits per sample %d found, need 8 or 16\n",pathname,bitsamp); fclose(ffp); return;}
    if (samrate*bytsamp!=bytsek) {fprintf(stderr,"ATTA: \"%s\": strange: %d*%d!=%d, going on ...\n",pathname,samrate,bytsamp,bytsek);}
    for (bsize-=16;bsize>0;) {  /* read rest of block */
      s1=(bsize>(int)sizeof(data)?sizeof(data):(size_t)bsize);
      if ((s1=fread(data,sizeof(char),s1,ffp))==0) {fprintf(stderr,"ATTA: short read \"%s\" (rest-of-fmt)\n",pathname); fclose(ffp); return;}
      bsize-=(int)s1;
    }
    /* +++ ???? + blocklength +++ */
    while (1) {
      if (fread(data,sizeof(char),4+4,ffp)!=4+4) {fprintf(stderr,"ATTA: short read \"%s\" (2.else)\n",pathname); fclose(ffp); return;}
#ifdef ENDIAN_IS_BIG
      c2i.c[0]=data[4+3]; c2i.c[1]=data[4+2]; c2i.c[2]=data[4+1]; c2i.c[3]=data[4+0];
#else
      c2i.c[0]=data[4+0]; c2i.c[1]=data[4+1]; c2i.c[2]=data[4+2]; c2i.c[3]=data[4+3];
#endif
      bsize=c2i.i;
      if (strncmp(data,"data",4)==0) {break;}
      fseek(ffp,(long)bsize,SEEK_CUR);
    }
    /* +++ "data" begin +++ */
    if ((l1=ftell(ffp))<0L) {fprintf(stderr,"ATTA: \"%s\": ftell: %s\n",pathname,strerror(errno)); fclose(ffp); return;}
    sndbegin=(unsigned int)l1;
    sndlen=(unsigned int)bsize;

  } else if (fext==FEXT_AU) {
    /* AU-File Header (big endian):
    ** ".snd" (char[4])
    ** begin of data (uint)
    ** size of data optional (uint)
    ** data encoding (uint)
    ** sample rate (uint)
    ** channels (uint)
    ** optional information (any)
    */
    union {
      unsigned int i;
      char c[4];
    } c2i;
    char data[512];
    int bsize;
    long l1;

    /* +++ ".snd" +++ */
    if (fread(data,sizeof(char),4,ffp)!=4) {fprintf(stderr,"ATTA: short read \"%s\" (.snd)\n",pathname); fclose(ffp); return;}
    if (strncmp(data,".snd",4)!=0) {fprintf(stderr,"ATTA: \"%s\" is no AU-file.\n",pathname); fclose(ffp); return;}

    /* +++ begin of data +++ */
    if (fread(data,sizeof(char),4,ffp)!=4) {fprintf(stderr,"ATTA: short read \"%s\" (begin of data)\n",pathname); fclose(ffp); return;}
#ifdef ENDIAN_IS_LITTLE
    c2i.c[0]=data[3]; c2i.c[1]=data[2]; c2i.c[2]=data[1]; c2i.c[3]=data[0];
#else
    c2i.c[0]=data[0]; c2i.c[1]=data[1]; c2i.c[2]=data[2]; c2i.c[3]=data[3];
#endif
    bsize=c2i.i;
    if (bsize<24) {bsize=24;}

    /* +++ size of data +++ */
    if (fread(data,sizeof(char),4,ffp)!=4) {fprintf(stderr,"ATTA: short read \"%s\" (size of data)\n",pathname); fclose(ffp); return;}

    /* +++ data encoding +++ */
    if (fread(data,sizeof(char),4,ffp)!=4) {fprintf(stderr,"ATTA: short read \"%s\" (data encoding)\n",pathname); fclose(ffp); return;}
#ifdef ENDIAN_IS_LITTLE
    c2i.c[0]=data[3]; c2i.c[1]=data[2]; c2i.c[2]=data[1]; c2i.c[3]=data[0];
#else
    c2i.c[0]=data[0]; c2i.c[1]=data[1]; c2i.c[2]=data[2]; c2i.c[3]=data[3];
#endif
    if (c2i.i==1) {
      pcmcode=PCMCODE_MULAW;
      bitsamp=8;
    } else if (c2i.i==2) {
      pcmcode=PCMCODE_LINEAR;
      bitsamp=8;
    } else if (c2i.i==3) {
      pcmcode=PCMCODE_LINEAR;
      bitsamp=16;  /* big endian */
    } else if (c2i.i==27) {
      pcmcode=PCMCODE_ALAW;
      bitsamp=8;
    } else {fprintf(stderr,"ATTA: \"%s\": data encoding %d unsupported\n",pathname,c2i.i); fclose(ffp); return;}

    /* +++ sample rate +++ */
    if (fread(data,sizeof(char),4,ffp)!=4) {fprintf(stderr,"ATTA: short read \"%s\" (sample rate)\n",pathname); fclose(ffp); return;}
#ifdef ENDIAN_IS_LITTLE
    c2i.c[0]=data[3]; c2i.c[1]=data[2]; c2i.c[2]=data[1]; c2i.c[3]=data[0];
#else
    c2i.c[0]=data[0]; c2i.c[1]=data[1]; c2i.c[2]=data[2]; c2i.c[3]=data[3];
#endif
    samrate=c2i.i;
    if (samrate!=11025 && samrate!=22050 && samrate!=44100) {fprintf(stderr,"ATTA: \"%s\": warning: sample rate %d found (valid are 11025, 22050, 44100)\n",pathname,samrate);}

    /* +++ channels +++ */
    if (fread(data,sizeof(char),4,ffp)!=4) {fprintf(stderr,"ATTA: short read \"%s\" (channels)\n",pathname); fclose(ffp); return;}
#ifdef ENDIAN_IS_LITTLE
    c2i.c[0]=data[3]; c2i.c[1]=data[2]; c2i.c[2]=data[1]; c2i.c[3]=data[0];
#else
    c2i.c[0]=data[0]; c2i.c[1]=data[1]; c2i.c[2]=data[2]; c2i.c[3]=data[3];
#endif
    channel=c2i.i;
    if ((channel!=1) && (channel!=2)) {fprintf(stderr,"ATTA: \"%s\": %d channels found, need 1 or 2\n",pathname,channel); fclose(ffp); return;}

    /* +++ begin of data +++ */
    fseek(ffp,0L,SEEK_END);
    if ((l1=ftell(ffp))<0L) {fprintf(stderr,"ATTA: \"%s\": ftell: %s\n",pathname,strerror(errno)); fclose(ffp); return;}
    sndbegin=bsize;
    sndlen=(unsigned int)l1-bsize;

  } else if (fext==FEXT_MID) {
    pcmcode=PCMCODE_LINEAR; bitsamp=16; channel=sdptr->channel; samrate=sdptr->srate; sndbegin=sndlen=0;

  } else {pcmcode=PCMCODE_LINEAR; bitsamp=sdptr->bitsamp; channel=sdptr->channel; samrate=sdptr->srate; sndbegin=sndlen=0;}
  fclose(ffp);

  /* install s_name */
  if (flag) {
    struct h_instc * ielm,** iptr;
    if ((ielm=calloc(1,sizeof(*ielm)))==NULL) {fprintf(stderr,"ATTA: \"%s\": calloc: %s\n",s_name,strerror(errno)); return;}
    s1=strlen(s_name);
    strncpy(ielm->s_name,s_name,s1); ielm->s_name[(int)s1]='\0';
    ielm->fext=fext;
    ielm->filename=strdup(pathname);
    ielm->sndbegin=sndbegin;
    ielm->pcmcode=pcmcode;
    ielm->bitsamp=bitsamp;
    ielm->channel=channel;
    ielm->samrate=samrate;
    ielm->volpc=volpc;
    ielm->status=0;
    ielm->nptr=NULL;
    for (iptr=&sf.hn;*iptr!=NULL;iptr=&(*iptr)->nptr) {;}
    *iptr=ielm;
  } else {
    if (++sf.sn_max==1) {
      sf.sn=malloc(sizeof(*sf.sn));
    } else {
      sf.sn=realloc(sf.sn,sizeof(*sf.sn)*sf.sn_max);
    }
    if (sf.sn==NULL) {fprintf(stderr,"ATTA: \"%s\": malloc/realloc: %s\n",s_name,strerror(errno)); return;}
    memset(&sf.sn[sf.sn_max-1],0,sizeof(*sf.sn));
    s1=strlen(s_name);
    strncpy(sf.sn[sf.sn_max-1].s_name,s_name,s1); sf.sn[sf.sn_max-1].s_name[(int)s1]='\0';
    sf.sn[sf.sn_max-1].fext=fext;
    sf.sn[sf.sn_max-1].filename=strdup(pathname);
    sf.sn[sf.sn_max-1].sndbegin=sndbegin;
    sf.sn[sf.sn_max-1].pcmcode=pcmcode;
    sf.sn[sf.sn_max-1].bitsamp=bitsamp;
    sf.sn[sf.sn_max-1].channel=channel;
    sf.sn[sf.sn_max-1].samrate=samrate;
    sf.sn[sf.sn_max-1].volpc=volpc;
    sf.sn[sf.sn_max-1].volume=100;
    sf.sn[sf.sn_max-1].imax=0;
    sf.sn[sf.sn_max-1].nr=NULL;
    inthash_add(sf.ihs,s_name,sf.sn_max-1);
    fprintf(stderr,"\"%s\" attached as \"%s\": OK\n",pathname,s_name);
    sf.volcalc=1;  /* recalculate volume */
  }
} /* Ende cmd_atta */


void cmd_copy(const char * s_dupname,const char * s_name) {
  struct h_instc * ielm,* nelm,** iptr;
  size_t s1;
  if ((s_name==NULL) || ((int)strlen(s_name)>15) || (strchr(s_name,'#')!=NULL) || (strncmp(s_name,"h-",2)!=0)) {return;}
  if ((s_dupname==NULL) || ((int)strlen(s_dupname)>15) || (strchr(s_dupname,'#')!=NULL) || (strncmp(s_dupname,"h-",2)!=0)) {return;}
  for (ielm=sf.hn;ielm!=NULL;ielm=ielm->nptr) {
    if (strcmp(s_name,ielm->s_name)==0) {break;}
  }
  if (ielm==NULL) {return;}
  if ((nelm=calloc(1,sizeof(*nelm)))==NULL) {fprintf(stderr,"_COPY: \"%s\": calloc: %s\n",s_dupname,strerror(errno)); return;}
  s1=strlen(s_dupname);
  strncpy(nelm->s_name,s_dupname,s1); nelm->s_name[(int)s1]='\0';
  nelm->fext=ielm->fext;
  nelm->filename=strdup(ielm->filename);
  nelm->sndbegin=ielm->sndbegin;
  nelm->pcmcode=ielm->pcmcode;
  nelm->bitsamp=ielm->bitsamp;
  nelm->channel=ielm->channel;
  nelm->samrate=ielm->samrate;
  nelm->volpc=ielm->volpc;
  nelm->status=0;
  nelm->nptr=NULL;
  for (iptr=&sf.hn;*iptr!=NULL;iptr=&(*iptr)->nptr) {;}
  *iptr=nelm;
} /* Ende cmd_copy */


void cmd_play(const char * s_name,int cresc,int loop,int flag) {
/* give out to stdout >0=instance-number or 0=error */
  int idx=0;
  if ((s_name==NULL) || ((int)strlen(s_name)>15)) {
    fprintf(stderr,"PLAY: invalid parameter\n");
#ifndef NO_INR
    if (flag==0) {printf("0\n"); fflush(stdout);}
#endif
    return;
  }
  if (cresc<0) {cresc=0;}
  if (strchr(s_name,'#')!=NULL) {
    fprintf(stderr,"PLAY: instance-number not valid here\n");
#ifndef NO_INR
    if (flag==0) {printf("0\n"); fflush(stdout);}
#endif
    return;
  }
  if (strspn(s_name,"#0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-")!=strlen(s_name)) {
    fprintf(stderr,"PLAY: invalid character \"%s\"\n",s_name);
    return;
  }
  if (((flag==0) && (strncmp(s_name,"s-",2)!=0)) || ((flag) && (strncmp(s_name,"h-",2)!=0))) {
    fprintf(stderr,"PLAY: \"%s\" does not begin with \"s-\"\n",s_name);
#ifndef NO_INR
    if (flag==0) {printf("0\n"); fflush(stdout);}
#endif
    return;
  }
  if ((flag==0) && (inthash_get(sf.ihs,s_name,&idx)!=0)) {
    fprintf(stderr,"PLAY: \"%s\" unknown\n",s_name);
#ifndef NO_INR
    if (flag==0) {printf("0\n"); fflush(stdout);}
#endif
    return;
  }

  if (flag) {
    struct h_instc * ielm;
    for (ielm=sf.hn;ielm!=NULL;ielm=ielm->nptr) {
      if (strcmp(s_name,ielm->s_name)==0) {break;}
    }
    if (ielm==NULL) {
      fprintf(stderr,"PLAY: \"%s\" not found\n",s_name);
      return;
    }
    if (ielm->status==1) {  /* still playing */
      return;
    }
    if ((ielm->ffp=start_sound(ielm->filename,ielm->fext,ielm->sndbegin,&ielm->bitsamp,&ielm->channel,&ielm->samrate))==NULL) {
      return;
    }
    ielm->status=1;
    ielm->dpos=ielm->dend=0;
    ielm->howloop=(loop?0:1);
  } else {  /* create a new instance */
    struct s_instc * ielm,** iptr;
    int simanz=0;
    for (iptr=&sf.sn[idx].nr;*iptr!=NULL;iptr=&(*iptr)->nptr) {simanz++;}
    if (simanz>=MAX_INSTANCES) {
      fprintf(stderr,"PLAY: \"%s\": already too many instances running\n",s_name);
#ifndef NO_INR
      printf("0\n"); fflush(stdout);
#endif
      return;
    }
    if ((ielm=calloc(1,sizeof(*ielm)))==NULL) {
      fprintf(stderr,"PLAY: \"%s\": calloc: %s\n",s_name,strerror(errno));
#ifndef NO_INR
      printf("0\n"); fflush(stdout);
#endif
      return;
    }
    ielm->bitsamp=sf.sn[idx].bitsamp;
    ielm->channel=sf.sn[idx].channel;
    ielm->samrate=sf.sn[idx].samrate;
    if ((ielm->ffp=start_sound(sf.sn[idx].filename,sf.sn[idx].fext,sf.sn[idx].sndbegin,&ielm->bitsamp,&ielm->channel,&ielm->samrate))==NULL) {
#ifndef NO_INR
      printf("0\n"); fflush(stdout);
#endif
      return;
    }
    ielm->status=1;
    ielm->paused=0;
    ielm->dpos=ielm->dend=0;
    ielm->loop=loop;
    ielm->nptr=NULL;
    if (cresc>0) {
      ielm->cr_pc=100000/(sdptr->srate*cresc*sdptr->channel*sdptr->bitsamp/(8*sdptr->frag_size));  /* increasing percent */
      if (ielm->cr_pc<1) {ielm->cr_pc=1;}
      ielm->cr_val=0;  /* 0% */
    } else {ielm->cr_pc=0; ielm->cr_val=100000;}
    ielm->inr=++sf.sn[idx].imax;
    sf.sn[idx].imax%=99999;
    *iptr=ielm;
#ifndef NO_INR
    printf("%d\n",ielm->inr); fflush(stdout);
#endif
    fprintf(stderr,"\"%s#%d\" playing: OK\n",s_name,ielm->inr);
  }
} /* Ende cmd_play */


void cmd_deta(const char * s_name) {
  struct h_instc * ielm,* velm;
  if ((s_name==NULL) || ((int)strlen(s_name)>15) || (strchr(s_name,'#')!=NULL) || (strncmp(s_name,"h-",2)!=0)) {return;}
  velm=NULL;
  for (ielm=sf.hn;ielm!=NULL;ielm=ielm->nptr) {
    if (strcmp(s_name,ielm->s_name)==0) {break;}
    velm=ielm;
  }
  if (ielm==NULL) {return;}
  if ((ielm->status>0) && (ielm->ffp!=NULL)) {
    if ((ielm->fext==FEXT_WAV) || (ielm->fext==FEXT_AU)) {fclose(ielm->ffp);} else {pclose(ielm->ffp);}
  }
  if (velm==NULL) {sf.hn=ielm->nptr;} else {velm->nptr=ielm->nptr;}
  free(ielm->filename);
  free(ielm);
} /* Ende cmd_deta */


void cmd_wrapper(const char * sg_name,int argnr,const char * cmdname) {
  int aufruf=0;
  struct s_instc * ielm;
  const char * kp1;
  char sname[32],** pptr;
  int zahl,idx,i1;
  if ((sg_name==NULL) || (cmdname==NULL) || (*cmdname=='\0')) {
    fprintf(stderr,"cmd_wrapper: invalid parameter\n");
    return;
  }
  if ((kp1=strchr(sg_name,'#'))!=NULL) {zahl=atoi(kp1+1);} else {zahl=0;}
  if (strspn(sg_name,"#0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-")!=strlen(sg_name)) {
    fprintf(stderr,"%s: invalid character \"%s\"\n",cmdname,sg_name);
    return;
  }
  i1=(kp1==NULL?(int)strlen(sg_name):(int)(kp1-sg_name));
  if ((i1>15) \
  || ((strncmp(sg_name,"s-",2)!=0) && (strncmp(sg_name,"g-",2)!=0) && (strcmp(sg_name,GROUP_ALL)!=0)) \
  || ((strncmp(sg_name,"g-",2)==0) && (zahl>0))) {
    fprintf(stderr,"%s: invalid parameter\n",cmdname);
    return;
  }
  strncpy(sname,sg_name,i1); sname[i1]='\0';
  if (strcmp(cmdname,"PAUS")==0) {aufruf=1;}
  else if (strcmp(cmdname,"CONT")==0) {aufruf=2;}
  else if (strcmp(cmdname,"PATE")==0) {aufruf=3;}
  else if (strcmp(cmdname,"CATB")==0) {aufruf=4;}
  else if (strcmp(cmdname,"STOP")==0) {aufruf=5;}
  else {fprintf(stderr,"%s: invalid command name\n",cmdname); return;}

  if (strcmp(sname,GROUP_ALL)==0) {
    switch(aufruf) {
      case 1: cmd_paus(sname,argnr,NULL); break;
      case 2: cmd_cont(sname,argnr,NULL); break;
      case 3: cmd_pate(sname,argnr,NULL); break;
      case 4: cmd_catb(sname,argnr,NULL); break;
      case 5: cmd_stop(sname,argnr,NULL); break;
    }
    return;
  }
  if (inthash_get(sf.ihs,sname,&idx)!=0) {
    fprintf(stderr,"%s: \"%s\" unknown\n",cmdname,sname);
    return;
  }
  if (strncmp(sname,"g-",2)==0) {  /* group */
    int sdx;
    for (pptr=sf.gn[idx].s_name;*pptr!=NULL;pptr++) {
      if (inthash_get(sf.ihs,*pptr,&sdx)!=0) {
        fprintf(stderr,"%s: \"%s\" unknown\n",cmdname,*pptr);
        continue;
      }
      for (ielm=sf.sn[sdx].nr;ielm!=NULL;ielm=ielm->nptr) {
        switch(aufruf) {
          case 1: cmd_paus(sname,argnr,ielm); break;
          case 2: cmd_cont(sname,argnr,ielm); break;
          case 3: cmd_pate(sname,argnr,ielm); break;
          case 4: cmd_catb(sname,argnr,ielm); break;
          case 5: cmd_stop(sname,argnr,ielm); break;
        }
      }
    }
  } else {
    for (ielm=sf.sn[idx].nr;ielm!=NULL;ielm=ielm->nptr) {
      if ((zahl==0) || ((zahl>0) && (zahl==ielm->inr))) {
        switch(aufruf) {
          case 1: cmd_paus(sname,argnr,ielm); break;
          case 2: cmd_cont(sname,argnr,ielm); break;
          case 3: cmd_pate(sname,argnr,ielm); break;
          case 4: cmd_catb(sname,argnr,ielm); break;
          case 5: cmd_stop(sname,argnr,ielm); break;
        }
      }
    }
  }
} /* Ende cmd_wrapper */


void cmd_paus(const char * sname,int argnr,void * vptr) {
  if (sname==NULL) {return;}
  if (strcmp(sname,GROUP_ALL)==0) {
    int idx;
    struct s_instc * ielm;
    for (idx=0;idx<sf.sn_max;idx++) {
      for (ielm=sf.sn[idx].nr;ielm!=NULL;ielm=ielm->nptr) {
        cmd_paus(sf.sn[idx].s_name,argnr,ielm);
      }
    }
    paused=1;
    return;
  }
  if (vptr!=NULL) {
    struct s_instc * ielm;
    ielm=(struct s_instc *)vptr;
    if (ielm->paused==0) {
      ielm->paused=1;
      fprintf(stderr,"\"%s#%d\" paused: OK\n",sname,ielm->inr);
    }
  }
} /* Ende cmd_paus */


void cmd_cont(const char * sname,int argnr,void * vptr) {
  if (sname==NULL) {return;}
  paused=0;
  if (strcmp(sname,GROUP_ALL)==0) {
    int idx;
    struct s_instc * ielm;
    for (idx=0;idx<sf.sn_max;idx++) {
      for (ielm=sf.sn[idx].nr;ielm!=NULL;ielm=ielm->nptr) {
        cmd_cont(sf.sn[idx].s_name,argnr,ielm);
      }
    }
    return;
  }
  if (vptr!=NULL) {
    struct s_instc * ielm;
    ielm=(struct s_instc *)vptr;
    if (ielm->paused) {
      ielm->paused=0;
      fprintf(stderr,"\"%s#%d\" continued: OK\n",sname,ielm->inr);
    }
  }
} /* Ende cmd_cont */


void cmd_pate(const char * sname,int argnr,void * vptr) {
  if (sname==NULL) {return;}
  if (strcmp(sname,GROUP_ALL)==0) {  /* pause-at-end all but hidden s_name */
    int idx;
    struct s_instc * ielm;
    for (idx=0;idx<sf.sn_max;idx++) {
      for (ielm=sf.sn[idx].nr;ielm!=NULL;ielm=ielm->nptr) {
        cmd_pate(sf.sn[idx].s_name,argnr,ielm);
      }
    }
    return;
  }
  if (vptr!=NULL) {
    struct s_instc * ielm;
    ielm=(struct s_instc *)vptr;
    if (ielm->status==1) {
      ielm->status=2;
      fprintf(stderr,"\"%s#%d\" paused-at-end: OK\n",sname,ielm->inr);
    }
  }
} /* Ende cmd_pate */


void cmd_catb(const char * sname,int argnr,void * vptr) {
  if (sname==NULL) {return;}
  if (strcmp(sname,GROUP_ALL)==0) {  /* not for hidden s_name */
    int idx;
    struct s_instc * ielm;
    for (idx=0;idx<sf.sn_max;idx++) {
      for (ielm=sf.sn[idx].nr;ielm!=NULL;ielm=ielm->nptr) {
        cmd_catb(sf.sn[idx].s_name,argnr,ielm);
      }
    }
    return;
  }
  if (vptr!=NULL) {
    struct s_instc * ielm;
    ielm=(struct s_instc *)vptr;
    if ((ielm->status==2) || (ielm->status==3)) {
      ielm->status=1;
      fprintf(stderr,"\"%s#%d\" continued-at-begin: OK\n",sname,ielm->inr);
    }
  }
} /* Ende cmd_catb */


void cmd_stop(const char * sname,int argnr,void * vptr) {
  if (sname==NULL) {return;}
  if (strcmp(sname,GROUP_ALL)==0) {  /* stop all but hidden s_name */
    int idx;
    struct s_instc * ielm;
    for (idx=0;idx<sf.sn_max;idx++) {
      for (ielm=sf.sn[idx].nr;ielm!=NULL;ielm=ielm->nptr) {
        cmd_stop(sf.sn[idx].s_name,argnr,ielm);
      }
    }
    return;
  }
  if (vptr!=NULL) {
    struct s_instc * ielm;
    ielm=(struct s_instc *)vptr;
    if (ielm->status>0) {
      if (argnr>0) {
        ielm->cr_pc=-100000/(sdptr->srate*argnr*sdptr->channel*sdptr->bitsamp/(8*sdptr->frag_size));  /* decreasing percent */
        if (ielm->cr_pc>-1) {ielm->cr_pc=-1;}
        ielm->cr_val=100000;  /* 100% */
      } else {ielm->cr_pc=0; ielm->cr_val=0;}
      ielm->status=0;
      fprintf(stderr,"\"%s#%d\" stopped: OK\n",sname,ielm->inr);
    }
  }
} /* Ende cmd_stop */


void cmd_gclr(const char * g_name) {
  int idx=0;
  char ** pptr;
  size_t s1;
  if ((g_name==NULL) || ((int)strlen(g_name)>15)) {
    fprintf(stderr,"GCLR: invalid parameter\n");
    return;
  }
  if (strchr(g_name,'#')!=NULL) {
    fprintf(stderr,"GCLR: instance-number not valid for groups\n");
    return;
  }
  if (strspn(g_name,"#0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-")!=strlen(g_name)) {
    fprintf(stderr,"GCLR: invalid character \"%s\"\n",g_name);
    return;
  }
  if (strncmp(g_name,"g-",2)!=0) {
    fprintf(stderr,"GCLR: \"%s\" does not begin with \"g-\"\n",g_name);
    return;
  }
  if (inthash_get(sf.ihs,g_name,&idx)!=0) {  /* create group */
    if (++sf.gn_max==1) {
      sf.gn=malloc(sizeof(*sf.gn));
    } else {
      sf.gn=realloc(sf.gn,sizeof(*sf.gn)*sf.gn_max);
    }
    if (sf.gn==NULL) {fprintf(stderr,"GCLR: \"%s\": malloc/realloc: %s\n",g_name,strerror(errno)); return;}
    memset(&sf.gn[sf.gn_max-1],0,sizeof(*sf.gn));
    s1=strlen(g_name);
    strncpy(sf.gn[sf.gn_max-1].g_name,g_name,s1); sf.gn[sf.gn_max-1].g_name[(int)s1]='\0';
    sf.gn[sf.gn_max-1].volume=100;
    if ((sf.gn[sf.gn_max-1].s_name=malloc(sizeof(char *)))==NULL) {fprintf(stderr,"GCLR: \"%s\": malloc/realloc: %s\n",g_name,strerror(errno)); return;}
    sf.gn[sf.gn_max-1].s_name[0]=NULL;
    inthash_add(sf.ihs,g_name,sf.gn_max-1);
    idx=sf.gn_max-1;
  }
  /* clear group */
  for (pptr=sf.gn[idx].s_name;*pptr!=NULL;pptr++) {free(*pptr);}
  sf.gn[idx].s_name[0]=NULL;
  fprintf(stderr,"\"%s\" cleared: OK\n",g_name);
  sf.volcalc=1;  /* recalculate volume */
} /* Ende cmd_gclr */


void cmd_gadd(const char * g_name,const char * s_patt) {
  int idx=0,sdx=0,gnr;
  char ** pptr;
  size_t s1;
  if ((g_name==NULL) || ((int)strlen(g_name)>15) || (s_patt==NULL)) {
    fprintf(stderr,"GADD: invalid parameter\n");
    return;
  }
  if (strchr(g_name,'#')!=NULL) {
    fprintf(stderr,"GADD: instance-number not valid for groups\n");
    return;
  }
  if (strspn(g_name,"#0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-")!=strlen(g_name)) {
    fprintf(stderr,"GADD: invalid character \"%s\"\n",g_name);
    return;
  }
  if (strncmp(g_name,"g-",2)!=0) {
    fprintf(stderr,"GADD: \"%s\" does not begin with \"g-\"\n",g_name);
    return;
  }
  if (inthash_get(sf.ihs,g_name,&idx)!=0) {  /* create group */
    if (++sf.gn_max==1) {
      sf.gn=malloc(sizeof(*sf.gn));
    } else {
      sf.gn=realloc(sf.gn,sizeof(*sf.gn)*sf.gn_max);
    }
    if (sf.gn==NULL) {fprintf(stderr,"GADD: \"%s\": malloc/realloc: %s\n",g_name,strerror(errno)); return;}
    memset(&sf.gn[sf.gn_max-1],0,sizeof(*sf.gn));
    s1=strlen(g_name);
    strncpy(sf.gn[sf.gn_max-1].g_name,g_name,s1); sf.gn[sf.gn_max-1].g_name[(int)s1]='\0';
    sf.gn[sf.gn_max-1].volume=100;
    if ((sf.gn[sf.gn_max-1].s_name=malloc(sizeof(char *)))==NULL) {fprintf(stderr,"GADD: \"%s\": malloc/realloc: %s\n",g_name,strerror(errno)); return;}
    sf.gn[sf.gn_max-1].s_name[0]=NULL;
    inthash_add(sf.ihs,g_name,sf.gn_max-1);
    idx=sf.gn_max-1;
  }
  /* add sound-name(s) to group */
  for (gnr=1,pptr=sf.gn[idx].s_name;*pptr!=NULL;pptr++) {gnr++;}
  if (*s_patt=='&') {  /* shell pattern */
    for (sdx=0;sdx<sf.sn_max;sdx++) {
      if (fnpatt(s_patt+1,sf.sn[sdx].s_name)) {
        for (pptr=sf.gn[idx].s_name;*pptr!=NULL;pptr++) {
          if (strcmp(*pptr,sf.sn[sdx].s_name)==0) {break;}
        }
        if (*pptr!=NULL) {continue;}
        if ((sf.gn[idx].s_name=realloc(sf.gn[idx].s_name,sizeof(char *)*(gnr+1)))==NULL) {fprintf(stderr,"GADD: \"%s\": malloc/realloc: %s\n",g_name,strerror(errno)); return;}
        sf.gn[idx].s_name[gnr-1]=strdup(sf.sn[sdx].s_name);
        sf.gn[idx].s_name[gnr]=NULL;
        gnr++;
        fprintf(stderr,"\"%s\" added: OK\n",sf.sn[sdx].s_name);
      }
    }
  } else {  /* sound-name */
    if (inthash_get(sf.ihs,s_patt,&sdx)==0) {
      for (pptr=sf.gn[idx].s_name;*pptr!=NULL;pptr++) {
        if (strcmp(*pptr,sf.sn[sdx].s_name)==0) {break;}
      }
      if (*pptr==NULL) {
        if ((sf.gn[idx].s_name=realloc(sf.gn[idx].s_name,sizeof(char *)*(gnr+1)))==NULL) {fprintf(stderr,"GADD: \"%s\": malloc/realloc: %s\n",g_name,strerror(errno)); return;}
        sf.gn[idx].s_name[gnr-1]=strdup(sf.sn[sdx].s_name);
        sf.gn[idx].s_name[gnr]=NULL;
        gnr++;
        fprintf(stderr,"\"%s\" added: OK\n",sf.sn[sdx].s_name);
      }
    }
  }
  sf.volcalc=1;  /* recalculate volume */
} /* Ende cmd_gadd */


void cmd_gdel(const char * g_name,const char * s_patt) {
  int idx=0,sdx,gnr;
  char ** pptr;
  if ((g_name==NULL) || ((int)strlen(g_name)>15) || (s_patt==NULL)) {
    fprintf(stderr,"GDEL: invalid parameter\n");
    return;
  }
  if (strchr(g_name,'#')!=NULL) {
    fprintf(stderr,"GDEL: instance-number not valid for groups\n");
    return;
  }
  if (strspn(g_name,"#0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-")!=strlen(g_name)) {
    fprintf(stderr,"GDEL: invalid character \"%s\"\n",g_name);
    return;
  }
  if (strncmp(g_name,"g-",2)!=0) {
    fprintf(stderr,"GDEL: \"%s\" does not begin with \"g-\"\n",g_name);
    return;
  }
  if (inthash_get(sf.ihs,g_name,&idx)!=0) {return;}
  /* delete sound-name(s) from group */
  for (gnr=1,pptr=sf.gn[idx].s_name;*pptr!=NULL;pptr++) {gnr++;}
  for (sdx=0;sdx<gnr-1;sdx++) {
    if (*s_patt=='&') {  /* shell pattern */
      if (fnpatt(s_patt+1,sf.gn[idx].s_name[sdx])) {
        fprintf(stderr,"\"%s\" deleted: OK\n",sf.gn[idx].s_name[sdx]);
        free(sf.gn[idx].s_name[sdx]);
        memmove(sf.gn[idx].s_name+sdx,sf.gn[idx].s_name+sdx+1,sizeof(char *)*(gnr-sdx-1));
        gnr--;
        sdx--;
      }
    } else {
      if (strcmp(s_patt,sf.gn[idx].s_name[sdx])==0) {
        fprintf(stderr,"\"%s\" deleted: OK\n",sf.gn[idx].s_name[sdx]);
        free(sf.gn[idx].s_name[sdx]);
        memmove(sf.gn[idx].s_name+sdx,sf.gn[idx].s_name+sdx+1,sizeof(char *)*(gnr-sdx-1));
        break;
      }
    }
  }
  sf.volcalc=1;  /* recalculate volume */
} /* Ende cmd_gdel */


void cmd_volm(const char * sg0_name,int vol) {
  int idx,erg;
  if (sg0_name==NULL) {
    fprintf(stderr,"VOLM: invalid parameter\n");
    printf("0\n"); fflush(stdout);
    return;
  }
  if (strchr(sg0_name,'#')!=NULL) {
    fprintf(stderr,"VOLM: instance-number not valid here\n");
    printf("0\n"); fflush(stdout);
    return;
  }
  if (strspn(sg0_name,"#0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-")!=strlen(sg0_name)) {
    fprintf(stderr,"VOLM: invalid character \"%s\"\n",sg0_name);
    printf("0\n"); fflush(stdout);
    return;
  }
  if (vol>100) {vol=100;}
  if (strncmp(sg0_name,"s-",2)==0) {  /* sound-name */
    if (inthash_get(sf.ihs,sg0_name,&idx)!=0) {
      fprintf(stderr,"VOLM: \"%s\" not found\n",sg0_name);
      printf("0\n"); fflush(stdout);
      return;
    }
    erg=sf.sn[idx].volume;
    if (vol>=0) {sf.sn[idx].volume=vol*2; sf.volcalc=1;}
  } else if (strncmp(sg0_name,"g-",2)==0) {  /* group */
    if (inthash_get(sf.ihs,sg0_name,&idx)!=0) {
      fprintf(stderr,"VOLM: \"%s\" not found\n",sg0_name);
      printf("0\n"); fflush(stdout);
      return;
    }
    erg=sf.gn[idx].volume;
    if (vol>=0) {sf.gn[idx].volume=vol*2; sf.volcalc=1;}
  } else if (strcmp(sg0_name,"0")==0) {  /* main volume */
    if ((erg=sdptr->mixer_dev(sdptr,vol<0?-1:vol))<0) {erg=0;}
    erg*=2;
  } else if (strcmp(sg0_name,GROUP_ALL)==0) {  /* all groups */
    erg=0;
    for (idx=0;idx<sf.gn_max;idx++) {
      erg+=sf.gn[idx].volume;
      if (vol>=0) {sf.gn[idx].volume=vol*2;}
    }
    erg/=idx;
    if (vol>=0) {sf.volcalc=1;}
  } else {
    fprintf(stderr,"VOLM: invalid parameter\n");
    printf("0\n"); fflush(stdout);
    return;
  }
  printf("%d\n",erg/2); fflush(stdout);
  fprintf(stderr,"Last volume value: %d\n",erg/2);
} /* Ende cmd_volm */


void cmd_frgs(int samplerate,int channel,int frg_anz,int frg_sz) {
  if (frg_anz==0) {
    if ((frg_anz=sdptr->frag_anz)<2) {frg_anz=2;}
  } else {
    if (frg_anz<2) {frg_anz=2;}
  }
  if (frg_sz==0) {
    if ((frg_sz=sdptr->frag_size)<128) {frg_sz=128;}
  } else {
    if (frg_sz<128) {frg_sz=128;}
  }
  if (samplerate==0) {
    samplerate=sdptr->srate;
  } else {
    if (samplerate<11025+11025/2) {samplerate=11025;}
    else if (samplerate>22050+22050/2) {samplerate=44100;}
    else {samplerate=22050;}
  }
  if (channel==0) {
    channel=sdptr->channel;
  } else {
    if (channel!=1 && channel!=2) {channel=1;}
  }
  if ((frg_anz>=2) && (frg_sz>=128)) {
    int i1,mixa;
    setegid(saved_gid);
    seteuid(saved_uid);
    mixa=sdptr->mixer_dev(sdptr,-1);
    sdptr->frag_anz=frg_anz;
    sdptr->frag_size=frg_sz;
    sdptr->srate=samplerate;
    sdptr->channel=channel;
    sdptr->close_dev(sdptr);
    if (sdptr->open_dev(sdptr,sdptr->srate,sdptr->channel)<0) {
      fprintf(stderr,"cmd_frgs: error reopening device\n");
      extsoundserver(1);  /* resume external sound servers */
      fclose(stderr);
      exit(1);
    }
    for (i1=1;(sdptr->frag_size>>i1)>0;i1++) {;}
    if ((sdptr->frag_size=(1<<(i1-1)))<128) {sdptr->frag_size=128;}
    if (((frg_uc=realloc(frg_uc,sizeof(unsigned char)*sdptr->frag_size))==NULL) \
    || ((frg_si=realloc(frg_si,sizeof(int)*sdptr->frag_size))==NULL)) {
      fprintf(stderr,"cmd_frgs: realloc: %s\n",strerror(errno));
      sdptr->mixer_dev(sdptr,-2);
      sdptr->close_dev(sdptr);
      extsoundserver(1);  /* resume external sound servers */
      fclose(stderr);
      exit(1);
    }
    if (mixa>=0) {sdptr->mixer_dev(sdptr,mixa);}
    fprintf(stderr,"  Sample rate=%d, channels=%d, %d Bit, Fragments=%d, Fragment-Size=%d\n",sdptr->srate,sdptr->channel,sdptr->bitsamp,sdptr->frag_anz,sdptr->frag_size);
    for (i1=0;i1<99;i1++) {  /* close-on-exec for every non default file */
      if ((i1==STDIN_FILENO)||(i1==STDOUT_FILENO)||(i1==STDERR_FILENO)) {continue;}
      fcntl(i1,F_SETFD,fcntl(i1,F_GETFD,0)|FD_CLOEXEC);
    }
    setegid(getgid());
    seteuid(getuid());
  }
  printf("%d %d %d %d\n",sdptr->srate,sdptr->channel,sdptr->frag_anz,sdptr->frag_size); fflush(stdout);
} /* Ende cmd_frgs */


void output_block() {
/* give out fragment of data to device */
  int i1;
  double volx;

  /* initialize frg_si */
  memset(frg_si,0,sizeof(int)*sdptr->frag_size);

  /* calculate all volumes if necessary */
  volofsound(-1);

  /* add into frg_si for all hidden s_name */
  {struct h_instc * ielm;
   int erg;
   for (ielm=sf.hn;ielm!=NULL;ielm=ielm->nptr) {
     if ((ielm->status==0) || (paused)) {continue;}
     volx=(double)ielm->volpc/100.;  /* volumepercent */
     erg=fill_fragment(volx, \
                  ielm->fext, \
                  ielm->pcmcode, \
                  ielm->bitsamp, \
                  ielm->channel, \
                  ielm->samrate, \
                  ielm->ffp, \
                  ielm->data, \
                  &ielm->dpos, \
                  &ielm->dend);
     if (erg==0) {  /* no more sound data */
       if ((ielm->fext==FEXT_WAV) || (ielm->fext==FEXT_AU)) {fclose(ielm->ffp);} else {pclose(ielm->ffp);}
       ielm->ffp=NULL;
       if (ielm->status==0) {ielm->howloop=0;}
       if (ielm->howloop) {  /* start new sound */
         ielm->ffp=start_sound(ielm->filename,ielm->fext,ielm->sndbegin,&ielm->bitsamp,&ielm->channel,&ielm->samrate);
       }
       if (ielm->ffp!=NULL) {  /* new sound started */
         ielm->status=1;
         ielm->dpos=ielm->dend=0;
       } else {ielm->status=0;}
     }
   }
  }

  /* add into frg_si for all s_name */
  {struct s_instc * ielm,* velm;
   int idx,erg;
   for (idx=0;idx<sf.sn_max;idx++) {
     volx=volofsound(idx);  /* volumepercent */
     velm=NULL;
     for (ielm=sf.sn[idx].nr;ielm!=NULL;velm=ielm,ielm=ielm->nptr) {
       if ((ielm->status==0) && (ielm->cr_pc==0)) {
         if (ielm->ffp!=NULL) {
           if ((sf.sn[idx].fext==FEXT_WAV) || (sf.sn[idx].fext==FEXT_AU)) {fclose(ielm->ffp);} else {pclose(ielm->ffp);}
           ielm->ffp=NULL;
           if (velm==NULL) {sf.sn[idx].nr=ielm->nptr;} else {velm->nptr=ielm->nptr;}
           free(ielm);
           if ((ielm=(velm==NULL?sf.sn[idx].nr:velm))==NULL) {break;}
         }
         continue;
       }
       if ((ielm->status==3) || (ielm->paused)) {continue;}
       erg=fill_fragment(volx*((double)ielm->cr_val/100000.), \
                    sf.sn[idx].fext, \
                    sf.sn[idx].pcmcode, \
                    ielm->bitsamp, \
                    ielm->channel, \
                    ielm->samrate, \
                    ielm->ffp, \
                    ielm->data, \
                    &ielm->dpos, \
                    &ielm->dend);
       ielm->cr_val+=ielm->cr_pc;
       if (ielm->cr_val>100000) {ielm->cr_val=100000; ielm->cr_pc=0;}
       else if (ielm->cr_val<0) {ielm->cr_val=0; ielm->cr_pc=0;}
       if (erg==0) {  /* no more sound data */
         if ((sf.sn[idx].fext==FEXT_WAV) || (sf.sn[idx].fext==FEXT_AU)) {fclose(ielm->ffp);} else {pclose(ielm->ffp);}
         ielm->ffp=NULL;
         if ((ielm->status==0) || ((ielm->loop>0) && (--ielm->loop==0))) {ielm->loop=-1;}
         if (ielm->loop>=0) {  /* start new sound */
           ielm->ffp=start_sound(sf.sn[idx].filename,sf.sn[idx].fext,sf.sn[idx].sndbegin,&ielm->bitsamp,&ielm->channel,&ielm->samrate);
         }
         if (ielm->ffp!=NULL) {  /* new sound started */
           if (ielm->status==2) {ielm->status=3;} else {ielm->status=1;}
           ielm->paused=0;
           ielm->dpos=ielm->dend=0;
           ielm->cr_pc=0; ielm->cr_val=100000;
         } else {  /* free this instance */
           if (velm==NULL) {sf.sn[idx].nr=ielm->nptr;} else {velm->nptr=ielm->nptr;}
           if (ielm->status) {fprintf(stderr,"[\"%s#%d\" ended]\n",sf.sn[idx].s_name,ielm->inr);}
           free(ielm);
           if ((ielm=(velm==NULL?sf.sn[idx].nr:velm))==NULL) {break;}
         }
       }
     }
   }
  }

  /* collect sound samples */
  if (sdptr->bitsamp==8) {  /* sound device U8 bit */
    for (i1=0;i1<sdptr->frag_size;i1++) {
      frg_si[i1]=(frg_si[i1]<-32768?-32768:(frg_si[i1]>32767?32767:frg_si[i1]));
      frg_uc[i1]=((frg_si[i1]>>8)&0xff)+128;
    }
    sdptr->write_dev(sdptr,frg_uc,sdptr->frag_size);
  } else {  /* sound device 16LE/16BE bit */
    union {
      short h;
      unsigned char c[2];
    } * u_hc;
    u_hc=(void *)frg_uc;
    for (i1=0;i1<sdptr->frag_size/2;i1++) {
      frg_si[i1]=(frg_si[i1]<-32768?-32768:(frg_si[i1]>32767?32767:frg_si[i1]));
      u_hc[i1].h=(short)frg_si[i1];
    }
    sdptr->write_dev(sdptr,frg_uc,sdptr->frag_size);
  }
} /* Ende output_block */


int fill_fragment(double volx,int fext,int pcmcode,int bitsamp,int channel,int samrate,FILE * ffp,unsigned char * data,int * dpos,int * dend) {
  unsigned char * uptr;
  int readbyte,maxidx,idx,i1;
  short h1[2];
  int sfakt=calc_sfakt(samrate);
  maxidx=sdptr->frag_size/(sdptr->bitsamp/8);
  idx=0;
  if (bitsamp==8) {  /* 8bit soundfile */
    if (channel==1) {  /* mono soundfile */
      readbyte=1;
      while (idx<maxidx) {
        if ((uptr=read_data(readbyte,ffp,data,dpos,dend))==NULL) {return(0);}  /* no more data */
        if (pcmcode==PCMCODE_MULAW) {  /* MULAW8 bit */
          h1[0]=AUDIO_U2S(uptr[0]);
        } else if (pcmcode==PCMCODE_ALAW) {  /* ALAW8 bit */
          h1[0]=AUDIO_A2S(uptr[0]);
        } else {  /* U8 bit */
          h1[0]=(uptr[0]-128)<<8;
        }
        if (sdptr->channel==1) {  /* sound device mono */
          h1[0]=(short)((double)h1[0]*volx);
          frg_si[idx++]+=(int)h1[0];
        } else {  /* sound device stereo */
          h1[0]=(short)((double)h1[0]*volx);
          frg_si[idx++]+=(int)h1[0];
          frg_si[idx++]+=(int)h1[0];
        }
        if (sfakt<0) {  /* soundfile's rate is greater than devices's */
          for (i1=-2;i1>=sfakt;i1--) {
            if ((uptr=read_data(readbyte,ffp,data,dpos,dend))==NULL) {return(0);}  /* no more data */
          }
        } else {
          for (i1=2;i1<=sfakt;i1++) {
            frg_si[idx++]+=(int)h1[0];
            if (sdptr->channel==2) {frg_si[idx++]+=(int)h1[0];}
          }
        }
      }
    } else {  /* stereo soundfile */
      readbyte=2;
      while (idx<maxidx) {
        if ((uptr=read_data(readbyte,ffp,data,dpos,dend))==NULL) {return(0);}  /* no more data */
        if (pcmcode==PCMCODE_MULAW) {  /* MULAW8 bit */
          h1[0]=AUDIO_U2S(uptr[0]);
          h1[1]=AUDIO_U2S(uptr[1]);
        } else if (pcmcode==PCMCODE_ALAW) {  /* ALAW8 bit */
          h1[0]=AUDIO_A2S(uptr[0]);
          h1[1]=AUDIO_A2S(uptr[1]);
        } else {  /* U8 bit */
          h1[0]=(uptr[0]-128)<<8;
          h1[1]=(uptr[1]-128)<<8;
        }
        if (sdptr->channel==1) {  /* sound device mono */
          h1[0]=(h1[0]+h1[1])/2;
          h1[0]=(short)((double)h1[0]*volx);
          frg_si[idx++]+=(int)h1[0];
        } else {  /* sound device stereo */
          h1[0]=(short)((double)h1[0]*volx);
          frg_si[idx++]+=(int)h1[0];
          h1[1]=(short)((double)h1[1]*volx);
          frg_si[idx++]+=(int)h1[1];
        }
        if (sfakt<0) {  /* soundfile's rate is greater than devices's */
          for (i1=-2;i1>=sfakt;i1--) {
            if ((uptr=read_data(readbyte,ffp,data,dpos,dend))==NULL) {return(0);}  /* no more data */
          }
        } else {
          for (i1=2;i1<=sfakt;i1++) {
            frg_si[idx++]+=(int)h1[0];
            if (sdptr->channel==2) {frg_si[idx++]+=(int)h1[1];}
          }
        }
      }
    }
  } else {  /* 16bit soundfile */
    if (channel==1) {  /* mono soundfile */
      readbyte=2;
      while (idx<maxidx) {
        if ((uptr=read_data(readbyte,ffp,data,dpos,dend))==NULL) {return(0);}  /* no more data */
#ifdef ENDIAN_IS_BIG
        if (fext==FEXT_WAV) {  /* always 16LE bit */
          h1[0]=(uptr[1]<<8)|uptr[0];
        } else {  /* already 16BE bit */
          h1[0]=(uptr[0]<<8)|uptr[1];
        }
#else
        if (fext==FEXT_AU) {  /* always 16BE bit */
          h1[0]=(uptr[0]<<8)|uptr[1];
        } else {  /* already 16LE bit */
          h1[0]=(uptr[1]<<8)|uptr[0];
        }
#endif
        if (sdptr->channel==1) {  /* sound device mono */
          h1[0]=(short)((double)h1[0]*volx);
          frg_si[idx++]+=(int)h1[0];
        } else {  /* sound device stereo */
          h1[0]=(short)((double)h1[0]*volx);
          frg_si[idx++]+=(int)h1[0];
          frg_si[idx++]+=(int)h1[0];
        }
        if (sfakt<0) {  /* soundfile's rate is greater than devices's */
          for (i1=-2;i1>=sfakt;i1--) {
            if ((uptr=read_data(readbyte,ffp,data,dpos,dend))==NULL) {return(0);}  /* no more data */
          }
        } else {
          for (i1=2;i1<=sfakt;i1++) {
            frg_si[idx++]+=(int)h1[0];
            if (sdptr->channel==2) {frg_si[idx++]+=(int)h1[0];}
          }
        }
      }
    } else {  /* stereo soundfile */
      readbyte=4;
      while (idx<maxidx) {
        if ((uptr=read_data(readbyte,ffp,data,dpos,dend))==NULL) {return(0);}  /* no more data */
#ifdef ENDIAN_IS_BIG
        if (fext==FEXT_WAV) {  /* always 16LE bit */
          h1[0]=(uptr[1]<<8)|uptr[0];
          h1[1]=(uptr[3]<<8)|uptr[2];
        } else {  /* already 16BE bit */
          h1[0]=(uptr[0]<<8)|uptr[1];
          h1[1]=(uptr[2]<<8)|uptr[3];
        }
#else
        if (fext==FEXT_AU) {  /* always 16BE bit */
          h1[0]=(uptr[0]<<8)|uptr[1];
          h1[1]=(uptr[2]<<8)|uptr[3];
        } else {  /* already 16LE bit */
          h1[0]=(uptr[1]<<8)|uptr[0];
          h1[1]=(uptr[3]<<8)|uptr[2];
        }
#endif
        if (sdptr->channel==1) {  /* sound device mono */
          h1[0]=(h1[0]+h1[1])/2;
          h1[0]=(short)((double)h1[0]*volx);
          frg_si[idx++]+=(int)h1[0];
        } else {  /* sound device stereo */
          h1[0]=(short)((double)h1[0]*volx);
          frg_si[idx++]+=(int)h1[0];
          h1[1]=(short)((double)h1[1]*volx);
          frg_si[idx++]+=(int)h1[1];
        }
        if (sfakt<0) {  /* soundfile's rate is greater than devices's */
          for (i1=-2;i1>=sfakt;i1--) {
            if ((uptr=read_data(readbyte,ffp,data,dpos,dend))==NULL) {return(0);}  /* no more data */
          }
        } else {
          for (i1=2;i1<=sfakt;i1++) {
            frg_si[idx++]+=(int)h1[0];
            if (sdptr->channel==2) {frg_si[idx++]+=(int)h1[1];}
          }
        }
      }
    }
  }
  return(1);
} /* Ende fill_fragment */


int calc_sfakt(int samrate) {
  int sfakt;
  if (samrate>sdptr->srate) {
    if (samrate/2>sdptr->srate) {
      if (samrate/4>sdptr->srate) {
        sfakt=-4;
      } else if (samrate/2-sdptr->srate<sdptr->srate-samrate/4) {
        sfakt=-2;
      } else {sfakt=-4;}
    } else if (samrate-sdptr->srate<sdptr->srate-samrate/2) {
      sfakt=-1;
    } else {sfakt=-2;}
  } else {
    if (sdptr->srate/2>samrate) {
      if (sdptr->srate/4>samrate) {
        sfakt=4;
      } else if (sdptr->srate/2-samrate<samrate-sdptr->srate/4) {
        sfakt=2;
      } else {sfakt=4;}
    } else if (sdptr->srate-samrate<samrate-sdptr->srate/2) {
      sfakt=1;
    } else {sfakt=2;}
  }
  return(sfakt);
} /* Ende calc_sfakt */


unsigned char * read_data(int readbyte,FILE * ffp,unsigned char * data,int * dpos,int * dend) {
  unsigned char * rpos=NULL;
  for (;;) {
    if (*dpos+readbyte>*dend) {
      size_t rin;
      if (*dpos>0 && *dend>*dpos) {
        memmove(data,data+*dpos,*dend-*dpos);
        *dend-=*dpos;
      } else {*dend=0;}
      *dpos=0;
      if ((rin=fread(data+*dend,sizeof(unsigned char),BLOCK_LEN-*dend,ffp))==0) {return(NULL);}
      *dend+=rin;
    } else {rpos=data+*dpos; *dpos+=readbyte; break;}
  }
  return(rpos);
} /* Ende read_data */


double volofsound(int vdx) {
/* return real volume-percent of sound file */
  static int fmax=0,* fvol=NULL,* fanz=NULL;
  int idx,sdx;
  double retw;
  if (vdx<0) {  /* read in volume */
    char ** pptr;
    if (sf.volcalc) {sf.volcalc=0;} else {return(0);}
    if (fmax==0) {
      if ((fvol=malloc(sizeof(int)*sf.sn_max))==NULL) {return(0);}
      if ((fanz=malloc(sizeof(int)*sf.sn_max))==NULL) {return(0);}
      fmax=sf.sn_max;
    }
    if (fmax<sf.sn_max) {
      if ((fvol=realloc(fvol,sizeof(int)*sf.sn_max))==NULL) {fmax=0; return(0);}
      if ((fanz=realloc(fanz,sizeof(int)*sf.sn_max))==NULL) {fmax=0; return(0);}
      fmax=sf.sn_max;
    }
    memset(fvol,0,sizeof(int)*fmax);
    memset(fanz,0,sizeof(int)*fmax);
    for (idx=0;idx<sf.gn_max;idx++) {
      for (pptr=sf.gn[idx].s_name;*pptr!=NULL;pptr++) {
        if (inthash_get(sf.ihs,*pptr,&sdx)!=0) {continue;}
        fvol[sdx]+=sf.gn[idx].volume;
        fanz[sdx]++;
      }
    }
    retw=0.;
  } else {  /* give back volume */
    if (vdx>=fmax) {return(0);}
    retw=((double)sf.sn[vdx].volume+(double)fvol[vdx])/(1.+(double)fanz[vdx]);
    retw*=(double)sf.sn[vdx].volpc;
    retw/=10000.;
  }
  return(retw);
} /* Ende volofsound */


void helptopic(const char * topic) {
  if (topic!=NULL) {
    if (strcmp(topic,"ATTA")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  ATTA <soundfile> <s_name> <volumepercent>\n");
      fprintf(stderr,"    Attach a soundfile to a sound-name (a shortcut).\n");
      fprintf(stderr,"    This command loads the file with the given sound-name as a reference.\n");
      fprintf(stderr,"    The Extension of the soundfile has to be:\n");
      fprintf(stderr,"     - for wave:           .wav or .wave\n");
      fprintf(stderr,"     - for Sun/NeXT audio: .au\n");
      fprintf(stderr,"     - for midi:           .mid or .midi\n");
      fprintf(stderr,"     - for mp3:            .mp3\n");
      fprintf(stderr,"    If the soundfile does not have a / at the beginning,\n");
      fprintf(stderr,"    the prefix \"%s\" is put before.\n",workpath);
      fprintf(stderr,"    Use the sound-name (s_name) to refer to this soundfile with later\n");
      fprintf(stderr,"    commands. It may contain only letters, digits, underscores (_) and\n");
      fprintf(stderr,"    hyphens (-), and has to begin with \"s-\" with a maximum of 15 characters.\n");
      fprintf(stderr,"    The <volumepercent> defines the real volume compared to normal volume.\n");
      fprintf(stderr,"    It may be set from 0 to 200. If it is e.g. 150, this means,\n");
      fprintf(stderr,"    the file will be played one and a half as loud as normal.\n");
      fprintf(stderr,"    (Example: ATTA sounds/file1.wav s-shot 100)\n");
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"PLAY")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  PLAY <s_name> <crescendo> <loop>\n");
      fprintf(stderr,"    Play a soundfile referred to by the sound-name s_name once or more times.\n");
      fprintf(stderr,"    The crescendo is a time in seconds to become louder to the normal volume\n");
      fprintf(stderr,"    or 0. The loop is the number of times, how often to play it or 0=infinite.\n");
      fprintf(stderr,"    PLAY creates a new instance-number of this soundfile.\n");
      fprintf(stderr,"    As an attached soundfile can be played more than once at the same time,\n");
      fprintf(stderr,"    the instance-number refers to this playing instance of the soundfile.\n");
      fprintf(stderr,"    Some commands allow you to refer not only to the soundfile, but also to\n");
      fprintf(stderr,"    a specified instance of the soundfile with \"sound-name#instance-number\".\n");
      fprintf(stderr,"    (Example: PLAY s-shot 0 1)\n");
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"PAUS")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  PAUS <s_name | s_name#<inr> | g_name | %s>\n",GROUP_ALL);
      fprintf(stderr,"    Pause a soundfile.\n");
      fprintf(stderr,"    The soundfile can be referred to\n");
      fprintf(stderr,"    - by a sound-name (s_name)\n");
      fprintf(stderr,"      then all playing sounds of this soundfile are paused\n");
      fprintf(stderr,"      (Example: PAUS s-shot)\n");
      fprintf(stderr,"    - by a specified instance of a sound-name (s_name#inr)\n");
      fprintf(stderr,"      then only the specified instance of this soundfile is paused\n");
      fprintf(stderr,"      (Example: PAUS s-shot#13)\n");
      fprintf(stderr,"    - by a selfdefined group of soundfiles (g_name)\n");
      fprintf(stderr,"      then all soundfiles of this group are paused\n");
      fprintf(stderr,"      (Example: PAUS g-music)\n");
      fprintf(stderr,"    - by \"%s\" for all soundfiles\n",GROUP_ALL);
      fprintf(stderr,"      (Example: PAUS %s)\n",GROUP_ALL);
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"CONT")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  CONT <s_name | s_name#<inr> | g_name | %s>\n",GROUP_ALL);
      fprintf(stderr,"    Continue a with command PAUS paused soundfile.\n");
      fprintf(stderr,"    The soundfile can be referred to\n");
      fprintf(stderr,"    - by a sound-name (s_name)\n");
      fprintf(stderr,"      then all paused sounds of this soundfile are continued\n");
      fprintf(stderr,"      (Example: CONT s-shot)\n");
      fprintf(stderr,"    - by a specified instance of a sound-name (s_name#inr)\n");
      fprintf(stderr,"      then only the specified instance of this soundfile is continued\n");
      fprintf(stderr,"      (Example: CONT s-shot#13)\n");
      fprintf(stderr,"    - by a selfdefined group of soundfiles (g_name)\n");
      fprintf(stderr,"      then all soundfiles of this group are continued\n");
      fprintf(stderr,"      (Example: CONT g-music)\n");
      fprintf(stderr,"    - by \"%s\" for all soundfiles\n",GROUP_ALL);
      fprintf(stderr,"      (Example: CONT %s)\n",GROUP_ALL);
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"PATE")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  PATE <s_name | s_name#<inr> | g_name | %s>\n",GROUP_ALL);
      fprintf(stderr,"    Pause a soundfile at its end.\n");
      fprintf(stderr,"    This is only useful, if it would be started again (looping).\n");
      fprintf(stderr,"    The soundfile can be referred to\n");
      fprintf(stderr,"    - by a sound-name (s_name)\n");
      fprintf(stderr,"      then all playing sounds of this soundfile are paused-at-end\n");
      fprintf(stderr,"      (Example: PATE s-shot)\n");
      fprintf(stderr,"    - by a specified instance of a sound-name (s_name#inr)\n");
      fprintf(stderr,"      then only the specified instance of this soundfile is paused-at-end\n");
      fprintf(stderr,"      (Example: PATE s-shot#13)\n");
      fprintf(stderr,"    - by a selfdefined group of soundfiles (g_name)\n");
      fprintf(stderr,"      then all soundfiles of this group are paused-at-end\n");
      fprintf(stderr,"      (Example: PATE g-music)\n");
      fprintf(stderr,"    - by \"%s\" for all soundfiles\n",GROUP_ALL);
      fprintf(stderr,"      (Example: PATE %s)\n",GROUP_ALL);
      fprintf(stderr,"    To play on when paused-at-end, send a CATB.\n");
      fprintf(stderr,"    If you send a CATB before end playing, the command PATE will be cancelled.\n");
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"CATB")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  CATB <s_name | s_name#<inr> | g_name | %s>\n",GROUP_ALL);
      fprintf(stderr,"    Continue a with PATE paused-at-end soundfile.\n");
      fprintf(stderr,"    The soundfile can be referred to\n");
      fprintf(stderr,"    - by a sound-name (s_name)\n");
      fprintf(stderr,"      then all paused-at-end sounds of this soundfile are continued\n");
      fprintf(stderr,"      (Example: CATB s-shot)\n");
      fprintf(stderr,"    - by a specified instance of a sound-name (s_name#inr)\n");
      fprintf(stderr,"      then only the specified instance of this soundfile is continued\n");
      fprintf(stderr,"      (Example: CATB s-shot#13)\n");
      fprintf(stderr,"    - by a selfdefined group of soundfiles (g_name)\n");
      fprintf(stderr,"      then all soundfiles of this group are continued\n");
      fprintf(stderr,"      (Example: CATB g-music)\n");
      fprintf(stderr,"    - by \"%s\" for all soundfiles\n",GROUP_ALL);
      fprintf(stderr,"      (Example: CATB %s)\n",GROUP_ALL);
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"STOP")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  STOP <s_name | s_name#<inr> | g_name | %s> <diminuendo>\n",GROUP_ALL);
      fprintf(stderr,"    End (immediatelly) a playing soundfile.\n");
      fprintf(stderr,"    The <diminuendo> is a time in seconds to become lower\n");
      fprintf(stderr,"    or set it to 0 means: stop immediatelly.\n");
      fprintf(stderr,"    The soundfile can be referred to\n");
      fprintf(stderr,"    - by a sound-name (s_name)\n");
      fprintf(stderr,"      then all playing sounds of this soundfile are stopped\n");
      fprintf(stderr,"      (Example: STOP s-shot 0)\n");
      fprintf(stderr,"    - by a specified instance of a sound-name (s_name#inr)\n");
      fprintf(stderr,"      then only the specified instance of this soundfile is stopped\n");
      fprintf(stderr,"      (Example: STOP s-shot#13 0)\n");
      fprintf(stderr,"    - by a selfdefined group of soundfiles (g_name)\n");
      fprintf(stderr,"      then all soundfiles of this group are stopped\n");
      fprintf(stderr,"      (Example: STOP g-music 0)\n");
      fprintf(stderr,"    - by \"%s\" for all soundfiles\n",GROUP_ALL);
      fprintf(stderr,"      (Example: STOP %s 0)\n",GROUP_ALL);
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"GCLR")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  GCLR <g_name>\n");
      fprintf(stderr,"    Clear a group with group-name g_name.\n");
      fprintf(stderr,"    The group-name may contain letters, digits, underscores (_) and hyphens (-)\n");
      fprintf(stderr,"    and has to begin with \"g-\".\n");
      fprintf(stderr,"    (Example: GCLR g-music)\n");
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"GADD")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  GADD <g_name> <s_name | &s_patt>\n");
      fprintf(stderr,"    Add a soundfile to a group with group-name g_name.\n");
      fprintf(stderr,"    The group-name may contain letters, digits, underscores (_) and hyphens (-)\n");
      fprintf(stderr,"    and has to begin with \"g-\".\n");
      fprintf(stderr,"    The soundfile to be added can be referred to\n");
      fprintf(stderr,"    - by a sound-name (s_name)\n");
      fprintf(stderr,"      (Example: GADD g-music s-song1)\n");
      fprintf(stderr,"    - by a (shell)pattern which matches one or more sound-names (&s_patt)\n");
      fprintf(stderr,"      (pattern characters: ?*[])\n");
      fprintf(stderr,"      To distinguish a pattern from a sound-name, a \"&\" must be put before it.\n");
      fprintf(stderr,"      (Example: GADD g-music &s-song?)\n");
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"GDEL")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  GDEL <g_name> <s_name | &s_patt>\n");
      fprintf(stderr,"    Delete a soundfile from a group with group-name g_name.\n");
      fprintf(stderr,"    The group-name may contain letters, digits, underscores (_) and hyphens (-)\n");
      fprintf(stderr,"    and has to begin with \"g-\".\n");
      fprintf(stderr,"    The soundfile to be deleted can be referred to\n");
      fprintf(stderr,"    - by a sound-name (s_name)\n");
      fprintf(stderr,"      (Example: GDEL g-music s-song1)\n");
      fprintf(stderr,"    - by a (shell)pattern which matches one or more sound-names (&s_patt)\n");
      fprintf(stderr,"      (pattern characters: ?*[])\n");
      fprintf(stderr,"      To distinguish a pattern from a sound-name, a \"&\" must be put before it.\n");
      fprintf(stderr,"      (Example: GDEL g-music &s-song?)\n");
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"VOLM")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  VOLM <s_name | g_name | 0>[ <volume>]\n");
      fprintf(stderr,"    Change volume and return previous value.\n");
      fprintf(stderr,"    <volume> is a number from 0 to 100.\n");
      fprintf(stderr,"    If missing, only the current volume is returned.\n");
      fprintf(stderr,"    The volume can be changed for\n");
      fprintf(stderr,"    - a sound-name (s_name)\n");
      fprintf(stderr,"      (Example: VOLM s-shot 90)\n");
      fprintf(stderr,"    - a group-name (g_name), which affects all sound-names of this group\n");
      fprintf(stderr,"      (Example: VOLM g-music 90)\n");
      fprintf(stderr,"    - all by changing the mixer volume (0)\n");
      fprintf(stderr,"      (Example: VOLM 0 90)\n");
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"FRGS")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  FRGS [<samplerate>,<channels>,<number of fragments>,<fragment size>]\n");
      fprintf(stderr,"    Change samplerate, channels and fragments and return previous values.\n");
      fprintf(stderr,"    <samplerate> is 11025, 22050 or 44100.\n");
      fprintf(stderr,"    <channels> is 1 (for mono) or 2 (for stereo).\n");
      fprintf(stderr,"    <number of fragments> is a number from 128 to ?.\n");
      fprintf(stderr,"    <fragment size> is a number from 2 to ?.\n");
      fprintf(stderr,"    The more fragments resp. the bigger the size,\n");
      fprintf(stderr,"    the slower are the reactions to a sound command,\n");
      fprintf(stderr,"    but the smaller is the possibility to hear 'clicks'.\n");
      fprintf(stderr,"    Each of the arguments may be 0 (=don't change).\n");
      fprintf(stderr,"    (Example: FRGS 22050,1,3,1024)\n");
      fprintf(stderr,"\n");
    } else if (strcmp(topic,"HELP")==0) {
      fprintf(stderr,"Command help:\n");
      fprintf(stderr,"  HELP\n");
      fprintf(stderr,"    This help.\n");
      fprintf(stderr,"\n");
    } else {topic=NULL;}
  }
  if (topic==NULL) {
    fprintf(stderr,"Following commands are defined:\n");
    fprintf(stderr,"  \"ATTA <soundfile> <s_name> <volumepercent>\"\n");
    fprintf(stderr,"  \"PLAY <s_name> <crescendo> <loop>\"\n");
    fprintf(stderr,"  \"PAUS <s_name | g_name>\"\n");
    fprintf(stderr,"  \"CONT <s_name | g_name>\"\n");
    fprintf(stderr,"  \"PATE <s_name | g_name>\"\n");
    fprintf(stderr,"  \"CATB <s_name | g_name>\"\n");
    fprintf(stderr,"  \"STOP <s_name | g_name> <diminuendo>\"\n");
    fprintf(stderr,"  \"GCLR <g_name>\"\n");
    fprintf(stderr,"  \"GADD <g_name> <s_name | &s_patt>\"\n");
    fprintf(stderr,"  \"GDEL <g_name> <s_name | &s_patt>\"\n");
    fprintf(stderr,"  \"VOLM <s_name | g_name | 0> <volume>\"\n");
    fprintf(stderr,"  \"FRGS [<samplerate>,<channels>,<number of fragments>,<fragment size>]\"\n");
    fprintf(stderr,"  \"HELP [<topic>]\"\n");
    fprintf(stderr,"To quit close stdin (CTRL-D)\n");
    fprintf(stderr,"\n");
  }
} /* Ende helptopic */
