/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* main file */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include "config.h"
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#include "vgagames2.h"
#include "cwdir.h"
#include "sysmenu.h"
#include "textformat.h"
#ifdef HAVE_X11
# include "video_x11.h"
#endif
#ifdef HAVE_SVGALIB
# include "video_svgalib.h"
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
# include "video_vgl.h"
#endif

extern int vg_nw_isopen();
extern int vg_nw_update(unsigned char *);

static int smenu=0;
char _cwdir[512]="",_cwdir2[512]="",_sharedir[512]="";
int _rgbblack,_rgbwhite,_rgbdark;
int block_sysmenu;
struct vol_name volume_name;
struct w_mode window_mode;
struct s_mode sound_mode;
struct l_mode lang_mode;

static void propfile_in(const char *);
void propfile_out(void);

int vg_init_vgagames(const char *,int,const char *);
int vg_wait_time(int);
int vg_window_open(const char *,int,int);
void vg_window_flush(void);
void vg_window_close(void);
void vg_key_discard(void);
int vg_key_update(void);
int vg_key_pressed(int,int);
int vg_mouse_found(void);
int vg_mouse_x(void);
int vg_mouse_y(void);
int vg_mouse_pressed(int,int);
int vg_mouse_speed(int);
void vg_menu_volume(void);
void vg_menu_language(void);
int vg_timestamp(int *,int *,int *);


/* +++ internal functions +++ */

static void propfile_in(const char * lkz) {
/* read in vgagames-properties file */
  char buf[512];
  FILE * ffp;
  default_textformat(lkz);
  snprintf(buf,sizeof(buf),"%s/share/.vgag2.rc",CWDIR);
  if ((ffp=fopen(buf,"r"))!=NULL) {
    if (fread(&window_mode,sizeof(window_mode),1,ffp)==1) {window_mode.isset=1;} else {window_mode.isset=0;}
    fread(&sound_mode,sizeof(sound_mode),1,ffp);
    fread(&lang_mode,sizeof(lang_mode),1,ffp);
    if (*lang_mode.lkz!='\0') {lkz=lang_mode.lkz;}
    load_textformat(lkz);
    fclose(ffp);
  } else {load_textformat(lkz);}
} /* Ende propfile_in */


void propfile_out() {
/* write out vgagames-properties file */
  char buf[512];
  FILE * ffp;
  mode_t m1,m2;
  snprintf(buf,sizeof(buf),"%s/share",CWDIR);
  m1=umask(0);
  m2=0777;
  mkdir(buf,m2);
  snprintf(buf,sizeof(buf),"%s/share/.vgag2.rc",CWDIR);
  unlink(buf);
  if ((ffp=fopen(buf,"w"))!=NULL) {
    fwrite(&window_mode,sizeof(window_mode),1,ffp);
    fwrite(&sound_mode,sizeof(sound_mode),1,ffp);
    fwrite(&lang_mode,sizeof(lang_mode),1,ffp);
    m2=0666;
    fchmod(fileno(ffp),m2);
    fclose(ffp);
  }
  umask(m1);
} /* Ende propfile_out */


/* +++ functions +++ */

int vg_init_vgagames(const char * argv0,int flag,const char * lkz) {
/* some initializations
** 1.arg: argv[0]
** 2.arg: following values or'ed:
**        INITVGA_NOFILE=do not read vgagames-properties file
** 3.arg: use this language (2-characters lowercase), if no language is set
**          in the vgagames-properties file
**        or NULL
** return: 0=OK or -1=error
** vgagames-properties file: <CWDIR>/share/.vgag2.rc
*/
  static int neu=1;
  int i1;
  if (neu==0) {return(0);}
  neu=0;
  block_sysmenu=0;
  if (get_cwdir(argv0,CWDIR,sizeof(CWDIR))<0) {fprintf(stderr,"vg_init_vgagames: error calling get_cwdir.\n"); return(-1);}
  *_cwdir2='\0';
  memset(&window_mode,0,sizeof(window_mode));
  memset(&sound_mode,0,sizeof(sound_mode));
  memset(&lang_mode,0,sizeof(lang_mode));
  memset(&volume_name,0,sizeof(volume_name));
  for (i1=0;i1<=VOLUME_MAXNO;i1++) {sound_mode.vol[i1]=-1;}
  if ((flag&INITVGA_NOFILE)==0) {propfile_in(lkz);}
  return(0);
} /* Ende vg_init_vgagames */


int vg_wait_time(int m_sec) {
/* wait rest of 1.arg milliseconds since last call
** return: slept time in milliseconds or -1=error
*/
  static struct timeval tv;
  static struct timezone tz;
  static int vs=0,ms,ws,slt=0;
  if (gettimeofday(&tv,&tz)!=0) {fprintf(stderr,"vg_wait_time: gettimeofday: %s.\n",strerror(errno)); return(-1);}
  ms=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  if (ms<vs) {ws=60000+ms-vs;} else {ws=ms-vs;}
  if (ws<m_sec) {
    if ((m_sec-=TIME_OF_SELECT)<0) {m_sec=0;}
    if (ws<m_sec) {
      tv.tv_usec=(m_sec-ws)%1000*1000;
      tv.tv_sec=(m_sec-ws)/1000;
    } else {tv.tv_usec=1; tv.tv_sec=0;}
    select(0,NULL,NULL,NULL,&tv);
    slt=tv.tv_sec*1000+tv.tv_usec/1000;
  } else {slt=0;}
  if (gettimeofday(&tv,&tz)!=0) {fprintf(stderr,"vg_wait_time: gettimeofday: %s.\n",strerror(errno)); return(-1);}
  ms=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  if (ms<vs) {ws=60000+ms-vs;} else {ws=ms-vs;}
  vs=ms;
  return(slt);
} /* Ende vg_wait_time */


int vg_window_open(const char * wnam,int dp,int vidlib) {
/* create window
** for X-window:
**   1.arg: name of the window (NULL=same name as before)
**   2.arg: window properties
**          0:              get largest window size, no full screen
**          VGAWINDOW_1:    window scaling factor 1  (original size)
**          VGAWINDOW_2:    window scaling factor 2  (4 pixels per pixel)
**          VGAWINDOW_3:    window scaling factor 3  (9 pixels per pixel)
**          VGAWINDOW_FULL: full screen modus
**          VGAWINDOW_NOSWITCH: don't switch to another mode
**          VGAWINDOW_DIRECTDRAW: activate directdraw
** for console libraries:
**   1.arg: NULL
**   2.arg: 0
** for all:
**   3.arg: one of VIDEOLIB_??? to use only this video library
**          or 0=try automatically to find one
** return: 0=ok or -1=error
**
** Note: the values of the 2.arg will be ignored if the vgagames-properties
**       already have been read from file (see vg_init_vgagames())
*/
  static int firstcall=1;
  static int lastmode=0;
  int erg=1;
  uid_t uid;
  if (lastmode>0) {vidlib=lastmode;}
  uid=geteuid();

#ifdef HAVE_SVGALIB
  /* svgalib makes an exit() if its fails */
  if ((erg) && (uid==0) && ((vidlib==0) || (vidlib==VIDEOLIB_SVGALIB))) {
    printf("Trying svgalib ...\n"); fflush(stdout);
    erg=window_open_svgalib();
  }
#endif

#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if ((erg) && (uid==0) && ((vidlib==0) || (vidlib==VIDEOLIB_VGL))) {
    printf("Trying libvgl ...\n"); fflush(stdout);
    erg=window_open_vgl();
  }
#endif

#ifdef HAVE_X11
  (void)uid;
  if ((erg) && ((vidlib==0) || (vidlib==VIDEOLIB_X11))) {
    printf("Trying X11 ...\n"); fflush(stdout);
    erg=window_open_x11(wnam,dp);
  }
#endif

  if (erg) {
    fprintf(stderr,"vg_window_open: cannot open window (video library: %d).\n",vidlib);
    return(-1);
  }

  if (vg_load_colormap(RGB256,NULL,0)<0) {return(-1);}

  propfile_out();
  lastmode=window_mode.videolib;

  if (firstcall) {printf("Press [ESCAPE] for a system menu.\n"); fflush(stdout);}
  firstcall=0;
  return(0);
} /* Ende vg_window_open */


void vg_window_flush() {
/* give backbuffer out to screen */
  if (window_mode.videolib==0) {return;}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    window_flush_x11();
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    window_flush_svgalib();
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    window_flush_vgl();
  }
#endif
} /* Ende vg_window_flush */


void vg_window_close() {
/* close window */
  if (window_mode.videolib==0) {return;}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    window_close_x11();
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    window_close_svgalib();
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    window_close_vgl();
  }
#endif
  window_mode.videolib=0;
} /* Ende vg_window_close */


void vg_key_discard() {
/* discard keys/mousebuttons */
  if (window_mode.videolib==0) {return;}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    key_discard_x11();
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    key_discard_svgalib();
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    key_discard_vgl();
  }
#endif
} /* Ende vg_key_discard */


int vg_key_update() {
/* get pressed or released keys/mousebuttons
** return:  0=OK
**         -1=network closed (only if network available)
*/
  unsigned char * ktn=NULL;
  if (window_mode.videolib==0) {return(0);}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    key_update_x11();
    if (vg_nw_isopen()) {ktn=keystonw_x11();}
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    key_update_svgalib();
    if (vg_nw_isopen()) {ktn=keystonw_svgalib();}
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    key_update_vgl();
    if (vg_nw_isopen()) {ktn=keystonw_vgl();}
  }
#endif
  if ((smenu==0) && (vg_key_pressed(KEYS_INDEX,SHORTKEY))) {  /* Escape: system menu */
    smenu=1;
    system_menu(0);
    smenu=0;
    return(0);
  }
  return(vg_nw_update(ktn));
} /* Ende vg_key_update */


int vg_key_pressed(int key,int art) {
/* whether a key is pressed
** 1.arg: key-define (KEY_0, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
*/
  int rval=0;
  if (window_mode.videolib==0) {return(0);}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    rval=key_pressed_x11(key,art);
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    rval=key_pressed_svgalib(key,art);
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    rval=key_pressed_vgl(key,art);
  }
#endif
  return(rval);
} /* Ende vg_key_pressed */


int vg_mouse_found() {
/* whether a mouse is found
** return: 0=not found, 1=found
*/
  int rval=0;
  if (window_mode.videolib==0) {return(0);}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    rval=mouse_found_x11();
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    rval=mouse_found_svgalib();
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    rval=mouse_found_vgl();
  }
#endif
  return(rval);
} /* Ende vg_mouse_found */


int vg_mouse_x() {
/* return mouse x-position or -1=not in window */
  int rval=0;
  if (window_mode.videolib==0) {return(0);}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    rval=mouse_x_x11();
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    rval=mouse_x_svgalib();
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    rval=mouse_x_vgl();
  }
#endif
  return(rval);
} /* Ende vg_mouse_x */


int vg_mouse_y() {
/* return mouse y-position or -1=not in window */
  int rval=0;
  if (window_mode.videolib==0) {return(0);}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    rval=mouse_y_x11();
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    rval=mouse_y_svgalib();
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    rval=mouse_y_vgl();
  }
#endif
  return(rval);
} /* Ende vg_mouse_y */


int vg_mouse_pressed(int bnr,int art) {
/* whether a mousebutton is pressed
** 1.arg: mousebutton-define (MOUSE_LEFT, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
*/
  int rval=0;
  if (window_mode.videolib==0) {return(0);}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    rval=mouse_pressed_x11(bnr,art);
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    rval=mouse_pressed_svgalib(bnr,art);
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    rval=mouse_pressed_vgl(bnr,art);
  }
#endif
  return(rval);
} /* Ende vg_mouse_pressed */


int vg_mouse_speed(int index) {
/* set speed of mouse
** 1.arg: 1=slow, 2=middle, 3=fast
**        0=return only current speed
** return: previous speed number
**         or 0=speed cannot be changed
*/
  int rval=0;
  if (window_mode.videolib==0) {return(0);}
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    rval=mouse_speed_x11(index);
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    rval=mouse_speed_svgalib(index);
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    rval=mouse_speed_vgl(index);
  }
#endif
  return(rval);
} /* Ende vg_mouse_speed */


void vg_menu_volume() {
/* call sound volume menu directly */
  if (smenu==0) {
    smenu=1;
    system_menu(1);
    smenu=0;
  }
} /* Ende vg_menu_volume */


void vg_menu_language() {
/* call language menu directly */
  if (smenu==0) {
    smenu=1;
    system_menu(2);
    smenu=0;
  }
} /* Ende vg_menu_language */


int vg_timestamp(int * min,int * sek,int * msek) {
/* return the actual time
** 1.arg: address for minutes or NULL
** 2.arg: address for seconds or NULL
** 3.arg: address for milliseconds or NULL
** return: 0=OK or -1=error
*/
  struct timeval tv;
  int m1,m2,m3;
  if (gettimeofday(&tv,NULL)<0) {fprintf(stderr,"vg_timestamp: gettimeofday: %s\n",strerror(errno)); return(-1);}
  m1=(tv.tv_sec%3600)/60;
  m2=tv.tv_sec%60;
  m3=tv.tv_usec/1000;
  if (min!=NULL) {*min=m1;}
  if (sek!=NULL) {*sek=m2;}
  if (msek!=NULL) {*msek=m3;}
  return(0);
} /* Ende vg_timestamp */
