/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* window and input functions for svgalib */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vga.h>
#include <vgagl.h>
#include <vgamouse.h>
#include <vgakeyboard.h>
#include "config.h"
#include "vgagames2.h"
#include "bitmap.h"
#include "color.h"


/* +++ declarations +++ */

static struct {  /* for window */
  int              is_open;       /* window has been opened? */
  GraphicsContext  * pys;         /* window */
  bitmap           * backbuffer;  /* backbuffer */
} wd={0};

static struct {  /* for input */
  int k_cmp[KEYS_INDEX+1];
  char k_press[KEYS_INDEX+1],k_mody[KEYS_INDEX+1],k_flush[KEYS_INDEX+1];
  int m_missing;
  int m_press[3+1],m_mody[3+1],m_flush[3+1];
  int m_xpos,m_ypos;
} inp;


static int init_keys(void);

int window_open_svgalib(void);
void window_flush_svgalib(void);
void window_close_svgalib(void);

void key_discard_svgalib(void);
void key_update_svgalib(void);
int key_pressed_svgalib(int,int);
int mouse_found_svgalib(void);
int mouse_x_svgalib(void);
int mouse_y_svgalib(void);
int mouse_pressed_svgalib(int,int);
int mouse_speed_svgalib(int);
unsigned char * keystonw_svgalib(void);

int brightness_svgalib(int);


/* +++ internal functions +++ */

static int init_keys() {
/* initalize keys, called from window_open_svgalib() */
  static int firstcall=1;
  if (!firstcall) {return(0);}
  firstcall=0;
  inp.k_cmp[KEY_0]=SCANCODE_0;
  inp.k_cmp[KEY_1]=SCANCODE_1;
  inp.k_cmp[KEY_2]=SCANCODE_2;
  inp.k_cmp[KEY_3]=SCANCODE_3;
  inp.k_cmp[KEY_4]=SCANCODE_4;
  inp.k_cmp[KEY_5]=SCANCODE_5;
  inp.k_cmp[KEY_6]=SCANCODE_6;
  inp.k_cmp[KEY_7]=SCANCODE_7;
  inp.k_cmp[KEY_8]=SCANCODE_8;
  inp.k_cmp[KEY_9]=SCANCODE_9;
  inp.k_cmp[KEY_A]=SCANCODE_A;
  inp.k_cmp[KEY_B]=SCANCODE_B;
  inp.k_cmp[KEY_C]=SCANCODE_C;
  inp.k_cmp[KEY_D]=SCANCODE_D;
  inp.k_cmp[KEY_E]=SCANCODE_E;
  inp.k_cmp[KEY_F]=SCANCODE_F;
  inp.k_cmp[KEY_G]=SCANCODE_G;
  inp.k_cmp[KEY_H]=SCANCODE_H;
  inp.k_cmp[KEY_I]=SCANCODE_I;
  inp.k_cmp[KEY_J]=SCANCODE_J;
  inp.k_cmp[KEY_K]=SCANCODE_K;
  inp.k_cmp[KEY_L]=SCANCODE_L;
  inp.k_cmp[KEY_M]=SCANCODE_M;
  inp.k_cmp[KEY_N]=SCANCODE_N;
  inp.k_cmp[KEY_O]=SCANCODE_O;
  inp.k_cmp[KEY_P]=SCANCODE_P;
  inp.k_cmp[KEY_Q]=SCANCODE_Q;
  inp.k_cmp[KEY_R]=SCANCODE_R;
  inp.k_cmp[KEY_S]=SCANCODE_S;
  inp.k_cmp[KEY_T]=SCANCODE_T;
  inp.k_cmp[KEY_U]=SCANCODE_U;
  inp.k_cmp[KEY_V]=SCANCODE_V;
  inp.k_cmp[KEY_W]=SCANCODE_W;
  inp.k_cmp[KEY_X]=SCANCODE_X;
  inp.k_cmp[KEY_Y]=SCANCODE_Y;
  inp.k_cmp[KEY_Z]=SCANCODE_Z;
  inp.k_cmp[KEY_KP_0]=SCANCODE_KEYPAD0;
  inp.k_cmp[KEY_KP_1]=SCANCODE_KEYPAD1;
  inp.k_cmp[KEY_KP_2]=SCANCODE_KEYPAD2;
  inp.k_cmp[KEY_KP_3]=SCANCODE_KEYPAD3;
  inp.k_cmp[KEY_KP_4]=SCANCODE_KEYPAD4;
  inp.k_cmp[KEY_KP_5]=SCANCODE_KEYPAD5;
  inp.k_cmp[KEY_KP_6]=SCANCODE_KEYPAD6;
  inp.k_cmp[KEY_KP_7]=SCANCODE_KEYPAD7;
  inp.k_cmp[KEY_KP_8]=SCANCODE_KEYPAD8;
  inp.k_cmp[KEY_KP_9]=SCANCODE_KEYPAD9;
  inp.k_cmp[KEY_TAB]=SCANCODE_TAB;
  inp.k_cmp[KEY_LCTRL]=SCANCODE_LEFTCONTROL;
  inp.k_cmp[KEY_RCTRL]=SCANCODE_RIGHTCONTROL;
  inp.k_cmp[KEY_LALT]=SCANCODE_LEFTALT;
  inp.k_cmp[KEY_RALT]=SCANCODE_RIGHTALT;
  inp.k_cmp[KEY_SPACE]=SCANCODE_SPACE;
  inp.k_cmp[KEY_ENTER]=SCANCODE_ENTER;
  inp.k_cmp[KEY_BSP]=SCANCODE_BACKSPACE;
  inp.k_cmp[KEY_RCURS]=SCANCODE_CURSORRIGHT;
  inp.k_cmp[KEY_LCURS]=SCANCODE_CURSORLEFT;
  inp.k_cmp[KEY_UCURS]=SCANCODE_CURSORUP;
  inp.k_cmp[KEY_DCURS]=SCANCODE_CURSORDOWN;
  inp.k_cmp[KEY_F1]=SCANCODE_F1;
  inp.k_cmp[KEY_F2]=SCANCODE_F2;
  inp.k_cmp[KEY_F3]=SCANCODE_F3;
  inp.k_cmp[KEY_F4]=SCANCODE_F4;
  inp.k_cmp[KEY_F5]=SCANCODE_F5;
  inp.k_cmp[KEY_F6]=SCANCODE_F6;
  inp.k_cmp[KEY_F7]=SCANCODE_F7;
  inp.k_cmp[KEY_F8]=SCANCODE_F8;
  inp.k_cmp[KEY_F9]=SCANCODE_F9;
  inp.k_cmp[KEY_F10]=SCANCODE_F10;
  inp.k_cmp[KEY_F11]=SCANCODE_F11;
  inp.k_cmp[KEY_F12]=SCANCODE_F12;
  inp.k_cmp[KEYS_INDEX]=SCANCODE_ESCAPE;  /* system menu */
  return(0);
} /* Ende init_keys */


/* +++ functions +++ */

int window_open_svgalib() {
/* switch console screen into graphic mode
** return: 0=ok or -1=error
*/
  static char firstcall=1;

  if (firstcall) {
    memset(&wd,0,sizeof(wd));
    memset(&inp,0,sizeof(inp));
    wd.is_open=0;
  }

  /* set graphic mode */
  if (vga_init()!=0) {fprintf(stderr,"window_open_svgalib: error calling vga_init.\n"); return(-1);}
  if (!vga_hasmode(G320x200x256)) {fprintf(stderr,"window_open_svgalib: vga_hasmode: mode G320x200x256 is not supported.\n"); return(-1);}
  if (vga_setmode(G320x200x256)!=0) {fprintf(stderr,"window_open_svgalib: vga_setmode: set mode G320x200x256 failed.\n"); return(-1);}
  if (gl_setcontextvga(G320x200x256)!=0) {fprintf(stderr,"window_open_svgalib: gl_setcontextvga with mode G320x200x256 failed.\n"); vga_setmode(TEXT); return(-1);}
  gl_enableclipping();
  wd.pys=gl_allocatecontext();
  gl_getcontext(wd.pys);

  /* create backbuffer */
  wd.backbuffer=vg_bitmap_createnew(SC_WIDTH,SC_HEIGHT);
  if (wd.backbuffer==NULL) {fprintf(stderr,"window_open_svgalib: can't create backbuffer.\n"); vga_setmode(TEXT); return(-1);}
  backbuffer=wd.backbuffer;

  /* initialize keyboard and mouse */
  if (init_keys()<0) {fprintf(stderr,"window_open_svgalib: init_keys error.\n"); vga_setmode(TEXT); return(-1);}
  memset(inp.k_press,0,sizeof(inp.k_press));
  memset(inp.k_mody,0,sizeof(inp.k_mody));
  memset(inp.k_flush,0,sizeof(inp.k_flush));
  if (keyboard_init()<0) {fprintf(stderr,"window_open_svgalib: error calling keyboard_init.\n"); vga_setmode(TEXT); return(-1);}
  keyboard_translatekeys(TRANSLATE_CURSORKEYS|TRANSLATE_KEYPADENTER);

  if ((vga_getmousetype()&MOUSE_TYPE_MASK)==MOUSE_NONE) {inp.m_missing=1;} else {inp.m_missing=0; vga_setmousesupport(1);}
  inp.m_xpos=inp.m_ypos=-1;
  memset(inp.m_press,0,sizeof(inp.m_press));
  memset(inp.m_mody,0,sizeof(inp.m_mody));
  memset(inp.m_flush,0,sizeof(inp.m_flush));
  if (!inp.m_missing) {mouse_setwrap(MOUSE_NOWRAP);}
  keyboard_clearstate();
  wd.is_open=1;
  if (!inp.m_missing) {mouse_speed_svgalib(2);}

  window_mode.isset=1;
  window_mode.videolib=VIDEOLIB_SVGALIB;
  window_mode.vlb[VIDEOLIB_SVGALIB].scale=VGAWINDOW_1;
  window_mode.vlb[VIDEOLIB_SVGALIB].full=1;
  window_mode.vlb[VIDEOLIB_SVGALIB].noswitch=1;
  window_mode.vlb[VIDEOLIB_SVGALIB].directdraw=0;
  window_mode.vlb[VIDEOLIB_SVGALIB].modenr=window_mode.vlb[VIDEOLIB_SVGALIB].modemax=0;
  window_mode.vlb[VIDEOLIB_SVGALIB].modewidth=SC_WIDTH;
  window_mode.vlb[VIDEOLIB_SVGALIB].modeheight=SC_HEIGHT;
  return(0);
} /* Ende window_open_svgalib */


void window_flush_svgalib() {
/* give backbuffer out to screen */
  if (!wd.is_open) {return;}
  memmove(wd.pys->vbuf,wd.backbuffer->pxm,SC_WIDTH*SC_HEIGHT);
} /* Ende window_flush_svgalib */


void window_close_svgalib() {
/* close window */
  if (!wd.is_open) {return;}
  vg_bitmap_free(wd.backbuffer); wd.backbuffer=NULL;
  if (wd.pys!=NULL) {gl_freecontext(wd.pys);}
  keyboard_close();
  vga_setmode(TEXT);
  wd.is_open=0;
} /* Ende window_close_svgalib */


void key_discard_svgalib() {
  if (!wd.is_open) {return;}
  key_update_svgalib();
  memset(inp.k_flush,1,sizeof(inp.k_flush));
  memset(inp.m_flush,1,sizeof(inp.m_flush));
} /* Ende key_discard_svgalib */


void key_update_svgalib() {
/* get pressed or released keys/mousebuttons */
  int i1,i2;

  if (!wd.is_open) {return;}
  memset(inp.k_mody,0,sizeof(inp.k_mody));
  memset(inp.m_mody,0,sizeof(inp.m_mody));

  keyboard_update();  /* get keyboard input */
  for (i1=0;i1<=KEYS_INDEX;i1++) {
    i2=keyboard_keypressed(inp.k_cmp[i1]);
    if (i2==KEY_PRESSED) {
      if (inp.k_press[i1]==0) {inp.k_mody[i1]=1;} else {inp.k_mody[i1]=0;}
      inp.k_press[i1]=1;
    } else if (i2==KEY_NOTPRESSED) {
      if (inp.k_press[i1]==1) {inp.k_mody[i1]=1;} else {inp.k_mody[i1]=0;}
      inp.k_press[i1]=0;
    }
  }

  if (!inp.m_missing) {  /* mouse found */
    static int bzx=0,bzy=0;
    mouse_update();  /* get mouse events */
    i1=mouse_getbutton();  /* check whether button pressed */
    inp.m_xpos=mouse_getx()-bzx;  /* set x position of mouse */
    if (inp.m_xpos>SC_WIDTH) {
       bzx+=inp.m_xpos-SC_WIDTH; inp.m_xpos=SC_WIDTH;
    } else if (inp.m_xpos<0) {bzx+=inp.m_xpos; inp.m_xpos=0;}
    inp.m_ypos=mouse_gety()-bzy;  /* set y position of mouse */
    if (inp.m_ypos>SC_HEIGHT) {
      bzy+=inp.m_ypos-SC_HEIGHT; inp.m_ypos=SC_HEIGHT;
    } else if (inp.m_ypos<0) {bzy+=inp.m_ypos; inp.m_ypos=0;}
    if (i1&MOUSE_LEFTBUTTON) {  /* left button pressed */
      if (inp.m_press[MOUSE_LEFT]==0) {inp.m_mody[MOUSE_LEFT]=1;} else {inp.m_mody[MOUSE_LEFT]=0;}
      inp.m_press[MOUSE_LEFT]=1;
    } else {
      if (inp.m_press[MOUSE_LEFT]==1) {inp.m_mody[MOUSE_LEFT]=1;} else {inp.m_mody[MOUSE_LEFT]=0;}
      inp.m_press[MOUSE_LEFT]=0;
    }
    if (i1&MOUSE_RIGHTBUTTON) {  /* right button pressed */
      if (inp.m_press[MOUSE_RIGHT]==0) {inp.m_mody[MOUSE_RIGHT]=1;} else {inp.m_mody[MOUSE_RIGHT]=0;}
      inp.m_press[MOUSE_RIGHT]=1;
    } else {
      if (inp.m_press[MOUSE_RIGHT]==1) {inp.m_mody[MOUSE_RIGHT]=1;} else {inp.m_mody[MOUSE_RIGHT]=0;}
      inp.m_press[MOUSE_RIGHT]=0;
    }
    if (i1&MOUSE_MIDDLEBUTTON) {  /* middle button pressed */
      if (inp.m_press[MOUSE_MIDDLE]==0) {inp.m_mody[MOUSE_MIDDLE]=1;} else {inp.m_mody[MOUSE_MIDDLE]=0;}
      inp.m_press[MOUSE_MIDDLE]=1;
    } else {
      if (inp.m_press[MOUSE_MIDDLE]==1) {inp.m_mody[MOUSE_MIDDLE]=1;} else {inp.m_mody[MOUSE_MIDDLE]=0;}
      inp.m_press[MOUSE_MIDDLE]=0;
    }
  }
} /* Ende key_update_svgalib */


int key_pressed_svgalib(int key,int art) {
/* whether a key is pressed
** 1.arg: key-define (KEY_0, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
*/
  if (!wd.is_open) {return(0);}
  if ((key<0) || (key>KEYS_INDEX)) {return(0);}
  if (inp.k_flush[key]) {
    if (inp.k_press[key]&inp.k_mody[key]) {inp.k_flush[key]=0;} else {return(0);}
  }
  if (art==LONGKEY) {
    return(inp.k_press[key]);
  } else if (art==SHORTKEY) {
    return(inp.k_press[key]&inp.k_mody[key]);
  }
  return(0);
} /* Ende key_pressed_svgalib */


int mouse_found_svgalib() {
/* whether a mouse is found */
  if (!wd.is_open) {return(0);}
  return(!inp.m_missing);
} /* Ende mouse_found_svgalib */


int mouse_x_svgalib() {
/* mouse x-position or -1=not in window */
  return(inp.m_xpos);
} /* Ende mouse_x_svgalib */


int mouse_y_svgalib() {
/* mouse y-position or -1=not in window */
  return(inp.m_ypos);
} /* Ende mouse_y_svgalib */


int mouse_pressed_svgalib(int bnr,int art) {
/* whether a mousebutton is pressed
** 1.arg: mousebutton-define (MOUSE_LEFT, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
*/
  if (!wd.is_open) {return(0);}
  switch(bnr) {
    case MOUSE_LEFT:
    case MOUSE_RIGHT:
    case MOUSE_MIDDLE:
      if (inp.m_flush[bnr]) {
        if (inp.m_press[bnr]&inp.m_mody[bnr]) {inp.m_flush[bnr]=0;} else {return(0);}
      }
      if (art==LONGKEY) {
        return(inp.m_press[bnr]);
      } else if (art==SHORTKEY) {
        return(inp.m_press[bnr]&inp.m_mody[bnr]);
      }
  }
  return(0);
} /* Ende mouse_pressed_svgalib */


int mouse_speed_svgalib(int index) {
/* set speed of mouse
** 1.arg: 1=slow, 2=middle, 3=fast
**        0=return only current speed
** return: previous speed number
**         or 0=speed cannot be changed
*/
  static int mvs=1;
  int vs;
  if (!wd.is_open) {return(0);}
  if (index<0) {index=0;} else if (index>3) {index=3;}
  if (index>0) {
    mouse_setscale(index==1?120:(index==2?70:50));
    vs=mvs;
    mvs=index;
  } else {vs=mvs;}
  return(vs);
} /* Ende mouse_speed_svgalib */


unsigned char * keystonw_svgalib() {
/* only for internal use with networking:
** returns a static unsigned char field for (non)pressed keys and mouse buttons
** format: unsigned char (flag for new or old): 1
**         unsigned char[KEYS_INDEX] (for keystrokes):
**           bit0=k_press[], bit1=k_mody[], bit2=k_flush[]
**         unsigned char (for MOUSE_LEFT):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char (for MOUSE_MIDDLE):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char (for MOUSE_RIGHT):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char[2] (for mouse_x): <0|1>+<0-255> or <255>+<255>=-1
**         unsigned char (for mouse_y): <0-SC_HEIGHT> or <255>=-1
*/
  static unsigned char retw[1+KEYS_INDEX+3+2+1]="";  /* NW_KEYSIZE */
  int i1;
  /* flag */
  retw[0]=1;
  /* keys */
  for (i1=0;i1<KEYS_INDEX;i1++) {
    if (inp.k_flush[i1]) {
      if (inp.k_press[i1]&inp.k_mody[i1]) {inp.k_flush[i1]=0;}
    }
    retw[1+i1]=(!!inp.k_press[i1]|((!!inp.k_mody[i1])<<1)|((!!inp.k_flush[i1])<<2));
  }
  i1=1+KEYS_INDEX;
  /* mouse buttons */
  if (inp.m_flush[MOUSE_LEFT]) {
    if (inp.m_press[MOUSE_LEFT]&inp.m_mody[MOUSE_LEFT]) {inp.m_flush[MOUSE_LEFT]=0;}
  }
  retw[i1++]=(!!inp.m_press[MOUSE_LEFT]|((!!inp.m_mody[MOUSE_LEFT])<<1)|((!!inp.m_flush[MOUSE_LEFT])<<2));
  if (inp.m_flush[MOUSE_MIDDLE]) {
    if (inp.m_press[MOUSE_MIDDLE]&inp.m_mody[MOUSE_MIDDLE]) {inp.m_flush[MOUSE_MIDDLE]=0;}
  }
  retw[i1++]=(!!inp.m_press[MOUSE_MIDDLE]|((!!inp.m_mody[MOUSE_MIDDLE])<<1)|((!!inp.m_flush[MOUSE_MIDDLE])<<2));
  if (inp.m_flush[MOUSE_RIGHT]) {
    if (inp.m_press[MOUSE_RIGHT]&inp.m_mody[MOUSE_RIGHT]) {inp.m_flush[MOUSE_RIGHT]=0;}
  }
  retw[i1++]=(!!inp.m_press[MOUSE_RIGHT]|((!!inp.m_mody[MOUSE_RIGHT])<<1)|((!!inp.m_flush[MOUSE_RIGHT])<<2));
  /* mouse position */
  retw[i1++]=(inp.m_xpos<0?255:inp.m_xpos/256);
  retw[i1++]=(inp.m_xpos<0?255:inp.m_xpos%256);
  retw[i1++]=(inp.m_ypos<0?255:inp.m_ypos%SC_HEIGHT);
  return(retw);
} /* Ende keystonw_svgalib */


int brightness_svgalib(int vol) {
/* reduces or increases brightness of colormap
** 1.arg: -63 (darkest) to 62 (lightest), 0 is default, or 99=previous
** return: previous brightness value (-63 to 62)
*/
  static int lastvol=0;
  int r,g,b;
  int i1,i2,i3;
  float f1;
  if (!wd.is_open) {return(0);}
  if (vol==99) {vol=lastvol;}
  else if (vol<-63) {vol=-63;}
  else if (vol>62) {vol=62;}
  for (i1=0;i1<256;i1++) {  /* fcpix: color map of e.g. RGB256_FILE */
    i3=vol;
    r=fcpix[i1].r;
    g=fcpix[i1].g;
    b=fcpix[i1].b;
    if (vol>0) {
      if (r==0) {r=1;}
      if (g==0) {g=1;}
      if (b==0) {b=1;}
    }
    i2=(r>g?(r>b?r:b):(g>b?g:b));  /* get highest value */
    if (i2>0) {
      if (i2+i3>63) {i3=63-i2;}
      f1=(float)i3/(float)i2;
      r+=(f1*r);
      g+=(f1*g);
      b+=(f1*b);
      if (r<0) {r=0;} else if (r>63) {r=63;}
      if (g<0) {g=0;} else if (g>63) {g=63;}
      if (b<0) {b=0;} else if (b>63) {b=63;}
      if ((vol>0) && ((r==63) || (g==63) || (b==63))) {
        float f2;
        i3=(fcpix[i1].r>fcpix[i1].g?(fcpix[i1].r>fcpix[i1].b?fcpix[i1].r:fcpix[i1].b):(fcpix[i1].g>fcpix[i1].b?fcpix[i1].g:fcpix[i1].b));
        if (i3==0) {i3=1;}
        f2=63/i3;
        i3=vol-(63-i3);
        if (i3>0) {
          i2=(r<g?(r<b?r:b):(g<b?g:b));  /* get lowest value */
          if (i2>0) {
            if (i2+i3>63) {i3=63-i2;}
            f1=(float)i3/(float)i2*f2;
            r+=(f1*r);
            g+=(f1*g);
            b+=(f1*b);
            if (r<0) {r=0;} else if (r>63) {r=63;}
            if (g<0) {g=0;} else if (g>63) {g=63;}
            if (b<0) {b=0;} else if (b>63) {b=63;}
          }
        }
      }
    }
    gl_setpalettecolor(i1,r,g,b);
  }
  i1=lastvol; lastvol=vol;
  return(i1);
} /* Ende brightness_svgalib */
