/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* window and input functions for libvgl */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#if defined(HAVE_VGL4)
# include <machine/console.h>
#elif defined(HAVE_VGL5)
# include <sys/fbio.h>
# include <sys/consio.h>
# include <sys/kbio.h>
#endif
#include <sys/ioctl.h>
#include <sys/signal.h>
#include <signal.h>
#include <vgl.h>
#include "vgagames2.h"
#include "bitmap.h"
#include "color.h"


/* +++ declarations +++ */

static struct {  /* for window */
  int              is_open;       /* window has been opened? */
  bitmap           * backbuffer;  /* backbuffer */
} wd={0};

static struct {  /* for input */
  int k_cmp[KEYS_INDEX+1];
  char k_last[KEYS_INDEX+1],k_press[KEYS_INDEX+1],k_mody[KEYS_INDEX+1],k_flush[KEYS_INDEX+1];
  int m_missing;
  int m_press[3+1],m_mody[3+1],m_flush[3+1];
  int m_xpos,m_ypos;
} inp;


static int init_keys(void);

int window_open_vgl(void);
void window_flush_vgl(void);
void window_close_vgl(void);

void key_discard_vgl(void);
void key_update_vgl(void);
int key_pressed_vgl(int,int);
int mouse_found_vgl(void);
int mouse_x_vgl(void);
int mouse_y_vgl(void);
int mouse_pressed_vgl(int,int);
int mouse_speed_vgl(int);
unsigned char * keystonw_vgl(void);

int brightness_vgl(int);


/* +++ internal functions +++ */

static int init_keys() {
/* initalize keys, called from window_open_vgl() */
  static int firstcall=1;
  if (!firstcall) {return(0);}
  firstcall=0;
  inp.k_cmp[KEY_0]=11;
  inp.k_cmp[KEY_1]=2;
  inp.k_cmp[KEY_2]=3;
  inp.k_cmp[KEY_3]=4;
  inp.k_cmp[KEY_4]=5;
  inp.k_cmp[KEY_5]=6;
  inp.k_cmp[KEY_6]=7;
  inp.k_cmp[KEY_7]=8;
  inp.k_cmp[KEY_8]=9;
  inp.k_cmp[KEY_9]=10;
  inp.k_cmp[KEY_A]=30;
  inp.k_cmp[KEY_B]=48;
  inp.k_cmp[KEY_C]=46;
  inp.k_cmp[KEY_D]=32;
  inp.k_cmp[KEY_E]=18;
  inp.k_cmp[KEY_F]=33;
  inp.k_cmp[KEY_G]=34;
  inp.k_cmp[KEY_H]=35;
  inp.k_cmp[KEY_I]=23;
  inp.k_cmp[KEY_J]=36;
  inp.k_cmp[KEY_K]=37;
  inp.k_cmp[KEY_L]=38;
  inp.k_cmp[KEY_M]=50;
  inp.k_cmp[KEY_N]=49;
  inp.k_cmp[KEY_O]=24;
  inp.k_cmp[KEY_P]=25;
  inp.k_cmp[KEY_Q]=16;
  inp.k_cmp[KEY_R]=19;
  inp.k_cmp[KEY_S]=31;
  inp.k_cmp[KEY_T]=20;
  inp.k_cmp[KEY_U]=22;
  inp.k_cmp[KEY_V]=47;
  inp.k_cmp[KEY_W]=17;
  inp.k_cmp[KEY_X]=45;
  inp.k_cmp[KEY_Y]=44;
  inp.k_cmp[KEY_Z]=21;
  inp.k_cmp[KEY_KP_0]=82;
  inp.k_cmp[KEY_KP_1]=79;
  inp.k_cmp[KEY_KP_2]=80;
  inp.k_cmp[KEY_KP_3]=81;
  inp.k_cmp[KEY_KP_4]=75;
  inp.k_cmp[KEY_KP_5]=76;
  inp.k_cmp[KEY_KP_6]=77;
  inp.k_cmp[KEY_KP_7]=71;
  inp.k_cmp[KEY_KP_8]=72;
  inp.k_cmp[KEY_KP_9]=73;
  inp.k_cmp[KEY_TAB]=15;
  inp.k_cmp[KEY_LCTRL]=29;
  inp.k_cmp[KEY_RCTRL]=90;
  inp.k_cmp[KEY_LALT]=56;
  inp.k_cmp[KEY_RALT]=93;
  inp.k_cmp[KEY_SPACE]=57;
  inp.k_cmp[KEY_ENTER]=28;
  inp.k_cmp[KEY_BSP]=14;
  inp.k_cmp[KEY_RCURS]=98;
  inp.k_cmp[KEY_LCURS]=97;
  inp.k_cmp[KEY_UCURS]=95;
  inp.k_cmp[KEY_DCURS]=100;
  inp.k_cmp[KEY_F1]=59;
  inp.k_cmp[KEY_F2]=60;
  inp.k_cmp[KEY_F3]=61;
  inp.k_cmp[KEY_F4]=62;
  inp.k_cmp[KEY_F5]=63;
  inp.k_cmp[KEY_F6]=64;
  inp.k_cmp[KEY_F7]=65;
  inp.k_cmp[KEY_F8]=66;
  inp.k_cmp[KEY_F9]=67;
  inp.k_cmp[KEY_F10]=68;
  inp.k_cmp[KEY_F11]=87;
  inp.k_cmp[KEY_F12]=88;
  inp.k_cmp[KEYS_INDEX]=1;  /* system menu */
  return(0);
} /* Ende init_keys */


/* +++ functions +++ */

int window_open_vgl() {
/* switch console screen into graphic mode
** return: 0=ok or -1=error
*/
  static char firstcall=1;

  if (firstcall) {
    memset(&wd,0,sizeof(wd));
    memset(&inp,0,sizeof(inp));
    wd.is_open=0;
    firstcall=0;
  }

  /* set graphic mode */
  if (VGLInit(SW_VGA_CG320)!=0) {fprintf(stderr,"window_open_vgl: error calling VGLInit.\n"); return(-1);}
  if (VGLDisplay->Type!=VIDBUF8) {fprintf(stderr,"window_open_vgl: wrong video mode.\n"); VGLEnd(); return(-1);}

  /* create backbuffer */
  wd.backbuffer=vg_bitmap_createnew(SC_WIDTH,SC_HEIGHT);
  if (wd.backbuffer==NULL) {fprintf(stderr,"window_open_vgl: can't create backbuffer.\n"); VGLEnd(); return(-1);}
  backbuffer=wd.backbuffer;

  /* initialize keyboard and mouse */
  if (init_keys()<0) {fprintf(stderr,"window_open_vgl: init_keys error.\n"); VGLEnd(); return(-1);}
  memset(inp.k_last,0,sizeof(inp.k_last));
  memset(inp.k_press,0,sizeof(inp.k_press));
  memset(inp.k_mody,0,sizeof(inp.k_mody));
  memset(inp.k_flush,0,sizeof(inp.k_flush));
  if (VGLKeyboardInit(VGL_CODEKEYS)<0) {fprintf(stderr,"window_open_vgl: error calling VGLKeyboardInit.\n"); VGLEnd(); return(-1);}
  if (VGLMouseInit(VGL_MOUSEHIDE)<0) {inp.m_missing=1;} else {inp.m_missing=0;}
  inp.m_xpos=inp.m_ypos=-1;
  memset(inp.m_press,0,sizeof(inp.m_press));
  memset(inp.m_mody,0,sizeof(inp.m_mody));
  memset(inp.m_flush,0,sizeof(inp.m_flush));
  if (!inp.m_missing) {
    mouse_info_t mi;
    mi.operation=MOUSE_MODE;
    mi.u.mode.signal=0;
    ioctl(0,CONS_MOUSECTL,&mi);  /* don't want to be interrupted by signal */
    signal(SIGUSR2,SIG_IGN);
  }
  while (VGLKeyboardGetCh()!=0) {;}
  wd.is_open=1;
  if (!inp.m_missing) {mouse_speed_vgl(2);}

  window_mode.isset=1;
  window_mode.videolib=VIDEOLIB_VGL;
  window_mode.vlb[VIDEOLIB_VGL].scale=VGAWINDOW_1;
  window_mode.vlb[VIDEOLIB_VGL].full=1;
  window_mode.vlb[VIDEOLIB_VGL].noswitch=1;
  window_mode.vlb[VIDEOLIB_VGL].directdraw=0;
  window_mode.vlb[VIDEOLIB_VGL].modenr=window_mode.vlb[VIDEOLIB_VGL].modemax=0;
  window_mode.vlb[VIDEOLIB_VGL].modewidth=SC_WIDTH;
  window_mode.vlb[VIDEOLIB_VGL].modeheight=SC_HEIGHT;
  return(0);
} /* Ende window_open_vgl */


void window_flush_vgl() {
/* give backbuffer out to screen */
  static int firstcall=1;
  if (!wd.is_open) {return;}
  if (firstcall) {VGLSetBorder(RGB_BLACK); firstcall=0;}
  memmove(VGLDisplay->Bitmap,wd.backbuffer->pxm,SC_WIDTH*SC_HEIGHT);
} /* Ende window_flush_vgl */


void window_close_vgl() {
/* close window */
  if (!wd.is_open) {return;}
  vg_bitmap_free(wd.backbuffer); wd.backbuffer=NULL;
  VGLKeyboardEnd();
  VGLEnd();
  wd.is_open=0;
} /* Ende window_close_vgl */


void key_discard_vgl() {
  if (!wd.is_open) {return;}
  key_update_vgl();
  memset(inp.k_flush,1,sizeof(inp.k_flush));
  memset(inp.m_flush,1,sizeof(inp.m_flush));
} /* Ende key_discard_vgl */


void key_update_vgl() {
/* get pressed or released keys/mousebuttons */
  int kchar,i1,pressed;

  if (!wd.is_open) {return;}
  memmove(inp.k_last,inp.k_press,sizeof(inp.k_last));
  memset(inp.k_mody,0,sizeof(inp.k_mody));
  memset(inp.m_mody,0,sizeof(inp.m_mody));

  while ((kchar=VGLKeyboardGetCh())!=0) {  /* get all keyboard input */
    if (kchar>=128) {kchar-=128; pressed=0;} else {pressed=1;}
    for (i1=0;i1<=KEYS_INDEX;i1++) {
      if (kchar==inp.k_cmp[i1]) {
        if (pressed) {
          if (inp.k_last[i1]==0) {inp.k_mody[i1]=1;} else {inp.k_mody[i1]=0;}
          inp.k_press[i1]=1;
        } else {
          if (inp.k_last[i1]==1) {inp.k_mody[i1]=1;} else {inp.k_mody[i1]=0;}
          inp.k_press[i1]=0;
        }
        break;
      }
    }
  }

  if (!inp.m_missing) {  /* mouse found */
#define MOUSE_LEFTBUTTON   1
#define MOUSE_MIDDLEBUTTON 2
#define MOUSE_RIGHTBUTTON  4
    static int bzx=0,bzy=0;
    mouse_info_t mi;
    mi.operation=MOUSE_GETINFO;
    ioctl(0,CONS_MOUSECTL,&mi);  /* get mouse events */
    i1=mi.u.data.buttons;  /* get pressed buttons */
    inp.m_xpos=mi.u.data.x-bzx;  /* set x position of mouse */
    if (inp.m_xpos>SC_WIDTH) {
       bzx+=inp.m_xpos-SC_WIDTH; inp.m_xpos=SC_WIDTH;
    } else if (inp.m_xpos<0) {bzx+=inp.m_xpos; inp.m_xpos=0;}
    inp.m_ypos=mi.u.data.y-bzy;  /* set y position of mouse */
    if (inp.m_ypos>SC_HEIGHT) {
      bzy+=inp.m_ypos-SC_HEIGHT; inp.m_ypos=SC_HEIGHT;
    } else if (inp.m_ypos<0) {bzy+=inp.m_ypos; inp.m_ypos=0;}
    if (i1&MOUSE_LEFTBUTTON) {  /* left button pressed */
      if (inp.m_press[MOUSE_LEFT]==0) {inp.m_mody[MOUSE_LEFT]=1;} else {inp.m_mody[MOUSE_LEFT]=0;}
      inp.m_press[MOUSE_LEFT]=1;
    } else {
      if (inp.m_press[MOUSE_LEFT]==1) {inp.m_mody[MOUSE_LEFT]=1;} else {inp.m_mody[MOUSE_LEFT]=0;}
      inp.m_press[MOUSE_LEFT]=0;
    }
    if (i1&MOUSE_RIGHTBUTTON) {  /* right button pressed */
      if (inp.m_press[MOUSE_RIGHT]==0) {inp.m_mody[MOUSE_RIGHT]=1;} else {inp.m_mody[MOUSE_RIGHT]=0;}
      inp.m_press[MOUSE_RIGHT]=1;
    } else {
      if (inp.m_press[MOUSE_RIGHT]==1) {inp.m_mody[MOUSE_RIGHT]=1;} else {inp.m_mody[MOUSE_RIGHT]=0;}
      inp.m_press[MOUSE_RIGHT]=0;
    }
    if (i1&MOUSE_MIDDLEBUTTON) {  /* middle button pressed */
      if (inp.m_press[MOUSE_MIDDLE]==0) {inp.m_mody[MOUSE_MIDDLE]=1;} else {inp.m_mody[MOUSE_MIDDLE]=0;}
      inp.m_press[MOUSE_MIDDLE]=1;
    } else {
      if (inp.m_press[MOUSE_MIDDLE]==1) {inp.m_mody[MOUSE_MIDDLE]=1;} else {inp.m_mody[MOUSE_MIDDLE]=0;}
      inp.m_press[MOUSE_MIDDLE]=0;
    }
  }
} /* Ende key_update_vgl */


int key_pressed_vgl(int key,int art) {
/* whether a key is pressed
** 1.arg: key-define (KEY_0, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
*/
  if (!wd.is_open) {return(0);}
  if ((key<0) || (key>KEYS_INDEX)) {return(0);}
  if (inp.k_flush[key]) {
    if (inp.k_press[key]&inp.k_mody[key]) {inp.k_flush[key]=0;} else {return(0);}
  }
  if (art==LONGKEY) {
    return(inp.k_press[key]);
  } else if (art==SHORTKEY) {
    return(inp.k_press[key]&inp.k_mody[key]);
  }
  return(0);
} /* Ende key_pressed_vgl */


int mouse_found_vgl() {
/* whether a mouse is found
** to be able to use the mouse, the server moused must run
*/
  if (!wd.is_open) {return(0);}
  return(!inp.m_missing);
} /* Ende mouse_found_vgl */


int mouse_x_vgl() {
/* mouse x-position or -1=not in window */
  return(inp.m_xpos);
} /* Ende mouse_x_vgl */


int mouse_y_vgl() {
/* mouse y-position or -1=not in window */
  return(inp.m_ypos);
} /* Ende mouse_y_vgl */


int mouse_pressed_vgl(int bnr,int art) {
/* whether a mousebutton is pressed
** 1.arg: mousebutton-define (MOUSE_LEFT, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
*/
  if (!wd.is_open) {return(0);}
  switch(bnr) {
    case MOUSE_LEFT:
    case MOUSE_RIGHT:
    case MOUSE_MIDDLE:
      if (inp.m_flush[bnr]) {
        if (inp.m_press[bnr]&inp.m_mody[bnr]) {inp.m_flush[bnr]=0;} else {return(0);}
      }
      if (art==LONGKEY) {
        return(inp.m_press[bnr]);
      } else if (art==SHORTKEY) {
        return(inp.m_press[bnr]&inp.m_mody[bnr]);
      }
  }
  return(0);
} /* Ende mouse_pressed_vgl */


unsigned char * keystonw_vgl() {
/* only for internal use with networking:
** returns a static unsigned char field for (non)pressed keys and mouse buttons
** format: unsigned char (flag for new or old): 1
**         unsigned char[KEYS_INDEX] (for keystrokes):
**           bit0=k_press[], bit1=k_mody[], bit2=k_flush[]
**         unsigned char (for MOUSE_LEFT):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char (for MOUSE_MIDDLE):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char (for MOUSE_RIGHT):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char[2] (for mouse_x): <0|1>+<0-255> or <255>+<255>=-1
**         unsigned char (for mouse_y): <0-SC_HEIGHT> or <255>=-1
*/
  static unsigned char retw[1+KEYS_INDEX+3+2+1]="";  /* NW_KEYSIZE */
  int i1;
  /* flag */
  retw[0]=1;
  /* keys */
  for (i1=0;i1<KEYS_INDEX;i1++) {
    if (inp.k_flush[i1]) {
      if (inp.k_press[i1]&inp.k_mody[i1]) {inp.k_flush[i1]=0;}
    }
    retw[1+i1]=(!!inp.k_press[i1]|((!!inp.k_mody[i1])<<1)|((!!inp.k_flush[i1])<<2));
  }
  i1=1+KEYS_INDEX;
  /* mouse buttons */
  if (inp.m_flush[MOUSE_LEFT]) {
    if (inp.m_press[MOUSE_LEFT]&inp.m_mody[MOUSE_LEFT]) {inp.m_flush[MOUSE_LEFT]=0;}
  }
  retw[i1++]=(!!inp.m_press[MOUSE_LEFT]|((!!inp.m_mody[MOUSE_LEFT])<<1)|((!!inp.m_flush[MOUSE_LEFT])<<2));
  if (inp.m_flush[MOUSE_MIDDLE]) {
    if (inp.m_press[MOUSE_MIDDLE]&inp.m_mody[MOUSE_MIDDLE]) {inp.m_flush[MOUSE_MIDDLE]=0;}
  }
  retw[i1++]=(!!inp.m_press[MOUSE_MIDDLE]|((!!inp.m_mody[MOUSE_MIDDLE])<<1)|((!!inp.m_flush[MOUSE_MIDDLE])<<2));
  if (inp.m_flush[MOUSE_RIGHT]) {
    if (inp.m_press[MOUSE_RIGHT]&inp.m_mody[MOUSE_RIGHT]) {inp.m_flush[MOUSE_RIGHT]=0;}
  }
  retw[i1++]=(!!inp.m_press[MOUSE_RIGHT]|((!!inp.m_mody[MOUSE_RIGHT])<<1)|((!!inp.m_flush[MOUSE_RIGHT])<<2));
  /* mouse position */
  retw[i1++]=(inp.m_xpos<0?255:inp.m_xpos/256);
  retw[i1++]=(inp.m_xpos<0?255:inp.m_xpos%256);
  retw[i1++]=(inp.m_ypos<0?255:inp.m_ypos%SC_HEIGHT);
  return(retw);
} /* Ende keystonw_vgl */


int mouse_speed_vgl(int index) {
/* set speed of mouse
** 1.arg: 1=slow, 2=middle, 3=fast
**        0=return only current speed
** return: previous speed number
**         or 0=speed cannot be changed
*/
  if (!wd.is_open) {return(0);}
  return(0);  /* cannot be changed */
} /* Ende mouse_speed_vgl */


int brightness_vgl(int vol) {
/* reduces or increases brightness of colormap
** 1.arg: -63 (darkest) to 62 (lightest), 0 is default, or 99=previous
** return: previous brightness value (-63 to 62)
*/
  static int lastvol=0;
  int r,g,b;
  int i1,i2,i3;
  float f1;
  if (!wd.is_open) {return(0);}
  if (vol==99) {vol=lastvol;}
  else if (vol<-63) {vol=-63;}
  else if (vol>62) {vol=62;}
  for (i1=0;i1<256;i1++) {  /* fcpix: color map of e.g. RGB256_FILE */
    i3=vol;
    r=fcpix[i1].r;
    g=fcpix[i1].g;
    b=fcpix[i1].b;
    if (vol>0) {
      if (r==0) {r=1;}
      if (g==0) {g=1;}
      if (b==0) {b=1;}
    }
    i2=(r>g?(r>b?r:b):(g>b?g:b));  /* get highest value */
    if (i2>0) {
      if (i2+i3>63) {i3=63-i2;}
      f1=(float)i3/(float)i2;
      r+=(f1*r);
      g+=(f1*g);
      b+=(f1*b);
      if (r<0) {r=0;} else if (r>63) {r=63;}
      if (g<0) {g=0;} else if (g>63) {g=63;}
      if (b<0) {b=0;} else if (b>63) {b=63;}
      if ((vol>0) && ((r==63) || (g==63) || (b==63))) {
        float f2;
        i3=(fcpix[i1].r>fcpix[i1].g?(fcpix[i1].r>fcpix[i1].b?fcpix[i1].r:fcpix[i1].b):(fcpix[i1].g>fcpix[i1].b?fcpix[i1].g:fcpix[i1].b));
        if (i3==0) {i3=1;}
        f2=63/i3;
        i3=vol-(63-i3);
        if (i3>0) {
          i2=(r<g?(r<b?r:b):(g<b?g:b));  /* get lowest value */
          if (i2>0) {
            if (i2+i3>63) {i3=63-i2;}
            f1=(float)i3/(float)i2*f2;
            r+=(f1*r);
            g+=(f1*g);
            b+=(f1*b);
            if (r<0) {r=0;} else if (r>63) {r=63;}
            if (g<0) {g=0;} else if (g>63) {g=63;}
            if (b<0) {b=0;} else if (b>63) {b=63;}
          }
        }
      }
    }
    VGLSetPaletteIndex(i1,r,g,b);
  }
  i1=lastvol; lastvol=vol;
  return(i1);
} /* Ende brightness_vgl */
