/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* window and input functions for X11 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include "config.h"
#ifdef HAVE_X11_EXTS
# include <X11/extensions/xf86vmode.h>
#endif
#include "vgagames2.h"
#include "bitmap.h"
#include "color.h"
#include "inthash.h"


/* +++ declarations +++ */

union _u_s {  /* mapping uint2/chars */
  unsigned short w;
  unsigned char c[2];
};
union _u_i {  /* mapping uint4/chars */
  unsigned int w;
  unsigned char c[4];
};

static struct {  /* for window */
  int           is_open;            /* window has been opened? */
  Display       * display;          /* Display */
  int           screen;             /* Screen of Display */
  Window        window;             /* Window of Display */
  GC            gc;                 /* GC of window */
  Colormap      colormap;           /* Colormap */
  XVisualInfo   * visual;           /* Visual of Display */
  XImage        * ximage;           /* Image for screenflush */
  int           bp_doppl;           /* scaling factor */
  int           pos_x,pos_y;        /* pixel start position x,y of window */
  int           noswitch;           /* don't switch mode */
  int           fullscr;            /* fullscreen? */
  unsigned long colpix[256];        /* pixel colors */
  unsigned long px_black,px_white;  /* black and white pixel */
  bitmap        * bpuff[2];         /* backbuffers */
  int           ipuff;              /* index backbuffer */
  int           brpuff;             /* redraw all pixels? */
  union _u_s u_s[256],* pu_s;       /* for color pixels 16bit */
  union _u_i u_i[256],* pu_i;       /* for color pixels 24/32bit */
#ifdef HAVE_X11_EXTS
  int virtwidth,virtheight;         /* mode switched to width and height */
  XF86VidModeModeInfo mi_res;       /* for restoring old mode */
  int modenr,modemax;               /* mode number to switch to, max. modenr */
  int modewidth,modeheight;         /* mode width and height */
#endif
} wd={0};

static struct {  /* for input */
  struct ihash ** k_cmp;
  char k_last[KEYS_INDEX+1],k_press[KEYS_INDEX+1],k_mody[KEYS_INDEX+1],k_flush[KEYS_INDEX+1];
  unsigned int m_cmp[3+1];
  int m_press[3+1],m_mody[3+1],m_flush[3+1];
  int m_xpos,m_ypos,m_xold,m_yold;
} inp;

static char ** x11modes=NULL;


static int init_keys(void);
static void restore_mode(void);
void do_sig_x11(int);

int window_open_x11(const char *,int);
void window_flush_x11(void);
void window_close_x11(void);

void key_discard_x11(void);
void key_update_x11(void);
int key_pressed_x11(int,int);
int mouse_found_x11(void);
int mouse_x_x11(void);
int mouse_y_x11(void);
int mouse_pressed_x11(int,int);
int mouse_speed_x11(int);
unsigned char * keystonw_x11(void);

int brightness_x11(int);


/* +++ internal functions +++ */

static int init_keys() {
/* initalize keys, called from window_open_x11() */
  static int firstcall=1;
  if (!firstcall) {return(0);}
  firstcall=0;
  if ((inp.k_cmp=inthash_new(127))==NULL) {return(-1);}
  if (inthash_add(inp.k_cmp,"0",KEY_0)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"1",KEY_1)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"2",KEY_2)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"3",KEY_3)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"4",KEY_4)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"5",KEY_5)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"6",KEY_6)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"7",KEY_7)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"8",KEY_8)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"9",KEY_9)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"A",KEY_A)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"B",KEY_B)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"C",KEY_C)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"D",KEY_D)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"E",KEY_E)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F",KEY_F)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"G",KEY_G)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"H",KEY_H)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"I",KEY_I)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"J",KEY_J)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"K",KEY_K)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"L",KEY_L)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"M",KEY_M)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"N",KEY_N)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"O",KEY_O)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"P",KEY_P)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"Q",KEY_Q)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"R",KEY_R)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"S",KEY_S)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"T",KEY_T)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"U",KEY_U)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"V",KEY_V)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"W",KEY_W)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"X",KEY_X)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"Y",KEY_Y)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"Z",KEY_Z)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_0",KEY_KP_0)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_1",KEY_KP_1)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_2",KEY_KP_2)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_3",KEY_KP_3)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_4",KEY_KP_4)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_5",KEY_KP_5)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_6",KEY_KP_6)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_7",KEY_KP_7)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_8",KEY_KP_8)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_9",KEY_KP_9)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"TAB",KEY_TAB)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"CONTROL_L",KEY_LCTRL)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"CONTROL_R",KEY_RCTRL)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"ALT_L",KEY_LALT)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"ALT_R",KEY_RALT)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"ISO_LEVEL3_SHIFT",KEY_RALT)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"SPACE",KEY_SPACE)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"RETURN",KEY_ENTER)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"KP_ENTER",KEY_ENTER)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"BACKSPACE",KEY_BSP)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"RIGHT",KEY_RCURS)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"LEFT",KEY_LCURS)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"UP",KEY_UCURS)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"DOWN",KEY_DCURS)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F1",KEY_F1)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F2",KEY_F2)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F3",KEY_F3)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F4",KEY_F4)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F5",KEY_F5)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F6",KEY_F6)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F7",KEY_F7)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F8",KEY_F8)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F9",KEY_F9)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F10",KEY_F10)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F11",KEY_F11)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"F12",KEY_F12)<0) {return(-1);}
  if (inthash_add(inp.k_cmp,"ESCAPE",KEYS_INDEX)<0) {return(-1);}  /* system menu */
  return(0);
} /* Ende init_keys */


static void restore_mode() {
  if (!wd.is_open) {return;}
#ifdef HAVE_X11_EXTS
  if (!wd.noswitch) {
    XF86VidModeSwitchToMode(wd.display,wd.screen,&wd.mi_res);
    XF86VidModeSetViewPort(wd.display,wd.screen,0,0);
    XFlush(wd.display);
    XSync(wd.display,False);
  }
#endif
} /* Ende restore_mode */


void do_sig_x11(int signum) {
/* catches SIGSEGV,SIGBUS,SIGINT,SIGTERM for switching back to original mode */
  char buf[128];
  restore_mode(); 
  strlcpy(buf,"--> caught signal 000\n",sizeof(buf));
  buf[18]=(signum/100)%10+48;
  buf[19]=(signum/10)%10+48;
  buf[20]=signum%10+48;
  write(STDERR_FILENO,buf,strlen(buf));
  _exit(1);
} /* Ende do_sig_x11 */


/* +++ functions +++ */

int window_open_x11(const char * wnam,int dp) {
/* create window
** 1.arg: name of the window (NULL=same name as before)
** 2.arg: window properties
**        0:              get largest window size, no full screen
**        VGAWINDOW_1:    window scaling factor 1  (original size)
**        VGAWINDOW_2:    window scaling factor 2  (4 pixels per pixel)
**        VGAWINDOW_3:    window scaling factor 3  (9 pixels per pixel)
**        VGAWINDOW_FULL: full screen modus
**        VGAWINDOW_NOSWITCH: don't switch to another mode
**        VGAWINDOW_DIRECTDRAW: activate directdraw
** return: 0=ok or -1=error
**
** Note: the value of the 2.arg will be ignored if the vgagames-properties
**       already have been read from file (see vg_init_vgagames())
*/
  static char firstcall=1;
  static char * pnam=NULL;
  Window rootw;
  XSetWindowAttributes attributes;
  unsigned long attr_mask;
  int bs_w,bs_h;
  int px_w,px_h;
  unsigned int cdepth;
  struct sigaction sa;
  int directdraw;

  if (firstcall) {
    memset(&wd,0,sizeof(wd));
    memset(&inp,0,sizeof(inp));
    wd.is_open=0;
    wd.ximage=NULL;
  }

  /* connection to X */
  if (((wd.display=XOpenDisplay(NULL))==NULL) \
  && ((wd.display=XOpenDisplay(":0.0"))==NULL)) {fprintf(stderr,"window_open_x11: can't connect to %s.\n",XDisplayName(NULL)); return(-1);}

  if (firstcall) {
    firstcall=0;
    memset(&sa,0,sizeof(sa));
    sa.sa_handler=do_sig_x11;
    sigaction(SIGSEGV,&sa,NULL);
    sigaction(SIGBUS,&sa,NULL);
    sigaction(SIGINT,&sa,NULL);
    sigaction(SIGTERM,&sa,NULL);
    atexit(restore_mode);
  }

  if (wnam==NULL) {
    if (pnam==NULL) {pnam=strdup("VgaGames2");}
  } else {
    if (pnam!=NULL) {free(pnam);}
    pnam=strdup(wnam);
  }
  if (window_mode.isset) {  /* use window_mode instead of dp */
    dp=0;
    dp|=window_mode.vlb[VIDEOLIB_X11].scale;
    dp|=(window_mode.vlb[VIDEOLIB_X11].full?VGAWINDOW_FULL:0);
    dp|=(window_mode.vlb[VIDEOLIB_X11].noswitch?VGAWINDOW_NOSWITCH:0);
    dp|=(window_mode.vlb[VIDEOLIB_X11].directdraw?VGAWINDOW_DIRECTDRAW:0);
  } else {window_mode.vlb[VIDEOLIB_X11].modenr=0;}

  wd.noswitch=!!(dp&VGAWINDOW_NOSWITCH);
  directdraw=!!(dp&VGAWINDOW_DIRECTDRAW);

  /* get DefaultScreen number */
  wd.screen=DefaultScreen(wd.display);

  /* get RootWindow of screen */
  rootw=RootWindow(wd.display,wd.screen);

  if (x11modes!=NULL) {
    char ** mdp;
    for (mdp=x11modes;*mdp!=NULL;mdp++) {free(*mdp);}
    free(x11modes); x11modes=NULL;
  }
  if ((x11modes=malloc(sizeof(char *)*2))==NULL) {fprintf(stderr,"window_open_x11: malloc x11modes: %s.\n",strerror(errno)); return(-1);}
  x11modes[0]=strdup("(automatic mode)");
  x11modes[1]=NULL;
#ifdef HAVE_X11_EXTS
  if (!wd.noswitch) {
    static int firstcall=1;
    XF86VidModeModeInfo ** mi;
    XF86VidModeModeLine ml;
    void * vpt;
    int mc,i1,mw;
    int width=WidthOfScreen(DefaultScreenOfDisplay(wd.display));
    int height=HeightOfScreen(DefaultScreenOfDisplay(wd.display));
    char bf[32];
    wd.virtwidth=640; wd.virtheight=480;
    memset(&wd.mi_res,0,sizeof(wd.mi_res));
    XF86VidModeGetModeLine(wd.display,wd.screen,&mc,&ml);
    vpt=&wd.mi_res;
    wd.mi_res.dotclock=mc;
    memmove(vpt+sizeof(unsigned int),&ml,sizeof(ml));
    XF86VidModeGetAllModeLines(wd.display,wd.screen,&mc,&mi);
    wd.modewidth=wd.modeheight=0;
    if ((wd.modenr=window_mode.vlb[VIDEOLIB_X11].modenr)>mc) {fprintf(stderr,"window_open_x11: requested modenr > modemax: %d > %d.\n",wd.modenr,mc); wd.modenr=0;}
    wd.modemax=mc;
    if ((x11modes=realloc(x11modes,sizeof(char *)*(mc+2)))==NULL) {fprintf(stderr,"window_open_x11: realloc x11modes: %s.\n",strerror(errno)); return(-1);}
    if (firstcall) {printf("Modes found: ");}
    mw=-1;
    for (i1=0;i1<mc;i1++) {
      snprintf(bf,sizeof(bf),"%dx%d",mi[i1]->hdisplay,mi[i1]->vdisplay);
      if (firstcall) {printf("%s ",bf);}
      x11modes[i1+1]=strdup(bf);
      if (wd.modenr>0) {
        if (i1+1==wd.modenr) {
          width=mi[i1]->hdisplay; height=mi[i1]->vdisplay;
          wd.modewidth=mi[i1]->hdisplay;
          wd.modeheight=mi[i1]->vdisplay;
          mw=i1;
        }
      } else if ((mi[i1]->hdisplay<=width) && (mi[i1]->vdisplay<=height) \
        && (mi[i1]->hdisplay>=SC_WIDTH) && (mi[i1]->vdisplay>=SC_HEIGHT)) {
        width=mi[i1]->hdisplay; height=mi[i1]->vdisplay;
        mw=i1;
      }
    }
    x11modes[mc+1]=NULL;
    if (mw>=0) {
      if (firstcall) {printf("\n");}
      printf("Switching to %dx%d (%d).",mi[mw]->hdisplay,mi[mw]->vdisplay,mw+1);
      if (!firstcall) {printf("\n");}
      XF86VidModeSwitchToMode(wd.display,wd.screen,mi[mw]);
      XF86VidModeSetViewPort(wd.display,wd.screen,0,0);
      XWarpPointer(wd.display,None,None,0,0,0,0,-10000,-10000);
      XFlush(wd.display);
      XSync(wd.display,False);
    }
    wd.virtwidth=width; wd.virtheight=height;
    if (firstcall) {printf("\n");}
    firstcall=0;
  } else {
    wd.virtwidth=640; wd.virtheight=480;
    printf("Mode-switching deactivated.\n");
  }
  fflush(stdout);
#else
  wd.noswitch=1;
#endif

  /* get default bit number of RootWindow of DefaultScreen */
  cdepth=DefaultDepthOfScreen(DefaultScreenOfDisplay(wd.display));
  bs_w=WidthOfScreen(DefaultScreenOfDisplay(wd.display));
  bs_h=HeightOfScreen(DefaultScreenOfDisplay(wd.display));
#ifdef HAVE_X11_EXTS
  if (wd.noswitch) {wd.virtwidth=bs_w; wd.virtheight=bs_h;}
#endif
  /* set scaling factor */
  wd.pos_x=bs_w; wd.pos_y=bs_h;
  bs_w/=SC_WIDTH;
  bs_h/=SC_HEIGHT;
  wd.bp_doppl=(bs_w<bs_h?bs_w:bs_h);
  wd.fullscr=!!(dp&VGAWINDOW_FULL);
  if ((wd.bp_doppl>=3) && (dp&VGAWINDOW_3)) {wd.bp_doppl=3;}
  else if ((wd.bp_doppl>=2) && (dp&VGAWINDOW_2)) {wd.bp_doppl=2;}
  else if ((wd.bp_doppl>=1) && (dp&VGAWINDOW_1)) {wd.bp_doppl=1;}
  if (wd.bp_doppl>3) {wd.bp_doppl=3;}
#ifdef HAVE_X11_EXTS
  while (wd.bp_doppl>1) {
    if ((SC_WIDTH*wd.bp_doppl>wd.virtwidth) || (SC_HEIGHT*wd.bp_doppl>wd.virtheight)) {
      wd.bp_doppl--;
    } else {break;}
  }
#endif
  if (wd.fullscr) {
    px_w=wd.pos_x; px_h=wd.pos_y;
    wd.pos_x-=SC_WIDTH*wd.bp_doppl;
    wd.pos_x/=2;
    wd.pos_y-=SC_HEIGHT*wd.bp_doppl;
    wd.pos_y/=2;
  } else {
    wd.pos_x=wd.pos_y=0;
    px_w=SC_WIDTH*wd.bp_doppl; px_h=SC_HEIGHT*wd.bp_doppl;
  }
#ifdef HAVE_X11_EXTS
  if (wd.fullscr) {
    XWarpPointer(wd.display,None,None,0,0,0,0,-10000,-10000);
    XWarpPointer(wd.display,None,None,0,0,0,0,(WidthOfScreen(DefaultScreenOfDisplay(wd.display))+wd.virtwidth)/2,(HeightOfScreen(DefaultScreenOfDisplay(wd.display))+wd.virtheight)/2);
    XWarpPointer(wd.display,None,None,0,0,0,0,-wd.virtwidth/2,-wd.virtheight/2);
    XFlush(wd.display);
    XSync(wd.display,False);
  } else {
    XWarpPointer(wd.display,None,None,0,0,0,0,-10000,-10000);
    XWarpPointer(wd.display,None,None,0,0,0,0,SC_WIDTH*wd.bp_doppl/2,SC_HEIGHT*wd.bp_doppl/2);
    XFlush(wd.display);
    XSync(wd.display,False);
  }
#endif

  /* get Black/White pixel value of screen number */
  wd.px_black=BlackPixel(wd.display,wd.screen);
  wd.px_white=WhitePixel(wd.display,wd.screen);

  /* set default colormap */
  wd.colormap=DefaultColormap(wd.display,wd.screen);

  /* get a PseudoColor visual or not */
  {XVisualInfo xvt;
   int vn;
#if defined(__cplusplus) || defined(c_plusplus)
   xvt.c_class=PseudoColor;
#else
   xvt.class=PseudoColor;
#endif
   xvt.screen=wd.screen;
   xvt.depth=cdepth;
   wd.visual=XGetVisualInfo(wd.display,VisualScreenMask|VisualDepthMask|VisualClassMask,&xvt,&vn);
   if (vn==0) {
     if (wd.visual!=NULL) {XFree(wd.visual); wd.visual=NULL;}
   }
  }

  /* open window */
  if (wd.fullscr) {attributes.override_redirect=True;}
  attributes.background_pixel=wd.px_black;  /* background color */
  attributes.border_pixel=wd.px_white;  /* foreground color */
  attributes.event_mask=KeyPressMask|KeyReleaseMask;
  attributes.event_mask|=ButtonPressMask|ButtonReleaseMask|PointerMotionMask;
  attributes.event_mask|=ExposureMask|LeaveWindowMask|EnterWindowMask;
  attr_mask=CWEventMask|CWBackPixel|CWBorderPixel;
  if (wd.fullscr) {attr_mask|=CWOverrideRedirect;}
#ifdef HAVE_X11_EXTS
  wd.window=XCreateWindow(wd.display,rootw,(wd.fullscr?((WidthOfScreen(DefaultScreenOfDisplay(wd.display))-px_w)/2):0),(wd.fullscr?((HeightOfScreen(DefaultScreenOfDisplay(wd.display))-px_h)/2):0),px_w,px_h,0,CopyFromParent,InputOutput,(Visual *)CopyFromParent,attr_mask,&attributes);
#else
  wd.window=XCreateWindow(wd.display,rootw,0,0,px_w,px_h,0,CopyFromParent,InputOutput,(Visual *)CopyFromParent,attr_mask,&attributes);
#endif

  /* set mouse buttons */
  inp.m_xpos=inp.m_ypos=-1;
  inp.m_xold=inp.m_yold=0;
  memset(inp.m_press,0,sizeof(inp.m_press));
  memset(inp.m_mody,0,sizeof(inp.m_mody));
  memset(inp.m_flush,0,sizeof(inp.m_flush));
#ifdef HAVE_X11_EXTS
  inp.m_xpos=SC_WIDTH/2;
  inp.m_ypos=SC_HEIGHT/2;
#endif
  {unsigned char mapr[3],nmap;
   int erg;
   nmap=3;
   erg=XGetPointerMapping(wd.display,mapr,nmap);
   memset(inp.m_cmp,0,sizeof(inp.m_cmp));
   if (erg>0) {inp.m_cmp[MOUSE_LEFT]=mapr[0];}
   if (erg>1) {inp.m_cmp[MOUSE_MIDDLE]=mapr[1];}
   if (erg>2) {inp.m_cmp[MOUSE_RIGHT]=mapr[2];}
  }

  /* set window hints */
  {XSizeHints sh;
   XClassHint ch;
   XWMHints wh;
#ifdef HAVE_X11_EXTS
   sh.x=(wd.fullscr?((WidthOfScreen(DefaultScreenOfDisplay(wd.display))-px_w)/2):0);
   sh.y=(wd.fullscr?((HeightOfScreen(DefaultScreenOfDisplay(wd.display))-px_h)/2):0);
#else
   sh.x=0;
   sh.y=0;
#endif
   sh.height=px_h;
   sh.width=px_w;
   sh.min_height=px_h;
   sh.min_width=px_w;
   sh.base_height=px_h;
   sh.base_width=px_w;
   sh.flags=USPosition|USSize|PMinSize|PBaseSize;
   XSetWMNormalHints(wd.display,wd.window,&sh);
   XStoreName(wd.display,wd.window,pnam);
   ch.res_class=pnam;
   ch.res_name=pnam;
   XSetClassHint(wd.display,wd.window,&ch);
   wh.initial_state=NormalState;
   wh.input=True;
   wh.flags=InputHint|StateHint;
   XSetWMHints(wd.display,wd.window,&wh);
  }

  /* show window */
  XMapRaised(wd.display,wd.window);
  XFlush(wd.display);

  /* create backbuffer */
  wd.bpuff[0]=vg_bitmap_createnew(SC_WIDTH,SC_HEIGHT);
  if (wd.bpuff[0]==NULL) {fprintf(stderr,"window_open_x11: can't create backbuffer.\n"); return(-1);}
  wd.bpuff[1]=vg_bitmap_createnew(SC_WIDTH,SC_HEIGHT);
  if (wd.bpuff[1]==NULL) {fprintf(stderr,"window_open_x11: can't create backbuffer.\n"); return(-1);}
  backbuffer=wd.bpuff[0]; wd.ipuff=0;

  /* set cursor */
  {Pixmap pixcurs;
   XGCValues gcx;
   XColor xc;
   GC gc0;
   Cursor curs;
   xc.red=0; xc.green=0; xc.blue=0;
   xc.flags=DoRed|DoGreen|DoBlue;
   pixcurs=XCreatePixmap(wd.display,wd.window,1,1,1);
   gcx.foreground=0;
   gcx.background=wd.px_black;
   gc0=XCreateGC(wd.display,pixcurs,GCForeground|GCBackground,&gcx);
   XDrawPoint(wd.display,pixcurs,gc0,0,0);
   curs=XCreatePixmapCursor(wd.display,pixcurs,pixcurs,&xc,&xc,0,0);
   XDefineCursor(wd.display,wd.window,curs);
  }

  {XGCValues gcx;
   XGetGCValues(wd.display,XDefaultGC(wd.display,XDefaultScreen(wd.display)),GCForeground,&gcx);
   wd.gc=XCreateGC(wd.display,wd.window,GCForeground,&gcx);
  }
  XFillRectangle(wd.display,wd.window,wd.gc,0,0,px_w,px_h);

  /* set keys for keyboard */
  if (init_keys()<0) {fprintf(stderr,"window_open_x11: init_keys error.\n"); return(-1);}
  memset(inp.k_last,0,sizeof(inp.k_last));
  memset(inp.k_press,0,sizeof(inp.k_press));
  memset(inp.k_mody,0,sizeof(inp.k_mody));
  memset(inp.k_flush,0,sizeof(inp.k_flush));

  /* get an async grab on keyboard and pointer */
  if (wd.fullscr) {
    XGrabKeyboard(wd.display,wd.window,True,GrabModeAsync,GrabModeAsync,CurrentTime);
    XGrabPointer(wd.display,wd.window,True,ButtonPressMask|ButtonReleaseMask|PointerMotionMask|LeaveWindowMask,GrabModeAsync,GrabModeAsync,None,None,CurrentTime);
  }

  sleep(2);
  wd.is_open=1;
  mouse_speed_x11(2);

  if (wd.visual!=NULL) {  /* create a colormap */
    wd.colormap=XCreateColormap(wd.display,wd.window,wd.visual->visual,AllocAll);
  }
  if (wd.ximage!=NULL) {XDestroyImage(wd.ximage);}
  if ((wd.ximage=XGetImage(wd.display,wd.window,wd.pos_x,wd.pos_y,SC_WIDTH*wd.bp_doppl,SC_HEIGHT*wd.bp_doppl,0xffffffff,ZPixmap))==NULL) {fprintf(stderr,"window_open_x11: XGetImage: NULL\n"); return(-1);}

  /* if superuser with setuid-bit, fall back to original user */
  if ((getuid()!=0) && (geteuid()==0)) {
    if (getgid()!=getegid()) {setgid(getgid());}
    setuid(getuid());
  }
  window_mode.isset=1;
  window_mode.videolib=VIDEOLIB_X11;
  window_mode.vlb[VIDEOLIB_X11].scale=(wd.bp_doppl==1?VGAWINDOW_1:(wd.bp_doppl==2?VGAWINDOW_2:VGAWINDOW_3));
  window_mode.vlb[VIDEOLIB_X11].full=wd.fullscr;
  window_mode.vlb[VIDEOLIB_X11].noswitch=wd.noswitch;
  window_mode.vlb[VIDEOLIB_X11].directdraw=directdraw;
#ifdef HAVE_X11_EXTS
  window_mode.vlb[VIDEOLIB_X11].modenr=wd.modenr;
  window_mode.vlb[VIDEOLIB_X11].modemax=wd.modemax;
  window_mode.vlb[VIDEOLIB_X11].modewidth=wd.modewidth;
  window_mode.vlb[VIDEOLIB_X11].modeheight=wd.modeheight;
#else
  window_mode.vlb[VIDEOLIB_X11].modenr=window_mode.vlb[VIDEOLIB_X11].modemax=0;
  window_mode.vlb[VIDEOLIB_X11].modewidth=window_mode.vlb[VIDEOLIB_X11].modeheight=0;
#endif
  return(0);
} /* Ende window_open_x11 */


void window_flush_x11() {
/* give backbuffer out to screen */
  static int dirdraw=-1;
  int i1,i2,i3,i4;
  unsigned char * spt1,* spt2,* dptr;
  if (!wd.is_open) {return;}
  if (dirdraw!=window_mode.vlb[VIDEOLIB_X11].directdraw) {
    wd.brpuff=1;
    dirdraw=window_mode.vlb[VIDEOLIB_X11].directdraw;
  }

_fredo:
  if ((!wd.brpuff) && (window_mode.vlb[VIDEOLIB_X11].directdraw)) {  /* screenoutput direct */
    static XPoint * xpo=NULL;
    static int npo=0;
    if ((xpo==NULL) || (npo!=wd.bp_doppl*wd.bp_doppl)) {
      if (xpo!=NULL) {free(xpo);}
      npo=wd.bp_doppl*wd.bp_doppl;
      if ((xpo=malloc(sizeof(*xpo)*npo))==NULL) {wd.brpuff=1; goto _fredo;}
      for (i1=0;i1<wd.bp_doppl;i1++) {
        xpo[i1*wd.bp_doppl].x=-wd.bp_doppl+1;
        xpo[i1*wd.bp_doppl].y=1;
        for (i2=1;i2<wd.bp_doppl;i2++) {
          xpo[i2+i1*wd.bp_doppl].x=1;
          xpo[i2+i1*wd.bp_doppl].y=0;
        }
      }
    }
    for (i1=0;i1<SC_HEIGHT;i1++) {
      spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
      spt2=wd.bpuff[!wd.ipuff]->pxm+i1*SC_WIDTH;
      for (i2=0;i2<SC_WIDTH;i2++) {
        if (*spt1!=*spt2) {
          XSetForeground(wd.display,wd.gc,wd.colpix[*spt1]);
          xpo[0].x=i2*wd.bp_doppl+wd.pos_x;
          xpo[0].y=i1*wd.bp_doppl+wd.pos_y;
          XDrawPoints(wd.display,wd.window,wd.gc,xpo,npo,CoordModePrevious);
        }
        spt1++; spt2++;
      }
    }
  } else {  /* screenoutput with image */
    if (wd.ximage->bits_per_pixel==8) {
      switch(wd.bp_doppl) {
        case 1:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            dptr=(unsigned char *)wd.ximage->data+i1*wd.ximage->bytes_per_line;
            for (i2=0;i2<SC_WIDTH;i2++) {
              *dptr=wd.colpix[*spt1]; dptr++;
              spt1++;
            }
          }
          break;
        case 2:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            spt2=wd.bpuff[!wd.ipuff]->pxm+i1*SC_WIDTH;
            dptr=(unsigned char *)wd.ximage->data+(i1*wd.bp_doppl)*wd.ximage->bytes_per_line;
            for (i2=0;i2<SC_WIDTH;i2++) {
              if ((wd.brpuff) || (*spt1!=*spt2)) {
                *dptr=wd.colpix[*spt1]; *(dptr+wd.ximage->width)=wd.colpix[*spt1]; dptr++;
                *dptr=wd.colpix[*spt1]; *(dptr+wd.ximage->width)=wd.colpix[*spt1]; dptr++;
              } else {dptr+=2;}
              spt1++; spt2++;
            }
          }
          break;
        default:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            spt2=wd.bpuff[!wd.ipuff]->pxm+i1*SC_WIDTH;
            dptr=(unsigned char *)wd.ximage->data+(i1*wd.bp_doppl)*wd.ximage->bytes_per_line;
            for (i2=0;i2<SC_WIDTH;i2++) {
              if ((wd.brpuff) || (*spt1!=*spt2)) {
                *dptr=wd.colpix[*spt1]; *(dptr+wd.ximage->width)=wd.colpix[*spt1]; *(dptr+wd.ximage->width*2)=wd.colpix[*spt1]; dptr++;
                *dptr=wd.colpix[*spt1]; *(dptr+wd.ximage->width)=wd.colpix[*spt1]; *(dptr+wd.ximage->width*2)=wd.colpix[*spt1]; dptr++;
                *dptr=wd.colpix[*spt1]; *(dptr+wd.ximage->width)=wd.colpix[*spt1]; *(dptr+wd.ximage->width*2)=wd.colpix[*spt1]; dptr++;
              } else {dptr+=3;}
              spt1++; spt2++;
            }
          }
          break;
      }
    } else if (wd.ximage->bits_per_pixel==16) {
      switch(wd.bp_doppl) {
        case 1:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            wd.pu_s=(union _u_s *)(wd.ximage->data+i1*wd.ximage->bytes_per_line);
            for (i2=0;i2<SC_WIDTH;i2++) {
              wd.pu_s->w=wd.u_s[*spt1].w; wd.pu_s++;
              spt1++;
            }
          }
          break;
        case 2:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            spt2=wd.bpuff[!wd.ipuff]->pxm+i1*SC_WIDTH;
            wd.pu_s=(union _u_s *)(wd.ximage->data+(i1*wd.bp_doppl)*wd.ximage->bytes_per_line);
            for (i2=0;i2<SC_WIDTH;i2++) {
              if ((wd.brpuff) || (*spt1!=*spt2)) {
                wd.pu_s->w=wd.u_s[*spt1].w; (wd.pu_s+wd.ximage->width)->w=wd.u_s[*spt1].w; wd.pu_s++;
                wd.pu_s->w=wd.u_s[*spt1].w; (wd.pu_s+wd.ximage->width)->w=wd.u_s[*spt1].w; wd.pu_s++;
              } else {wd.pu_s+=2;}
              spt1++; spt2++;
            }
          }
          break;
        default:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            spt2=wd.bpuff[!wd.ipuff]->pxm+i1*SC_WIDTH;
            wd.pu_s=(union _u_s *)(wd.ximage->data+(i1*wd.bp_doppl)*wd.ximage->bytes_per_line);
            for (i2=0;i2<SC_WIDTH;i2++) {
              if ((wd.brpuff) || (*spt1!=*spt2)) {
                wd.pu_s->w=wd.u_s[*spt1].w; (wd.pu_s+wd.ximage->width)->w=wd.u_s[*spt1].w; (wd.pu_s+wd.ximage->width*2)->w=wd.u_s[*spt1].w; wd.pu_s++;
                wd.pu_s->w=wd.u_s[*spt1].w; (wd.pu_s+wd.ximage->width)->w=wd.u_s[*spt1].w; (wd.pu_s+wd.ximage->width*2)->w=wd.u_s[*spt1].w; wd.pu_s++;
                wd.pu_s->w=wd.u_s[*spt1].w; (wd.pu_s+wd.ximage->width)->w=wd.u_s[*spt1].w; (wd.pu_s+wd.ximage->width*2)->w=wd.u_s[*spt1].w; wd.pu_s++;
              } else {wd.pu_s+=3;}
              spt1++; spt2++;
            }
          }
          break;
      }
    } else if (wd.ximage->bits_per_pixel==32) {
      switch(wd.bp_doppl) {
        case 1:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            wd.pu_i=(union _u_i *)(wd.ximage->data+i1*wd.ximage->bytes_per_line);
            for (i2=0;i2<SC_WIDTH;i2++) {
              wd.pu_i->w=wd.u_i[*spt1].w; wd.pu_i++;
              spt1++;
            }
          }
          break;
        case 2:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            spt2=wd.bpuff[!wd.ipuff]->pxm+i1*SC_WIDTH;
            wd.pu_i=(union _u_i *)(wd.ximage->data+(i1*wd.bp_doppl)*wd.ximage->bytes_per_line);
            for (i2=0;i2<SC_WIDTH;i2++) {
              if ((wd.brpuff) || (*spt1!=*spt2)) {
                wd.pu_i->w=wd.u_i[*spt1].w; (wd.pu_i+wd.ximage->width)->w=wd.u_i[*spt1].w; wd.pu_i++;
                wd.pu_i->w=wd.u_i[*spt1].w; (wd.pu_i+wd.ximage->width)->w=wd.u_i[*spt1].w; wd.pu_i++;
              } else {wd.pu_i+=2;}
              spt1++; spt2++;
            }
          }
          break;
        default:
          for (i1=0;i1<SC_HEIGHT;i1++) {
            spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
            spt2=wd.bpuff[!wd.ipuff]->pxm+i1*SC_WIDTH;
            wd.pu_i=(union _u_i *)(wd.ximage->data+(i1*wd.bp_doppl)*wd.ximage->bytes_per_line);
            for (i2=0;i2<SC_WIDTH;i2++) {
              if ((wd.brpuff) || (*spt1!=*spt2)) {
                wd.pu_i->w=wd.u_i[*spt1].w; (wd.pu_i+wd.ximage->width)->w=wd.u_i[*spt1].w; (wd.pu_i+wd.ximage->width*2)->w=wd.u_i[*spt1].w; wd.pu_i++;
                wd.pu_i->w=wd.u_i[*spt1].w; (wd.pu_i+wd.ximage->width)->w=wd.u_i[*spt1].w; (wd.pu_i+wd.ximage->width*2)->w=wd.u_i[*spt1].w; wd.pu_i++;
                wd.pu_i->w=wd.u_i[*spt1].w; (wd.pu_i+wd.ximage->width)->w=wd.u_i[*spt1].w; (wd.pu_i+wd.ximage->width*2)->w=wd.u_i[*spt1].w; wd.pu_i++;
              } else {wd.pu_i+=3;}
              spt1++; spt2++;
            }
          }
          break;
      }
    } else {
      for (i1=0;i1<SC_HEIGHT;i1++) {
        spt1=wd.bpuff[wd.ipuff]->pxm+i1*SC_WIDTH;
        for (i2=0;i2<SC_WIDTH;i2++) {
          for (i3=0;i3<wd.bp_doppl;i3++) {
            for (i4=0;i4<wd.bp_doppl;i4++) {
              XPutPixel(wd.ximage,i2*wd.bp_doppl+i4,i1*wd.bp_doppl+i3,wd.colpix[*spt1]);
            }
          }
          spt1++;
        }
      }
    }
    XPutImage(wd.display,wd.window,wd.gc,wd.ximage,0,0,wd.pos_x,wd.pos_y,SC_WIDTH*wd.bp_doppl,SC_HEIGHT*wd.bp_doppl);
  }

  wd.ipuff=!wd.ipuff;
  backbuffer=wd.bpuff[wd.ipuff];
  memmove(wd.bpuff[wd.ipuff]->pxm,wd.bpuff[!wd.ipuff]->pxm,wd.bpuff[wd.ipuff]->width*wd.bpuff[wd.ipuff]->height);
  wd.brpuff=0;
  XFlush(wd.display);
} /* Ende window_flush_x11 */


char ** getmodes_x11() {
  return(x11modes);
} /* Ende getmodes_x11 */


void window_close_x11() {
/* close window */
  if (!wd.is_open) {return;}
  if (wd.fullscr) {
    XUngrabPointer(wd.display,CurrentTime);
    XUngrabKeyboard(wd.display,CurrentTime);
  }
  vg_bitmap_free(wd.bpuff[0]); wd.bpuff[0]=NULL;
  vg_bitmap_free(wd.bpuff[1]); wd.bpuff[1]=NULL;
  XFreeGC(wd.display,wd.gc);
  if (wd.visual!=NULL) {  /* created colormap */
    XFree(wd.visual);
    XFreeColormap(wd.display,wd.colormap);
    wd.visual=NULL;
  }
  mouse_speed_x11(-1);  /* restore old mouse speed */

  restore_mode();

  XDestroyWindow(wd.display,wd.window);
  XCloseDisplay(wd.display);
  if (x11modes!=NULL) {
    char ** mdp;
    for (mdp=x11modes;*mdp!=NULL;mdp++) {free(*mdp);}
    free(x11modes); x11modes=NULL;
  }
  wd.is_open=0;
} /* Ende window_close_x11 */


void key_discard_x11() {
  if (!wd.is_open) {return;}
  key_update_x11();
  memset(inp.k_flush,1,sizeof(inp.k_flush));
  memset(inp.m_flush,1,sizeof(inp.m_flush));
} /* Ende key_discard_x11 */


void key_update_x11() {
/* get pressed or released keys/mousebuttons */
  int i1;
  XEvent event;
  KeySym keysym;
  XComposeStatus costa;
  char keychar[128],* ptr;

  if (!wd.is_open) {return;}
  memmove(inp.k_last,inp.k_press,sizeof(inp.k_last));
  memset(inp.k_mody,0,sizeof(inp.k_mody));
  memset(inp.m_mody,0,sizeof(inp.m_mody));

  if (wd.fullscr) {
    Status sta;
    sta=XGrabKeyboard(wd.display,wd.window,True,GrabModeAsync,GrabModeAsync,CurrentTime);
    if (sta!=GrabSuccess) {return;}
    sta=XGrabPointer(wd.display,wd.window,True,ButtonPressMask|ButtonReleaseMask|PointerMotionMask|LeaveWindowMask,GrabModeAsync,GrabModeAsync,None,None,CurrentTime);
    if (sta!=GrabSuccess) {return;}
  }

  XSync(wd.display,False);
  while (XPending(wd.display)) {
    XNextEvent(wd.display,&event);
    if ((event.type==KeyPress) || (event.type==KeyRelease)) {
      XLookupString(&event.xkey,keychar,1,&keysym,&costa);
      if ((ptr=XKeysymToString(keysym))!=NULL) {
        if (inthash_get(inp.k_cmp,case_upper(ptr),&i1)==0) {
          if (event.type==KeyPress) {
            if (inp.k_last[i1]==0) {inp.k_mody[i1]=1;} else {inp.k_mody[i1]=0;}
            inp.k_press[i1]=1;
          } else if (event.type==KeyRelease) {
            if (inp.k_last[i1]==1) {inp.k_mody[i1]=1;} else {inp.k_mody[i1]=0;}
            inp.k_press[i1]=0;
          }
        }
      }
    } else if (event.type==ButtonPress) {
      if (event.xbutton.button>0) {
        if (event.xbutton.button==inp.m_cmp[MOUSE_LEFT]) {
          inp.m_press[MOUSE_LEFT]=1;
          inp.m_mody[MOUSE_LEFT]=1;
        } else if (event.xbutton.button==inp.m_cmp[MOUSE_RIGHT]) {
          inp.m_press[MOUSE_RIGHT]=1;
          inp.m_mody[MOUSE_RIGHT]=1;
        } else if (event.xbutton.button==inp.m_cmp[MOUSE_MIDDLE]) {
          inp.m_press[MOUSE_MIDDLE]=1;
          inp.m_mody[MOUSE_MIDDLE]=1;
        }
        inp.m_xpos=(event.xbutton.x-wd.pos_x)/wd.bp_doppl;
        if ((inp.m_xpos<0) || (inp.m_xpos>=SC_WIDTH)) {inp.m_xpos=-1;}
        inp.m_ypos=(event.xbutton.y-wd.pos_y)/wd.bp_doppl;
        if ((inp.m_ypos<0) || (inp.m_ypos>=SC_HEIGHT)) {inp.m_ypos=-1;}
      }
    } else if (event.type==ButtonRelease) {
      if (event.xbutton.button>0) {
        if (event.xbutton.button==inp.m_cmp[MOUSE_LEFT]) {
          inp.m_press[MOUSE_LEFT]=0;
          inp.m_mody[MOUSE_LEFT]=1;
        } else if (event.xbutton.button==inp.m_cmp[MOUSE_RIGHT]) {
          inp.m_press[MOUSE_RIGHT]=0;
          inp.m_mody[MOUSE_RIGHT]=1;
        } else if (event.xbutton.button==inp.m_cmp[MOUSE_MIDDLE]) {
          inp.m_press[MOUSE_MIDDLE]=0;
          inp.m_mody[MOUSE_MIDDLE]=1;
        }
        inp.m_xpos=(event.xbutton.x-wd.pos_x)/wd.bp_doppl;
        if ((inp.m_xpos<0) || (inp.m_xpos>=SC_WIDTH)) {inp.m_xpos=-1;}
        inp.m_ypos=(event.xbutton.y-wd.pos_y)/wd.bp_doppl;
        if ((inp.m_ypos<0) || (inp.m_ypos>=SC_HEIGHT)) {inp.m_ypos=-1;}
      }
    } else if (event.type==MotionNotify) {
      inp.m_xpos=(event.xbutton.x-wd.pos_x)/wd.bp_doppl;
      if ((inp.m_xpos<0) || (inp.m_xpos>=SC_WIDTH)) {inp.m_xpos=-1;}
      inp.m_ypos=(event.xbutton.y-wd.pos_y)/wd.bp_doppl;
      if ((inp.m_ypos<0) || (inp.m_ypos>=SC_HEIGHT)) {inp.m_ypos=-1;}
      if (inp.m_xpos>-1) {inp.m_xold=event.xbutton.x;}
      if (inp.m_ypos>-1) {inp.m_yold=event.xbutton.y;}
    } else if (event.type==EnterNotify) {
      window_flush_x11();
    } else if (event.type==LeaveNotify) {
      inp.m_xpos=inp.m_ypos=-1;
    } else if (event.type==Expose) {
      window_flush_x11();
    }
  }
#ifdef HAVE_X11_EXTS
  if (wd.fullscr) {
    if ((inp.m_xpos<0) || (inp.m_ypos<0)) {
      XWarpPointer(wd.display,None,None,0,0,0,0,-10000,-10000);
      XWarpPointer(wd.display,None,None,0,0,0,0,(WidthOfScreen(DefaultScreenOfDisplay(wd.display))+wd.virtwidth)/2,(HeightOfScreen(DefaultScreenOfDisplay(wd.display))+wd.virtheight)/2);
      XWarpPointer(wd.display,None,None,0,0,0,0,inp.m_xold-(WidthOfScreen(DefaultScreenOfDisplay(wd.display))+wd.virtwidth)/2,inp.m_yold-(HeightOfScreen(DefaultScreenOfDisplay(wd.display))+wd.virtheight)/2);
      XFlush(wd.display);
      XSync(wd.display,False);
      inp.m_xpos=(inp.m_xold-wd.pos_x)/wd.bp_doppl;
      inp.m_ypos=(inp.m_yold-wd.pos_y)/wd.bp_doppl;
    }
  }
#endif
} /* Ende key_update_x11 */


int key_pressed_x11(int key,int art) {
/* whether a key is pressed
** 1.arg: key-define (KEY_0, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
*/
  if (!wd.is_open) {return(0);}
  if ((key<0) || (key>KEYS_INDEX)) {return(0);}
  if (inp.k_flush[key]) {
    if (inp.k_press[key]&inp.k_mody[key]) {inp.k_flush[key]=0;} else {return(0);}
  }
  if (art==LONGKEY) {
    return(inp.k_press[key]);
  } else if (art==SHORTKEY) {
    return(inp.k_press[key]&inp.k_mody[key]);
  }
  return(0);
} /* Ende key_pressed_x11 */


int mouse_found_x11() {
/* whether a mouse is found */
  if (!wd.is_open) {return(0);}
  return(1);
} /* Ende mouse_found_x11 */


int mouse_x_x11() {
/* mouse x-position or -1=not in window */
  return(inp.m_xpos);
} /* Ende mouse_x_x11 */


int mouse_y_x11() {
/* mouse y-position or -1=not in window */
  return(inp.m_ypos);
} /* Ende mouse_y_x11 */


int mouse_pressed_x11(int bnr,int art) {
/* whether a mousebutton is pressed
** 1.arg: mousebutton-define (MOUSE_LEFT, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
*/
  if (!wd.is_open) {return(0);}
  switch(bnr) {
    case MOUSE_LEFT:
    case MOUSE_RIGHT:
    case MOUSE_MIDDLE:
      if (inp.m_flush[bnr]) {
        if (inp.m_press[bnr]&inp.m_mody[bnr]) {inp.m_flush[bnr]=0;} else {return(0);}
      }
      if (art==LONGKEY) {
        return(inp.m_press[bnr]);
      } else if (art==SHORTKEY) {
        return(inp.m_press[bnr]&inp.m_mody[bnr]);
      }
  }
  return(0);
} /* Ende mouse_pressed_x11 */


int mouse_speed_x11(int index) {
/* set speed of mouse
** 1.arg: 1=slow, 2=middle, 3=fast
**        0=return only current speed
**        -1=restore original speed
** return: previous speed number
**         or 0=speed cannot be changed
*/
  static int j1=0,j2,j3,mvs=1;
  int vs;
  if (!wd.is_open) {return(0);}
  if (j1==0) {XGetPointerControl(wd.display,&j1,&j2,&j3);}
  if (index<0) {  /* restore original mouse speed */
    XChangePointerControl(wd.display,True,True,j1,j2,j3);
    vs=mvs;
  } else {
    if (index>3) {index=3;}
    if (index>0) {
      XChangePointerControl(wd.display,True,True,10,(40-index*10)/wd.bp_doppl,5);
      vs=mvs;
      mvs=index;
    } else {vs=mvs;}
  }
  return(vs);
} /* Ende mouse_speed_x11 */


unsigned char * keystonw_x11() {
/* only for internal use with networking:
** returns a static unsigned char field for (non)pressed keys and mouse buttons
** format: unsigned char (flag for new or old): 1
**         unsigned char[KEYS_INDEX] (for keystrokes):
**           bit0=k_press[], bit1=k_mody[], bit2=k_flush[]
**         unsigned char (for MOUSE_LEFT):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char (for MOUSE_MIDDLE):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char (for MOUSE_RIGHT):
**           bit0=m_press[], bit1=m_mody[], bit2=m_flush[]
**         unsigned char[2] (for mouse_x): <0|1>+<0-255> or <255>+<255>=-1
**         unsigned char (for mouse_y): <0-SC_HEIGHT> or <255>=-1
*/
  static unsigned char retw[1+KEYS_INDEX+3+2+1]="";  /* NW_KEYSIZE */
  int i1;
  /* flag */
  retw[0]=1;
  /* keys */
  for (i1=0;i1<KEYS_INDEX;i1++) {
    if (inp.k_flush[i1]) {
      if (inp.k_press[i1]&inp.k_mody[i1]) {inp.k_flush[i1]=0;}
    }
    retw[1+i1]=((!!inp.k_press[i1])|((!!inp.k_mody[i1])<<1)|((!!inp.k_flush[i1])<<2));
  }
  i1=1+KEYS_INDEX;
  /* mouse buttons */
  if (inp.m_flush[MOUSE_LEFT]) {
    if (inp.m_press[MOUSE_LEFT]&inp.m_mody[MOUSE_LEFT]) {inp.m_flush[MOUSE_LEFT]=0;}
  }
  retw[i1++]=((!!inp.m_press[MOUSE_LEFT])|((!!inp.m_mody[MOUSE_LEFT])<<1)|((!!inp.m_flush[MOUSE_LEFT])<<2));
  if (inp.m_flush[MOUSE_MIDDLE]) {
    if (inp.m_press[MOUSE_MIDDLE]&inp.m_mody[MOUSE_MIDDLE]) {inp.m_flush[MOUSE_MIDDLE]=0;}
  }
  retw[i1++]=((!!inp.m_press[MOUSE_MIDDLE])|((!!inp.m_mody[MOUSE_MIDDLE])<<1)|((!!inp.m_flush[MOUSE_MIDDLE])<<2));
  if (inp.m_flush[MOUSE_RIGHT]) {
    if (inp.m_press[MOUSE_RIGHT]&inp.m_mody[MOUSE_RIGHT]) {inp.m_flush[MOUSE_RIGHT]=0;}
  }
  retw[i1++]=((!!inp.m_press[MOUSE_RIGHT])|((!!inp.m_mody[MOUSE_RIGHT])<<1)|((!!inp.m_flush[MOUSE_RIGHT])<<2));
  /* mouse position */
  retw[i1++]=(inp.m_xpos<0?255:inp.m_xpos/256);
  retw[i1++]=(inp.m_xpos<0?255:inp.m_xpos%256);
  retw[i1++]=(inp.m_ypos<0?255:inp.m_ypos%SC_HEIGHT);
  return(retw);
} /* Ende keystonw_x11 */


int brightness_x11(int vol) {
/* reduces or increases brightness of colormap
** 1.arg: -63 (darkest) to 62 (lightest), 0 is default, or 99=previous
** return: previous brightness value (-63 to 62)
*/
  static int lastvol=0;
  int r,g,b;
  int i1,i2,i3;
  float f1;
  XColor xc;
  unsigned long ul;
  if (!wd.is_open) {return(0);}
  if (vol==99) {vol=lastvol;}
  else if (vol<-63) {vol=-63;}
  else if (vol>62) {vol=62;}
  for (i1=0;i1<256;i1++) {  /* fcpix: color map of e.g. RGB256_FILE */
    i3=vol;
    r=fcpix[i1].r;
    g=fcpix[i1].g;
    b=fcpix[i1].b;
    if (vol>0) {
      if (r==0) {r=1;}
      if (g==0) {g=1;}
      if (b==0) {b=1;}
    }
    i2=(r>g?(r>b?r:b):(g>b?g:b));  /* get highest value */
    if (i2>0) {
      if (i2+i3>63) {i3=63-i2;}
      f1=(float)i3/(float)i2;
      r+=(f1*r);
      g+=(f1*g);
      b+=(f1*b);
      if (r<0) {r=0;} else if (r>63) {r=63;}
      if (g<0) {g=0;} else if (g>63) {g=63;}
      if (b<0) {b=0;} else if (b>63) {b=63;}
      if ((vol>0) && ((r==63) || (g==63) || (b==63))) {
        float f2;
        i3=(fcpix[i1].r>fcpix[i1].g?(fcpix[i1].r>fcpix[i1].b?fcpix[i1].r:fcpix[i1].b):(fcpix[i1].g>fcpix[i1].b?fcpix[i1].g:fcpix[i1].b));
        if (i3==0) {i3=1;}
        f2=63/i3;
        i3=vol-(63-i3);
        if (i3>0) {
          i2=(r<g?(r<b?r:b):(g<b?g:b));  /* get lowest value */
          if (i2>0) {
            if (i2+i3>63) {i3=63-i2;}
            f1=(float)i3/(float)i2*f2;
            r+=(f1*r);
            g+=(f1*g);
            b+=(f1*b);
            if (r<0) {r=0;} else if (r>63) {r=63;}
            if (g<0) {g=0;} else if (g>63) {g=63;}
            if (b<0) {b=0;} else if (b>63) {b=63;}
          }
        }
      }
    }
    r<<=2; g<<=2; b<<=2;
    xc.red=r*256; xc.green=g*256; xc.blue=b*256;
    xc.flags=DoRed|DoGreen|DoBlue;
    if (wd.visual!=NULL) {  /* created colormap */
      ul=(unsigned long)i1;
      xc.pixel=ul;
      XStoreColor(wd.display,wd.colormap,&xc);
    } else {  /* default colormap */
      if (XAllocColor(wd.display,wd.colormap,&xc)!=0) {ul=xc.pixel;} else {ul=wd.px_white;}
    }
    wd.colpix[i1]=ul;
  }
  if (wd.visual!=NULL) {  /* created colormap */
    XSetWindowColormap(wd.display,wd.window,wd.colormap);
    XInstallColormap(wd.display,wd.colormap);
  }
  if (wd.ximage->bits_per_pixel==16) {
    for (i1=0;i1<256;i1++) {
      XPutPixel(wd.ximage,0,0,wd.colpix[i1]);
      wd.pu_s=(union _u_s *)wd.ximage->data; wd.u_s[i1].w=wd.pu_s->w;
    }
  } else if (wd.ximage->bits_per_pixel==32) {
    for (i1=0;i1<256;i1++) {
      XPutPixel(wd.ximage,0,0,wd.colpix[i1]);
      wd.pu_i=(union _u_i *)wd.ximage->data; wd.u_i[i1].w=wd.pu_i->w;
    }
  }
  wd.brpuff=1;
  i1=lastvol; lastvol=vol;
  return(i1);
} /* Ende brightness_x11 */
