/* Copyright 2021 Kurt Nienhaus
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

extern struct vg3_ofunc * ofunc_new(void);

const char * get_oid_name(int);

#define MAX_PLAYER 4

struct sel_players {
  int exit;
  int plf[MAX_PLAYER];
};

static int play_intro(struct vg3_window *);
static void fade_out(struct vg3_window *, int);
static int play_extro(struct vg3_window *, int, int);
static void draw_bg(struct vg3_window *);
static void print_text(struct vg3_window *, const char *);
static int sel_single_nw(struct g_main *, char *, size_t, int *);
static int sel_maze(struct g_main *, char *, size_t, int);
static int sel_clockspeed(struct g_main *, int *, int);
static struct sel_players sel_player_nw(struct g_main *, int, int);
static struct sel_players sel_player_single(struct g_main *);
static int hilfe(void *);


/* get_oid_name:
 * returns object-id as a string
 * @param oidn  object-id (OID_NAMES)
 * @return  object-id as string (or empty)
 */
const char *
get_oid_name(int oidn)
{
  switch(oidn) {
    case OID_NAME_MAZE:      return "maze";
    case OID_NAME_PLAYER:    return "player";
    case OID_NAME_GHOST:     return "ghost";
    case OID_NAME_MINOTAUR:  return "minotaur";
    case OID_NAME_MUSIC:     return "music";
  }
  return "";
}


/* play intro film */
static int
play_intro(struct vg3_window *wstruct)
{
  struct vg3_multilang *mlang;
  char buf[128];
  int inr, erg;

  if (wstruct == NULL) { return 0; }

  mlang = VG3_multilang_new(NULL, "en");
  VG3_multilang_add(mlang, FILES_DIR "/mlang/intro");

  erg = 0;
  for (inr = 1; inr <= 4; inr++) {
    snprintf(buf, sizeof(buf), "%s/intro/intro%d.film", FILES_DIR, inr);
    if ((erg = VG3_film_play(wstruct, mlang, buf, 1)) > 0) { break; }
  }
  if (erg != 1) { erg = 0; }

  VG3_multilang_free(mlang);

  return erg;
}


static void
fade_out(struct vg3_window *wstruct, int bnr)
{
  struct vg3_image_attributes wattr;
  char buf[128];
  int anr;

  snprintf(buf, sizeof(buf), "%s/sound/%s-scream.wav", FILES_DIR, (bnr % 2 ? "f" : "m"));
  anr = VG3_audio_load(wstruct, buf, 100, VGAG3_AUDIO_VOLUME_SOUND);

  VG3_audio_stop(wstruct, VG3_audio_group(VGAG3_AUDIO_VOLUME_ALL), 0);
  VG3_audio_play(wstruct, anr, 0, 0);

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&wattr);
  for (;;) {
    if (!VG3_audio_isplaying(wstruct, anr)) { break; }
    if (wattr.bright < 10) { wattr.bright = 0; } else { wattr.bright -= 10; }
    VG3_window_attributes(wstruct, &wattr, NULL, -1, NULL);
    VG3_window_update(wstruct, 0, 0);
    VG3_wait_time(100);
  }
  VG3_audio_unload(wstruct, anr);
  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&wattr);
  VG3_window_attributes(wstruct, &wattr, NULL, -1, NULL);
}


/* play extro film */
static int
play_extro(struct vg3_window *wstruct, int filmnr, int bnr)
{
  char buf[128];
  int erg;

  if (wstruct == NULL) { return 0; }

  erg = 0;

  if (filmnr == 1) {
    snprintf(buf, sizeof(buf), "%s/extro/extro-gone-%d.film", FILES_DIR, bnr);
    erg = VG3_film_play(wstruct, NULL, buf, 1);

  } else if (filmnr == 2) {
    fade_out(wstruct, bnr);
    snprintf(buf, sizeof(buf), "%s/extro/extro-dead.film", FILES_DIR);
    erg = VG3_film_play(wstruct, NULL, buf, 1);

  } else if (filmnr == 3) {
    fade_out(wstruct, bnr);
    snprintf(buf, sizeof(buf), "%s/extro/extro-fear-%d.film", FILES_DIR, bnr);
    erg = VG3_film_play(wstruct, NULL, buf, 1);
  }

  if (erg != 1) { erg = 0; }

  return erg;
}


/* draw background */
static void
draw_bg(struct vg3_window *wstruct)
{
  if (wstruct == NULL) { return; }
  VG3_draw_clear(wstruct, NULL, VGAG3_COLOR_BLACK);
  VG3_window_update(wstruct, 0, 0);
}


/* output text centered onto the window */
static void
print_text(struct vg3_window *wstruct, const char *text)
{
  struct vg3_rect rect;
  struct vg3_text stxt;
  int w0, h0;

  if (wstruct == NULL || text == NULL || *text == '\0') { return; }

  VG3_window_getsize(wstruct, &w0, &h0);

  draw_bg(wstruct);
  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, "8x8.font+", '\n', 0, text);
  rect.x = rect.y = 0; rect.w = w0; rect.h = h0;
  rect = VG3_draw_text(wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLACK, 1);
  rect.x = w0 / 2 - rect.w / 2; rect.y = h0 / 2 - rect.h / 2;
  VG3_draw_text(wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLACK, 0);
  rect.x--; rect.y--; rect.w += 2; rect.h += 2;
  VG3_draw_rect(wstruct, NULL, &rect, 0, VGAG3_COLOR_YELLOW);
}


/* selection: single-player or network */
static int
sel_single_nw(struct g_main *gmain, char *svname, size_t svsize, int *is_master)
{
  char *seltext[6], **pseltext;
  int erg, hdw;

  if (gmain == NULL || svname == NULL || svsize == 0 || is_master == NULL) { return 0; }

  draw_bg(gmain->wstruct);

  seltext[0] = (char *)VG3_multilang_get(gmain->mlang, "help"),
  seltext[1] = (char *)VG3_multilang_get(gmain->mlang, "sel-single"),
  seltext[2] = (char *)VG3_multilang_get(gmain->mlang, "sel-nw"),
  seltext[3] = (char *)VG3_multilang_get(gmain->mlang, "system-menu"),
  seltext[4] = (char *)VG3_multilang_get(gmain->mlang, "exit"),
  seltext[5] = NULL;
  pseltext = seltext;

redomain:
  hdw = VG3_font_highdouble(gmain->wstruct, 1);
  erg = VG3_input_select(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE,
                         VG3_multilang_get(gmain->mlang, "main-menu"), pseltext, 1);
  VG3_font_highdouble(gmain->wstruct, hdw);

  if (erg < 1) { return 0; }
  if (erg == 5) { return 0; }

  if (erg == 1) {
    if (hilfe(gmain)) { return 0; }
    goto redomain;
  }

  if (erg == 4) {
    if (VG3_sysmenu_exec(gmain->sysm) > 0) { return 0; }
    goto redomain;
  }

  if (erg == 3) {  /* network */
    snprintf(svname, svsize, "?");
    seltext[0] = (char *)VG3_multilang_get(gmain->mlang, "sel-bcnw"),
    seltext[1] = (char *)VG3_multilang_get(gmain->mlang, "sel-svnw"),
    seltext[2] = (char *)VG3_multilang_get(gmain->mlang, "start-sv"),
    seltext[3] = NULL;
    pseltext = seltext;

redonw:
    hdw = VG3_font_highdouble(gmain->wstruct, 1);
    erg = VG3_input_select(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE,
                           VG3_multilang_get(gmain->mlang, "use-network"), pseltext, 1);
    VG3_font_highdouble(gmain->wstruct, hdw);

    if (erg < 1) { return 0; }

    if (erg == 3) {
      *is_master = 1;
      seltext[2] = (char *)VG3_multilang_get(gmain->mlang, "nostart-sv");
      goto redonw;
    }

    if (erg == 2) {  /* input server name */
      hdw = VG3_font_highdouble(gmain->wstruct, 1);
      erg = VG3_input_box(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE,
                          VG3_multilang_get(gmain->mlang, "enter-nwserver"), svname, svsize);
      VG3_font_highdouble(gmain->wstruct, hdw);
      if (erg < 1) { return 0; }
    }
  }

  return 1;
}


/* selection: maze */
static int
sel_maze(struct g_main *gmain, char *mazebuf, size_t mazesize, int clnr)
{
  if (gmain == NULL || mazebuf == NULL || mazesize == 0) { return 0; }

  draw_bg(gmain->wstruct);

  if (gmain->nwptr == NULL || clnr == 1) {
    char *seltext[4], **pseltext, bcom[256];
    int erg, hdw, msize;
    FILE *pfp;
    size_t blen;

    seltext[0] = (char *)VG3_multilang_get(gmain->mlang, "sel-little-maze"),
    seltext[1] = (char *)VG3_multilang_get(gmain->mlang, "sel-medium-maze"),
    seltext[2] = (char *)VG3_multilang_get(gmain->mlang, "sel-big-maze"),
    seltext[3] = NULL;
    pseltext = seltext;

    hdw = VG3_font_highdouble(gmain->wstruct, 1);
    erg = VG3_input_select(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE,
                           VG3_multilang_get(gmain->mlang, "create-maze"), pseltext, 1);
    VG3_font_highdouble(gmain->wstruct, hdw);

    if (erg < 1) {
      if (gmain->nwptr != NULL) { VG3_nw_send_data(gmain->nwptr, "0"); }
      return 0;
    }

    /* retrieve maze-data with msize * msize */

    msize = erg * 10 + 1;  /* 31 is maximum for network data-sending */
    snprintf(bcom, sizeof(bcom), "./mazegen -w %d -h %d", msize, msize);
    if ((pfp = popen(bcom, "r")) == NULL) {
      fprintf(stderr, "popen(%s): %s\n", bcom, strerror(errno));
      if (gmain->nwptr != NULL) { VG3_nw_send_data(gmain->nwptr, "0"); }
      return 0;
    }

    msize = 1;
    for (;;) {
      if ((fgets(mazebuf + msize, (int)mazesize - msize, pfp)) == NULL) { break; }
      blen = strlen(mazebuf + msize);
      if (blen > 1 && mazebuf[msize + blen - 1] == '\n' && mazebuf[msize + blen - 2] == '\r') {
        mazebuf[msize + blen - 2] = '\n';
        mazebuf[msize + blen - 1] = '\0';
        blen--;
      }
      msize += (int)blen;
      if (msize >= (int)mazesize - 1) { break; }
    }
    if (pclose(pfp) == -1) {
      fprintf(stderr, "pclose(%s): %s\n", bcom, strerror(errno));
      if (gmain->nwptr != NULL) { VG3_nw_send_data(gmain->nwptr, "0"); }
      return 0;
    }
    mazebuf[0] = '1';
    if (gmain->nwptr != NULL) { VG3_nw_send_data(gmain->nwptr, mazebuf); }

  /* receive maze-data */
  } else if (gmain->nwptr != NULL) {
    print_text(gmain->wstruct, "Receiving maze-data...");
    VG3_window_update(gmain->wstruct, 0, 0);

    VG3_discard_input(gmain->wstruct);
    for (;;) {
      if (VG3_inputevent_update(gmain->wstruct) > 0) { return 0; }

      /* ALT+Q or ESC: exit */
      if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
        && VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) { return 0; }
      if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) { return 0; }

      /* check for and get maze-data */
      if (VG3_nw_recv_data(gmain->nwptr)) { return 0; }
      if (VG3_nw_fetch_data(gmain->nwptr, mazebuf, mazesize) > 0) {
        if (mazebuf[0] != '1') { return 0; }
        draw_bg(gmain->wstruct);
        VG3_window_update(gmain->wstruct, 0, 0);
        break;
      }

      VG3_window_update(gmain->wstruct, 0, 0);
      VG3_wait_time(50);
    }
    VG3_discard_input(gmain->wstruct);
  }

  return 1;
}


/* selection: clock speed */
static int
sel_clockspeed(struct g_main *gmain, int *clockmax, int clnr)
{
  char bdata[64];

  if (gmain == NULL || clockmax == NULL) { return 0; }

  draw_bg(gmain->wstruct);

  if (gmain->nwptr == NULL || clnr == 1) {
    char *seltext[4], **pseltext;
    int erg, hdw;

    seltext[0] = (char *)VG3_multilang_get(gmain->mlang, "sel-clock-slow"),
    seltext[1] = (char *)VG3_multilang_get(gmain->mlang, "sel-clock-medium"),
    seltext[2] = (char *)VG3_multilang_get(gmain->mlang, "sel-clock-quick"),
    seltext[3] = NULL;
    pseltext = seltext;

    hdw = VG3_font_highdouble(gmain->wstruct, 1);
    erg = VG3_input_select(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE,
                           VG3_multilang_get(gmain->mlang, "clock-speed"), pseltext, 1);
    VG3_font_highdouble(gmain->wstruct, hdw);

    if (erg < 1) {
      if (gmain->nwptr != NULL) { VG3_nw_send_data(gmain->nwptr, "0"); }
      return 0;
    }

    /* set clock speed */
    *clockmax = 0;
    if (erg == 1) {
      *clockmax = gmain->maze.wsize * gmain->maze.hsize * 36 / 49 / 5;
    } else if (erg == 2) {
      *clockmax = gmain->maze.wsize * gmain->maze.hsize * 36 / 49 / 8;
    } else if (erg == 3) {
      *clockmax = gmain->maze.wsize * gmain->maze.hsize * 36 / 49 / 11;
    }

    if (gmain->nwptr != NULL) {
      snprintf(bdata, sizeof(bdata), "1%d", *clockmax);
      VG3_nw_send_data(gmain->nwptr, bdata);
    }

  /* receive clock-speed */
  } else if (gmain->nwptr != NULL) {
    print_text(gmain->wstruct, "Receiving clock-speed...");
    VG3_window_update(gmain->wstruct, 0, 0);

    VG3_discard_input(gmain->wstruct);
    for (;;) {
      if (VG3_inputevent_update(gmain->wstruct) > 0) { return 0; }

      /* ALT+Q or ESC: exit */
      if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
        && VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) { return 0; }
      if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) { return 0; }

      /* check for and get clock-speed */
      if (VG3_nw_recv_data(gmain->nwptr)) { return 0; }
      if (VG3_nw_fetch_data(gmain->nwptr, bdata, sizeof(bdata)) > 0) {
        if (bdata[0] != '1') { return 0; }
        *clockmax = atoi(bdata + 1);
        draw_bg(gmain->wstruct);
        VG3_window_update(gmain->wstruct, 0, 0);
        break;
      }

      VG3_window_update(gmain->wstruct, 0, 0);
      VG3_wait_time(50);
    }
    VG3_discard_input(gmain->wstruct);
  }

  return 1;
}


/* selection for network: player-bitmap */
static struct sel_players
sel_player_nw(struct g_main *gmain, int clnr, int clmax)
{
  struct sel_players selp;
  struct vg3_image *imgload[MAX_PLAYER], *imgsel[MAX_PLAYER], **pimgsel;
  int erg, hdw, cli, firstloop, datada;
  char buf[256];

  selp.exit = 1;
  if (gmain == NULL || gmain->nwptr == NULL || clmax < 1 || clmax > MAX_PLAYER || clnr < 1 || clnr > clmax) { return selp; }
  selp.exit = 0;

  /* imgload[<1 to MAX_PLAYER> - 1] = <player-bitmap> */
  for (erg = 1; erg <= MAX_PLAYER; erg++) {
    snprintf(buf, sizeof(buf), "%s/bmp/player/pl%d.bmp", FILES_DIR, erg);
    imgsel[erg - 1] = imgload[erg - 1] = VG3_image_load(gmain->wstruct, buf, 1);
    if (imgload[erg - 1] == NULL) { goto spnw_exit; }
  }
  pimgsel = imgsel;

  /* selp.plf[<client-number> - 1] = <player-bitmap-number> */
  memset(selp.plf, 0, sizeof(selp.plf));

sp_redo:
  draw_bg(gmain->wstruct);

  hdw = VG3_font_highdouble(gmain->wstruct, 1);
  erg = VG3_input_image_select(gmain->wstruct, VG3_multilang_get(gmain->mlang, "sel-player"), pimgsel, MAX_PLAYER, 0);
  VG3_font_highdouble(gmain->wstruct, hdw);

  if (erg < 1) { goto spnw_exit; }

  /* set my selection */
  selp.plf[clnr - 1] = erg;
  imgsel[selp.plf[clnr - 1] - 1] = NULL;

  /* receive selections of all other players (and send my selection) */
  print_text(gmain->wstruct, "Waiting for other players...");
  VG3_window_update(gmain->wstruct, 0, 0);
  VG3_discard_input(gmain->wstruct);
  for (firstloop = 1;; firstloop = 0) {
    if (VG3_inputevent_update(gmain->wstruct) > 0) { goto spnw_exit; }

    /* ALT+Q or ESC: exit */
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
      && VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) { goto spnw_exit; }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) { goto spnw_exit; }

    /* check for and get other player-selections */
    if (VG3_nw_recv_data(gmain->nwptr)) { goto spnw_exit; }
    datada = 0;
    if (VG3_nw_fetch_data(gmain->nwptr, NULL, 0) > 0) { datada = 1; }  /* data available */

    if (datada) {
      while ((cli = VG3_nw_fetch_data(gmain->nwptr, buf, sizeof(buf))) > 0) {
        if (cli > MAX_PLAYER) { continue; }
        if (buf[0] == '0') {  /* player quit */
          if (selp.plf[cli - 1] > 0) { imgsel[selp.plf[cli - 1] - 1] = imgload[selp.plf[cli - 1] - 1]; }
          selp.plf[cli - 1] = -1;
        } else {  /* set player's selection */
          selp.plf[cli - 1] = atoi(buf + 1);
          imgsel[selp.plf[cli - 1] - 1] = NULL;
        }
      }

      /* check if another player has chosen my selection */
      for (cli = 1; cli <= clmax; cli++) {
        if (cli != clnr && selp.plf[cli - 1] == selp.plf[clnr - 1]) { break; }
      }

      if (cli <= clmax) {  /* yes, go back to selection menu */
        if (!firstloop) {  /* selection collision */
          imgsel[selp.plf[cli - 1] - 1] = imgload[selp.plf[cli - 1] - 1];
          selp.plf[cli - 1] = 0;
          print_text(gmain->wstruct, "Collision of selections, please redo");
        } else {  /* another player has already selected my selection */
          print_text(gmain->wstruct, "Another player has it already selected");
        }
        VG3_window_update(gmain->wstruct, 0, 0);
        selp.plf[clnr - 1] = 0;
        sleep(3);
        goto sp_redo;
      }
    }

    if (firstloop) {  /* inform other players of my selection */
      snprintf(buf, sizeof(buf), "1%d", selp.plf[clnr - 1]);
      VG3_nw_send_data(gmain->nwptr, buf);
      datada = 1;
    }

    if (datada) {
      /* check if at least one other player has not taken a selection */
      for (cli = 1; cli <= clmax; cli++) {
        if (selp.plf[cli - 1] == 0) { break; }
      }
      if (cli > clmax) { break; }  /* all done */
    }

    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(50);
  }
  VG3_discard_input(gmain->wstruct);

  /* unload player-bitmaps */
  for (erg = 1; erg <= MAX_PLAYER; erg++) {
    VG3_image_unload(gmain->wstruct, imgload[erg - 1]);
  }

  draw_bg(gmain->wstruct);
  VG3_window_update(gmain->wstruct, 0, 0);

  return selp;

spnw_exit:
  VG3_nw_send_data(gmain->nwptr, "0");
  selp.exit = 1;
  return selp;
}


/* selection for single-player: player-bitmap */
static struct sel_players
sel_player_single(struct g_main *gmain)
{
  struct sel_players selp;
  struct vg3_image *imgload[MAX_PLAYER], *imgsel[MAX_PLAYER], **pimgsel;
  int erg, hdw;
  char buf[256];

  selp.exit = 1;
  if (gmain == NULL) { return selp; }
  selp.exit = 0;

  /* imgload[<1 to MAX_PLAYER> - 1] = <player-bitmap> */
  for (erg = 1; erg <= MAX_PLAYER; erg++) {
    snprintf(buf, sizeof(buf), "%s/bmp/player/pl%d.bmp", FILES_DIR, erg);
    imgsel[erg - 1] = imgload[erg - 1] = VG3_image_load(gmain->wstruct, buf, 1);
    if (imgload[erg - 1] == NULL) { goto spsg_exit; }
  }
  pimgsel = imgsel;

  /* selp.plf[0] = <player-bitmap-number> */
  memset(selp.plf, 0, sizeof(selp.plf));

  draw_bg(gmain->wstruct);

  hdw = VG3_font_highdouble(gmain->wstruct, 1);
  erg = VG3_input_image_select(gmain->wstruct, VG3_multilang_get(gmain->mlang, "sel-player"), pimgsel, MAX_PLAYER, 0);
  VG3_font_highdouble(gmain->wstruct, hdw);

  if (erg < 1) { goto spsg_exit; }

  /* set my selection */
  selp.plf[0] = erg;

  /* unload player-bitmaps */
  for (erg = 1; erg <= MAX_PLAYER; erg++) {
    VG3_image_unload(gmain->wstruct, imgload[erg - 1]);
  }

  draw_bg(gmain->wstruct);
  VG3_window_update(gmain->wstruct, 0, 0);

  return selp;

spsg_exit:
  selp.exit = 1;
  return selp;
}


int main(int argc, char **argv) {
  struct g_main gmain;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_window *wstruct;
  struct vg3_sysmenu_submenu *subm_keyb, *subm_gc;
  struct vg3_ofunc_object *objp;
  int i1, retw, scale, clnr, jid, clockmax, dowait, is_master, extrofilm, brovdr;
  char buf[256], svname[128], mazebuf[1024];
  time_t zt;

  scale = VGAG3_WINSCALE_FULLSCREEN;

  zt = time(NULL);
  brovdr = 5;

  opterr = opterr ? opterr : 1;
  while ((i1 = getopt(argc, argv, "+s:b:h")) != -1) {
    switch(i1) {
      case 's':  if (strcmp(optarg, "no") == 0) {
                   scale = VGAG3_WINSCALE_NOSCALE;
                 } else if (strcmp(optarg, "best") == 0) {
                   scale = VGAG3_WINSCALE_BESTSCALE;
                 } else if (strcmp(optarg, "max") == 0) {
                   scale = VGAG3_WINSCALE_MAXIMIZE;
                 } else if (strcmp(optarg, "full") == 0) {
                   scale = VGAG3_WINSCALE_FULLSCREEN;
                 }
                 break;
      case 'b':  brovdr = atoi(optarg);
                 break;
      case 'h':
      default:  fprintf(stderr, "Usage: %s [<options>]\n", argv[0]);
                fprintf(stderr, "options:\n");
                fprintf(stderr, " -s <scaling>:    scale window according to <scaling>:\n");
                fprintf(stderr, "                  - no: no scaling\n");
                fprintf(stderr, "                  - best: best scaling\n");
                fprintf(stderr, "                  - max: maximize window\n");
                fprintf(stderr, "                  - full: fullscreen\n");
                fprintf(stderr, " -b <brightness>: 0 = dark, 1 = medium, 2 = bright, missing = according to hour\n");
                exit(1);
    }
  }

  if (brovdr < 0) { brovdr = 0; }
  if (brovdr > 9) { brovdr = 9; }
  printf("Brightness (-b): %d\n", brovdr);

  srand(zt % 1000);

  memset(&gmain, 0, sizeof(gmain));

  /* open window */
  wstruct = VG3_window_new(argv[0], VGAG3_VGAVERSION_HIGH, scale);
  if (wstruct == NULL) { fprintf(stderr, "%s\n", VG3_error()); exit(1); }

  /* play intro */
  gmain.wstruct = wstruct;
  settings_readfromfile(&gmain, 1);
  if (play_intro(wstruct)) { retw = 1; goto endgame; }

startgame:
  memset(&gmain, 0, sizeof(gmain));
  gmain.wstruct = wstruct;
  gmain.brovdr = brovdr;
  extrofilm = 0;
  retw = 1;

  /* get the size of the window */
  VG3_window_getsize(gmain.wstruct, &gmain.winw, &gmain.winh);

  /* create multilanguage */
  gmain.mlang = VG3_multilang_new(NULL, "en");
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/sysmenu");
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/game");

  /* create system-menu */
  gmain.sysm = VG3_sysmenu_new(gmain.wstruct, NULL, VG3_color_brightness(VGAG3_COLOR_GREEN, 50), VGAG3_COLOR_GREEN);
  VG3_sysmenu_simple_exitmenu(gmain.sysm, NULL, VG3_multilang_get(gmain.mlang, "exit-game"));
  VG3_sysmenu_simple_volumemenu(gmain.sysm, NULL,
    VG3_multilang_get(gmain.mlang, "main-vol"),
    VG3_multilang_get(gmain.mlang, "sound-vol"),
    VG3_multilang_get(gmain.mlang, "music-vol"),
    NULL
  );
  VG3_sysmenu_simple_windowmenu(gmain.sysm, NULL, VG3_multilang_get(gmain.mlang, "window-change"));
  subm_keyb = VG3_sysmenu_simple_keyboardmenu(gmain.sysm, NULL);
  subm_gc = NULL;
  jid = 0;
  { int *jidf;
    VG3_gamecontroller_getall(gmain.wstruct, &jidf);
    if (jidf[0] > 0) {
      subm_gc = VG3_sysmenu_simple_gcmenu(gmain.sysm, NULL);
      jid = jidf[1];  /* use first found gamecontroller/joystick instead of keyboard */
    }
    free(jidf);
  }
  /* insert help into system-menu */
  { struct vg3_text skey;
    VGAG3_TEXT_ATTRIBUTES_SET(&skey, "10x17.font+", 0, 0, VG3_multilang_get(gmain.mlang, "help"));
    VG3_sysmenu_insert_callentry(gmain.sysm, NULL, &skey, hilfe, &gmain);
  }

  /* read saved values */
  settings_readfromfile(&gmain, 0);

  /* install keys */
  gmain.skeys = VG3_keys_new(gmain.wstruct, gmain.sysm);
  VG3_keys_menu_insert(gmain.skeys, gmain.sysm_string, gmain.mlang, KEYDEF_GO_FORWARD, "fwd", "go-fwd", subm_keyb, VGAG3_KEY_UCURS, subm_gc, VGAG3_GC_AXIS_LEFTY_UP, "Axis-2:neg");
  VG3_keys_menu_insert(gmain.skeys, gmain.sysm_string, gmain.mlang, KEYDEF_GO_BACKWARD, "bwd", "go-bwd", subm_keyb, VGAG3_KEY_DCURS, subm_gc, VGAG3_GC_AXIS_LEFTY_DOWN, "Axis-2:pos");
  VG3_keys_menu_insert(gmain.skeys, gmain.sysm_string, gmain.mlang, KEYDEF_TURN_RIGHT, "turn-right", "turn-right", subm_keyb, VGAG3_KEY_RCURS, subm_gc, VGAG3_GC_AXIS_RIGHTX_RIGHT, "Axis-1:pos");
  VG3_keys_menu_insert(gmain.skeys, gmain.sysm_string, gmain.mlang, KEYDEF_TURN_LEFT, "turn-left", "turn-left", subm_keyb, VGAG3_KEY_LCURS, subm_gc, VGAG3_GC_AXIS_RIGHTX_LEFT, "Axis-1:neg");
  VG3_keys_menu_insert(gmain.skeys, gmain.sysm_string, gmain.mlang, KEYDEF_TOGGLE_MAP, "map", "toggle-map", subm_keyb, VGAG3_KEY_M, subm_gc, VGAG3_GC_BUTTON_RIGHTSHOULDER, "Button-1");
  VG3_keys_menu_insert(gmain.skeys, gmain.sysm_string, gmain.mlang, KEYDEF_TOGGLE_LAMP, "lamp", "toggle-lamp", subm_keyb, VGAG3_KEY_SPACE, subm_gc, VGAG3_GC_AXIS_TRIGGERRIGHT, "Button-2");
  VG3_keys_menu_insert(gmain.skeys, gmain.sysm_string, gmain.mlang, KEYDEF_USE_ITEM, "item", "use-item", subm_keyb, VGAG3_KEY_ENTER, subm_gc, VGAG3_GC_BUTTON_LEFTSHOULDER, "Button-3");

  /* single-player or network? */
  *svname = '\0';
  is_master = 0;
  if (!sel_single_nw(&gmain, svname, sizeof(svname), &is_master)) { goto endgame; }

  /* using network */
  clnr = 0;
  if (*svname != '\0') {
    char remhost[128], *ipsrv;
    int masterbyte = 1;

    /* master starts evtl. multicast/broadcast server and network server */
    if (is_master) {
      if (strcmp(svname, "?") == 0) { VG3_nw_mbcast_start(); }
      VG3_nw_server(NULL);
    }

    /* retrieve ip of network server via multicast/broadcast */
    if (strcmp(svname, "?") == 0) {
      if (VG3_inputevent_update(gmain.wstruct)) { goto endgame; }
      print_text(gmain.wstruct, "Searching for network-server...");
      VG3_window_update(gmain.wstruct, 0, 0);
      ipsrv = VG3_nw_mbcast_getip(remhost, sizeof(remhost));
      if (ipsrv == NULL) { fprintf(stderr, "%s\n", VG3_error()); goto endgame; }

      /* show server name and ip */
      if (VG3_inputevent_update(gmain.wstruct)) { goto endgame; }
      snprintf(buf, sizeof(buf), "Network-server found:\n%s: %s", remhost, ipsrv);
      print_text(gmain.wstruct, buf);
      VG3_window_update(gmain.wstruct, 0, 0);
      sleep(2);
    } else {
      ipsrv = strdup(svname);
    }

    /* get initialized network struct */
    gmain.nwptr = VG3_nw_open(gmain.wstruct);

    /* add keys installed into the system menu to network server */
    VG3_keys_nw_addkeys(gmain.skeys, gmain.nwptr, jid);

    /* clear window and connect to network server */
    if (VG3_inputevent_update(gmain.wstruct)) { goto endgame; }
    VG3_draw_clear(gmain.wstruct, NULL, VGAG3_COLOR_BLACK);
    VG3_window_update(gmain.wstruct, 0, 0);
    clnr = VG3_nw_connect(gmain.nwptr, ipsrv, NULL, NULL, MAX_PLAYER, &masterbyte);
    if (clnr < 0) { fprintf(stderr, "%s\n", VG3_error()); goto endgame; }
    if (clnr == 0) { goto endgame; }
    if (VG3_inputevent_update(gmain.wstruct)) { goto endgame; }
    free(ipsrv);

    /* master stops multicast/broadcast server */
    if (is_master && strcmp(svname, "?") == 0) { VG3_nw_mbcast_stop(); }
  }

  /* select maze-size and retrieve maze-data (single-player or first connected client) */
  if (!sel_maze(&gmain, mazebuf, sizeof(mazebuf), clnr)) { goto endgame; }

  /* create main-struct for object-functions */
  gmain.ofstruct = ofunc_new();

  /* create maze */
  ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_MAZE));
  if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_MAZE)); goto endgame; }
  if (ofc->f_new(&gmain, 0, mazebuf + 1, (gmain.nwptr != NULL ? VG3_nw_numberofclients(gmain.nwptr) : 1)) == NULL) {
    fprintf(stderr, "%s\n", VG3_error());
    goto endgame;
  }

  /* select clock speed */
  if (!sel_clockspeed(&gmain, &clockmax, clnr)) { goto endgame; }

  /* create players */
  ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_PLAYER));
  if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_PLAYER)); goto endgame; }
  if (gmain.nwptr != NULL) {
    int clmax, cli;
    struct sel_players selp;
    clmax = VG3_nw_numberofclients(gmain.nwptr);
    selp = sel_player_nw(&gmain, clnr, clmax);
    if (selp.exit) { goto endgame; }
    for (cli = 1; cli <= clmax; cli++) {
      if (selp.plf[cli - 1] < 1) { continue; }
      if ((objp = ofc->f_new(&gmain, 0, cli, (cli == clnr), selp.plf[cli - 1], clockmax)) == NULL) {
        fprintf(stderr, "%s\n", VG3_error());
        goto endgame;
      }
    }
  } else {
    struct sel_players selp;
    selp = sel_player_single(&gmain);
    if ((objp = ofc->f_new(&gmain, 0, 0, 1, selp.plf[0], clockmax)) == NULL) { fprintf(stderr, "%s\n", VG3_error()); goto endgame; }
  }

  print_text(gmain.wstruct, "Loading data...");
  VG3_window_update(gmain.wstruct, 0, 0);

  /* activate ghost-object-management */
  if (VG3_ofunc_mgmt_activate(gmain.ofstruct, &gmain, get_oid_name(OID_NAME_GHOST)) != 0) {
    fprintf(stderr, "%s\n", VG3_error());
    goto endgame;
  }

  /* create minotaur */
  ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_MINOTAUR));
  if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_MINOTAUR)); goto endgame; }
  if (ofc->f_new(&gmain, 0) == NULL) { fprintf(stderr, "%s\n", VG3_error()); goto endgame; }

  /* create music-instance */
  ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_MUSIC));
  if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_MUSIC)); goto endgame; }
  if (ofc->f_new(&gmain, 0) == NULL) { fprintf(stderr, "%s\n", VG3_error()); goto endgame; }


  /* +++ game loop +++ */

  gmain.clockdelta = 0;
  VG3_discard_input(gmain.wstruct);
  for (;;) {
    if (gmain.nwptr != NULL) {
      if (VG3_nw_update(gmain.nwptr, &dowait) > 0) { break; }
    } else {
      if (VG3_inputevent_update(gmain.wstruct) > 0) { break; }
      dowait = 1;
    }

    /* ALT+Q: exit */
    if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
      && VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) { break; }

    /* ESC: system-menu */
    if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) {
      if (VG3_sysmenu_exec(gmain.sysm) > 0) { break; }
    }

    /* P: pause game */
    if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_P, VGAG3_IS_NEW_PRESSED)) {
      if (gmain.nwptr != NULL) {
        VG3_nw_pause(gmain.nwptr);
      } else {
        VG3_pause(gmain.wstruct, NULL, VGAG3_KEY_NOKEY);
      }
    }

    VG3_draw_clear(gmain.wstruct, NULL, VGAG3_COLOR_BLACK);

    VG3_ofunc_mgmt_run(gmain.ofstruct, &gmain);

    VG3_ofunc_objlist_call_run(gmain.ofstruct, &gmain);

    VG3_ofunc_objlist_call_draw(gmain.ofstruct, &gmain);

    if (dowait) {
      VG3_window_update(gmain.wstruct, 0, 0);
      VG3_wait_time(80);
    }

    if (gmain.ply.gone) { extrofilm = 1; retw = 0; break; }
    if (gmain.ply.dead) { extrofilm = 1 + gmain.ply.dead; retw = 0; break; }

    if (++gmain.clockdelta == clockmax) {
      if (++gmain.clock < 0) { gmain.clock = 100; }
      gmain.clockdelta = 0;
    }
  }
  VG3_discard_input(gmain.wstruct);

  /* close network */
  if (gmain.nwptr != NULL) { VG3_nw_close(gmain.nwptr); gmain.nwptr = NULL; }

  /* reset and play extro */
  { struct vg3_image_attributes wattr;
    VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&wattr);
    VG3_window_attributes(gmain.wstruct, &wattr, NULL, VGAG3_COLORDEF_DEFAULT, NULL);
    VG3_audio_stop(gmain.wstruct, VG3_audio_group(VGAG3_AUDIO_VOLUME_ALL), 0);
    if (extrofilm > 0) { play_extro(gmain.wstruct, extrofilm, gmain.ply.bnr); }
    VG3_draw_clear(gmain.wstruct, NULL, VGAG3_COLOR_BLACK);
    VG3_window_update(wstruct, 0, 0);
  }

  /* save values to file */
  { char *sysm_string_new = VG3_sysmenu_savestring_insert(gmain.sysm, gmain.sysm_string);
    if (gmain.sysm_string != NULL) { free(gmain.sysm_string); }
    gmain.sysm_string = sysm_string_new;
    settings_savetofile(&gmain);
  }
  if (gmain.sysm_string != NULL) { free(gmain.sysm_string); }

  /* clean up */
  VG3_ofunc_mgmt_deactivate(gmain.ofstruct, &gmain, NULL);
  VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, NULL);
  VG3_ofunc_free(gmain.ofstruct);
  VG3_image_unload(gmain.wstruct, NULL);
  VG3_keys_free(gmain.skeys);
  VG3_sysmenu_free(gmain.sysm);
  VG3_multilang_free(gmain.mlang);

endgame:
  if (gmain.nwptr != NULL) { VG3_nw_close(gmain.nwptr); }
  if (retw == 0) { goto startgame; }

  /* close window and exit */
  VG3_window_free(gmain.wstruct);
  exit(0);
}


/* help */
static int
hilfe(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;
  const char *locale;
  int retw, ypos, ysize, fcolor, bcolor;
  struct vg3_image *imgp;
  struct vg3_text stxt;
  char text[8192], path[256], *fontname, *ptr;
  size_t slen;
  FILE *ffp;

  if (gmain == NULL) { return 0; }

  fontname = NULL;
  fcolor = VGAG3_COLOR_YELLOW;
  bcolor = VG3_color_brightness(VGAG3_COLOR_TURQUOISE, 40);

  locale = VG3_multilang_locale(gmain->mlang, 0);
  if (*locale == '\0') {
    locale = VG3_multilang_locale(gmain->mlang, 1);
  }

  snprintf(path, sizeof(path), "%s/help/help-text-%s.txt", FILES_DIR, locale);

  if ((ffp = fopen(path, "r")) == NULL) { fprintf(stderr, "%s not found\n", path); return 0; }
  slen = fread(text, 1, sizeof(text) - 1, ffp);
  fclose(ffp);
  if (slen == 0) { return 0; }
  text[slen] = '\0';

  if ((ptr = strrchr(path, '/')) != NULL) { *ptr = '\0'; } else { strcpy(path, "."); }

  VG3_utf8_to_iso(text, text, slen + 1);

  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, fontname, '\n', 0, text);
  imgp = VG3_text_with_images(gmain->wstruct, &stxt, path, fcolor, bcolor);
  if (imgp == NULL) { fprintf(stderr, "%s\n", VG3_error()); return 0; }
  VG3_image_getsize(gmain->wstruct, imgp, NULL, NULL, &ysize);

  retw = 0;
  ypos = (ysize - gmain->winh) / 2;
  for (;;) {
    if (VG3_inputevent_update(gmain->wstruct)) { retw = 1; break; }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)) { break; }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) { break; }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_UCURS, VGAG3_IS_PRESSED)) {
      if (ypos < (ysize - gmain->winh) / 2) {
        ypos += 8;
        if (ypos > (ysize - gmain->winh) / 2) { ypos = (ysize - gmain->winh) / 2; }
      }
    }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_DCURS, VGAG3_IS_PRESSED)) {
      if (ypos > -(ysize - gmain->winh) / 2) {
        ypos -= 8;
        if (ypos < -(ysize - gmain->winh) / 2) { ypos = -(ysize - gmain->winh) / 2; }
      }
    }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_PGUP, VGAG3_IS_NEW_PRESSED)
        || VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_BACKSPACE, VGAG3_IS_NEW_PRESSED)) {
      if (ypos < (ysize - gmain->winh) / 2) {
        ypos += (gmain->winh - 8);
        if (ypos > (ysize - gmain->winh) / 2) { ypos = (ysize - gmain->winh) / 2; }
      }
    }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_PGDOWN, VGAG3_IS_NEW_PRESSED)
        || VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) {
      if (ypos > -(ysize - gmain->winh) / 2) {
        ypos -= (gmain->winh - 8);
        if (ypos < -(ysize - gmain->winh) / 2) { ypos = -(ysize - gmain->winh) / 2; }
      }
    }

    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);

    VG3_image_copy(gmain->wstruct, NULL, imgp, gmain->winw / 2, ypos + gmain->winh / 2, NULL, 0);

    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(50);
  }

  VG3_image_unload(gmain->wstruct, imgp);
  draw_bg(gmain->wstruct);
  return retw;
}
