#include "breadthfirstsearch.h"
#include "depthfirstsearch.h"
#include "kruskal.h"
#include "looperasedrandomwalk.h"
#include "prim.h"
#include "rectangularmaze.h"
#include <cstring>
#include <iostream>
#include <map>
#include <string>

void usage(std::ostream &out) {
  out << "Usage: mazegen [--help] [-a <algorithm type>]"
      << std::endl;
  out << "               [-w <width> -h <height>]" << std::endl;

  out << std::endl;
  out << "Optional arguments" << std::endl;
  out << "  --help  "
      << "Show this message and exit" << std::endl;
  out << "  -a      "
      << "Algorithm type" << std::endl;
  out << "          "
      << "0: Kruskal's algorithm (default)" << std::endl;
  out << "          "
      << "1: Depth-first search" << std::endl;
  out << "          "
      << "2: Breadth-first search" << std::endl;
  out << "          "
      << "3: Loop-erased random walk" << std::endl;
  out << "          "
      << "4: Prim's algorithm" << std::endl;
  out << "  -w,-h   "
      << "Width and height (rectangular maze, default: 20)" << std::endl;
}

int main(int argc, char *argv[]) {
  std::map<std::string, int> optionmap{{"-a", 0}, {"-w", 20}, {"-h", 20}, {"--help", 0}};

  for (int i = 1; i < argc; i++) {
    if (optionmap.find(argv[i]) == optionmap.end()) {
      std::cerr << "Unknown argument " << argv[i] << "\n";
      usage(std::cerr);
      return 1;
    }

    if (strcmp("--help", argv[i]) == 0) {
      usage(std::cout);
      return 0;
    }

    if (i + 1 == argc) {
      std::cerr << "Missing option for argument " << argv[i] << std::endl;
      usage(std::cerr);
      return 1;
    }
    int x;
    try {
      x = std::stoi(argv[i + 1]);
    } catch (...) {
      std::cerr << "Invalid argument " << argv[i + 1] << " for option "
                << argv[i] << "\n";
      usage(std::cerr);
      return 1;
    }
    optionmap[argv[i++]] = x;
  }

  Maze *maze;
  SpanningtreeAlgorithm *algorithm;

  if (optionmap["-w"] < 1 or optionmap["-h"] < 1) {
    std::cerr << "Invalide size " << optionmap["-w"] << "x"
              << optionmap["-h"] << " for rectangular maze\n";
    usage(std::cerr);
    return 1;
  }
  maze = new RectangularMaze(optionmap["-w"], optionmap["-h"]);

  switch (optionmap["-a"]) {
    case 0:
      algorithm = new Kruskal;
      break;

    case 1:
      algorithm = new DepthFirstSearch;
      break;

    case 2:
      algorithm = new BreadthFirstSearch;
      break;

    case 3:
      algorithm = new LoopErasedRandomWalk;
      break;

    case 4:
      algorithm = new Prim;
      break;

    default:
      std::cerr << "Unknown algorithm type " << optionmap["-a"];
      usage(std::cerr);
      return 1;
  }

  maze->InitialiseGraph();
  maze->GenerateMaze(algorithm);
  maze->PrintMazeSVG();
  return 0;
}
