#include "maze.h"
#include <algorithm>
#include <fstream>
#include <iostream>

extern "C" {
  extern void makemaze(const char *);
}

Maze::Maze(int vertices, int startvertex, int endvertex)
    : vertices_(vertices), startvertex_(startvertex), endvertex_(endvertex) {}

void Maze::InitialiseGraph() {
  adjacencylist_.clear();
  adjacencylist_.resize(vertices_);
}

void Maze::GenerateMaze(SpanningtreeAlgorithm* algorithm) {
  auto spanningtree = algorithm->SpanningTree(vertices_, adjacencylist_);
  RemoveBorders(spanningtree);
}

void Maze::RemoveBorders(const std::vector<std::pair<int, int>>& edges) {
  for (const auto& edge : edges) {
    int u = edge.first, v = edge.second;
    for (int i = 0; i < (int)adjacencylist_[u].size(); ++i) {
      if (adjacencylist_[u][i].first == v) {
        adjacencylist_[u].erase(adjacencylist_[u].begin() + i);
        break;
      }
    }
    for (int i = 0; i < (int)adjacencylist_[v].size(); ++i) {
      if (adjacencylist_[v][i].first == u) {
        adjacencylist_[v].erase(adjacencylist_[v].begin() + i);
        break;
      }
    }
  }
}

void Maze::PrintMazeSVG(void) const {
  double xmin, ymin, xmax, ymax;
  std::tie(xmin, ymin, xmax, ymax) = GetCoordinateBounds();
  int xresolution = (xmax - xmin + 2) * 30,
      yresolution = (ymax - ymin + 2) * 30;
  std::string mazebuf;

  mazebuf = "<svg width=\"";
  mazebuf += std::to_string(xresolution);
  mazebuf += "\" height=\"";
  mazebuf += std::to_string(yresolution);
  mazebuf += "\" xmlns=\"http://www.w3.org/2000/svg\">";
  mazebuf += "\n";
  mazebuf += "<g transform=\"translate(";
  mazebuf += std::to_string(((1 - xmin) * 30));
  mazebuf += ",";
  mazebuf += std::to_string((yresolution - (1 - ymin) * 30));
  mazebuf += ") scale(1,-1)\">";
  mazebuf += "\n";
  mazebuf += "<rect x=\"";
  mazebuf += std::to_string(((xmin - 1) * 30));
  mazebuf += "\" y=\"";
  mazebuf += std::to_string(((ymin - 1) * 30));
  mazebuf += "\" width=\"";
  mazebuf += std::to_string(xresolution);
  mazebuf += "\" height=\"";
  mazebuf += std::to_string(yresolution);
  mazebuf += "\" fill=\"white\"/>";
  mazebuf += "\n";

  for (int i = 0; i < vertices_; ++i) {
    for (const auto& edge : adjacencylist_[i]) {
      if (edge.first < i) {
        mazebuf += edge.second->SVGPrintString();
        mazebuf += "\n";
      }
    }
  }
  mazebuf += "</g>\n";
  mazebuf += "</svg>\n";
  makemaze(mazebuf.c_str());
}
