#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-ghost.h"

void getofc_ghost(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);

static int follow_player(struct g_main *, struct g_obj_ghost *);


/* +++ get-function +++ */

void
getofc_ghost(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_GHOST));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_data = f_data;
}


/* +++ object-functions +++ */

/* new-function
 * no variable parameter
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_ghost *gobj;
  char buf[256];
  int i1;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->pos.x = ZUFALL(0, gmain->maze.wsize - 1);
  gobj->pos.y = ZUFALL(0, gmain->maze.hsize - 1);
  gobj->pos.dir = WAND_NORD;
  gobj->pos.prz = 0;
  gobj->wway.anz = 0;
  gobj->wway.pos = -1;
  gobj->follow_id = 0;
  gobj->pause = 100;  /* should be length of appearing sprite */
  gobj->horrorloop = 0;
  gobj->horrorzoom = 0;
  if (ZUFALL(0, 1) == 0) { gobj->walknr = GHOST_SPRITE_WALK1; } else { gobj->walknr = GHOST_SPRITE_WALK2; }

  /* load ghost-sprites */
  snprintf(buf, sizeof(buf), FILES_DIR "/bmp/ghost/g1/geist-appear.sprite");
  gobj->sprtapp = VG3_sprite_load(gmain->wstruct, buf);
  if (gobj->sprtapp == NULL) { free(gobj); return NULL; }
  for (i1 = 0; i1 < GHOST_SPRITE_MAX; i1++) {
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/ghost/g%d/geist.sprite", i1 + 1);
    gobj->sprt[i1] = VG3_sprite_load(gmain->wstruct, buf);
    if (gobj->sprt[i1] == NULL) { free(gobj); return NULL; }
  }
  gobj->sprt_mom = gobj->sprtapp;

  /* load sound */
  gobj->sound.follow = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/ghost-follow.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  gobj->sound.frighten = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/ghost-frighten.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);

  /* create and fill ghost-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_GHOST));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert ghost-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_ghost *gobj;
  int i1;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_ghost *)objp->ostruct;

  /* remove ghost-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free ghost-sprites */
  if (gobj->sprtapp != NULL) { VG3_sprite_free(gobj->sprtapp); }
  for (i1 = 0; i1 < GHOST_SPRITE_MAX; i1++) {
    if (gobj->sprt[i1] != NULL) { VG3_sprite_free(gobj->sprt[i1]); }
  }

  if (gobj->sound.follow > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.follow); }
  if (gobj->sound.frighten > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.frighten); }

  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_ghost *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_ghost *)objp->ostruct;

  if (gobj->sprt_mom == gobj->sprtapp) {
    if (gobj->pause == 0) {
      gobj->sprt_mom = gobj->sprt[GHOST_SPRITE_REST];
      VG3_sprite_rewind(gobj->sprt_mom);
    } else {
      gobj->pause--;
      return;
    }
  }

  if (gobj->horrorloop > 0) {
    if (--gobj->horrorloop == 0) {
      f_free(gmain, objp);
    }
    return;
  }

  if (follow_player(gmain, gobj)) {
    gobj->pause = 0;
    gobj->sprt_mom = gobj->sprt[gobj->walknr];
    VG3_sprite_rewind(gobj->sprt_mom);
    if (gobj->follow_id == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.follow, 0, 0); }
  }

  if (gobj->pause > 0) {
    if (--gobj->pause > 0) { return; }
    gobj->sprt_mom = gobj->sprt[gobj->walknr];
    VG3_sprite_rewind(gobj->sprt_mom);
  }

  if (gobj->wway.anz == 0) {
    int xbis, ybis, maxdo = 0;
    while (gobj->wway.anz < 2) {
      xbis = ZUFALL(0, gmain->maze.wsize - 1);
      ybis = ZUFALL(0, gmain->maze.hsize - 1);
      create_way(gmain, &gobj->wway, WALK_MAX, gobj->pos.x, gobj->pos.y, xbis, ybis);
      gobj->pos.prz = 0;
      gobj->follow_id = 0;
      if (gobj->wway.way[1].x > gobj->wway.way[0].x) {
        gobj->pos.dir = WAND_OST;
      } else if (gobj->wway.way[1].x < gobj->wway.way[0].x) {
        gobj->pos.dir = WAND_WEST;
      } else if (gobj->wway.way[1].y > gobj->wway.way[0].y) {
        gobj->pos.dir = WAND_SUED;
      } else {
        gobj->pos.dir = WAND_NORD;
      }
      if (++maxdo == 7) { return; }
    }
    gobj->sprt_mom = gobj->sprt[gobj->walknr];
    VG3_sprite_rewind(gobj->sprt_mom);
  }

  gobj->pos.prz += 4; if (gobj->follow_id) { gobj->pos.prz += 1; }
  if (gobj->pos.prz >= 100) {
    gobj->pos.prz = 0;
    if (++gobj->wway.pos >= gobj->wway.anz) { gobj->wway.anz = 0; return; }
    gobj->pos.x = gobj->wway.way[gobj->wway.pos].x;
    gobj->pos.y = gobj->wway.way[gobj->wway.pos].y;
    if (gobj->wway.pos + 1 < gobj->wway.anz) {
      if (gobj->wway.way[gobj->wway.pos + 1].x > gobj->wway.way[gobj->wway.pos].x) {
        gobj->pos.dir = WAND_OST;
      } else if (gobj->wway.way[gobj->wway.pos + 1].x < gobj->wway.way[gobj->wway.pos].x) {
        gobj->pos.dir = WAND_WEST;
      } else if (gobj->wway.way[gobj->wway.pos + 1].y > gobj->wway.way[gobj->wway.pos].y) {
        gobj->pos.dir = WAND_SUED;
      } else {
        gobj->pos.dir = WAND_NORD;
      }
    }
    if (gobj->wway.pos == gobj->wway.anz - 1) {
      gobj->wway.anz = gobj->wway.pos = 0;
      gobj->follow_id = 0;
      gobj->pause = 50 + ZUFALL(0, 150);
      gobj->sprt_mom = gobj->sprt[GHOST_SPRITE_REST];
      VG3_sprite_rewind(gobj->sprt_mom);
      return;
    }
  }
}


/* data-exchanging-function */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *vdata)
{
  struct g_main *gmain = vmain;
  struct g_obj_ghost *gobj;
  struct obj_pos *opos = (struct obj_pos *)vdata;
  int flag;

  if (gmain == NULL || objp == NULL || opos == NULL) { return 0; }

  (void)gmain;
  gobj = (struct g_obj_ghost *)objp->ostruct;

  flag = opos->dir;

  if (flag == 2) {  /* cancel following player */
    if (gobj->follow_id == opos->u.gh.follow_id) { gobj->follow_id = 0; }
  }

  memset(opos, 0, sizeof(*opos));
  opos->x = gobj->pos.x;
  opos->y = gobj->pos.y;
  opos->prz = gobj->pos.prz;
  opos->dir = gobj->pos.dir;
  opos->img = NULL;
  opos->u_pg = 2;
  opos->u.gh.wway = &gobj->wway;
  opos->u.gh.follow_id = gobj->follow_id;

  if (flag == 1) {
    if (!VG3_sprite_get(gobj->sprt_mom, &opos->img, &opos->attr, NULL)) {
      VG3_sprite_rewind(gobj->sprt_mom);
      VG3_sprite_get(gobj->sprt_mom, &opos->img, &opos->attr, NULL);
    }
    if (gobj->horrorzoom) {
     opos->attr.zoom += (HORRORLOOP_MAX - gobj->horrorloop) * 8;
    }
  }

  return 1;
}


/* check whether to follow player, return 1 = following or 0 = not following */
static int
follow_player(struct g_main *gmain, struct g_obj_ghost *gobj_ghost)
{
  const struct vg3_ofunc_objfunc *ofc;
  struct walking_way wway1, wway0;

  if (gmain == NULL || gobj_ghost == NULL || gobj_ghost->follow_id) { return 0; }

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
  if (ofc != NULL && ofc->f_data != NULL) {
    struct vg3_ofunc_objsnap *osnap;
    struct vg3_ofunc_object *objp_player;
    struct obj_pos opos;
    int walk_go;
    unsigned int player_instanceid;

    wway0.anz = 0;
    player_instanceid = 0;

    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
    while ((objp_player = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      opos.dir = 0;
      if (ofc->f_data(gmain, objp_player, &opos) && !opos.u.ply.garlic) {
        walk_go = red_distance(gmain, objp_player->instanceid);
        create_way(gmain, &wway1, walk_go, gobj_ghost->pos.x, gobj_ghost->pos.y, opos.x, opos.y);
        if (wway1.anz >= 2 && wway1.anz < walk_go && (wway0.anz == 0 || wway1.anz < wway0.anz)) {
          wway0 = wway1;
          player_instanceid = objp_player->instanceid;
        }
      }
    }
    VG3_ofunc_objlist_freelist(osnap);

    if (wway0.anz > 0) {
      gobj_ghost->wway = wway0;
      gobj_ghost->pos.prz = 0;
      gobj_ghost->follow_id = player_instanceid;
      if (gobj_ghost->wway.way[1].x > gobj_ghost->wway.way[0].x) {
        gobj_ghost->pos.dir = WAND_OST;
      } else if (gobj_ghost->wway.way[1].x < gobj_ghost->wway.way[0].x) {
        gobj_ghost->pos.dir = WAND_WEST;
      } else if (gobj_ghost->wway.way[1].y > gobj_ghost->wway.way[0].y) {
        gobj_ghost->pos.dir = WAND_SUED;
      } else {
        gobj_ghost->pos.dir = WAND_NORD;
      }
      return 1;
    }
  }
  return 0;
}
