#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <math.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-maze.h"

void getofc_maze(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);

struct box_obj_pos {
  int distance;
  struct obj_pos opos;
};

static int sort_oposp(const void *, const void *);
static void draw_ghost(struct g_main *, struct obj_pos *, int, int);
static int load_maze(struct g_main *, const char *, int);
static int load_images(struct vg3_window *, struct g_obj_maze *);


/* +++ get-function +++ */

void
getofc_maze(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_MAZE));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_draw = f_draw;
}


/* +++ object-functions +++ */

/* new-function
 * variable parameter:
 *  - const char *mazebuf = maze descripting
 *  - int planz = number of players
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_maze *gobj;
  const char *mazebuf;
  int planz;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments: mazebuf:const_char_* */
  va_start(ap, iparent);
  mazebuf = va_arg(ap, const char *);
  planz = va_arg(ap, int);
  va_end(ap);
  if (mazebuf == NULL || *mazebuf == '\0') { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }
  if (planz < 1) { planz = 1; }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  if (!load_maze(gmain, mazebuf, planz)) { return NULL; }
  if (!load_images(gmain->wstruct, gobj)) { return NULL; }
  gobj->w_col = MAZE_WCOL_NORMAL;

  /* create and fill maze-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_MAZE));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert maze-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_maze *gobj;
  int i1;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_maze *)objp->ostruct;

  /* remove maze-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free maze-instance */
  if (gmain->maze.map != NULL) {
    for (i1 = 0; i1 < gmain->maze.hsize; i1++) {
      if (gmain->maze.map[i1] != NULL) { free(gmain->maze.map[i1]); }
    }
    if (gmain->maze.map != NULL) { free(gmain->maze.map); }
  }
  for (i1 = 0; i1 < MAZEBLICK_MAX; i1++) {
    if (gobj->mazeimg.s[i1].wand_links.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].wand_links.img); }
    if (gobj->mazeimg.s[i1].wand_links.imgexit != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].wand_links.imgexit); }
    if (gobj->mazeimg.s[i1].wand_links.imgportal != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].wand_links.imgportal); }
    if (gobj->mazeimg.s[i1].wand_rechts.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].wand_rechts.img); }
    if (gobj->mazeimg.s[i1].wand_rechts.imgexit != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].wand_rechts.imgexit); }
    if (gobj->mazeimg.s[i1].wand_rechts.imgportal != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].wand_rechts.imgportal); }
    if (gobj->mazeimg.s[i1].saeule_links.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].saeule_links.img); }
    if (gobj->mazeimg.s[i1].saeule_rechts.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].saeule_rechts.img); }
    if (gobj->mazeimg.s[i1].halbwand_links.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].halbwand_links.img); }
    if (gobj->mazeimg.s[i1].halbwand_links.noimg != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].halbwand_links.noimg); }
    if (gobj->mazeimg.s[i1].halbwand_links.imgexit != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].halbwand_links.imgexit); }
    if (gobj->mazeimg.s[i1].halbwand_links.imgportal != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].halbwand_links.imgportal); }
    if (gobj->mazeimg.s[i1].halbwand_rechts.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].halbwand_rechts.img); }
    if (gobj->mazeimg.s[i1].halbwand_rechts.noimg != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].halbwand_rechts.noimg); }
    if (gobj->mazeimg.s[i1].halbwand_rechts.imgexit != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].halbwand_rechts.imgexit); }
    if (gobj->mazeimg.s[i1].halbwand_rechts.imgportal != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].halbwand_rechts.imgportal); }
    if (gobj->mazeimg.s[i1].vorne.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].vorne.img); }
    if (gobj->mazeimg.s[i1].vorne.imgexit != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].vorne.imgexit); }
    if (gobj->mazeimg.s[i1].vorne.imgportal != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].vorne.imgportal); }
    if (gobj->mazeimg.s[i1].torch.left.img_torch != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].torch.left.img_torch); }
    if (gobj->mazeimg.s[i1].torch.right.img_torch != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].torch.right.img_torch); }
    if (gobj->mazeimg.s[i1].torch.vorne.img_torch != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].torch.vorne.img_torch); }
    if (gobj->mazeimg.s[i1].torch.left.sprt_flame != NULL) { VG3_sprite_free(gobj->mazeimg.s[i1].torch.left.sprt_flame); }
    if (gobj->mazeimg.s[i1].torch.right.sprt_flame != NULL) { VG3_sprite_free(gobj->mazeimg.s[i1].torch.right.sprt_flame); }
    if (gobj->mazeimg.s[i1].torch.vorne.sprt_flame != NULL) { VG3_sprite_free(gobj->mazeimg.s[i1].torch.vorne.sprt_flame); }
    if (gobj->mazeimg.s[i1].item.garlic.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].item.garlic.img); }
    if (gobj->mazeimg.s[i1].item.valerian.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].item.valerian.img); }
    if (gobj->mazeimg.s[i1].item.jewel.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.s[i1].item.jewel.img); }
  }
  if (gobj->mazeimg.dreh != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.dreh); }
  if (gobj->mazeimg.bg != NULL) { VG3_image_unload(gmain->wstruct, gobj->mazeimg.bg); }
  if (gobj->img_jewel_klein != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_jewel_klein); }
  free(gobj);
  free(objp);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_maze *gobj;
  struct vg3_ofunc_object *objq;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_maze *)objp->ostruct;

  /* draw maze-instance */
  { const int mazeblick_extend = 10;
    int mazeblick;
    struct vg3_image *imgp;
    int ypos, xpos, imgw, imgh, yexit_links, xexit_links, yexit_rechts, xexit_rechts, wand_links, wand_rechts;

    /* background */
    VG3_image_copy(gmain->wstruct, NULL, gobj->mazeimg.bg, gmain->winw / 2, gmain->winh / 2, NULL, 0);

    if (gmain->ply.turn == 0) {
      const struct vg3_ofunc_objfunc *ofc;
      struct obj_pos opos;
      struct box_obj_pos *boposp_ghost = NULL;
      int opospmax_ghost = 0;
      int oidx, icol, distance;
      struct vg3_image_attributes gesattr, iattr, zattr;
      struct vg3_rect rect;
      struct vg3_text stxt;
      char buf[64];

      /* find next front wall */
      for (mazeblick = 0; mazeblick < mazeblick_extend; mazeblick++) {
        if (gmain->ply.dir == WAND_NORD) {
          ypos = gmain->ply.y - mazeblick; xpos = gmain->ply.x;
          if (ypos == 0) { break; }
        } else if (gmain->ply.dir == WAND_OST) {
          ypos = gmain->ply.y; xpos = gmain->ply.x + mazeblick;
          if (xpos == gmain->maze.wsize) { break; }
        } else if (gmain->ply.dir == WAND_SUED) {
          ypos = gmain->ply.y + mazeblick; xpos = gmain->ply.x;
          if (ypos == gmain->maze.hsize) { break; }
        } else if (gmain->ply.dir == WAND_WEST) {
          ypos = gmain->ply.y; xpos = gmain->ply.x - mazeblick;
          if (xpos == 0) { break; }
        } else { break; }
        if ((gmain->maze.map[ypos][xpos] & gmain->ply.dir)) { break; }
      }

      /* get relevant remote players into boposp_ghost[] */
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
      if (ofc != NULL && ofc->f_data != NULL) {
        struct vg3_ofunc_objsnap *osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
        while ((objq = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
          if (objq->instanceid == gmain->ply.instanceid) { continue; }
          opos.dir = 1;
          if (ofc->f_data(gmain, objq, &opos)) {
            distance = 0;
            if (gmain->ply.dir == WAND_NORD) {
              if (opos.y > gmain->ply.y || opos.y < gmain->ply.y - mazeblick || opos.x != gmain->ply.x) {
                continue;
              } else {
                distance = gmain->ply.y - opos.y;
              }
            }
            if (gmain->ply.dir == WAND_OST) {
              if (opos.x < gmain->ply.x || opos.x > gmain->ply.x + mazeblick || opos.y != gmain->ply.y) {
                continue;
              } else {
                distance = opos.x - gmain->ply.x;
              }
            }
            if (gmain->ply.dir == WAND_SUED) {
              if (opos.y < gmain->ply.y || opos.y > gmain->ply.y + mazeblick || opos.x != gmain->ply.x) {
                continue;
              } else {
                distance = opos.y - gmain->ply.y;
              }
            }
            if (gmain->ply.dir == WAND_WEST) {
              if (opos.x > gmain->ply.x || opos.x < gmain->ply.x - mazeblick || opos.y != gmain->ply.y) {
                continue;
              } else {
                distance = gmain->ply.x - opos.x;
              }
            }
            opospmax_ghost++;
            if (boposp_ghost == NULL) {
              boposp_ghost = malloc(sizeof(*boposp_ghost));
            } else {
              boposp_ghost = realloc(boposp_ghost, sizeof(*boposp_ghost) * opospmax_ghost);
            }
            if (boposp_ghost == NULL) { fprintf(stderr, "malloc/calloc: %s\n", strerror(errno)); opospmax_ghost = 0; break; }
            boposp_ghost[opospmax_ghost - 1].opos = opos;
            boposp_ghost[opospmax_ghost - 1].distance = distance;
          }
        }
        VG3_ofunc_objlist_freelist(osnap);
      }

      /* add relevant ghosts into boposp_ghost[] */
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
      if (ofc != NULL && ofc->f_data != NULL) {
        struct vg3_ofunc_objsnap *osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
        while ((objq = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
          opos.dir = 1;
          if (ofc->f_data(gmain, objq, &opos)) {
            distance = 0;
            if (gmain->ply.dir == WAND_NORD) {
              if (opos.y > gmain->ply.y || opos.y < gmain->ply.y - mazeblick
                  || opos.x < gmain->ply.x - 1 || opos.x > gmain->ply.x + 1) {
                continue;
              } else {
                distance = gmain->ply.y - opos.y;
              }
            }
            if (gmain->ply.dir == WAND_OST) {
              if (opos.x < gmain->ply.x || opos.x > gmain->ply.x + mazeblick
                  || opos.y < gmain->ply.y - 1 || opos.y > gmain->ply.y + 1) {
                continue;
              } else {
                distance = opos.x - gmain->ply.x;
              }
            }
            if (gmain->ply.dir == WAND_SUED) {
              if (opos.y < gmain->ply.y || opos.y > gmain->ply.y + mazeblick
                  || opos.x < gmain->ply.x - 1 || opos.x > gmain->ply.x + 1) {
                continue;
              } else {
                distance = opos.y - gmain->ply.y;
              }
            }
            if (gmain->ply.dir == WAND_WEST) {
              if (opos.x > gmain->ply.x || opos.x < gmain->ply.x - mazeblick
                  || opos.y < gmain->ply.y - 1 || opos.y > gmain->ply.y + 1) {
                continue;
              } else {
                distance = gmain->ply.x - opos.x;
              }
            }
            opospmax_ghost++;
            if (boposp_ghost == NULL) {
              boposp_ghost = malloc(sizeof(*boposp_ghost));
            } else {
              boposp_ghost = realloc(boposp_ghost, sizeof(*boposp_ghost) * opospmax_ghost);
            }
            if (boposp_ghost == NULL) { fprintf(stderr, "malloc/calloc: %s\n", strerror(errno)); opospmax_ghost = 0; break; }
            boposp_ghost[opospmax_ghost - 1].opos = opos;
            boposp_ghost[opospmax_ghost - 1].distance = distance;
          }
        }
        VG3_ofunc_objlist_freelist(osnap);
      }

      /* add minotaur into boposp_ghost[] if relevant */
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_MINOTAUR));
      if (ofc != NULL && ofc->f_data != NULL) {
        if (VG3_ofunc_objlist_find_obj(gmain->ofstruct, get_oid_name(OID_NAME_MINOTAUR), &objq) > 0) {
          opos.dir = 1;
          if (ofc->f_data(gmain, objq, &opos)) {
            distance = 0;
            if (gmain->ply.dir == WAND_NORD) {
              if (opos.y > gmain->ply.y || opos.y < gmain->ply.y - mazeblick
                  || opos.x < gmain->ply.x - 1 || opos.x > gmain->ply.x + 1) {
                opos.img = NULL;
              } else {
                distance = gmain->ply.y - opos.y;
              }
            }
            if (gmain->ply.dir == WAND_OST) {
              if (opos.x < gmain->ply.x || opos.x > gmain->ply.x + mazeblick
                  || opos.y < gmain->ply.y - 1 || opos.y > gmain->ply.y + 1) {
                opos.img = NULL;
              } else {
                distance = opos.x - gmain->ply.x;
              }
            }
            if (gmain->ply.dir == WAND_SUED) {
              if (opos.y < gmain->ply.y || opos.y > gmain->ply.y + mazeblick
                  || opos.x < gmain->ply.x - 1 || opos.x > gmain->ply.x + 1) {
                opos.img = NULL;
              } else {
                distance = opos.y - gmain->ply.y;
              }
            }
            if (gmain->ply.dir == WAND_WEST) {
              if (opos.x > gmain->ply.x || opos.x < gmain->ply.x - mazeblick
                  || opos.y < gmain->ply.y - 1 || opos.y > gmain->ply.y + 1) {
                opos.img = NULL;
              } else {
                distance = gmain->ply.x - opos.x;
              }
            }
            if (opos.img != NULL) {
              opospmax_ghost++;
              if (boposp_ghost == NULL) {
                boposp_ghost = malloc(sizeof(*boposp_ghost));
              } else {
                boposp_ghost = realloc(boposp_ghost, sizeof(*boposp_ghost) * opospmax_ghost);
              }
              if (boposp_ghost == NULL) { fprintf(stderr, "malloc/calloc: %s\n", strerror(errno)); opospmax_ghost = 0; }
              if (boposp_ghost != NULL) {
                boposp_ghost[opospmax_ghost - 1].opos = opos;
                boposp_ghost[opospmax_ghost - 1].distance = distance;
              }
            }
          }
        }
      }

      /* sort according distance */
      qsort(boposp_ghost, opospmax_ghost, sizeof(*boposp_ghost), sort_oposp);

      if (mazeblick >= MAZEBLICK_MAX) {
        for (; mazeblick >= MAZEBLICK_MAX; mazeblick--) {
          for (oidx = 0; oidx < opospmax_ghost; oidx++) {
            draw_ghost(gmain, &boposp_ghost[oidx].opos, mazeblick, 0);
          }
        }
      }

      for (; mazeblick >= 0; mazeblick--) {
        if (gmain->ply.dir == WAND_NORD) {
          ypos = gmain->ply.y - mazeblick; xpos = gmain->ply.x;
          yexit_links = ypos; xexit_links = xpos - 1;
          yexit_rechts = ypos; xexit_rechts = xpos + 1;
          wand_links = WAND_WEST; wand_rechts = WAND_OST;
        } else if (gmain->ply.dir == WAND_OST) {
          ypos = gmain->ply.y; xpos = gmain->ply.x + mazeblick;
          yexit_links = ypos - 1; xexit_links = xpos;
          yexit_rechts = ypos + 1; xexit_rechts = xpos;
          wand_links = WAND_NORD; wand_rechts = WAND_SUED;
        } else if (gmain->ply.dir == WAND_SUED) {
          ypos = gmain->ply.y + mazeblick; xpos = gmain->ply.x;
          yexit_links = ypos; xexit_links = xpos + 1;
          yexit_rechts = ypos; xexit_rechts = xpos - 1;
          wand_links = WAND_OST; wand_rechts = WAND_WEST;
        } else if (gmain->ply.dir == WAND_WEST) {
          ypos = gmain->ply.y; xpos = gmain->ply.x - mazeblick;
          yexit_links = ypos + 1; xexit_links = xpos;
          yexit_rechts = ypos - 1; xexit_rechts = xpos;
          wand_links = WAND_SUED; wand_rechts = WAND_NORD;
        } else { break; }

        /* vorne */
        if ((gmain->maze.map[ypos][xpos] & gmain->ply.dir)) {
          if (gmain->maze.exit_y == ypos && gmain->maze.exit_x == xpos && gmain->maze.exit_dir == gmain->ply.dir) {
            imgp = gobj->mazeimg.s[mazeblick].vorne.imgexit;
          } else {
            imgp = gobj->mazeimg.s[mazeblick].vorne.img;
            if (gmain->maze.map[ypos][xpos] & (gmain->ply.dir << PORTAL_SHIFT)) {
              if (gobj->w_col == MAZE_WCOL_LAMP) { imgp = gobj->mazeimg.s[mazeblick].vorne.imgportal; }
            }
          }
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].vorne.xm, gobj->mazeimg.s[mazeblick].vorne.ym, NULL, 0);
        }

        /* halbwand */

        if (!(gmain->maze.map[ypos][xpos] & wand_links) && gobj->mazeimg.s[mazeblick].halbwand_links.img != NULL) {
          if (gmain->maze.exit_y == yexit_links && gmain->maze.exit_x == xexit_links && gmain->maze.exit_dir == gmain->ply.dir) {
            imgp = gobj->mazeimg.s[mazeblick].halbwand_links.imgexit;
          } else {
            imgp = gobj->mazeimg.s[mazeblick].halbwand_links.img;
            if (!(gmain->maze.map[yexit_links][xexit_links] & gmain->ply.dir)) {
              imgp = gobj->mazeimg.s[mazeblick].halbwand_links.noimg;
            } else if (gmain->maze.map[yexit_links][xexit_links] & (gmain->ply.dir << PORTAL_SHIFT)) {
              if (gobj->w_col == MAZE_WCOL_LAMP) { imgp = gobj->mazeimg.s[mazeblick].halbwand_links.imgportal; }
            }
          }
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].halbwand_links.xm, gobj->mazeimg.s[mazeblick].halbwand_links.ym, NULL, 0);
        }

        if (!(gmain->maze.map[ypos][xpos] & wand_rechts) && gobj->mazeimg.s[mazeblick].halbwand_rechts.img != NULL) {
          if (gmain->maze.exit_y == yexit_rechts && gmain->maze.exit_x == xexit_rechts && gmain->maze.exit_dir == gmain->ply.dir) {
            imgp = gobj->mazeimg.s[mazeblick].halbwand_rechts.imgexit;
          } else {
            imgp = gobj->mazeimg.s[mazeblick].halbwand_rechts.img;
            if (!(gmain->maze.map[yexit_rechts][xexit_rechts] & gmain->ply.dir)) {
              imgp = gobj->mazeimg.s[mazeblick].halbwand_rechts.noimg;
            } else if (gmain->maze.map[yexit_rechts][xexit_rechts] & (gmain->ply.dir << PORTAL_SHIFT)) {
              if (gobj->w_col == MAZE_WCOL_LAMP) { imgp = gobj->mazeimg.s[mazeblick].halbwand_rechts.imgportal; }
            }
          }
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].halbwand_rechts.xm, gobj->mazeimg.s[mazeblick].halbwand_rechts.ym, NULL, 0);
        }

        /* saeule */
        VG3_image_copy(gmain->wstruct, NULL, gobj->mazeimg.s[mazeblick].saeule_links.img,
                       gobj->mazeimg.s[mazeblick].saeule_links.xm, gobj->mazeimg.s[mazeblick].saeule_links.ym, NULL, 0);
        VG3_image_copy(gmain->wstruct, NULL, gobj->mazeimg.s[mazeblick].saeule_rechts.img,
                       gobj->mazeimg.s[mazeblick].saeule_rechts.xm, gobj->mazeimg.s[mazeblick].saeule_rechts.ym, NULL, 0);

        /* players+ghosts+minotaur before wand */
        for (oidx = 0; oidx < opospmax_ghost; oidx++) {
          draw_ghost(gmain, &boposp_ghost[oidx].opos, mazeblick, 1);
        }

        /* wand */

        if (gmain->maze.map[ypos][xpos] & wand_links) {
          if (gmain->maze.exit_y == ypos && gmain->maze.exit_x == xpos && gmain->maze.exit_dir == wand_links) {
            imgp = gobj->mazeimg.s[mazeblick].wand_links.imgexit;
          } else {
            imgp = gobj->mazeimg.s[mazeblick].wand_links.img;
            if (gmain->maze.map[ypos][xpos] & (wand_links << PORTAL_SHIFT)) {
              if (gobj->w_col == MAZE_WCOL_LAMP) { imgp = gobj->mazeimg.s[mazeblick].wand_links.imgportal; }
            }
          }
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].wand_links.xm, gobj->mazeimg.s[mazeblick].wand_links.ym, NULL, 0);
        }

        if (gmain->maze.map[ypos][xpos] & wand_rechts) {
          if (gmain->maze.exit_y == ypos && gmain->maze.exit_x == xpos && gmain->maze.exit_dir == wand_rechts) {
            imgp = gobj->mazeimg.s[mazeblick].wand_rechts.imgexit;
          } else {
            imgp = gobj->mazeimg.s[mazeblick].wand_rechts.img;
            if (gmain->maze.map[ypos][xpos] & (wand_rechts << PORTAL_SHIFT)) {
              if (gobj->w_col == MAZE_WCOL_LAMP) { imgp = gobj->mazeimg.s[mazeblick].wand_rechts.imgportal; }
            }
          }
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].wand_rechts.xm, gobj->mazeimg.s[mazeblick].wand_rechts.ym, NULL, 0);
        }

        /* torch */

        if (gmain->maze.map[ypos][xpos] & gmain->ply.dir
          && gmain->maze.map[ypos][xpos] & (gmain->ply.dir << TORCH_SHIFT)
        ) {
          imgp = gobj->mazeimg.s[mazeblick].torch.vorne.img_torch;
          VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gesattr);
          gesattr.zoom = gobj->mazeimg.s[mazeblick].torch.percent;
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].torch.vorne.xm_torch, gobj->mazeimg.s[mazeblick].torch.vorne.ym_torch, &gesattr, 0);
          if (gmain->clock <= 100) {
            VG3_sprite_get(gobj->mazeimg.s[mazeblick].torch.vorne.sprt_flame, &imgp, &iattr, NULL);
            if (imgp != NULL) {
              VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&zattr);
              zattr.zoom = gobj->mazeimg.s[mazeblick].torch.percent * (100 - gmain->clock) / 100;
              VG3_image_attr_sum(&gesattr, &iattr, &zattr);
              VG3_image_copy(gmain->wstruct, NULL, imgp,
                             gobj->mazeimg.s[mazeblick].torch.vorne.xm_flame,
                             gobj->mazeimg.s[mazeblick].torch.vorne.ym_flame
                               + gobj->mazeimg.s[mazeblick].torch.vorne.h_flame / 2
                               * (100 - zattr.zoom) / 100
                               * gobj->mazeimg.s[mazeblick].torch.percent / 100,
                             &gesattr, 0);
            }
          }
        }

        if (gmain->maze.map[ypos][xpos] & wand_links
          && gmain->maze.map[ypos][xpos] & (wand_links << TORCH_SHIFT)
        ) {
          imgp = gobj->mazeimg.s[mazeblick].torch.left.img_torch;
          VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gesattr);
          gesattr.zoom = gobj->mazeimg.s[mazeblick].torch.percent;
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].torch.left.xm_torch, gobj->mazeimg.s[mazeblick].torch.left.ym_torch, &gesattr, 0);
          if (gmain->clock <= 100) {
            VG3_sprite_get(gobj->mazeimg.s[mazeblick].torch.left.sprt_flame, &imgp, &iattr, NULL);
            if (imgp != NULL) {
              VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&zattr);
              zattr.zoom = gobj->mazeimg.s[mazeblick].torch.percent * (100 - gmain->clock) / 100;
              VG3_image_attr_sum(&gesattr, &iattr, &zattr);
              VG3_image_copy(gmain->wstruct, NULL, imgp,
                             gobj->mazeimg.s[mazeblick].torch.left.xm_flame
                               - gobj->mazeimg.s[mazeblick].torch.left.w_flame / 4
                               * (100 - zattr.zoom) / 100
                               * gobj->mazeimg.s[mazeblick].torch.percent / 100,
                             gobj->mazeimg.s[mazeblick].torch.left.ym_flame
                               + gobj->mazeimg.s[mazeblick].torch.left.h_flame / 2
                               * (100 - zattr.zoom) / 100
                               * gobj->mazeimg.s[mazeblick].torch.percent / 100,
                             &gesattr, 0);
            }
          }
        }

        if (gmain->maze.map[ypos][xpos] & wand_rechts
          && gmain->maze.map[ypos][xpos] & (wand_rechts << TORCH_SHIFT)
        ) {
          imgp = gobj->mazeimg.s[mazeblick].torch.right.img_torch;
          VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gesattr);
          gesattr.zoom = gobj->mazeimg.s[mazeblick].torch.percent;
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].torch.right.xm_torch, gobj->mazeimg.s[mazeblick].torch.right.ym_torch, &gesattr, 0);
          if (gmain->clock <= 100) {
            VG3_sprite_get(gobj->mazeimg.s[mazeblick].torch.right.sprt_flame, &imgp, &iattr, NULL);
            if (imgp != NULL) {
              VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&zattr);
              zattr.zoom = gobj->mazeimg.s[mazeblick].torch.percent * (100 - gmain->clock) / 100;
              VG3_image_attr_sum(&gesattr, &iattr, &zattr);
              VG3_image_copy(gmain->wstruct, NULL, imgp,
                             gobj->mazeimg.s[mazeblick].torch.right.xm_flame
                               + gobj->mazeimg.s[mazeblick].torch.right.w_flame / 4
                               * (100 - zattr.zoom) / 100
                               * gobj->mazeimg.s[mazeblick].torch.percent / 100,
                             gobj->mazeimg.s[mazeblick].torch.right.ym_flame
                               + gobj->mazeimg.s[mazeblick].torch.right.h_flame / 2
                               * (100 - zattr.zoom) / 100
                               * gobj->mazeimg.s[mazeblick].torch.percent / 100,
                             &gesattr, 0);
            }
          }
        }

        /* item: garlic */
        if (gmain->maze.map[ypos][xpos] & ITEM_GARLIC) {
          imgp = gobj->mazeimg.s[mazeblick].item.garlic.img;
          VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gesattr);
          gesattr.zoom = gobj->mazeimg.s[mazeblick].item.percent;
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].item.garlic.xm, gobj->mazeimg.s[mazeblick].item.garlic.ym, &gesattr, 0);
        }

        /* item: valerian */
        if (gmain->maze.map[ypos][xpos] & ITEM_VALERIAN) {
          imgp = gobj->mazeimg.s[mazeblick].item.valerian.img;
          VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gesattr);
          gesattr.zoom = gobj->mazeimg.s[mazeblick].item.percent;
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].item.valerian.xm, gobj->mazeimg.s[mazeblick].item.valerian.ym, &gesattr, 0);
        }

        /* item: jewel */
        if (gmain->maze.map[ypos][xpos] & ITEM_JEWEL) {
          imgp = gobj->mazeimg.s[mazeblick].item.jewel.img;
          VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gesattr);
          gesattr.zoom = gobj->mazeimg.s[mazeblick].item.percent;
          VG3_image_copy(gmain->wstruct, NULL, imgp,
                         gobj->mazeimg.s[mazeblick].item.jewel.xm, gobj->mazeimg.s[mazeblick].item.jewel.ym, &gesattr, 0);
        }

        /* players+ghosts+minotaur after wand */
        for (oidx = 0; oidx < opospmax_ghost; oidx++) {
          draw_ghost(gmain, &boposp_ghost[oidx].opos, mazeblick, 0);
        }
      }

      if (gmain->ply.go != 0) {
        VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
        VG3_window_attributes(gmain->wstruct, &iattr, &zattr, -1, NULL);
        if (gmain->ply.go < 0) {  /* go backward */
          iattr.zoom = 100 - gmain->ply.go / 2;
        } else if (gmain->ply.go > 0) {  /* go forward */
          iattr.zoom = 100 + (100 - gmain->ply.go) / 2;
        }
        if ((imgp = VG3_image_clone(gmain->wstruct, NULL, &iattr, NULL, NULL)) != NULL) {
          VG3_image_copy(gmain->wstruct, NULL, imgp, gmain->winw / 2, gmain->winh / 2, NULL, 0);
          VG3_image_unload(gmain->wstruct, imgp);
        }
        VG3_window_attributes(gmain->wstruct, &zattr, NULL, -1, NULL);
      }

      if (opospmax_ghost > 0) { free(boposp_ghost); }

      /* draw text: direction */
      rect.x = 0;
      rect.y = 10;
      rect.w = gmain->winw;
      rect.h = gmain->winh;
      if (gmain->ply.dir == WAND_NORD) {
        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, VG3_multilang_get(gmain->mlang, "north"));
      } else if (gmain->ply.dir == WAND_OST) {
        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, VG3_multilang_get(gmain->mlang, "east"));
      } else if (gmain->ply.dir == WAND_SUED) {
        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, VG3_multilang_get(gmain->mlang, "south"));
      } else if (gmain->ply.dir == WAND_WEST) {
        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, VG3_multilang_get(gmain->mlang, "west"));
      }
      rect = VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 1);
      rect.x = (gmain->winw - rect.w) / 2;
      VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 0);

      /* draw text: fear */
      rect.x = 10;
      rect.y = 10;
      rect.w = gmain->winw;
      rect.h = gmain->winh;
      snprintf(buf, sizeof(buf), "%s: %d%%", VG3_multilang_get(gmain->mlang, "fear"), gmain->ply.fear);
      if (gmain->ply.fear <= 30) {
        icol = VGAG3_COLOR_GREEN;
      } else if (gmain->ply.fear <= 60) {
        icol = VGAG3_COLOR_ORANGE;
      } else {
        icol = VGAG3_COLOR_RED;
      }
      VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, buf);
      rect = VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 1);
      VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 0);

      /* draw text: garlic */
      if (gmain->ply.garlic > 0) {
        rect.x = 10;
        rect.y = 24;
        rect.w = gmain->winw;
        rect.h = gmain->winh;
        snprintf(buf, sizeof(buf), "%s: %d%%", VG3_multilang_get(gmain->mlang, "garlic"), gmain->ply.garlic);
        if (gmain->ply.garlic >= 60) {
          icol = VGAG3_COLOR_GREEN;
        } else if (gmain->ply.garlic >= 30) {
          icol = VGAG3_COLOR_ORANGE;
        } else {
          icol = VGAG3_COLOR_RED;
        }
        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, buf);
        rect = VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 1);
        VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 0);
      }

      /* draw remote players as icon */
      rect.x = 10;
      rect.y = 38;
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
      if (ofc != NULL && ofc->f_data != NULL) {
        struct vg3_ofunc_objsnap *osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
        while ((objq = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
          if (objq->instanceid == gmain->ply.instanceid) { continue; }
          opos.dir = 2;
          if (ofc->f_data(gmain, objq, &opos) && opos.img != NULL) {
            VG3_image_getsize(gmain->wstruct, opos.img, NULL, &imgw, &imgh);
            rect.x += (imgw / 2);
            VG3_image_copy(gmain->wstruct, NULL, opos.img, rect.x, rect.y + imgh / 2, NULL, 0);
            rect.x += (imgw + 10);
          }
        }
        VG3_ofunc_objlist_freelist(osnap);
      }

      /* draw jewel */
      if (gmain->ply.jewel) {
        rect.x = gmain->winw / 2;
        rect.y = 30;
        VG3_image_getsize(gmain->wstruct, gobj->img_jewel_klein, NULL, NULL, &rect.h);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_jewel_klein, rect.x, rect.y + rect.h / 2, NULL, 0);
      }

      /* draw text: clock */
      rect.x = 0;
      rect.y = 10;
      rect.w = gmain->winw;
      rect.h = gmain->winh;
      if (gmain->clock < 100) {
        snprintf(buf, sizeof(buf), "%d %s", 60 - (gmain->clock * 60 / 100), VG3_multilang_get(gmain->mlang, "min_to_midnight"));
        icol = VGAG3_COLOR_GREEN;
      } else if (gmain->clock == 100) {
        snprintf(buf, sizeof(buf), "%s", VG3_multilang_get(gmain->mlang, "midnight"));
        icol = VGAG3_COLOR_ORANGE;
      } else {
        snprintf(buf, sizeof(buf), "%s", VG3_multilang_get(gmain->mlang, "after_midnight"));
        icol = VGAG3_COLOR_RED;
      }
      VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, buf);
      rect = VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 1);
      rect.x = gmain->winw - rect.w - 10;
      VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 0);

      /* draw text: ghosts */
      rect.x = 0;
      rect.y = 24;
      rect.w = gmain->winw;
      rect.h = gmain->winh;
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
      if (ofc != NULL) {
        int ghanz = 0;
        struct vg3_ofunc_objsnap *osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
        while ((objq = VG3_ofunc_objlist_nextlist(osnap)) != NULL) { ghanz++; }
        VG3_ofunc_objlist_freelist(osnap);
        snprintf(buf, sizeof(buf), "%s: %d", VG3_multilang_get(gmain->mlang, "ghosts"), ghanz);
        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, buf);
        icol = VGAG3_COLOR_BROWN;
        rect = VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 1);
        rect.x = gmain->winw - rect.w - 10;
        VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 0);
      }

      /* draw text: radar */
      rect.x = 0;
      rect.y = 38;
      rect.w = gmain->winw;
      rect.h = gmain->winh;
      snprintf(buf, sizeof(buf), "%s: %d %s",
               VG3_multilang_get(gmain->mlang, "radar"),
               red_distance(gmain, gmain->ply.instanceid),
               VG3_multilang_get(gmain->mlang, "steps"));
      icol = VGAG3_COLOR_BROWN;
      VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, buf);
      rect = VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 1);
      rect.x = gmain->winw - rect.w - 10;
      VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_TRANSPARENT, 0);

      /* draw text?: info */
      if (gmain->ply.info != NULL && *gmain->ply.info != '\0') {
        rect.x = 0;
        rect.y = 110;
        rect.w = gmain->winw;
        rect.h = gmain->winh;
        icol = VGAG3_COLOR_WHITE;
        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, "8x8.font+", '\n', 0, gmain->ply.info);
        rect = VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_BLUE, 1);
        rect.x = (gmain->winw - rect.w) / 2;
        VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, icol, VGAG3_COLOR_BLUE, 0);
      }

    } else {
      imgp = gobj->mazeimg.dreh;
      VG3_image_getsize(gmain->wstruct, imgp, NULL, &imgw, &imgh);
      if (gmain->ply.turn < 0) {  /* turn left */
        xpos = gmain->winw - gmain->winw * (-gmain->ply.turn) / 100;
      } else {  /* turn right */
        xpos = gmain->winw * gmain->ply.turn / 100;
      }
      VG3_image_copy(gmain->wstruct, NULL, imgp, xpos, gmain->winh / 2, NULL, 0);
    }
  }

  if (gmain->clock >= 0 && gmain->clock <= 100) {
    struct vg3_image_attributes wattr;
    VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&wattr);
    wattr.bright = 200 - gmain->clock*4/5;
    VG3_window_attributes(gmain->wstruct, &wattr, NULL, -1, NULL);
  }

  if (gmain->ply.map) { draw_mazemap(gmain); }

  if (gmain->ply.lamp) {
    VG3_window_attributes(gmain->wstruct, NULL, NULL, VG3_colordef_colorize_to("BLUE", gmain->brovdr + 12, gmain->brovdr), NULL);
    gobj->w_col = MAZE_WCOL_LAMP;
  } else {
    if (gmain->brovdr == 0) {
      VG3_window_attributes(gmain->wstruct, NULL, NULL, VGAG3_COLORDEF_DEFAULT, NULL);
    } else {
      VG3_window_attributes(gmain->wstruct, NULL, NULL, VG3_colordef_colorize_to("WHITE", 0, gmain->brovdr), NULL);
    }
    gobj->w_col = MAZE_WCOL_NORMAL;
    { const int walk_violet = 10;
      const struct vg3_ofunc_objfunc *ofc;
      struct walking_way wway;
      struct obj_pos opos;
      int walk_red, walk_max;
      int minway = -1, obfollow = 0;
      walk_red = red_distance(gmain, gmain->ply.instanceid);
      if (walk_red < walk_violet) { walk_max = walk_violet; } else { walk_max = walk_red; }
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
      if (ofc != NULL && ofc->f_data != NULL) {
        struct vg3_ofunc_objsnap *osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
        while ((objq = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
          opos.dir = 0;
          if (ofc->f_data(gmain, objq, &opos)) {
            create_way(gmain, &wway, walk_max, opos.x, opos.y, gmain->ply.x, gmain->ply.y);
            if ((wway.anz > 0 && wway.anz < walk_max) || (opos.x == gmain->ply.x && opos.y == gmain->ply.y)) {
              if (minway == -1 || wway.anz < minway) { minway = wway.anz; }
              if (opos.u.gh.follow_id == gmain->ply.instanceid && wway.anz <= walk_red) {
                if (obfollow == 0 || obfollow > 1 + wway.anz) { obfollow = 1 + wway.anz; }
              }
            }
          }
        }
        VG3_ofunc_objlist_freelist(osnap);
      }
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_MINOTAUR));
      if (ofc != NULL && ofc->f_data != NULL) {
        if (VG3_ofunc_objlist_find_obj(gmain->ofstruct, get_oid_name(OID_NAME_MINOTAUR), &objq) > 0) {
          opos.dir = 0;
          if (ofc->f_data(gmain, objq, &opos)) {
            create_way(gmain, &wway, walk_max, opos.x, opos.y, gmain->ply.x, gmain->ply.y);
            if ((wway.anz > 0 && wway.anz < walk_max) || (opos.x == gmain->ply.x && opos.y == gmain->ply.y)) {
              if (minway == -1 || wway.anz < minway) { minway = wway.anz; }
              if (opos.u.gh.follow_id == gmain->ply.instanceid && wway.anz <= walk_red) {
                if (obfollow == 0 || obfollow > 1 + wway.anz) { obfollow = 1 + wway.anz; }
              }
            }
          }
        }
      }
      if (minway >= 0) {
        if (obfollow > 0) { minway = obfollow - 1; }
        VG3_window_attributes(gmain->wstruct, NULL, NULL, VG3_colordef_colorize_to((obfollow ? "RED" : "VIOLET"), gmain->brovdr + walk_max - minway, gmain->brovdr), NULL);
      }
    }
  }
}


static int
sort_oposp(const void *vbopos1, const void *vbopos2)
{
  const struct box_obj_pos *bopos1, *bopos2;

  bopos1 = (struct box_obj_pos *)vbopos1;
  bopos2 = (struct box_obj_pos *)vbopos2;
  if (bopos1->distance < bopos2->distance) { return 1; }
  if (bopos1->distance > bopos2->distance) { return -1; }
  return 0;
}


static void
draw_ghost(struct g_main *gmain, struct obj_pos *opos, int mazeblick, int indirekt)
{
  struct g_pos { int xl, xr; };
  struct g_pos pos_ghost[MAZEBLICK_MAX] = {
    { -160, 800 }, { 150, 490 }, { 220, 420 }, { 255, 385 }
  };
  struct vg3_image_attributes gesattr, iattr;
  double dpos;
  int xpos, xprz, doagain;

  if (gmain == NULL || opos == NULL || opos->img == NULL || mazeblick < 0) { return; }

  if (gmain->ply.lamp && opos->u_pg == 2) { return; }

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
  xpos = gmain->winw / 2;
  doagain = 0;

  if (opos->dir == 0 && opos->prz == -1) {
    if (indirekt) { return; }
    xpos = opos->x;

  } else if (gmain->ply.dir == WAND_NORD && opos->y == gmain->ply.y - mazeblick) {
    if (opos->x != gmain->ply.x) {
      if ((opos->x == gmain->ply.x + 1 && opos->dir == WAND_WEST && opos->prz > 50)
        || (opos->x == gmain->ply.x - 1 && opos->dir == WAND_OST && opos->prz > 50)) {
        if (indirekt) { return; }
        indirekt = 1;
      }
    }
    if (indirekt && opos->x == gmain->ply.x) { return; }
    if (!indirekt && opos->x != gmain->ply.x) { return; }
    if (indirekt && (opos->dir == WAND_NORD || opos->dir == WAND_SUED)) { return; }
    dpos = (double)mazeblick;
    if (opos->dir == WAND_NORD) {
      dpos += ((double)opos->prz / 100.);
    } else if (opos->dir == WAND_SUED) {
      dpos -= ((double)opos->prz / 100.);
      if (dpos < .0) { return; }
      doagain = 1;
    }
    if (mazeblick < MAZEBLICK_MAX) {
      xprz = 0;
      if (opos->dir == WAND_WEST) { xprz = opos->prz; } else if (opos->dir == WAND_OST) { xprz = -opos->prz; }
      xprz = 100 - 100 * (opos->x - gmain->ply.x) + xprz;
      xpos = pos_ghost[mazeblick].xr - (pos_ghost[mazeblick].xr - pos_ghost[mazeblick].xl) * xprz / 200;
    }
    if (dpos > 6.) { iattr.zoom = -2. * ((dpos + 1.) - 12.); } else { iattr.zoom = 220. / ((dpos + 1.) * sqrt((dpos + 1.) + .2)); }

  } else if (gmain->ply.dir == WAND_OST && opos->x == gmain->ply.x + mazeblick) {
    if (opos->y != gmain->ply.y) {
      if ((opos->y == gmain->ply.y + 1 && opos->dir == WAND_NORD && opos->prz > 50)
        || (opos->y == gmain->ply.y - 1 && opos->dir == WAND_SUED && opos->prz > 50)) {
        if (indirekt) { return; }
        indirekt = 1;
      }
    }
    if (indirekt && opos->y == gmain->ply.y) { return; }
    if (!indirekt && opos->y != gmain->ply.y) { return; }
    if (indirekt && (opos->dir == WAND_OST || opos->dir == WAND_WEST)) { return; }
    dpos = (double)mazeblick;
    if (opos->dir == WAND_OST) {
      dpos += ((double)opos->prz / 100.);
    } else if (opos->dir == WAND_WEST) {
      dpos -= ((double)opos->prz / 100.);
      if (dpos < .0) { return; }
      doagain = 1;
    }
    if (mazeblick < MAZEBLICK_MAX) {
      xprz = 0;
      if (opos->dir == WAND_NORD) { xprz = opos->prz; } else if (opos->dir == WAND_SUED) { xprz = -opos->prz; }
      xprz = 100 - 100 * (opos->y - gmain->ply.y) + xprz;
      xpos = pos_ghost[mazeblick].xr - (pos_ghost[mazeblick].xr - pos_ghost[mazeblick].xl) * xprz / 200;
    }
    if (dpos > 6.) { iattr.zoom = -2. * ((dpos + 1.) - 12.); } else { iattr.zoom = 220. / ((dpos + 1.) * sqrt((dpos + 1.) + .2)); }

  } else if (gmain->ply.dir == WAND_SUED && opos->y == gmain->ply.y + mazeblick) {
    if (opos->x != gmain->ply.x) {
      if ((opos->x == gmain->ply.x + 1 && opos->dir == WAND_WEST && opos->prz > 50)
        || (opos->x == gmain->ply.x - 1 && opos->dir == WAND_OST && opos->prz > 50)) {
        if (indirekt) { return; }
        indirekt = 1;
      }
    }
    if (indirekt && opos->x == gmain->ply.x) { return; }
    if (!indirekt && opos->x != gmain->ply.x) { return; }
    if (indirekt && (opos->dir == WAND_SUED || opos->dir == WAND_NORD)) { return; }
    dpos = (double)mazeblick;
    if (opos->dir == WAND_SUED) {
      dpos += ((double)opos->prz / 100.);
    } else if (opos->dir == WAND_NORD) {
      dpos -= ((double)opos->prz / 100.);
      if (dpos < .0) { return; }
      doagain = 1;
    }
    if (mazeblick < MAZEBLICK_MAX) {
      xprz = 0;
      if (opos->dir == WAND_WEST) { xprz = -opos->prz; } else if (opos->dir == WAND_OST) { xprz = opos->prz; }
      xprz = 100 + 100 * (opos->x - gmain->ply.x) + xprz;
      xpos = pos_ghost[mazeblick].xr - (pos_ghost[mazeblick].xr - pos_ghost[mazeblick].xl) * xprz / 200;
    }
    if (dpos > 6.) { iattr.zoom = -2. * ((dpos + 1.) - 12.); } else { iattr.zoom = 220. / ((dpos + 1.) * sqrt((dpos + 1.) + .2)); }

  } else if (gmain->ply.dir == WAND_WEST && opos->x == gmain->ply.x - mazeblick) {
    if (opos->y != gmain->ply.y) {
      if ((opos->y == gmain->ply.y + 1 && opos->dir == WAND_NORD && opos->prz > 50)
        || (opos->y == gmain->ply.y - 1 && opos->dir == WAND_SUED && opos->prz > 50)) {
        if (indirekt) { return; }
        indirekt = 1;
      }
    }
    if (indirekt && opos->y == gmain->ply.y) { return; }
    if (!indirekt && opos->y != gmain->ply.y) { return; }
    if (indirekt && (opos->dir == WAND_WEST || opos->dir == WAND_OST)) { return; }
    dpos = (double)mazeblick;
    if (opos->dir == WAND_WEST) {
      dpos += ((double)opos->prz / 100.);
    } else if (opos->dir == WAND_OST) {
      dpos -= ((double)opos->prz / 100.);
      if (dpos < .0) { return; }
      doagain = 1;
    }
    if (mazeblick < MAZEBLICK_MAX) {
      xprz = 0;
      if (opos->dir == WAND_NORD) { xprz = -opos->prz; } else if (opos->dir == WAND_SUED) { xprz = opos->prz; }
      xprz = 100 + 100 * (opos->y - gmain->ply.y) + xprz;
      xpos = pos_ghost[mazeblick].xr - (pos_ghost[mazeblick].xr - pos_ghost[mazeblick].xl) * xprz / 200;
    }
    if (dpos > 6.) { iattr.zoom = -2. * ((dpos + 1.) - 12.); } else { iattr.zoom = 220. / ((dpos + 1.) * sqrt((dpos + 1.) + .2)); }

  } else {
    return;
  }

  if (doagain && mazeblick == 0) { doagain = 0; }

  VG3_image_attr_sum(&gesattr, &opos->attr, &iattr);
  if (doagain) {  /* draw object-instance not now but at next call */
    opos->x = xpos; opos->attr = gesattr; opos->dir = 0; opos->prz = -1;
  } else {
    VG3_image_copy(gmain->wstruct, NULL, opos->img, xpos, gmain->winh / 2, &gesattr, 0);
    opos->img = NULL;
  }
}


static int
load_maze(struct g_main *gmain, const char *mazebuf, int planz)
{
  static const char *hexmapping = "0123456789abcdef";
  char buf[256], *ptr;
  size_t slen;
  int i1, i2, i3, wpos, hpos, anz_garlic, anz_valerian, anz_jewel;
  const char *pt1, *pt2;

  if (gmain == NULL || mazebuf == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return 0; }
  gmain->maze.wsize = gmain->maze.hsize = 0;
  gmain->maze.map = NULL;

  /* create maze */
  for (pt1 = mazebuf, pt2 = strchr(pt1, '\n'); ; pt1 = pt2 + 1, pt2 = strchr(pt1, '\n')) {
    if (pt2 == NULL) {
      pt2 = pt1 + strlen(pt1);
      if (pt1 == pt2) { break; }
    }
    snprintf(buf, sizeof(buf), "%.*s", (int)(pt2 - pt1), pt1);
    slen = strlen(buf);
    if (slen > 0 && buf[slen - 1] == '\n') { slen--; }
    if (slen > 0 && buf[slen - 1] == '\r') { slen--; }
    buf[slen] = '\0';
    if (slen == 0) { VG3_seterror(EINVAL, "maze-description invalid(1)"); return 0; }
    if (gmain->maze.wsize == 0) {
      gmain->maze.wsize = (int)slen;
    } else if (gmain->maze.wsize != (int)slen) {
      VG3_seterror(EINVAL, "maze-description invalid(2)"); 
      return 0;
    }
    if (gmain->maze.hsize == 0) {
      gmain->maze.map = malloc(sizeof(unsigned short *));
    } else {
      gmain->maze.map = realloc(gmain->maze.map, sizeof(unsigned short *) * (gmain->maze.hsize + 1));
    }
    if (gmain->maze.map == NULL) { VG3_seterror(ENOMEM, "malloc/realloc error"); return 0; }
    gmain->maze.map[gmain->maze.hsize] = malloc(sizeof(unsigned short) * gmain->maze.wsize);
    if (gmain->maze.map[gmain->maze.hsize] == NULL) { VG3_seterror(ENOMEM, "malloc error"); return 0; }
    for (wpos = 0; wpos < gmain->maze.wsize; wpos++) {
      if ((ptr = strchr(hexmapping, buf[wpos])) == NULL) { VG3_seterror(EINVAL, "maze-description invalid(3)"); return 0; }
      gmain->maze.map[gmain->maze.hsize][wpos] = (unsigned short)(int)(size_t)(ptr - hexmapping);
    }
    gmain->maze.hsize++;
    if (*pt2 == '\0') { break; }
  }
  if (gmain->maze.hsize == 0) { VG3_seterror(EINVAL, "maze-description invalid(4)"); return 0; }

  /* +++ addons +++ */

  for (hpos = 0; hpos < gmain->maze.hsize; hpos++) {
    for (wpos = 0; wpos < gmain->maze.wsize; wpos++) {
      if (gmain->maze.map[hpos][wpos] != 0) {
        int widx, wanz, zfl;
        for (wanz = 0, widx = 0; widx < MAP_SHIFT; widx++) {
          if (gmain->maze.map[hpos][wpos] & (unsigned short)(1 << widx)) { wanz++; }
        }
        if (wanz > 0) {
          /* torch */
          zfl = ZUFALL(0, 11) % wanz;  /* dep. on MAP_SHIFT */
          for (widx = 0; widx < MAP_SHIFT; widx++) {
            if (gmain->maze.map[hpos][wpos] & (unsigned short)(1 << widx)) {
              if (zfl-- == 0) {
                gmain->maze.map[hpos][wpos] |= (unsigned short)(1 << (TORCH_SHIFT + widx));
                break;
              }
            }
          }
          /* secret wall-portal */
          if (ZUFALL(0, 29) < 3) {
            zfl = ZUFALL(0, 11) % wanz;  /* dep. on MAP_SHIFT */
            for (widx = 0; widx < MAP_SHIFT; widx++) {
              if (gmain->maze.map[hpos][wpos] & (unsigned short)(1 << widx)) {
                if (zfl-- == 0) {
                  if ((hpos == 0 && widx == 0)
                      || (hpos == gmain->maze.hsize - 1 && widx == 2)
                      || (wpos == 0 && widx == 3)
                      || (wpos == gmain->maze.wsize - 1 && widx == 1)
                     ) {
                    ;
                  } else {
                    gmain->maze.map[hpos][wpos] |= (unsigned short)(1 << (PORTAL_SHIFT + widx));
                  }
                  break;
                }
              }
            }
          }
        }
      }
    }
  }

  /* items */
  if (planz > 4) { planz = 4; }
  i1 = (gmain->maze.wsize * gmain->maze.hsize) / 100;
  anz_garlic = (gmain->maze.wsize * gmain->maze.hsize) / (65 - (planz * 5));
  if (i1 > 0) { anz_garlic += (ZUFALL(1, i1) - (i1 / 2)); }
  anz_valerian = (gmain->maze.wsize * gmain->maze.hsize) / (70 - (planz * 5));
  if (i1 > 0) { anz_valerian += (ZUFALL(1, i1) - (i1 / 2)); }
  anz_jewel = (gmain->maze.wsize + gmain->maze.hsize) / 20 * planz;
  if (anz_jewel == 0) { anz_jewel = planz; }

  i1 = gmain->maze.wsize * gmain->maze.hsize;
  /* item:garlic */
  for (i2 = 0; i2 < anz_garlic; i2++) {
    i3 = ZUFALL(1, i1) - 1;
    hpos = i3 / gmain->maze.hsize;
    wpos = i3 % gmain->maze.hsize;
    if (!(gmain->maze.map[hpos][wpos] & ((unsigned short)ITEM_GARLIC | (unsigned short)ITEM_VALERIAN | (unsigned short)ITEM_JEWEL))) {
      gmain->maze.map[hpos][wpos] |= (unsigned short)ITEM_GARLIC;
    } else {
      i2--;
    }
  }
  /* item:valerian */
  for (i2 = 0; i2 < anz_valerian; i2++) {
    i3 = ZUFALL(1, i1) - 1;
    hpos = i3 / gmain->maze.hsize;
    wpos = i3 % gmain->maze.hsize;
    if (!(gmain->maze.map[hpos][wpos] & ((unsigned short)ITEM_GARLIC | (unsigned short)ITEM_VALERIAN | (unsigned short)ITEM_JEWEL))) {
      gmain->maze.map[hpos][wpos] |= (unsigned short)ITEM_VALERIAN;
    } else {
      i2--;
    }
  }
  /* item:jewel */
  for (i2 = 0; i2 < anz_jewel; i2++) {
    i3 = ZUFALL(1, i1) - 1;
    hpos = i3 / gmain->maze.hsize;
    wpos = i3 % gmain->maze.hsize;
    if (!(gmain->maze.map[hpos][wpos] & ((unsigned short)ITEM_GARLIC | (unsigned short)ITEM_VALERIAN | (unsigned short)ITEM_JEWEL))) {
      gmain->maze.map[hpos][wpos] |= (unsigned short)ITEM_JEWEL;
    } else {
      i2--;
    }
  }

  /* get start-position and exit */
  gmain->ply.x = gmain->ply.y = gmain->ply.dir = 0;
  gmain->maze.exit_x = gmain->maze.exit_y = gmain->maze.exit_dir = 0;
  for (wpos = 0; wpos < gmain->maze.wsize; wpos++) {
    if (!(gmain->maze.map[0][wpos] & WAND_NORD)) {
      if (gmain->ply.dir == 0) {
        gmain->ply.x = wpos; gmain->ply.y = 0; gmain->ply.dir = WAND_SUED;
      } else {
        gmain->maze.exit_x = wpos; gmain->maze.exit_y = 0; gmain->maze.exit_dir = WAND_NORD;
      }
      gmain->maze.map[0][wpos] |= WAND_NORD;
    }
    if (!(gmain->maze.map[gmain->maze.hsize - 1][wpos] & WAND_SUED)) {
      if (gmain->ply.dir == 0) {
        gmain->ply.x = wpos; gmain->ply.y = gmain->maze.hsize - 1; gmain->ply.dir = WAND_NORD;
      } else {
        gmain->maze.exit_x = wpos; gmain->maze.exit_y = gmain->maze.hsize - 1; gmain->maze.exit_dir = WAND_SUED;
      }
      gmain->maze.map[gmain->maze.hsize - 1][wpos] |= WAND_SUED;
    }
  }
  for (wpos = 0; wpos < gmain->maze.hsize; wpos++) {
    if (!(gmain->maze.map[wpos][0] & WAND_WEST)) {
      if (gmain->ply.dir == 0) {
        gmain->ply.x = 0; gmain->ply.y = wpos; gmain->ply.dir = WAND_OST;
      } else {
        gmain->maze.exit_x = 0; gmain->maze.exit_y = wpos; gmain->maze.exit_dir = WAND_WEST;
      }
      gmain->maze.map[wpos][0] |= WAND_WEST;
    }
    if (!(gmain->maze.map[wpos][gmain->maze.wsize - 1] & WAND_OST)) {
      if (gmain->ply.dir == 0) {
        gmain->ply.x = gmain->maze.wsize - 1; gmain->ply.y = wpos; gmain->ply.dir = WAND_WEST;
      } else {
        gmain->maze.exit_x = gmain->maze.wsize - 1; gmain->maze.exit_y = wpos; gmain->maze.exit_dir = WAND_OST;
      }
      gmain->maze.map[wpos][gmain->maze.wsize - 1] |= WAND_OST;
    }
  }
  if (gmain->ply.dir == 0 || gmain->maze.exit_dir == 0) { VG3_seterror(EINVAL, "Maze: found no start or no exit"); return 0; }

  return 1;
}


static int
load_images(struct vg3_window *wstruct, struct g_obj_maze *gobj)
{
  struct i_pos { int xm, ym; };
  struct i_pos pos_wand_links[MAZEBLICK_MAX] = {
    { 81, 240 }, { 211, 238 }, { 256, 241 }, { 276, 241 }
  };
  struct i_pos pos_wand_rechts[MAZEBLICK_MAX] = {
    { 559, 240 }, { 429, 238 }, { 384, 241 }, { 363, 241 }
  };
  struct i_pos pos_saeule_links[MAZEBLICK_MAX] = {
    { 169, 239 }, { 234, 239 }, { 264, 241 }, { 277, 242 }
  };
  struct i_pos pos_saeule_rechts[MAZEBLICK_MAX] = {
    { 471, 239 }, { 406, 239 }, { 376, 241 }, { 363, 242 }
  };
  struct i_pos pos_halbwand_links[MAZEBLICK_MAX] = {
    { 72, 240 }, { 0, 0 }, { 0, 0 }, { 0, 0 }
  };
  struct i_pos pos_halbwand_rechts[MAZEBLICK_MAX] = {
    { 567, 240 }, { 0, 0 }, { 0, 0 }, { 0, 0 }
  };
  struct i_pos pos_vorne[MAZEBLICK_MAX] = {
    { 320, 240 }, { 319, 239 }, { 319, 241 }, { 319, 241 }
  };
  struct i_pos pos_torch_links[MAZEBLICK_MAX] = {
    { 148, 168 }, { 240, 202 }, { 272, 218 }, { 286, 228 }
  };
  struct i_pos pos_torch_rechts[MAZEBLICK_MAX] = {
    { 491, 168 }, { 399, 202 }, { 367, 218 }, { 353, 228 }
  };
  struct i_pos pos_torch_vorne[MAZEBLICK_MAX] = {
    { 320, 170 }, { 320, 200 }, { 320, 215 }, { 320, 223 }
  };
  struct i_pos pos_item_garlic[MAZEBLICK_MAX] = {
    { 320, 420 }, { 320, 310 }, { 320, 290 }, { 320, 275 }
  };
  struct i_pos pos_item_valerian[MAZEBLICK_MAX] = {
    { 320, 420 }, { 320, 310 }, { 320, 290 }, { 320, 275 }
  };
  struct i_pos pos_item_jewel[MAZEBLICK_MAX] = {
    { 320, 420 }, { 320, 310 }, { 320, 290 }, { 320, 275 }
  };
  int mazeblick_percent[MAZEBLICK_MAX] = { 100, 55, 33, 22 };
  int i1;
  char buf[256];

  if (wstruct == NULL || gobj == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return 0; }

  gobj->img_jewel_klein = VG3_image_load(wstruct, FILES_DIR "/bmp/jewel-klein.bmp", 1);
  if (gobj->img_jewel_klein  == NULL) { return 0; }

  gobj->mazeimg.bg = VG3_image_load(wstruct, FILES_DIR "/bmp/maze/hintergrund.bmp", 1);
  if (gobj->mazeimg.bg == NULL) { return 0; }

  gobj->mazeimg.dreh = VG3_image_load(wstruct, FILES_DIR "/bmp/maze/dreh.bmp", 1);
  if (gobj->mazeimg.dreh == NULL) { return 0; }

  for (i1 = 0; i1 < MAZEBLICK_MAX; i1++) {
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-wand-links.bmp", i1 + 1);
    gobj->mazeimg.s[i1].wand_links.img = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].wand_links.img == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-ausgang-links.bmp", i1 + 1);
    gobj->mazeimg.s[i1].wand_links.imgexit = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].wand_links.imgexit == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-wand-links-portal.bmp", i1 + 1);
    gobj->mazeimg.s[i1].wand_links.imgportal = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].wand_links.imgportal == NULL) { return 0; }
    gobj->mazeimg.s[i1].wand_links.xm = pos_wand_links[i1].xm;
    gobj->mazeimg.s[i1].wand_links.ym = pos_wand_links[i1].ym;

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-wand-rechts.bmp", i1 + 1);
    gobj->mazeimg.s[i1].wand_rechts.img = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].wand_rechts.img == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-ausgang-rechts.bmp", i1 + 1);
    gobj->mazeimg.s[i1].wand_rechts.imgexit = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].wand_rechts.imgexit == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-wand-rechts-portal.bmp", i1 + 1);
    gobj->mazeimg.s[i1].wand_rechts.imgportal = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].wand_rechts.imgportal == NULL) { return 0; }
    gobj->mazeimg.s[i1].wand_rechts.xm = pos_wand_rechts[i1].xm;
    gobj->mazeimg.s[i1].wand_rechts.ym = pos_wand_rechts[i1].ym;

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-saeule-links.bmp", i1 + 1);
    gobj->mazeimg.s[i1].saeule_links.img = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].saeule_links.img == NULL) { return 0; }
    gobj->mazeimg.s[i1].saeule_links.xm = pos_saeule_links[i1].xm;
    gobj->mazeimg.s[i1].saeule_links.ym = pos_saeule_links[i1].ym;

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-saeule-rechts.bmp", i1 + 1);
    gobj->mazeimg.s[i1].saeule_rechts.img = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].saeule_rechts.img == NULL) { return 0; }
    gobj->mazeimg.s[i1].saeule_rechts.xm = pos_saeule_rechts[i1].xm;
    gobj->mazeimg.s[i1].saeule_rechts.ym = pos_saeule_rechts[i1].ym;

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-halbwand.bmp", i1 + 1);
    if (access(buf, F_OK) == 0) {
      gobj->mazeimg.s[i1].halbwand_links.img = VG3_image_load(wstruct, buf, 1);
      gobj->mazeimg.s[i1].halbwand_rechts.img = VG3_image_load(wstruct, buf, 1);
      snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-ausgang-halbwand.bmp", i1 + 1);
      gobj->mazeimg.s[i1].halbwand_links.imgexit = VG3_image_load(wstruct, buf, 1);
      gobj->mazeimg.s[i1].halbwand_rechts.imgexit = VG3_image_load(wstruct, buf, 1);
      if (gobj->mazeimg.s[i1].halbwand_rechts.imgexit == NULL) { return 0; }
      snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-nohalbwand-links.bmp", i1 + 1);
      gobj->mazeimg.s[i1].halbwand_links.noimg = VG3_image_load(wstruct, buf, 1);
      if (gobj->mazeimg.s[i1].halbwand_links.noimg == NULL) { return 0; }
      snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-nohalbwand-rechts.bmp", i1 + 1);
      gobj->mazeimg.s[i1].halbwand_rechts.noimg = VG3_image_load(wstruct, buf, 1);
      if (gobj->mazeimg.s[i1].halbwand_rechts.noimg == NULL) { return 0; }
      snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-halbwand-portal.bmp", i1 + 1);
      gobj->mazeimg.s[i1].halbwand_links.imgportal = VG3_image_load(wstruct, buf, 1);
      gobj->mazeimg.s[i1].halbwand_rechts.imgportal = VG3_image_load(wstruct, buf, 1);
      if (gobj->mazeimg.s[i1].halbwand_rechts.imgportal == NULL) { return 0; }
      gobj->mazeimg.s[i1].halbwand_links.xm = pos_halbwand_links[i1].xm;
      gobj->mazeimg.s[i1].halbwand_links.ym = pos_halbwand_links[i1].ym;
      gobj->mazeimg.s[i1].halbwand_rechts.xm = pos_halbwand_rechts[i1].xm;
      gobj->mazeimg.s[i1].halbwand_rechts.ym = pos_halbwand_rechts[i1].ym;
    }

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-vorne.bmp", i1 + 1);
    gobj->mazeimg.s[i1].vorne.img = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].vorne.img == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-ausgang-vorne.bmp", i1 + 1);
    gobj->mazeimg.s[i1].vorne.imgexit = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].vorne.imgexit == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/maze/s%d-vorne-portal.bmp", i1 + 1);
    gobj->mazeimg.s[i1].vorne.imgportal = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].vorne.imgportal == NULL) { return 0; }
    gobj->mazeimg.s[i1].vorne.xm = pos_vorne[i1].xm;
    gobj->mazeimg.s[i1].vorne.ym = pos_vorne[i1].ym;

    gobj->mazeimg.s[i1].torch.percent = mazeblick_percent[i1];

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/torch/torch-left.bmp");
    gobj->mazeimg.s[i1].torch.left.img_torch = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].torch.left.img_torch == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/torch/flame-left.sprite");
    gobj->mazeimg.s[i1].torch.left.sprt_flame = VG3_sprite_load(wstruct, buf);
    if (gobj->mazeimg.s[i1].torch.left.sprt_flame == NULL) { return 0; }
    gobj->mazeimg.s[i1].torch.left.xm_torch = pos_torch_links[i1].xm;
    gobj->mazeimg.s[i1].torch.left.ym_torch = pos_torch_links[i1].ym;
    gobj->mazeimg.s[i1].torch.left.xm_flame = pos_torch_links[i1].xm - 1;
    gobj->mazeimg.s[i1].torch.left.ym_flame = pos_torch_links[i1].ym - (16 * mazeblick_percent[i1] / 100);
    VG3_sprite_imagesize(gobj->mazeimg.s[i1].torch.left.sprt_flame, "avg", 100, &gobj->mazeimg.s[i1].torch.left.w_flame, &gobj->mazeimg.s[i1].torch.left.h_flame);

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/torch/torch-right.bmp");
    gobj->mazeimg.s[i1].torch.right.img_torch = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].torch.right.img_torch == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/torch/flame-right.sprite");
    gobj->mazeimg.s[i1].torch.right.sprt_flame = VG3_sprite_load(wstruct, buf);
    if (gobj->mazeimg.s[i1].torch.right.sprt_flame == NULL) { return 0; }
    gobj->mazeimg.s[i1].torch.right.xm_torch = pos_torch_rechts[i1].xm;
    gobj->mazeimg.s[i1].torch.right.ym_torch = pos_torch_rechts[i1].ym;
    gobj->mazeimg.s[i1].torch.right.xm_flame = pos_torch_rechts[i1].xm + 1;
    gobj->mazeimg.s[i1].torch.right.ym_flame = pos_torch_rechts[i1].ym - (16 * mazeblick_percent[i1] / 100);
    VG3_sprite_imagesize(gobj->mazeimg.s[i1].torch.right.sprt_flame, "avg", 100, &gobj->mazeimg.s[i1].torch.right.w_flame, &gobj->mazeimg.s[i1].torch.right.h_flame);

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/torch/torch-vorne.bmp");
    gobj->mazeimg.s[i1].torch.vorne.img_torch = VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].torch.vorne.img_torch == NULL) { return 0; }
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/torch/flame-vorne.sprite");
    gobj->mazeimg.s[i1].torch.vorne.sprt_flame = VG3_sprite_load(wstruct, buf);
    if (gobj->mazeimg.s[i1].torch.vorne.sprt_flame == NULL) { return 0; }
    gobj->mazeimg.s[i1].torch.vorne.xm_torch = pos_torch_vorne[i1].xm;
    gobj->mazeimg.s[i1].torch.vorne.ym_torch = pos_torch_vorne[i1].ym;
    gobj->mazeimg.s[i1].torch.vorne.xm_flame = pos_torch_vorne[i1].xm;
    gobj->mazeimg.s[i1].torch.vorne.ym_flame = pos_torch_vorne[i1].ym - (28 * mazeblick_percent[i1] / 100);
    VG3_sprite_imagesize(gobj->mazeimg.s[i1].torch.vorne.sprt_flame, "avg", 100, &gobj->mazeimg.s[i1].torch.vorne.w_flame, &gobj->mazeimg.s[i1].torch.vorne.h_flame);

    gobj->mazeimg.s[i1].item.percent = mazeblick_percent[i1];

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/garlic.bmp");
    gobj->mazeimg.s[i1].item.garlic.img= VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].item.garlic.img == NULL) { return 0; }
    gobj->mazeimg.s[i1].item.garlic.xm = pos_item_garlic[i1].xm;
    gobj->mazeimg.s[i1].item.garlic.ym = pos_item_garlic[i1].ym;

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/valerian.bmp");
    gobj->mazeimg.s[i1].item.valerian.img= VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].item.valerian.img == NULL) { return 0; }
    gobj->mazeimg.s[i1].item.valerian.xm = pos_item_valerian[i1].xm;
    gobj->mazeimg.s[i1].item.valerian.ym = pos_item_valerian[i1].ym;

    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/jewel.bmp");
    gobj->mazeimg.s[i1].item.jewel.img= VG3_image_load(wstruct, buf, 1);
    if (gobj->mazeimg.s[i1].item.jewel.img == NULL) { return 0; }
    gobj->mazeimg.s[i1].item.jewel.xm = pos_item_jewel[i1].xm;
    gobj->mazeimg.s[i1].item.jewel.ym = pos_item_jewel[i1].ym;
  }

  return 1;
}
