#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-minotaur.h"

void getofc_minotaur(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);

static int follow_player(struct g_main *, struct g_obj_minotaur *);


/* +++ get-function +++ */

void
getofc_minotaur(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_MINOTAUR));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_data = f_data;
}


/* +++ object-functions +++ */

/* new-function
 * no variable parameter
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_minotaur *gobj;
  char buf[256];
  int i1;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->pos.x = ZUFALL(0, gmain->maze.wsize - 1);
  gobj->pos.y = ZUFALL(0, gmain->maze.hsize - 1);
  gobj->pos.dir = WAND_NORD;
  gobj->pos.prz = 0;
  gobj->wway.anz = 0;
  gobj->wway.pos = -1;
  gobj->follow_id = 0;
  gobj->pause = 150;  /* should be length of appearing sprite */
  gobj->walknr = MINOTAUR_SPRITE_WALK1;

  /* load minotaur-sprites */
  snprintf(buf, sizeof(buf), FILES_DIR "/bmp/minotaur/minotaur-appear.sprite");
  gobj->sprtapp = VG3_sprite_load(gmain->wstruct, buf);
  if (gobj->sprtapp == NULL) { free(gobj); return NULL; }
  for (i1 = 0; i1 < MINOTAUR_SPRITE_MAX; i1++) {
    snprintf(buf, sizeof(buf), FILES_DIR "/bmp/minotaur/minotaur%d.sprite", i1 + 1);
    gobj->sprt[i1] = VG3_sprite_load(gmain->wstruct, buf);
    if (gobj->sprt[i1] == NULL) { free(gobj); return NULL; }
  }
  gobj->sprt_mom = gobj->sprtapp;

  /* load sound */
  gobj->sound.appear = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/minotaur-appear.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  gobj->sound.follow = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/minotaur-follow.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);

  /* create and fill minotaur-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_MINOTAUR));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert minotaur-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_minotaur *gobj;
  int i1;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_minotaur *)objp->ostruct;

  /* remove minotaur-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free minotaur-sprites */
  if (gobj->sprtapp != NULL) { VG3_sprite_free(gobj->sprtapp); }
  for (i1 = 0; i1 < MINOTAUR_SPRITE_MAX; i1++) {
    if (gobj->sprt[i1] != NULL) { VG3_sprite_free(gobj->sprt[i1]); }
  }

  if (gobj->sound.appear > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.appear); }
  if (gobj->sound.follow > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.follow); }

  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_minotaur *gobj;

  if (gmain == NULL || objp == NULL) { return; }
  if (gmain->clock < 100) { return; }  /* deactivated */

  gobj = (struct g_obj_minotaur *)objp->ostruct;

  if (gobj->sprt_mom == gobj->sprtapp) {
    if (gobj->pause == 0) {
      gobj->sprt_mom = gobj->sprt[MINOTAUR_SPRITE_REST];
      VG3_sprite_rewind(gobj->sprt_mom);
    } else {
      if (gmain->clock == 100 && gmain->clockdelta == 0) {
        VG3_audio_play(gmain->wstruct, gobj->sound.appear, 0, 0);
      }
      gobj->pause--;
      return;
    }
  }

  if (follow_player(gmain, gobj)) {
    gobj->pause = 0;
    gobj->sprt_mom = gobj->sprt[gobj->walknr];
    VG3_sprite_rewind(gobj->sprt_mom);
    if (gobj->follow_id == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.follow, 0, 0); }
  }

  if (gobj->pause > 0) {
    if (--gobj->pause > 0) { return; }
    gobj->sprt_mom = gobj->sprt[gobj->walknr];
    VG3_sprite_rewind(gobj->sprt_mom);
  }

  if (gobj->wway.anz == 0) {
    int xbis, ybis, maxdo = 0;
    while (gobj->wway.anz < 2) {
      xbis = ZUFALL(0, gmain->maze.wsize - 1);
      ybis = ZUFALL(0, gmain->maze.hsize - 1);
      create_way(gmain, &gobj->wway, WALK_MAX, gobj->pos.x, gobj->pos.y, xbis, ybis);
      gobj->pos.prz = 0;
      gobj->follow_id = 0;
      if (gobj->wway.way[1].x > gobj->wway.way[0].x) {
        gobj->pos.dir = WAND_OST;
      } else if (gobj->wway.way[1].x < gobj->wway.way[0].x) {
        gobj->pos.dir = WAND_WEST;
      } else if (gobj->wway.way[1].y > gobj->wway.way[0].y) {
        gobj->pos.dir = WAND_SUED;
      } else {
        gobj->pos.dir = WAND_NORD;
      }
      if (++maxdo == 7) { return; }
    }
    gobj->sprt_mom = gobj->sprt[gobj->walknr];
    VG3_sprite_rewind(gobj->sprt_mom);
  }

  gobj->pos.prz += 8; if (gobj->follow_id) { gobj->pos.prz += 2; }
  if (gobj->pos.prz >= 100) {
    gobj->pos.prz = 0;
    if (++gobj->wway.pos >= gobj->wway.anz) { gobj->wway.anz = 0; return; }
    gobj->pos.x = gobj->wway.way[gobj->wway.pos].x;
    gobj->pos.y = gobj->wway.way[gobj->wway.pos].y;
    if (gobj->wway.pos + 1 < gobj->wway.anz) {
      if (gobj->wway.way[gobj->wway.pos + 1].x > gobj->wway.way[gobj->wway.pos].x) {
        gobj->pos.dir = WAND_OST;
      } else if (gobj->wway.way[gobj->wway.pos + 1].x < gobj->wway.way[gobj->wway.pos].x) {
        gobj->pos.dir = WAND_WEST;
      } else if (gobj->wway.way[gobj->wway.pos + 1].y > gobj->wway.way[gobj->wway.pos].y) {
        gobj->pos.dir = WAND_SUED;
      } else {
        gobj->pos.dir = WAND_NORD;
      }
    }
    if (gobj->wway.pos == gobj->wway.anz - 1) {
      gobj->wway.anz = gobj->wway.pos = 0;
      gobj->follow_id = 0;
      gobj->pause = 50 + ZUFALL(0, 50);
      gobj->sprt_mom = gobj->sprt[MINOTAUR_SPRITE_REST];
      VG3_sprite_rewind(gobj->sprt_mom);
      return;
    }
  }
}


/* data-exchanging-function */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *vdata)
{
  struct g_main *gmain = vmain;
  struct g_obj_minotaur *gobj;
  struct obj_pos *opos = (struct obj_pos *)vdata;
  int flag;

  if (gmain == NULL || objp == NULL || opos == NULL) { return 0; }
  if (gmain->clock < 100) { return 0; }  /* deactivated */

  (void)gmain;
  gobj = (struct g_obj_minotaur *)objp->ostruct;

  flag = opos->dir;
  memset(opos, 0, sizeof(*opos));
  opos->x = gobj->pos.x;
  opos->y = gobj->pos.y;
  opos->prz = gobj->pos.prz;
  opos->dir = gobj->pos.dir;
  opos->img = NULL;
  opos->u_pg = 2;
  opos->u.gh.wway = &gobj->wway;
  opos->u.gh.follow_id = gobj->follow_id;

  if (flag == 1) {
    if (!VG3_sprite_get(gobj->sprt_mom, &opos->img, &opos->attr, NULL)) {
      VG3_sprite_rewind(gobj->sprt_mom);
      VG3_sprite_get(gobj->sprt_mom, &opos->img, &opos->attr, NULL);
    }
  }

  return 1;
}


/* check whether to follow player, return 1 = following or 0 = not following */
static int
follow_player(struct g_main *gmain, struct g_obj_minotaur *gobj_minotaur)
{
  const struct vg3_ofunc_objfunc *ofc;
  struct walking_way wway1, wway0;

  if (gmain == NULL || gobj_minotaur == NULL || gobj_minotaur->follow_id) { return 0; }

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
  if (ofc != NULL && ofc->f_data != NULL) {
    struct vg3_ofunc_objsnap *osnap;
    struct vg3_ofunc_object *objp_player;
    struct obj_pos opos;
    int walk_go, player_instanceid;

    wway0.anz = 0;
    player_instanceid = 0;

    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
    while ((objp_player = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      opos.dir = 0;
      if (ofc->f_data(gmain, objp_player, &opos)) {
        walk_go = red_distance(gmain, objp_player->instanceid);
        create_way(gmain, &wway1, walk_go, gobj_minotaur->pos.x, gobj_minotaur->pos.y, opos.x, opos.y);
        if (wway1.anz >= 2 && wway1.anz < walk_go && (wway0.anz == 0 || wway1.anz < wway0.anz)) {
          wway0 = wway1;
          player_instanceid = objp_player->instanceid;
        }
      }
    }
    VG3_ofunc_objlist_freelist(osnap);

    if (wway0.anz > 0) {
      gobj_minotaur->wway = wway0;
      gobj_minotaur->pos.prz = 0;
      gobj_minotaur->follow_id = player_instanceid;
      if (gobj_minotaur->wway.way[1].x > gobj_minotaur->wway.way[0].x) {
        gobj_minotaur->pos.dir = WAND_OST;
      } else if (gobj_minotaur->wway.way[1].x < gobj_minotaur->wway.way[0].x) {
        gobj_minotaur->pos.dir = WAND_WEST;
      } else if (gobj_minotaur->wway.way[1].y > gobj_minotaur->wway.way[0].y) {
        gobj_minotaur->pos.dir = WAND_SUED;
      } else {
        gobj_minotaur->pos.dir = WAND_NORD;
      }
      return 1;
    }
  }
  return 0;
}
