#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-music.h"

void getofc_music(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_music(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_MUSIC));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
}


/* +++ object-functions +++ */

/* new-function
 * no variable parameter
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_music *gobj;
  char buf[256];
  int i1;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* load music */
  for (i1 = 0; i1 < MUSIC_MUS_MAX; i1++) {
    snprintf(buf, sizeof(buf), "%s/sound/mus%d.flac", FILES_DIR, i1 + 1);
    gobj->mus.id[i1] = VG3_audio_load(gmain->wstruct, buf, 60 + i1 * 5, VGAG3_AUDIO_VOLUME_MUSIC);
  }
  gobj->mus.id_aftermid = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-aftermid.flac", 80, VGAG3_AUDIO_VOLUME_MUSIC);

  /* load water-drip */
  for (i1 = 0; i1 < MUSIC_WATER_MAX; i1++) {
    snprintf(buf, sizeof(buf), "%s/sound/water%d.wav", FILES_DIR, i1 + 1);
    gobj->water.id[i1] = VG3_audio_load(gmain->wstruct, buf, 100, VGAG3_AUDIO_VOLUME_MUSIC);
  }

  /* get initial seed-number for water */
  gobj->water.seed = VG3_nw_random_initseed(SEED_WATER_MODIF);

  gobj->mus.pause = (gmain->maze.wsize * gmain->maze.hsize) / 2;
  gobj->water.pause = 50;

  /* create and fill music-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_MUSIC));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert music-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_music *gobj;
  int i1;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_music *)objp->ostruct;

  /* remove music-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* unload music */
  for (i1 = 0; i1 < MUSIC_MUS_MAX; i1++) {
    if (gobj->mus.id[i1] > 0) { VG3_audio_unload(gmain->wstruct, gobj->mus.id[i1]); }
  }
  if (gobj->mus.id_aftermid > 0) { VG3_audio_unload(gmain->wstruct, gobj->mus.id_aftermid); }

  /* unload water-drip */
  for (i1 = 0; i1 < MUSIC_WATER_MAX; i1++) {
    if (gobj->water.id[i1] > 0) { VG3_audio_unload(gmain->wstruct, gobj->water.id[i1]); }
  }

  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_music *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_music *)objp->ostruct;

  /* music */
  if (gobj->mus.pause > 0) {
    if (--gobj->mus.pause == 0) {
      if (gobj->mus.idrun == 0) {
        gobj->mus.idrun = 1;
        VG3_audio_play(gmain->wstruct, gobj->mus.id[gobj->mus.idrun - 1], 1, 1);
      } else {
        VG3_audio_play(gmain->wstruct, gobj->mus.id_aftermid, 1, 0);
      }
    }
  } else if (gmain->clockdelta == 0) {
    if (gmain->clock == 20 || gmain->clock == 60) {
      VG3_audio_stop(gmain->wstruct, gobj->mus.id[gobj->mus.idrun - 1], 1);
      gobj->mus.idrun++;
      VG3_audio_play(gmain->wstruct, gobj->mus.id[gobj->mus.idrun - 1], 1, 1);
    } else if (gmain->clock == 100) {
      VG3_audio_stop(gmain->wstruct, gobj->mus.id[gobj->mus.idrun - 1], 0);
      gobj->mus.pause = 50;
    }
  }

  /* water */
  if (gobj->water.idrun == 0 || !VG3_audio_isplaying(gmain->wstruct, gobj->water.id[gobj->water.idrun - 1])) {
    if (gobj->water.pause > 0) { gobj->water.pause--; }
    if (gobj->water.pause == 0) {
      gobj->water.pause = (int)VG3_nw_random_getnext(10, 50, &gobj->water.seed);
      gobj->water.idrun = (int)VG3_nw_random_getnext(1, MUSIC_WATER_MAX, &gobj->water.seed);
      VG3_audio_play(gmain->wstruct, gobj->water.id[gobj->water.idrun - 1], 0, 0);
    }
  }
}
