#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-player.h"

void getofc_player(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);


/* +++ get-function +++ */

void
getofc_player(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_PLAYER));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_data = f_data;
}


/* +++ object-functions +++ */

/* new-function
 * variable parameter:
 *  - network client-number or 0
 *  - whether player is local
 *  - player-bitmap-number
 *  - clockmax
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_player *gobj;
  int clnr, islocal, hpos, bnr, clockmax;
  char buf[256];
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  clnr = va_arg(ap, int);
  islocal = va_arg(ap, int);
  bnr = va_arg(ap, int);
  clockmax = va_arg(ap, int);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  snprintf(buf, sizeof(buf), "%s/bmp/player/pl%d.bmp", FILES_DIR, bnr);
  gobj->img = VG3_image_load(gmain->wstruct, buf, 1);
  if (gobj->img == NULL) { return NULL; }
  snprintf(buf, sizeof(buf), "%s/bmp/player/pl%d-klein.bmp", FILES_DIR, bnr);
  gobj->imgklein = VG3_image_load(gmain->wstruct, buf, 1);
  if (gobj->imgklein == NULL) { return NULL; }
  gobj->pos.x = gmain->ply.x;
  gobj->pos.y = gmain->ply.y;
  gobj->pos.dir = gmain->ply.dir;
  gobj->go = gobj->turn = 0;
  gobj->clnr = clnr;
  gobj->fear = 0;
  gobj->fearmulti = 100 + clockmax * 2;
  gobj->dead = 0;
  gobj->map = 2;
  if (islocal) { gmain->ply.bnr = bnr; }
  if (islocal) { gmain->ply.map = gobj->map; }
  gobj->ghost_instanceid = 0;
  gobj->lamp_ontime = 0;
  gobj->garlic_smell = 0;
  gobj->garlic_smellmax = (clockmax / 2 + 50) / 5;
  gobj->meeting_instanceid = 0;
  gobj->meeting_status = 'D';
  gobj->has_jewel = 0;

  /* create mapvisited */
  gobj->mapvisited = calloc((unsigned int)gmain->maze.hsize, sizeof(char *));
  if (gobj->mapvisited == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  for (hpos = 0; hpos < gmain->maze.hsize; hpos++) {
    gobj->mapvisited[hpos] = calloc(gmain->maze.wsize, sizeof(char));
    if (gobj->mapvisited[hpos] == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  }
  gobj->mapvisited[gobj->pos.y][gobj->pos.x] |= 128;
  if (islocal) { gmain->ply.mapvisited = gobj->mapvisited; }

  /* load sounds if local player */
  if (islocal) {
    gobj->sound.footstep[0] = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/footstep1.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
    gobj->sound.footstep[1] = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/footstep2.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
    gobj->sound.step = 0;
    gobj->sound.transfer = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/kn-transfer.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
    gobj->sound.eat = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/eat.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
    gobj->sound.drink = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/drink.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
    gobj->sound.wall_portal = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/wall-portal.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
    gobj->sound.jewel = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/jewel.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  }

  /* create and fill player-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_PLAYER));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert player-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  if (islocal) { gmain->ply.instanceid = objp->instanceid; }

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_player *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_player *)objp->ostruct;

  /* remove player-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  if (gobj->img != NULL) { VG3_image_unload(gmain->wstruct, gobj->img); }
  if (gobj->imgklein != NULL) { VG3_image_unload(gmain->wstruct, gobj->imgklein); }

  if (gobj->mapvisited != NULL) {
    int hpos;
    for (hpos = 0; hpos < gmain->maze.hsize; hpos++) {
      if (gobj->mapvisited[hpos] != NULL) { free(gobj->mapvisited[hpos]); }
    }
    free(gobj->mapvisited);
    if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.mapvisited = NULL; }
  }

  if (gobj->sound.footstep[0] > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.footstep[0]); }
  if (gobj->sound.footstep[1] > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.footstep[1]); }
  if (gobj->sound.transfer > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.transfer); }
  if (gobj->sound.eat > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.eat); }
  if (gobj->sound.drink > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.drink); }
  if (gobj->sound.wall_portal > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.wall_portal); }
  if (gobj->sound.jewel > 0) { VG3_audio_unload(gmain->wstruct, gobj->sound.jewel); }

  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_player *gobj;
  const struct vg3_ofunc_objfunc *ofc;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_player *)objp->ostruct;

  if (gmain->nwptr != NULL && !VG3_nw_getclientinfo(gmain->nwptr, gobj->clnr, NULL)) { return; }

  if (gobj->ghost_instanceid > 0) {
    struct vg3_ofunc_object *objp_ghost;
    objp_ghost = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gobj->ghost_instanceid);
    if (objp_ghost != NULL) { goto endrun; }
    gobj->ghost_instanceid = 0;
    if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.fear = gobj->fear / gobj->fearmulti; }
    if (gobj->fear >= 100 * gobj->fearmulti) {
      gobj->dead = 2;
      if (objp->instanceid == gmain->ply.instanceid) {
        gmain->ply.dead = gobj->dead;
      }
      goto endrun;
    }
  }

  /* check collisions with ghosts */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
  if (ofc != NULL && ofc->f_data != NULL) {
    struct vg3_ofunc_objsnap *osnap;
    struct vg3_ofunc_object *objp_ghost;
    struct obj_pos opos;

    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
    while ((objp_ghost = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      opos.dir = 0;
      if (ofc->f_data(gmain, objp_ghost, &opos)) {
        if (gobj->pos.x == opos.x && gobj->pos.y == opos.y) {  /* collided with ghost */
          const struct vg3_ofunc_objobjfunc *oofc;
          int collretw;
          struct vg3_coll_ret collrets;
          oofc = VG3_ofunc_get_objobjfunc(gmain->ofstruct, objp->oid, objp_ghost->oid);
          if (oofc == NULL || oofc->f_collision == NULL) { continue; }
          collretw = oofc->f_collision(gmain, objp, objp_ghost, &collrets);
          if (collretw < 0) { fprintf(stderr, "%s\n", VG3_error()); continue; }
          break;
        }
      }
    }
    VG3_ofunc_objlist_freelist(osnap);
  }

  if (gobj->ghost_instanceid > 0) {
    gobj->go = gobj->turn = 0;
    if (objp->instanceid == gmain->ply.instanceid) {
      gmain->ply.go = gmain->ply.turn = 0;
      gmain->ply.x = gobj->pos.x;
      gmain->ply.y = gobj->pos.y;
      gobj->mapvisited[gobj->pos.y][gobj->pos.x] |= 128;
    }
    goto endrun;
  }

  /* check collision with minotaur */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_MINOTAUR));
  if (ofc != NULL && ofc->f_data != NULL) {
    struct vg3_ofunc_object *objp_minotaur;
    struct obj_pos opos;

    if (VG3_ofunc_objlist_find_obj(gmain->ofstruct, get_oid_name(OID_NAME_MINOTAUR), &objp_minotaur) > 0) {
      opos.dir = 0;
      if (ofc->f_data(gmain, objp_minotaur, &opos)) {
        if (gobj->pos.x == opos.x && gobj->pos.y == opos.y) {  /* collided with minotaur */
          const struct vg3_ofunc_objobjfunc *oofc;
          int collretw;
          struct vg3_coll_ret collrets;
          oofc = VG3_ofunc_get_objobjfunc(gmain->ofstruct, objp->oid, objp_minotaur->oid);
          if (oofc != NULL && oofc->f_collision != NULL) {
            collretw = oofc->f_collision(gmain, objp, objp_minotaur, &collrets);
            if (collretw < 0) { fprintf(stderr, "%s\n", VG3_error()); }
          }
        }
      }
    }
  }

  if (gobj->dead == 1) {
    if (objp->instanceid == gmain->ply.instanceid) {
      gmain->ply.dead = gobj->dead;
    }
    goto endrun;
  }

  if (gobj->lamp_ontime != 0) { gobj->lamp_ontime++; }
  if (gobj->fear > 0) {
    gobj->fear--;
    if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.fear = gobj->fear / gobj->fearmulti; }
  }
  if (gobj->garlic_smell > 0) {
    if (++gobj->garlic_smellmom == gobj->garlic_smellmax) {
      gobj->garlic_smell -= 5;
      gobj->garlic_smellmom = 0;
      if (gobj->garlic_smell < 0) { gobj->garlic_smell = 0; }
      if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.garlic = gobj->garlic_smell; }
    }
  }

  /* lamp */
  if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, gobj->clnr, -1, KEYDEF_TOGGLE_LAMP, VGAG3_IS_NEW_PRESSED)) {
    if (!gobj->map) {
      if (gobj->lamp_ontime != 0) { gobj->lamp_ontime = -gobj->lamp_ontime; } else { gobj->lamp_ontime = 1; }
      if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.lamp = (gobj->lamp_ontime > 0 ? 1 : 0); }
    }
  }

  if (gobj->go != 0) {
    if (gobj->go < 0) { /* going backward */
      gobj->go += 20;
    } else if (gobj->go > 0) { /* going forward */
      gobj->go -= 20;
      if (gobj->go == 0) {
        if (objp->instanceid == gmain->ply.instanceid) {
          gmain->ply.x = gobj->pos.x;
          gmain->ply.y = gobj->pos.y;
        }
      }
    }
    if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.go = gobj->go; }
    if (gobj->go != 0) { return; }
    gobj->mapvisited[gobj->pos.y][gobj->pos.x] |= 128;
  }

  if (gobj->turn != 0) {
    if (gobj->turn < 0) { /* turning left */
      gobj->turn += 25;
    } else if (gobj->turn > 0) { /* turning right */
      gobj->turn -= 25;
    }
    if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.turn = gobj->turn; }
    if (gobj->turn != 0) { return; }
  }

  /* map */
  if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, gobj->clnr, -1, KEYDEF_TOGGLE_MAP, VGAG3_IS_NEW_PRESSED)) {
    if (gobj->map) {
      gobj->map = 0;
    } else if (!gobj->map) {
      gobj->map = 1;
      if (gobj->lamp_ontime > 0) { gobj->lamp_ontime = -gobj->lamp_ontime; }
    }
    if (objp->instanceid == gmain->ply.instanceid) {
      gmain->ply.map = gobj->map;
      gmain->ply.lamp = (gobj->lamp_ontime > 0 ? 1 : 0);
    }
  }
  if (gobj->map && VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_LSHIFT, VGAG3_IS_PRESSED)
      && VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_Y, VGAG3_IS_NEW_PRESSED)) {
    if (++gobj->map > 4) { gobj->map = 1; }
    gmain->ply.map = gobj->map;
  }

  if (gobj->map) {
    if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, gobj->clnr, -1, KEYDEF_USE_ITEM, VGAG3_IS_NEW_PRESSED)) {
      if (gobj->has_jewel) {
        gobj->map = 4;
        gobj->has_jewel = 0;
        if (objp->instanceid == gmain->ply.instanceid) {
          gmain->ply.jewel = gobj->has_jewel;
          gmain->ply.map = gobj->map;
        }
      }
    }
    return;
  }

  /* meeting? */
  if (gmain->nwptr != NULL) {
    ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
    if (ofc != NULL) {
      struct vg3_ofunc_objsnap *osnap;
      struct vg3_ofunc_object *objp_player;
      struct g_obj_player *gobj_test, *gobj_player;
      unsigned int instanceid = 0;

      gobj_player = NULL;
      osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
      while ((objp_player = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
        if (objp_player->instanceid == objp->instanceid) { continue; }
        gobj_test = (struct g_obj_player *)objp_player->ostruct;
        if (!VG3_nw_getclientinfo(gmain->nwptr, gobj_test->clnr, NULL)) { continue; }
        if (gobj->pos.x == gobj_test->pos.x && gobj->pos.y == gobj_test->pos.y) {
          if (instanceid == 0) {
            gobj_player = gobj_test;
            instanceid = objp_player->instanceid;
          } else {
            gobj_player = NULL;
            instanceid = (unsigned int)-1;
          }
        }
      }
      VG3_ofunc_objlist_freelist(osnap);

      if (instanceid == 0) {  /* no meeting */
        if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.info = NULL; }
        gobj->meeting_instanceid = 0;
        gobj->meeting_status = '?';
      } else if (instanceid == (unsigned int)-1) {  /* meeting with more than one player */
        if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.info = NULL; }
        gobj->meeting_instanceid = 0;
        gobj->meeting_status = 'D';
      } else {  /* meeting with another player */
        if (instanceid != gobj->meeting_instanceid) {  /* meeting player changed */
          gobj->meeting_instanceid = instanceid;
          gobj->meeting_status = '?';
        }
        if (gobj->meeting_status == '?') {  /* do i want to exchange data? */
          if (gobj_player->meeting_status == '?') {  /* does meeting player want to exchange data? */
            if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.info = VG3_multilang_get(gmain->mlang, "meeting-??"); }
          } else if (gobj_player->meeting_status == 'Y') {  /* meeting player wants to exchange data */
            if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.info = VG3_multilang_get(gmain->mlang, "meeting-?Y"); }
          } else {  /* meeting player doesn't want to exchange data */
            if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.info = NULL; }
            gobj->meeting_status = 'D';
          }
        } else if (gobj->meeting_status == 'Y') {  /* i want to exchange data */
          if (gobj_player->meeting_status == '?') {  /* does meeting player want to exchange data? */
            if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.info = VG3_multilang_get(gmain->mlang, "meeting-Y?"); }
          } else if (gobj_player->meeting_status == 'Y') {  /* meeting player wants to exchange data */
            if (gobj->mapvisited != NULL && gobj_player->mapvisited != NULL) {
              int hpos, wpos;
              for (hpos = 0; hpos < gmain->maze.hsize; hpos++) {
                if (gobj->mapvisited[hpos] != NULL && gobj_player->mapvisited[hpos] != NULL) {
                  for (wpos = 0; wpos < gmain->maze.wsize; wpos++) {
                    gobj->mapvisited[hpos][wpos] |= gobj_player->mapvisited[hpos][wpos];
                    gobj_player->mapvisited[hpos][wpos] = gobj->mapvisited[hpos][wpos];
                  }
                }
              }
              if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.info = VG3_multilang_get(gmain->mlang, "meeting-YY"); }
              gobj->fear = (gobj->fear + gobj_player->fear) / 2;
              gobj_player->fear = gobj->fear;
              if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.fear = gobj->fear / gobj->fearmulti; }
              gobj->meeting_status = 'D';
              if (objp->instanceid == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.transfer, 0, 0); }
            }
          } else {  /* meeting player has exchanged data */
            if (objp->instanceid == gmain->ply.instanceid) {
              gmain->ply.info = VG3_multilang_get(gmain->mlang, "meeting-YY");
              gmain->ply.fear = gobj->fear / gobj->fearmulti;
            }
            gobj->meeting_status = 'D';
            if (objp->instanceid == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.transfer, 0, 0); }
          }
        }
      }
    }
  }

  /* use item */
  if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, gobj->clnr, -1, KEYDEF_USE_ITEM, VGAG3_IS_NEW_PRESSED)) {
    if (gmain->nwptr != NULL && gobj->meeting_instanceid > 0) {
      if (gobj->meeting_status == '?') { gobj->meeting_status = 'Y'; }  /* i want to exchange data */
    } else if (gmain->maze.map[gobj->pos.y][gobj->pos.x] & ITEM_GARLIC) {
      if (gobj->garlic_smell == 0) {
        gobj->garlic_smell = 100;
        gobj->garlic_smellmom = 0;
        /* cancel following ghosts */
        ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
        if (ofc != NULL && ofc->f_data != NULL) {
          struct vg3_ofunc_objsnap *osnap;
          struct vg3_ofunc_object *objp_ghost;
          struct obj_pos opos;
          osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_GHOST));
          while ((objp_ghost = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
            opos.dir = 2;
            opos.u.gh.follow_id = objp->instanceid;
            ofc->f_data(gmain, objp_ghost, &opos);
          }
          VG3_ofunc_objlist_freelist(osnap);
        }
        if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.garlic = gobj->garlic_smell; }
        gmain->maze.map[gobj->pos.y][gobj->pos.x] &= ~ITEM_GARLIC;
        if (objp->instanceid == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.eat, 0, 0); }
      }
    } else if (gmain->maze.map[gobj->pos.y][gobj->pos.x] & ITEM_VALERIAN) {
      if (gobj->fear >= 10 * gobj->fearmulti) {
        gobj->fear /= 2;
        if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.fear = gobj->fear / gobj->fearmulti; }
        gmain->maze.map[gobj->pos.y][gobj->pos.x] &= ~ITEM_VALERIAN;
        if (objp->instanceid == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.drink, 0, 0); }
      }
    } else if (gmain->maze.map[gobj->pos.y][gobj->pos.x] & ITEM_JEWEL) {
      if (!gobj->has_jewel) {
        gobj->has_jewel = 1;
        if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.jewel = gobj->has_jewel; }
        gmain->maze.map[gobj->pos.y][gobj->pos.x] &= ~ITEM_JEWEL;
        if (objp->instanceid == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.jewel, 0, 0); }
      }
    }
  }

  /* go forward */
  if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, gobj->clnr, -1, KEYDEF_GO_FORWARD, VGAG3_IS_PRESSED)) {
    if (!(gmain->maze.map[gobj->pos.y][gobj->pos.x] & gobj->pos.dir)
        || gmain->maze.map[gobj->pos.y][gobj->pos.x] & (gobj->pos.dir << PORTAL_SHIFT)
       ) {
      if (gmain->maze.map[gobj->pos.y][gobj->pos.x] & (gobj->pos.dir << PORTAL_SHIFT)) { 
        gobj->mapvisited[gobj->pos.y][gobj->pos.x] |= gobj->pos.dir;
        if (objp->instanceid == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.wall_portal, 0, 0); }
      }
      if (gobj->pos.dir == WAND_NORD) {
        gobj->pos.y--;
      } else if (gobj->pos.dir == WAND_OST) {
        gobj->pos.x++;
      } else if (gobj->pos.dir == WAND_SUED) {
        gobj->pos.y++;
      } else if (gobj->pos.dir == WAND_WEST) {
        gobj->pos.x--;
      }
      gobj->go = 100;
      if (objp->instanceid == gmain->ply.instanceid) {
        VG3_audio_play(gmain->wstruct, gobj->sound.footstep[gobj->sound.step], 0, 0);
        gobj->sound.step = !gobj->sound.step;
      }

    } else if (gmain->maze.exit_x == gobj->pos.x
               && gmain->maze.exit_y == gobj->pos.y
               && gmain->maze.exit_dir == gobj->pos.dir) {
      if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.gone = 1; }
      goto endrun;
    }
  }

  /* go backward */
  if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, gobj->clnr, -1, KEYDEF_GO_BACKWARD, VGAG3_IS_PRESSED)) {
    int idir = 0;
    if (gobj->pos.dir == WAND_NORD) {
      idir = WAND_SUED;
    } else if (gobj->pos.dir == WAND_OST) {
      idir = WAND_WEST;
    } else if (gobj->pos.dir == WAND_SUED) {
      idir = WAND_NORD;
    } else if (gobj->pos.dir == WAND_WEST) {
      idir = WAND_OST;
    }
    if (!(gmain->maze.map[gobj->pos.y][gobj->pos.x] & idir)
        || gmain->maze.map[gobj->pos.y][gobj->pos.x] & (idir << PORTAL_SHIFT)
       ) {
      if (gmain->maze.map[gobj->pos.y][gobj->pos.x] & (idir << PORTAL_SHIFT)) { 
        gobj->mapvisited[gobj->pos.y][gobj->pos.x] |= idir;
        if (objp->instanceid == gmain->ply.instanceid) { VG3_audio_play(gmain->wstruct, gobj->sound.wall_portal, 0, 0); }
      }
      if (gobj->pos.dir == WAND_NORD) {
        gobj->pos.y++;
      } else if (gobj->pos.dir == WAND_OST) {
        gobj->pos.x--;
      } else if (gobj->pos.dir == WAND_SUED) {
        gobj->pos.y--;
      } else if (gobj->pos.dir == WAND_WEST) {
        gobj->pos.x++;
      }
      gobj->go = -100;
      if (objp->instanceid == gmain->ply.instanceid) {
        gmain->ply.x = gobj->pos.x;
        gmain->ply.y = gobj->pos.y;
        gmain->ply.go = gobj->go;
        VG3_audio_play(gmain->wstruct, gobj->sound.footstep[gobj->sound.step], 0, 0);
        gobj->sound.step = !gobj->sound.step;
      }
    } else if (gmain->maze.exit_x == gobj->pos.x
               && gmain->maze.exit_y == gobj->pos.y
               && gmain->maze.exit_dir == idir) {
      if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.gone = 1; }
      goto endrun;
    }
  }

  if (gobj->pos.x < 0 || gobj->pos.x >= gmain->maze.wsize || gobj->pos.y < 0 || gobj->pos.y >= gmain->maze.hsize) {
    if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.gone = 1; }
    goto endrun;
  }

  /* turn right */
  if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, gobj->clnr, -1, KEYDEF_TURN_RIGHT, VGAG3_IS_NEW_PRESSED)) {
    if (gobj->pos.dir == WAND_NORD) {
      gobj->pos.dir = WAND_OST;
    } else if (gobj->pos.dir == WAND_OST) {
      gobj->pos.dir = WAND_SUED;
    } else if (gobj->pos.dir == WAND_SUED) {
      gobj->pos.dir = WAND_WEST;
    } else if (gobj->pos.dir == WAND_WEST) {
      gobj->pos.dir = WAND_NORD;
    }
    gobj->turn = 100;
    if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.dir = gobj->pos.dir; gmain->ply.turn = gobj->turn; }
  }

  /* turn left */
  if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, gobj->clnr, -1, KEYDEF_TURN_LEFT, VGAG3_IS_NEW_PRESSED)) {
    if (gobj->pos.dir == WAND_NORD) {
      gobj->pos.dir = WAND_WEST;
    } else if (gobj->pos.dir == WAND_OST) {
      gobj->pos.dir = WAND_NORD;
    } else if (gobj->pos.dir == WAND_SUED) {
      gobj->pos.dir = WAND_OST;
    } else if (gobj->pos.dir == WAND_WEST) {
      gobj->pos.dir = WAND_SUED;
    }
    gobj->turn = -100;
    if (objp->instanceid == gmain->ply.instanceid) { gmain->ply.dir = gobj->pos.dir; gmain->ply.turn = gobj->turn; }
  }

  if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_1, VGAG3_IS_NEW_PRESSED)) {
    if (gmain->brovdr > 0) { gmain->brovdr--; }
  }
  if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_2, VGAG3_IS_NEW_PRESSED)) {
    if (gmain->brovdr < 10) { gmain->brovdr++; }
  }

endrun:
  ;
}


/* data-exchanging-function */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *vdata)
{
  struct g_main *gmain = vmain;
  struct g_obj_player *gobj;
  struct obj_pos *opos = (struct obj_pos *)vdata;
  int flag;

  if (gmain == NULL || objp == NULL || opos == NULL) { return 0; }

  gobj = (struct g_obj_player *)objp->ostruct;

  flag = opos->dir;

  if (gmain->nwptr != NULL && !VG3_nw_getclientinfo(gmain->nwptr, gobj->clnr, NULL)) { return 0; }

  memset(opos, 0, sizeof(*opos));
  opos->x = gobj->pos.x;
  opos->y = gobj->pos.y;
  opos->dir = gobj->pos.dir;
  if (flag == 1) {
    opos->img = gobj->img;
  } else if (flag == 2) {
    opos->img = gobj->imgklein;
  } else {
    opos->img = NULL;
  }
  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&opos->attr);
  opos->u_pg = 1;
  opos->u.ply.lamp_ontime = (gobj->lamp_ontime >= 0 ? gobj->lamp_ontime : -gobj->lamp_ontime);
  opos->u.ply.garlic = !!gobj->garlic_smell;

  return 1;
}
