/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"


/* abstract key definition and default resp. current value */
struct keydefs {
  int keyref;           /* abstract key definition */
  int keyidx;           /* (default resp. current) keyboard resp. gamecontroller/joystick key */
  const char *keyname;  /* name of key */
  const char *info;     /* key description in change-menu */
};

#define MAX_DEVICES  8
#define MAX_KEYS    16
struct {  /* input-devices (indx: 0 = keyboard, 1 to MAX_DEVICES = gamecontroller/joystick) */
  int enabled;                    /* whether input-device is enabled */
  int jid;                        /* input-device number: 0 = keyboard, >0 = gamecontroller/joystick */
  struct keydefs kdef[MAX_KEYS + 1];  /* definitions for each abstract key (last entry = end of key definitions) */
} abstractkey[MAX_DEVICES + 1];


void menu_insert_input(struct g_main *, int);
void nw_key(struct g_main *, int);
int event_update(struct g_main *, int *);
int key_ispressed(struct g_main *, int, int, int);

static void set_abstractkeys(struct g_main *);


/* set key definitions and corresponding values */
static void
set_abstractkeys(struct g_main *gmain)
{
  int *jidf, i1, i2;
  struct vg3_gamecontroller gcs;

  /* disable all input-devices */
  for (i1 = 0; i1 <= MAX_DEVICES; i1++) {
    abstractkey[i1].enabled = 0;
  }

  /* input-devices: 0 = reserved for keyboard, 1 to ? = gamecontroller/joysticks */
  VG3_gamecontroller_getall(gmain->wstruct, &jidf);
  if (jidf[0] > MAX_DEVICES) { jidf[0] = MAX_DEVICES; }

  /* 0 = keyboard; 1 to ? = gamecontroller/joystick */
  for (i1 = 0; i1 <= jidf[0]; i1++) {
    abstractkey[i1].enabled = 1;  /* enable input-device */

    if (i1 > 0) {  /* get info of gamecontroller/joystick */
      VG3_gamecontroller_getinfo(gmain->wstruct, jidf[i1], &gcs);
    } else {
      gcs.is_gamecontroller = 0;
    }

    /* set 0=keyboard or gamecontroller-/joystick-device-number */
    if (i1 == 0) {  /* keyboard */
      abstractkey[i1].jid = 0;
    } else {  /* gamecontroller/joystick */
      abstractkey[i1].jid = jidf[i1];
    }

    /* go forward */
    i2 = 0;
    if (i2 >= MAX_KEYS) { goto endofdef; }
    abstractkey[i1].kdef[i2].keyref = KEYDEF_GO_FORWARD;
    if (i1 == 0) {  /* keyboard */
      abstractkey[i1].kdef[i2].keyidx = VGAG3_KEY_UCURS;
    } else {  /* gamecontroller/joystick */
      if (gcs.is_gamecontroller) {
        abstractkey[i1].kdef[i2].keyidx = VGAG3_GC_AXIS_LEFTY_UP;
      } else {
        abstractkey[i1].kdef[i2].keyidx = VG3_gamecontroller_name2num(gmain->wstruct, jidf[i1], "Axis-2:neg");
      }
    }
    abstractkey[i1].kdef[i2].keyname = "fwd";
    abstractkey[i1].kdef[i2].info = "go-fwd";

    /* go backward */
    i2++;
    if (i2 >= MAX_KEYS) { goto endofdef; }
    abstractkey[i1].kdef[i2].keyref = KEYDEF_GO_BACKWARD;
    if (i1 == 0) {  /* keyboard */
      abstractkey[i1].kdef[i2].keyidx = VGAG3_KEY_DCURS;
    } else {  /* gamecontroller/joystick */
      if (gcs.is_gamecontroller) {
        abstractkey[i1].kdef[i2].keyidx = VGAG3_GC_AXIS_LEFTY_DOWN;
      } else {
        abstractkey[i1].kdef[i2].keyidx = VG3_gamecontroller_name2num(gmain->wstruct, jidf[i1], "Axis-2:pos");
      }
    }
    abstractkey[i1].kdef[i2].keyname = "bwd";
    abstractkey[i1].kdef[i2].info = "go-bwd";

    /* turn right */
    i2++;
    if (i2 >= MAX_KEYS) { goto endofdef; }
    abstractkey[i1].kdef[i2].keyref = KEYDEF_TURN_RIGHT;
    if (i1 == 0) {  /* keyboard */
      abstractkey[i1].kdef[i2].keyidx = VGAG3_KEY_RCURS;
    } else {  /* gamecontroller/joystick */
      if (gcs.is_gamecontroller) {
        abstractkey[i1].kdef[i2].keyidx = VGAG3_GC_AXIS_RIGHTX_RIGHT;
      } else {
        abstractkey[i1].kdef[i2].keyidx = VG3_gamecontroller_name2num(gmain->wstruct, jidf[i1], "Axis-1:pos");
      }
    }
    abstractkey[i1].kdef[i2].keyname = "turn-right";
    abstractkey[i1].kdef[i2].info = "turn-right";

    /* turn left */
    i2++;
    if (i2 >= MAX_KEYS) { goto endofdef; }
    abstractkey[i1].kdef[i2].keyref = KEYDEF_TURN_LEFT;
    if (i1 == 0) {  /* keyboard */
      abstractkey[i1].kdef[i2].keyidx = VGAG3_KEY_LCURS;
    } else {  /* gamecontroller/joystick */
      if (gcs.is_gamecontroller) {
        abstractkey[i1].kdef[i2].keyidx = VGAG3_GC_AXIS_RIGHTX_LEFT;
      } else {
        abstractkey[i1].kdef[i2].keyidx = VG3_gamecontroller_name2num(gmain->wstruct, jidf[i1], "Axis-1:neg");
      }
    }
    abstractkey[i1].kdef[i2].keyname = "turn-left";
    abstractkey[i1].kdef[i2].info = "turn-left";

    /* gunfire */
    i2++;
    if (i2 >= MAX_KEYS) { goto endofdef; }
    abstractkey[i1].kdef[i2].keyref = KEYDEF_FIRE;
    if (i1 == 0) {  /* keyboard */
      abstractkey[i1].kdef[i2].keyidx = VGAG3_KEY_SPACE;
    } else {  /* gamecontroller/joystick */
      if (gcs.is_gamecontroller) {
        abstractkey[i1].kdef[i2].keyidx = VGAG3_GC_BUTTON_RIGHTSHOULDER;
      } else {
        abstractkey[i1].kdef[i2].keyidx = VG3_gamecontroller_name2num(gmain->wstruct, jidf[i1], "Button-1");
      }
    }
    abstractkey[i1].kdef[i2].keyname = "gunfire";
    abstractkey[i1].kdef[i2].info = "gunfire";

    /* end of key definitions (last entry!) */
    i2++;
endofdef:
    abstractkey[i1].kdef[i2].keyref = KEYDEF_NOKEY;
    if (i1 == 0) {  /* keyboard */
      abstractkey[i1].kdef[i2].keyidx = VGAG3_KEY_NOKEY;
    } else {  /* gamecontroller/joystick */
      abstractkey[i1].kdef[i2].keyidx = VGAG3_GC_NOKEY;
    }
    abstractkey[i1].kdef[i2].keyname = NULL;
    abstractkey[i1].kdef[i2].info = NULL;
  }

  free(jidf);
}


/* insert selecting keys for input-devices into system-menu */
void
menu_insert_input(struct g_main *gmain, int use_network)
{
  struct vg3_sysmenu_submenu *subm;
  int ktnr, i1, obkeyb;

  if (gmain == NULL) { return; }

  /* set key definitions */
  set_abstractkeys(gmain);

  /* add keyboard menu-entry only if it is used by a player */
  if (!use_network) {
    obkeyb = 0;
    for (i1 = 0; i1 < MAXPLAYER; i1++) {
      if (gmain->player[i1].ply == PLY_IS_LOCAL) {
        if (gmain->player[i1].jid == -1 || gmain->player[i1].jid == 0) { obkeyb = 1; break; }
      }
    }
  } else {
    obkeyb = 1;
  }

  /* add menu-entry: keyboard-submenu (abstract key definition used as key reference) */
  if (abstractkey[0].enabled && obkeyb) {
    subm = VG3_sysmenu_simple_keyboardmenu(gmain->sysm, NULL);
    for (ktnr = 0; abstractkey[0].kdef[ktnr].keyref != KEYDEF_NOKEY; ktnr++) {
      /* find saved setting */
      abstractkey[0].kdef[ktnr].keyidx = VG3_sysmenu_savestring_find(gmain->sysm, gmain->sysm_string, 0, abstractkey[0].kdef[ktnr].keyref, abstractkey[0].kdef[ktnr].keyidx);
      /* add entry */
      VG3_sysmenu_simple_keyboardentry(gmain->sysm, subm, abstractkey[0].kdef[ktnr].keyref,
        VG3_multilang_get(gmain->mlang, abstractkey[0].kdef[ktnr].keyname),
        abstractkey[0].kdef[ktnr].keyidx,
        VG3_multilang_get(gmain->mlang, abstractkey[0].kdef[ktnr].info)
      );
    }
  }

  /* add menu-entry: gamecontroller-/joystick-submenu (abstract key definition used as key reference) */
  for (i1 = 1; i1 <= MAX_DEVICES; i1++) {
    if (abstractkey[i1].enabled) { break; }
  }
  if (i1 <= MAX_DEVICES) {
    subm = VG3_sysmenu_simple_gcmenu(gmain->sysm, NULL);
    for (i1 = 1; i1 <= MAX_DEVICES; i1++) {  /* for each gamecontroller/joystick */
      if (abstractkey[i1].enabled) {
        for (ktnr = 0; abstractkey[i1].kdef[ktnr].keyref != KEYDEF_NOKEY; ktnr++) {
          /* find saved setting */
          abstractkey[i1].kdef[ktnr].keyidx = VG3_sysmenu_savestring_find(gmain->sysm, gmain->sysm_string, abstractkey[i1].jid, abstractkey[i1].kdef[ktnr].keyref, abstractkey[i1].kdef[ktnr].keyidx);
          /* add entry */
          VG3_sysmenu_simple_gcentry(gmain->sysm, subm, abstractkey[i1].jid, abstractkey[i1].kdef[ktnr].keyref,
            VG3_multilang_get(gmain->mlang, abstractkey[i1].kdef[ktnr].keyname),
            abstractkey[i1].kdef[ktnr].keyidx,
            VG3_multilang_get(gmain->mlang, abstractkey[i1].kdef[ktnr].info)
          );
        }
      }
    }
  }
}


/* add keys for jid to network-struct resp. check for changed keys for all input-devices */
void
nw_key(struct g_main *gmain, int jid)
{
  int keyidx, devnr, ktnr;

  if (gmain == NULL) { return; }

  for (devnr = 0; devnr <= MAX_DEVICES; devnr++) {  /* for each input-device (0 = keyboard, 1 to ? = joystick) */
    if (!abstractkey[devnr].enabled) { continue; }
    for (ktnr = 0; abstractkey[devnr].kdef[ktnr].keyref != KEYDEF_NOKEY; ktnr++) {  /* for each defined key */
      if (jid < 0) {  /* check for changed key */
        /* get real key for abstractkey[devnr].kdef[ktnr].keyref */
        keyidx = VG3_sysmenu_getkeyval(gmain->sysm, abstractkey[devnr].jid, abstractkey[devnr].kdef[ktnr].keyref);
        /* is key from system-menu unequal to saved key in abstractkey[devnr].kdef[ktnr].keyidx? */
        if (keyidx != abstractkey[devnr].kdef[ktnr].keyidx) {
          /* change key in network-struct to key from system-menu */
          VG3_nw_changekey(gmain->nwclnt, abstractkey[devnr].jid, abstractkey[devnr].kdef[ktnr].keyidx, keyidx);
          abstractkey[devnr].kdef[ktnr].keyidx = keyidx;
        }
      } else if (jid == abstractkey[devnr].jid) {  /* add key to network-struct */
        VG3_nw_addkey(gmain->nwclnt, abstractkey[devnr].jid, abstractkey[devnr].kdef[ktnr].keyidx);
      }
    }
  }
}


/* call event-update, depending on network-game or not,
 * set in dowait, whether call VG3_wait_time()
 * return: 0 = ok or 1 = exit game
 */
int
event_update(struct g_main *gmain, int *dowait)
{
  if (gmain == NULL) { return 1; }

  if (gmain->nwclnt != NULL) {
    return VG3_nw_update(gmain->nwclnt, dowait);
  }
  if (dowait != NULL) { *dowait = 1; }
  return VG3_inputevent_update(gmain->wstruct);
}


/* check if a key is pressed
 * - plnumber:  player number (1 to MAXPLAYER)
 * - keyref:    abstract key (struct keydefs)
 * - flag:      kind of pressing (VGAG3_PRESSED)
 * return:      0 = key is not pressed, 1 = key is (new) pressed
 */
int
key_ispressed(struct g_main *gmain, int plnumber, int keyref, int flag)
{
  static time_t lastcall = 0;
  int ispressed, keyidx, devnr, ktnr, jid;

  if (gmain == NULL) { return 0; }
  if (plnumber < 1 || plnumber > MAXPLAYER) { return 0; }

  /* is player valid? */
  if (gmain->nwclnt != NULL) {  /* network active */
    if (gmain->player[plnumber - 1].ply != PLY_IS_LOCAL && gmain->player[plnumber - 1].ply != PLY_IS_REMOTE) {
      return 0;
    }
  } else {  /* no network */
    if (gmain->player[plnumber - 1].ply != PLY_IS_LOCAL) {
      return 0;
    }
  }
  if (!gmain->player[plnumber - 1].alive) { return 0; }  /* player dead */

  /* was system-menu called since last time? */
  if (lastcall < VG3_sysmenu_lastcall(gmain->sysm)) {
    nw_key(gmain, -1);
    lastcall = VG3_sysmenu_lastcall(gmain->sysm);
  }

  /* +++ get real key from keyref for player plnumber and check whether it is pressed +++ */

  ispressed = 0;

  for (devnr = 0; devnr <= MAX_DEVICES; devnr++) {  /* for each input-device (0 = keyboard, 1 to ? = joystick) */
    if (!abstractkey[devnr].enabled) { continue; }

    /* find struct for player's input-device
     * if gmain->player[plnumber - 1].jid is -1, then all input-devices are valid for the player (single-player game)
     */
    if (gmain->player[plnumber - 1].jid >= 0 && gmain->player[plnumber - 1].jid != abstractkey[devnr].jid) { continue; }

    /* find abstract key definition */
    for (ktnr = 0; abstractkey[devnr].kdef[ktnr].keyref != KEYDEF_NOKEY; ktnr++) {
      if (keyref == abstractkey[devnr].kdef[ktnr].keyref) { break; }
    }
    if (abstractkey[devnr].kdef[ktnr].keyref == KEYDEF_NOKEY) { continue; }  /* not found */

    /* get real key */
    jid = abstractkey[devnr].jid;
    keyidx = abstractkey[devnr].kdef[ktnr].keyidx;
    if (keyidx == VGAG3_KEY_NOKEY || keyidx == VGAG3_GC_NOKEY) { continue; }  /* not defined */

    /* is real key pressed? */
    if (VG3_nw_haskey(gmain->nwclnt, jid, keyidx)) {
      /* network is active and key was set with VG3_nw_addkey() */
      ispressed = VG3_nw_key_ispressed(gmain->nwclnt, plnumber, jid, keyidx, flag);
    } else {
      /* network is not active or key was not set with VG3_nw_addkey() */
      if (jid == 0) {
        ispressed = VG3_key_ispressed(gmain->wstruct, keyidx, flag);
      } else {
        ispressed = VG3_gamecontroller_ispressed(gmain->wstruct, jid, keyidx, flag);
      }
    }

    if (ispressed) { break; }
  }

  return ispressed;
}
