/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

extern struct vg3_ofunc * ofunc_new(void);

const char * get_oid_name(int);
const char * get_player_color(int);


/* get_oid_name:
 * returns object-id as a string
 * @param oidn  object-id (OID_NAMES)
 * @return  object-id as string (or empty)
 */
const char *
get_oid_name(int oidn)
{
  switch(oidn) {
    case OID_NAME_GROUND:  return "ground";
    case OID_NAME_WALL:    return "wall";
    case OID_NAME_TANK:    return "tank";
    case OID_NAME_SHOT:    return "shot";
  }
  return "";
}


/* get_player_color:
 * returns player color as a string
 * @param plcolor  player color (PLAYER_COLORS)
 * @return  player color as string (or empty)
 */
const char *
get_player_color(int plcolor)
{
  switch(plcolor) {
    case PLAYER_COLOR_RED:    return "red";
    case PLAYER_COLOR_YELLOW: return "yellow";
    case PLAYER_COLOR_GREEN:  return "green";
    case PLAYER_COLOR_BLUE:   return "blue";
  }
  return "";
}


int main(int argc, char **argv) {
  struct vg3_window *wstruct;
  struct g_main gmain;
  int retw, goend, scale, i1, dowait, use_network, nw_jid, geomw, geomh, bg_music;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_ofunc_object *objp;
  struct my_menu mymenu;

  scale = VGAG3_WINSCALE_NOSCALE;
  geomw = geomh = 0;

  opterr = opterr ? opterr : 1;
  while ((i1 = getopt(argc, argv, "+s:g:h")) != -1) {
    switch(i1) {
      case 's':  if (strcmp(optarg, "no") == 0) {
                   scale = VGAG3_WINSCALE_NOSCALE;
                 } else if (strcmp(optarg, "best") == 0) {
                   scale = VGAG3_WINSCALE_BESTSCALE;
                 } else if (strcmp(optarg, "max") == 0) {
                   scale = VGAG3_WINSCALE_MAXIMIZE;
                 } else if (strcmp(optarg, "full") == 0) {
                   scale = VGAG3_WINSCALE_FULLSCREEN;
                 }
                 break;
      case 'g':  { char *ptr = strchr(optarg, ':');
                   if (ptr != NULL) {
                     geomw = atoi(optarg);
                     geomh = atoi(ptr + 1);
                   } else {
                     geomw = atoi(optarg);
                   }
                 }
                 break;
      case 'h':
      default:  fprintf(stderr, "Usage: %s [<options>]\n", argv[0]);
                fprintf(stderr, "options:\n");
                fprintf(stderr, " -s <scaling>:   scale window according to <scaling>:\n");
                fprintf(stderr, "                 - no: no scaling\n");
                fprintf(stderr, "                 - best: best scaling\n");
                fprintf(stderr, "                 - max: maximize window\n");
                fprintf(stderr, "                 - full: fullscreen\n");
                fprintf(stderr, " -g <geometry>:  geometry: <width>:<height>\n");
                fprintf(stderr, "                 Size of playground in 128-pixel blocks\n");
                exit(1);
    }
  }

  if (geomw < 0 || geomw > 9) { geomw = 0; }
  if (geomh < 0 || geomh > 9) { geomh = 0; }

  /* open window */
  wstruct = VG3_window_new(argv[0], VGAG3_VGAVERSION_LOW, scale);
  if (wstruct == NULL) { fprintf(stderr, "%s\n", VG3_error()); exit(1); }

  /* don't capture mouse */
  VG3_mouserelease(wstruct, VGAG3_KEY_NOKEY);

  /* show intro */
  memset(&gmain, 0, sizeof(gmain));
  gmain.wstruct = wstruct;
  mymenu_readfromfile(&gmain, 1);  /* initial settings */
  VG3_film_play(wstruct, NULL, FILES_DIR "/film/intro.film", 1);

gostart:
  retw = 1;
  goend = 0;

  /* initialize game-struct */
  memset(&gmain, 0, sizeof(gmain));
  gmain.wstruct = wstruct;
  VG3_window_getsize(gmain.wstruct, &gmain.winw, &gmain.winh);
  gmain.nwclnt = NULL;

  /* initialize multi-language */
  gmain.mlang = VG3_multilang_new(NULL, "en");

  /* initialize system-menu (and additional menu) */
  memset(&mymenu, 0, sizeof(mymenu));
  mymenu.gmain = &gmain;
  gmain.sysm = VG3_sysmenu_new(gmain.wstruct, &mymenu, VG3_color_brightness(VGAG3_COLOR_GREEN, 50), VGAG3_COLOR_GREEN);
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/sysmenu");  /* add multi-language file for system-menu */

  /* read in saved system-menu values */
  mymenu_readfromfile(&gmain, 0);

  /* add another multi-language file */
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/game");

  /* insert exit-menu, volume-menu and window-menu into system-menu */
  VG3_sysmenu_simple_exitmenu(gmain.sysm, NULL, VG3_multilang_get(gmain.mlang, "exit-game"));
  VG3_sysmenu_simple_volumemenu(gmain.sysm, NULL,
    VG3_multilang_get(gmain.mlang, "main-vol"),
    VG3_multilang_get(gmain.mlang, "sound-vol"),
    VG3_multilang_get(gmain.mlang, "music-vol"),
    NULL
  );
  VG3_sysmenu_simple_windowmenu(gmain.sysm, NULL, VG3_multilang_get(gmain.mlang, "window-change"));

  /* insert additional menu: help */
  mymenu_insert_help(&gmain);

  /* show a demo */
  if (show_demo(&gmain)) { goto endgame; }

  /* get number of local players and their input-devices */
  for (i1 = 0; i1 < MAXPLAYER; i1++) { gmain.player[i1].ply = PLY_IS_NOTUSED; }
  if (player_select(&gmain, &use_network, &nw_jid)) { goto endgame; }

  /* insert input-devices into system-menu */
  menu_insert_input(&gmain, use_network);

  /* network */
  if (use_network) {
    int masterbyte;

    /* insert additional menu: network */
    mymenu_insert_network(&gmain);

    /* the geometry of the master shall be the geometry for all clients,
     * we use the masterbyte to inform all clients of the geometry if we are master,
     * we receive the masterbyte of the master and set the geometry
     */
    masterbyte = 1 + geomw * 10 + geomh;
    if (network_connect(&gmain, nw_jid, mymenu.network_name, &masterbyte)) { goto endgame; }
    masterbyte--;
    geomw = masterbyte / 10;
    geomh = masterbyte % 10;
  }

  /* create main-struct for object-functions */
  gmain.ofstruct = ofunc_new();

  /* create quadtree with dummy-size */
  gmain.hgrect.x = gmain.hgrect.y = 0;
  gmain.hgrect.w = gmain.hgrect.h = 1;
  gmain.qdtr = VG3_coll_q_new(&gmain.hgrect, 0, 0); 
  if (gmain.qdtr == NULL) { fprintf(stderr, "%s\n", VG3_error()); goto endgame; }

  /* create ground-object, which also recreates quadtree with correct size and creates wall-objects */
  ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_GROUND));
  if (ofc == NULL) { fprintf(stderr, "Need ground-object, but not found\n"); goto endgame; }
  if (ofc->f_new(&gmain, 0, geomw, geomh) == NULL) { fprintf(stderr, "%s\n", VG3_error()); goto endgame; }

  /* set not-used players to computer-players */
  for (i1 = 0; i1 < MAXPLAYER; i1++) {
    if (gmain.player[i1].ply == PLY_IS_NOTUSED) { gmain.player[i1].ply = PLY_IS_COMPUTER; }
  }

  /* create tanks and set remaining values of the players */
  ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_TANK));
  if (ofc == NULL) { fprintf(stderr, "Need tank-object, but not found\n"); goto endgame; }
  for (i1 = 0; i1 < MAXPLAYER; i1++) {
    if (gmain.player[i1].ply != PLY_IS_NOTUSED) {
      int xm, ym, rotdeg;
      gmain.player[i1].alive = 1;
      gmain.player[i1].hits = 0;
      gmain.player[i1].color = i1 + 1;  /* color of player according to player-number */
      if (i1 % 2 > 0) { xm = gmain.hgrect.w - 10; rotdeg = 270; } else { xm = 10; rotdeg = 90; }
      if (i1 / 2 > 0) { ym = gmain.hgrect.h - 10; } else { ym = 10; }
      objp = ofc->f_new(&gmain, 0, xm, ym, rotdeg, i1 + 1);
      if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); goto endgame; }
      gmain.player[i1].instanceid = objp->instanceid;
      if (*gmain.player[i1].name == '\0') {  /* set player name if still not set */
        snprintf(gmain.player[i1].name, sizeof(gmain.player[0].name), "%s-%d", VG3_multilang_get(gmain.mlang, "player"), i1 + 1);
      }
    }
  }
  gmain.followplayer = -1;  /* let the system decide which player to follow */

  /* load and start background music */
  bg_music = VG3_audio_load(gmain.wstruct, FILES_DIR "/sound/music.wav", 100, VGAG3_AUDIO_VOLUME_MUSIC);
  if (bg_music == 0) { fprintf(stderr, "loading music.wav: %s\n", VG3_error()); }
  if (bg_music > 0) {
    VG3_audio_play(gmain.wstruct, bg_music, 1, 1);
  }

  /* game loop */
  VG3_discard_input(gmain.wstruct);
  for (;;) {
    if (event_update(&gmain, &dowait)) { goend = 1; break; }

    /* ALT+Q: exit */
    if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
      && VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) { goend = 1; break; }

    /* ESC: system-menu */
    if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) {
      if (VG3_sysmenu_exec(gmain.sysm) > 0) { goend = 1; break; }
    }

    /* TAB: show player-names */
    if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_TAB, VGAG3_IS_NEW_PRESSED)) {
      if (gmain.shownames > 0) { gmain.shownames = 0; } else { gmain.shownames = 50; }
    } else if (gmain.shownames > 0) {
      gmain.shownames--;
    }

    /* P: pause game */
    if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_P, VGAG3_IS_NEW_PRESSED)) {
      pause_game(&gmain);
    }

    VG3_draw_clear(gmain.wstruct, NULL, VGAG3_COLOR_BLACK);

    /* execute run-function of loaded object-instances */
    VG3_ofunc_objlist_call_run(gmain.ofstruct, &gmain);

    /* execute draw-function of loaded object-instances */
    VG3_ofunc_objlist_call_draw(gmain.ofstruct, &gmain);

    VG3_window_update(gmain.wstruct, 0, 0);

    if (dowait) {
      /* count players and exit game if needed */
      static int countdown_alives = 0;
      int real_alives = 0, computer_alives = 0;
      if (countdown_alives > 0) {  /* wait a bit before showing winner */
        if (--countdown_alives == 0) {
          show_winner(&gmain);
          break;
        }
      } else {
        for (i1 = 0; i1 < MAXPLAYER; i1++) {
          if (gmain.player[i1].ply != PLY_IS_NOTUSED && gmain.player[i1].alive > 0) {
            if (gmain.player[i1].ply == PLY_IS_LOCAL || gmain.player[i1].ply == PLY_IS_REMOTE) {
              real_alives++;
            } else if (gmain.player[i1].ply == PLY_IS_COMPUTER) {
              computer_alives++;
            }
          }
        }
        if (real_alives == 0) { countdown_alives = 30; }
        if (real_alives == 1 && computer_alives == 0) { countdown_alives = 30; }
      }
      /* wait rest of wait-time */
      VG3_wait_time(50);
    }
  }
  VG3_discard_input(gmain.wstruct);

  if (!goend) {  /* stop background music with descrescendo, fade out, and unload background music */
    struct vg3_image_attributes wattr, iattr;
    if (bg_music > 0) {
      VG3_audio_stop(gmain.wstruct, bg_music, 1);
    }
    VG3_window_attributes(gmain.wstruct, NULL, &wattr, -1, NULL);  /* save window-settings */
    for (iattr = wattr; ;) {  /* fade out window */
      if (iattr.bright >= 2) { iattr.bright -= 2; } else { iattr.bright = 0; }
      VG3_window_attributes(gmain.wstruct, &iattr, NULL, -1, NULL);
      VG3_window_update(gmain.wstruct, 0, 0);
      VG3_wait_time(50);
      if (bg_music > 0) {  /* wait until music ended */
        if (!VG3_audio_isplaying(gmain.wstruct, bg_music)) { break; }
      } else {  /* no music: wait until window is dark */
        if (iattr.bright == 0) { break; }
      }
    }
    VG3_window_attributes(gmain.wstruct, &wattr, NULL, -1, NULL);  /* reset window to saved settings */
    if (bg_music > 0) {
      VG3_audio_unload(gmain.wstruct, bg_music);
    }
  } else {  /* stop and unload background music at once */
    if (bg_music > 0) {
      VG3_audio_stop(gmain.wstruct, bg_music, 0);
      VG3_audio_unload(gmain.wstruct, bg_music);
    }
  }

  /* unload and free still existing object-instances */
  VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, NULL);

  /* free quadtree */
  VG3_coll_q_free(gmain.qdtr);

  /* free main-struct for object-functions */
  VG3_ofunc_free(gmain.ofstruct);

  /* unload still existing images */
  VG3_image_unload(gmain.wstruct, NULL);

  retw = 0;
endgame:
  /* disconnect from network */
  if (gmain.nwclnt != NULL) { VG3_nw_close(gmain.nwclnt); }

  /* write out actual system-menu values */
  { char *ptr;
    ptr = VG3_sysmenu_savestring_insert(gmain.sysm, gmain.sysm_string);
    if (gmain.sysm_string != NULL) { free(gmain.sysm_string); }
    gmain.sysm_string = ptr;
    mymenu_savetofile(&gmain);
    free(gmain.sysm_string);
  }

  /* free system-menu */
  VG3_sysmenu_free(gmain.sysm);

  /* free multi-language */
  VG3_multilang_free(gmain.mlang);

  /* go to beginning? */
  if (retw == 0 && !goend) { goto gostart; }

  /* close window */
  VG3_window_free(gmain.wstruct);

  exit(retw);
}
