/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

extern struct vg3_ofunc * ofunc_new(void);

void draw_lastline(struct g_main *, const char *, struct vg3_rect, int);
void show_winner(struct g_main *);
int show_demo(struct g_main *);


/* draw last line - system-menu option */
void
draw_lastline(struct g_main *gmain, const char *fontfile, struct vg3_rect brect, int fcolor)
{
  int h_size;
  struct vg3_text stxt;
  char buf[64];

  if (gmain == NULL) { return; }

  VG3_fontsize(gmain->wstruct, fontfile, NULL, &h_size, NULL, NULL);

  brect.y += (brect.h - h_size);

  snprintf(buf, sizeof(buf), "ESC: %s", VG3_multilang_get(gmain->mlang, "to-sysmenu"));
  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, (char *)fontfile, 0, 0, buf);
  VG3_draw_textline(gmain->wstruct, NULL, &brect, 0, &stxt, fcolor, VGAG3_COLOR_TRANSPARENT, 0, 1, 0);
}


/* show the winner of the game */
void
show_winner(struct g_main *gmain)
{
  char buf[128], *name;
  int i1, hits;
  struct vg3_image *img;
  struct vg3_image_attributes iattr;

  if (gmain == NULL) { return; }

  for (i1 = 0; i1 < MAXPLAYER; i1++) {
    if (gmain->player[i1].ply == PLY_IS_LOCAL || gmain->player[i1].ply == PLY_IS_REMOTE) {
      if (gmain->player[i1].alive > 0) { break; }
    }
  }

  if (i1 == MAXPLAYER) { return; }  /* no winner */

  snprintf(buf, sizeof(buf), "%s/bmp/tank-%s1.bmp", FILES_DIR, get_player_color(gmain->player[i1].color));
  img = VG3_image_load(gmain->wstruct, buf, 0);
  if (img == NULL) { return; }

  name = gmain->player[i1].name;
  hits = gmain->player[i1].hits;

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
  for (i1 = 0; i1 < 80; i1++) {

    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    VG3_image_copy(gmain->wstruct, NULL, img, gmain->winw / 2, gmain->winh / 2, &iattr, 0);

    if (i1 < 20) {
      iattr.zoom += 20;
    } else {
      struct vg3_text stxt;
      struct vg3_rect brect;
      brect.x = 10;
      brect.y = 150;
      brect.w = gmain->winw - 20;
      brect.h = gmain->winh - 20;
      snprintf(buf, sizeof(buf), "%s: %s", VG3_multilang_get(gmain->mlang, "winner"), name);
      VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, 0, 0, buf);
      VG3_draw_textline(gmain->wstruct, NULL, &brect, 0, &stxt, VGAG3_COLOR_WHITE, VGAG3_COLOR_TRANSPARENT, 0, 0, 1);
      snprintf(buf, sizeof(buf), "%d %s", hits, VG3_multilang_get(gmain->mlang, "hits"));
      VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, 0, 0, buf);
      VG3_draw_textline(gmain->wstruct, NULL, &brect, 0, &stxt, VGAG3_COLOR_WHITE, VGAG3_COLOR_TRANSPARENT, 0, 0, 1);
    }

    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(50);
  }

  VG3_image_unload(gmain->wstruct, img);
}


/* show game-demo */
int
show_demo(struct g_main *gmain)
{
  int retw, sndvol, i1, dowait, blinking;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_ofunc_object *objp;
  struct vg3_text stxt;
  struct vg3_rect brect;
  char buf[128];

  if (gmain == NULL) { return 0; }

demostart:
  /* create main-struct for object-functions */
  gmain->ofstruct = ofunc_new();

  /* create quadtree with dummy-size */
  gmain->hgrect.x = gmain->hgrect.y = 0;
  gmain->hgrect.w = gmain->hgrect.h = 1;
  gmain->qdtr = VG3_coll_q_new(&gmain->hgrect, 0, 0); 
  if (gmain->qdtr == NULL) { fprintf(stderr, "%s\n", VG3_error()); return 1; }

  /* create ground-object, which also recreates quadtree with correct size and creates wall-objects */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_GROUND));
  if (ofc == NULL) { fprintf(stderr, "Need ground-object, but not found\n"); return 1; }
  if (ofc->f_new(gmain, 0, 2, 1) == NULL) { fprintf(stderr, "%s\n", VG3_error()); return 1; }

  /* set all players to computer-players */
  for (i1 = 0; i1 < MAXPLAYER; i1++) { gmain->player[i1].ply = PLY_IS_COMPUTER; }

  /* create tanks and set remaining values of the players */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_TANK));
  if (ofc == NULL) { fprintf(stderr, "Need tank-object, but not found\n"); return 1; }
  for (i1 = 0; i1 < MAXPLAYER; i1++) {
    if (gmain->player[i1].ply != PLY_IS_NOTUSED) {
      int xm, ym, rotdeg;
      gmain->player[i1].alive = 1;
      gmain->player[i1].hits = 0;
      gmain->player[i1].color = i1 + 1;  /* color of player according to player-number */
      if (i1 % 2 > 0) { xm = gmain->hgrect.w - 10; rotdeg = 270; } else { xm = 10; rotdeg = 90; }
      if (i1 / 2 > 0) { ym = gmain->hgrect.h - 10; } else { ym = 10; }
      objp = ofc->f_new(gmain, 0, xm, ym, rotdeg, i1 + 1);
      if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); return 1; }
      gmain->player[i1].instanceid = objp->instanceid;
    }
  }
  gmain->followplayer = -1;  /* let the system decide which player to follow */

  /* set volume to being quiet and save actual value */
  sndvol = VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_ALL, 0);

  /* game-demo loop */
  VG3_discard_input(gmain->wstruct);
  retw = 0;
  blinking = 0;
  for (;;) {
    if (event_update(gmain, &dowait)) { retw = 1; break; }

    /* ALT+Q: exit */
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
      && VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) { retw = 1; break; }

    /* ESC: system-menu */
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) {
      VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_ALL, sndvol);
      if (VG3_sysmenu_exec(gmain->sysm) > 0) { retw = 1; break; }
      sndvol = VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_ALL, 0);
    }

    /* end demo */
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) { break; }

    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);

    /* execute run-function of loaded object-instances */
    VG3_ofunc_objlist_call_run(gmain->ofstruct, gmain);

    /* execute draw-function of loaded object-instances */
    VG3_ofunc_objlist_call_draw(gmain->ofstruct, gmain);

    /* draw blinking text and last line */
    brect.x = 0;
    brect.y = gmain->winh / 2 - 4;
    brect.w = gmain->winw;
    brect.h = gmain->winh - brect.y;
    if (++blinking % 20 >= 10) {
      snprintf(buf, sizeof(buf), "%s", VG3_multilang_get(gmain->mlang, "game-demo"));
      VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, 0, 0, buf);
      VG3_draw_textline(gmain->wstruct, NULL, &brect, 0, &stxt, VGAG3_COLOR_WHITE, VGAG3_COLOR_TRANSPARENT, 0, 0, 1);
    }
    draw_lastline(gmain, NULL, brect, VGAG3_COLOR_WHITE);

    VG3_window_update(gmain->wstruct, 0, 0);

    if (dowait) {
      /* count players and exit game-demo if needed */
      int alives = 0;
      for (i1 = 0; i1 < MAXPLAYER; i1++) {
        if (gmain->player[i1].ply != PLY_IS_NOTUSED && gmain->player[i1].alive > 0) { alives++; }
      }
      if (alives <= 1) { retw = 2; break; }
      /* wait rest of wait-time */
      VG3_wait_time(50);
    }
  }

  VG3_discard_input(gmain->wstruct);

  /* set volume to saved value */
  VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_ALL, sndvol);

  /* unload and free still existing object-instances */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, NULL);

  /* free quadtree */
  VG3_coll_q_free(gmain->qdtr);

  /* free main-struct for object-functions */
  VG3_ofunc_free(gmain->ofstruct);

  /* unload still existing images */
  VG3_image_unload(gmain->wstruct, NULL);

  if (retw == 2) { goto demostart; }

  return retw;
}
