/* Copyright 2017-2022 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

void mymenu_insert_help(struct g_main *);
void mymenu_insert_network(struct g_main *);
void mymenu_savetofile(struct g_main *);
void mymenu_readfromfile(struct g_main *, int);

static void getf_nw_name(struct vg3_sysmenu *, int, int, struct vg3_text *, struct vg3_text *);
static int setf_nw_name(struct vg3_sysmenu *, int, int, int, int);
static size_t put_letter(int, char *, size_t);

static char keybuf[64], valbuf[64];
static const char *sv_file = FILES_DIR "/settings.save";

/* add additional submenu: help */
void
mymenu_insert_help(struct g_main *gmain)
{
  struct vg3_text skey, stxt;

  if (gmain == NULL) { return; }

  /* add multi-language help-file for menu */
  VG3_multilang_add(gmain->mlang, FILES_DIR "/mlang/help");

  /* textentry: instructions */
  VGAG3_TEXT_ATTRIBUTES_SET(&skey, "10x17.font", 0, 0, VG3_multilang_get(gmain->mlang, "help-title"));
  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, VG3_multilang_get(gmain->mlang, "help-text"));
  VG3_sysmenu_insert_textentry(gmain->sysm, NULL, &skey, &stxt, VG3_color_brightness(VGAG3_COLOR_REDORANGE, 50));
}


/* add additional submenu: network */
void
mymenu_insert_network(struct g_main *gmain)
{
  struct vg3_text stxt;
  struct vg3_sysmenu_submenu *subm;

  if (gmain == NULL) { return; }

  /* network-menu */
  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, "10x17.font", 0, 0, VG3_multilang_get(gmain->mlang, "network"));
  subm = VG3_sysmenu_insert_menuentry(gmain->sysm, NULL, &stxt);

  /* dataentry for network-menu: network name */
  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, 0, 0, VG3_multilang_get(gmain->mlang, "name-input"));
  VG3_sysmenu_insert_dataentry(gmain->sysm, subm, 0, 0, setf_nw_name, getf_nw_name, &stxt, 0);
}


/* save system-menu settings to file */
void
mymenu_savetofile(struct g_main *gmain)
{
  FILE *ffp;
  struct my_menu *mymenu;
  int ival;

  if (gmain == NULL || gmain->sysm == NULL) { return; }

  if ((ffp = fopen(sv_file, "w")) == NULL) { return; }

  /* write window-setting */
  ival = VG3_window_change(gmain->wstruct, -1);
  fprintf(ffp, "%d\n", ival);

  /* write volume-settings */
  ival = VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_ALL, -1);
  fprintf(ffp, "%d\n", ival);
  ival = VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_SOUND, -1);
  fprintf(ffp, "%d\n", ival);
  ival = VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_MUSIC, -1);
  fprintf(ffp, "%d\n", ival);
  ival = VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_SPEECH, -1);
  fprintf(ffp, "%d\n", ival);
  ival = VG3_audio_mute(gmain->wstruct, -1);
  fprintf(ffp, "%d\n", ival);

  /* write input-device settings */
  fprintf(ffp, "%s\n", (gmain->sysm_string == NULL ? "" : gmain->sysm_string));

  /* write network-name */
  mymenu = (struct my_menu *)gmain->sysm->vdata;
  if (mymenu != NULL) {
    fprintf(ffp, "%s\n", mymenu->network_name);
  } else {
    fprintf(ffp, "\n");
  }

  fclose(ffp);
}


/* read system-menu settings from file */
void
mymenu_readfromfile(struct g_main *gmain, int initial)
{
  FILE *ffp;
  struct my_menu *mymenu;
  int ival;
  char buf[4096];
  size_t blen;

  if (gmain == NULL) { return; }

  if ((ffp = fopen(sv_file, "r")) == NULL) { return; }

  /* read in window-setting */
  if (fgets(buf, sizeof(buf), ffp) == NULL) { fclose(ffp); return; }
  ival = atoi(buf);
  VG3_window_change(gmain->wstruct, ival);

  /* read and set volume-settings */
  /* all */
  if (fgets(buf, sizeof(buf), ffp) == NULL) { fclose(ffp); return; }
  ival = atoi(buf);
  VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_ALL, ival);
  /* sound */
  if (fgets(buf, sizeof(buf), ffp) == NULL) { fclose(ffp); return; }
  ival = atoi(buf);
  VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_SOUND, ival);
  /* music */
  if (fgets(buf, sizeof(buf), ffp) == NULL) { fclose(ffp); return; }
  ival = atoi(buf);
  VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_MUSIC, ival);
  /* speech */
  if (fgets(buf, sizeof(buf), ffp) == NULL) { fclose(ffp); return; }
  ival = atoi(buf);
  VG3_audio_volume(gmain->wstruct, VGAG3_AUDIO_VOLUME_SPEECH, ival);
  /* mute */
  if (fgets(buf, sizeof(buf), ffp) == NULL) { fclose(ffp); return; }
  ival = atoi(buf);
  VG3_audio_mute(gmain->wstruct, ival);

  if (initial || gmain->sysm == NULL) { fclose(ffp); return; }

  /* read input-device settings */
  if (fgets(buf, sizeof(buf), ffp) == NULL) { fclose(ffp); return; }
  blen = strlen(buf);
  if (blen > 0 && buf[blen - 1] == '\n') { blen--; }
  if (blen > 0 && buf[blen - 1] == '\r') { blen--; }
  buf[blen] = '\0';
  gmain->sysm_string = strdup(buf);

  /* read in network-name */
  if (fgets(buf, sizeof(buf), ffp) == NULL) { fclose(ffp); return; }
  mymenu = (struct my_menu *)gmain->sysm->vdata;
  if (mymenu != NULL) {
    blen = strlen(buf);
    if (blen > 0 && buf[blen - 1] == '\n') { blen--; }
    if (blen > 0 && buf[blen - 1] == '\r') { blen--; }
    if (blen >= sizeof(mymenu->network_name)) { blen = sizeof(mymenu->network_name) - 1; }
    buf[blen] = '\0';
    memmove(mymenu->network_name, buf, blen + 1);
  }

  fclose(ffp);
}
  

/* get key and value for network name */
static void
getf_nw_name(struct vg3_sysmenu *sysm, int inr, int jid, struct vg3_text *rkey, struct vg3_text *rval)
{
  struct my_menu *mymenu;

  if (sysm == NULL || rkey == NULL || rval == NULL) { return; }

  mymenu = (struct my_menu *)sysm->vdata;
  if (mymenu == NULL) { return; }

  /* get key: title for network name */
  VGAG3_TEXT_ATTRIBUTES_SET(rkey, NULL, '\n', 0, keybuf);
  snprintf(keybuf, sizeof(keybuf), "%s", VG3_multilang_get(mymenu->gmain->mlang, "network-name"));

  /* get value: network name */
  VGAG3_TEXT_ATTRIBUTES_SET(rval, NULL, 0, 0, valbuf);
  snprintf(valbuf, sizeof(valbuf), "%s", mymenu->network_name);
  
  (void)inr;
  (void)jid;
}


/* set value for network name
 * @return  0 = ok, 1 = return to parent-menu, 2 = exit game
 */
static int
setf_nw_name(struct vg3_sysmenu *sysm, int inr, int key_pressed, int jid, int gc_pressed)
{
  struct my_menu *mymenu;

  if (sysm == NULL) { return 0; }

  mymenu = (struct my_menu *)sysm->vdata;
  if (mymenu == NULL) { return 1; }

  if (key_pressed == VGAG3_KEY_NOKEY) { return 0; }  /* first call */
  if (key_pressed == VGAG3_KEY_ENTER) { return 1; }  /* return */

  /* line input for network name */
  put_letter(key_pressed, mymenu->network_name, sizeof(mymenu->network_name));

  (void)inr;
  (void)jid;
  (void)gc_pressed;
  return 0;
}


/* do very simple line editing
 * put representation of key_pressed into rbuf with size rsize
 */
static size_t
put_letter(int key_pressed, char *rbuf, size_t rsize)
{
  static char zchlast = '\0';
  static struct timeval tvlast;
  char zch = '\0';
  struct timeval tv;
  size_t rlen;

  if (rbuf == NULL || rsize == 0) { return 0; }

  rlen = strlen(rbuf);

  switch(key_pressed) {
    case VGAG3_KEY_0: zch = '0'; break;
    case VGAG3_KEY_1: zch = '1'; break;
    case VGAG3_KEY_2: zch = '2'; break;
    case VGAG3_KEY_3: zch = '3'; break;
    case VGAG3_KEY_4: zch = '4'; break;
    case VGAG3_KEY_5: zch = '5'; break;
    case VGAG3_KEY_6: zch = '6'; break;
    case VGAG3_KEY_7: zch = '7'; break;
    case VGAG3_KEY_8: zch = '8'; break;
    case VGAG3_KEY_9: zch = '9'; break;
    case VGAG3_KEY_A: zch = 'A'; break;
    case VGAG3_KEY_B: zch = 'B'; break;
    case VGAG3_KEY_C: zch = 'C'; break;
    case VGAG3_KEY_D: zch = 'D'; break;
    case VGAG3_KEY_E: zch = 'E'; break;
    case VGAG3_KEY_F: zch = 'F'; break;
    case VGAG3_KEY_G: zch = 'G'; break;
    case VGAG3_KEY_H: zch = 'H'; break;
    case VGAG3_KEY_I: zch = 'I'; break;
    case VGAG3_KEY_J: zch = 'J'; break;
    case VGAG3_KEY_K: zch = 'K'; break;
    case VGAG3_KEY_L: zch = 'L'; break;
    case VGAG3_KEY_M: zch = 'M'; break;
    case VGAG3_KEY_N: zch = 'N'; break;
    case VGAG3_KEY_O: zch = 'O'; break;
    case VGAG3_KEY_P: zch = 'P'; break;
    case VGAG3_KEY_Q: zch = 'Q'; break;
    case VGAG3_KEY_R: zch = 'R'; break;
    case VGAG3_KEY_S: zch = 'S'; break;
    case VGAG3_KEY_T: zch = 'T'; break;
    case VGAG3_KEY_U: zch = 'U'; break;
    case VGAG3_KEY_V: zch = 'V'; break;
    case VGAG3_KEY_W: zch = 'W'; break;
    case VGAG3_KEY_X: zch = 'X'; break;
    case VGAG3_KEY_Y: zch = 'Y'; break;
    case VGAG3_KEY_Z: zch = 'Z'; break;
    case VGAG3_KEY_SPACE: zch = ' '; break;
    case VGAG3_KEY_BACKSPACE: zch = '\r'; break;
  }

  /* if the same key as last time, care for some time between */
  gettimeofday(&tv, NULL);
  if (zch != '\0' && zchlast == zch) {
    time_t t1 = (tv.tv_sec - tvlast.tv_sec) % 1000 * 1000 + (tv.tv_usec / 1000 - tvlast.tv_usec / 1000);
    if (t1 < 300) { zch = '\0'; } else { tvlast = tv; }
  } else {
    zchlast = zch;
    tvlast = tv;
  }

  if (zch != '\0') {
    if (zch == '\r') {  /* backspace */
      if (rlen > 0) { rlen--; }
    } else if (rlen < rsize - 1) {
      rbuf[rlen++] = zch;
    }
    rbuf[rlen] = '\0';
  }

  return rlen;
}
