/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

int network_connect(struct g_main *, int, const char *, int *);
int network_is_client_connected(struct g_main *, int);
void pause_game(struct g_main *);

static int ask_master(struct g_main *, int *);
static int ask_client(struct g_main *, char *, size_t);
static int infotext(struct g_main *, int, const char *);
static int waitit(struct g_main *, int);

static char *bgfile = "files/bmp/tank-bg.bmp";
static struct vg3_image *bgimage = NULL;
static struct vg3_image_attributes bgattr;


/* ask whether network-master and connect to network
 * @return  0 = ok, 1 = exit
 */
int
network_connect(struct g_main *gmain, int nw_jid, const char *network_name, int *masterbyte)
{
  static int fg_color = VGAG3_COLOR_YELLOW;  /* foreground color */
  int master, retw, plnumber, plmax, i1;
  char nwservername[64], *nwserverip, buf[512];
  size_t blen;

  if (gmain == NULL || masterbyte == NULL) { fprintf(stderr, "network_connect: parameter error\n"); return 1; }

  retw = 0;
  nwserverip = NULL;
  master = 0;
  gmain->nwclnt = NULL;

  /* add multi-language file for network connection */
  VG3_multilang_add(gmain->mlang, FILES_DIR "/mlang/network");

  /* load background image */
  bgimage = VG3_image_load(gmain->wstruct, bgfile, 1);
  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&bgattr);
  bgattr.bright = 128;

  /* want to be network-master? */
  if (ask_master(gmain, &master) > 0) { retw = 1; goto endnw; }

  /* network-master starts broadcast-/multicast-server and network-server */
  *nwservername = '\0';
  if (master) {
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    if (bgimage != NULL) {
      VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, &bgattr, 0);
    }
    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_nw_mbcast_start();
    VG3_nw_server(NULL);
  } else {
    if (ask_client(gmain, nwservername, sizeof(nwservername)) > 0) { retw = 1; goto endnw; }
    *masterbyte = 0;
  }

  /* get ip of network-server via broadcast-/multicast-server */
  if (infotext(gmain, fg_color, VG3_multilang_get(gmain->mlang, "get-ip-of-nwserver"))) { retw = 1; goto endnw; }
  if (*nwservername == '\0') {
    nwserverip = VG3_nw_mbcast_getip(nwservername, sizeof(nwservername));
  } else {
    nwserverip = strdup(nwservername);
    if (nwserverip == NULL) { VG3_seterror(ENOMEM, "allocation error"); }
  }
  if (nwserverip == NULL) { fprintf(stderr, "%s\n", VG3_error()); retw = 1; goto endnw; }

  /* show ip or network-server */
  snprintf(buf, sizeof(buf), "%s: %s", nwservername, nwserverip);
  if (infotext(gmain, fg_color, buf)) { retw = 1;  goto endnw; }
  if (waitit(gmain, 3)) { retw = 1;  goto endnw; }

  /* open network and set network-keys */
  gmain->nwclnt = VG3_nw_open(gmain->wstruct);
  nw_key(gmain, nw_jid);

  /* prepare connection background */
  if (VG3_inputevent_update(gmain->wstruct)) { retw = 1; goto endnw; }
  VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
  if (bgimage != NULL) {
    VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, &bgattr, 0);
  }
  VG3_window_update(gmain->wstruct, 0, 0);

  /* connect to network-server */
  if (network_name != NULL && *network_name == '\0') { network_name = NULL; }
  plnumber = VG3_nw_connect(gmain->nwclnt, nwserverip, NULL, network_name, MAXPLAYER, masterbyte);
  if (plnumber < 0) { fprintf(stderr, "%s\n", VG3_error()); retw = 1; goto endnw; }
  if (plnumber == 0) { retw = 1; goto endnw; }
  if (VG3_inputevent_update(gmain->wstruct)) { retw = 1; goto endnw; }

  /* set player values for each network player */
  plmax = VG3_nw_numberofclients(gmain->nwclnt);
  for (i1 = 1; i1 <= plmax; i1++) {
    const char *name;
    if (VG3_nw_getclientinfo(gmain->nwclnt, i1, &name)) {
      snprintf(gmain->player[i1 - 1].name, sizeof(gmain->player[0].name), "%s", name);
    }
    if (i1 == plnumber) {  /* myself */
      gmain->player[i1 - 1].ply = PLY_IS_LOCAL;
    } else {  /* remote player */
      gmain->player[i1 - 1].ply = PLY_IS_REMOTE;
    }
    gmain->player[i1 - 1].jid = nw_jid;
  }

  /* show infos */
  blen = 0;
  snprintf(buf + blen, sizeof(buf) - blen, "%s %s\n", VG3_multilang_get(gmain->mlang, "connected-to"), nwservername);
  blen = strlen(buf);
  snprintf(buf + blen, sizeof(buf) - blen, "\n");
  blen = strlen(buf);
  for (i1 = 1; i1 <= plmax; i1++) {
    snprintf(buf + blen, sizeof(buf) - blen, "%s %d: %s\n", VG3_multilang_get(gmain->mlang, "player"), i1, gmain->player[i1 - 1].name);
    blen = strlen(buf);
  }
  if (infotext(gmain, fg_color, buf)) { retw = 1;  goto endnw; }
  if (waitit(gmain, 5)) { retw = 1;  goto endnw; }

endnw:
  if (master) { VG3_nw_mbcast_stop(); }
  if (nwserverip != NULL) { free(nwserverip); }
  if (bgimage != NULL) { VG3_image_unload(gmain->wstruct, bgimage); }

  return retw;
}


/* is (network-)player still connected? */
int
network_is_client_connected(struct g_main *gmain, int plnumber)
{
  if (gmain == NULL) { return 0; }
  if (gmain->nwclnt == NULL) { return 1; }
  if (gmain->player[plnumber - 1].ply == PLY_IS_COMPUTER) { return 1; }
  if (VG3_nw_getclientinfo(gmain->nwclnt, plnumber, NULL)) { return 1; }
  return 0;
}


/* pause game depending on network-game or local game */
void
pause_game(struct g_main *gmain)
{
  if (gmain == NULL) { return; }
  if (gmain->nwclnt == NULL) {
    VG3_pause(gmain->wstruct, NULL, VGAG3_KEY_NOKEY);
  } else {
    VG3_nw_pause(gmain->nwclnt);
  }
}


/* be network-master?
 * @return  0 = ok, 1 = exit
 */
static int
ask_master(struct g_main *gmain, int *master)
{
  char **selarray;
  int erg;

  if (gmain == NULL || master == NULL) { return 1; }

  selarray = calloc(3, sizeof(*selarray));
  if (selarray == NULL) { fprintf(stderr, "%s\n", strerror(errno)); return 1; }
  selarray[0] = (char *)VG3_multilang_get(gmain->mlang, "no-master");
  selarray[1] = (char *)VG3_multilang_get(gmain->mlang, "yes-master");
  selarray[2] = NULL;

  /* loop: select from options */
  *master = 0;
  for (;;) {
    /* draw background */
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    if (bgimage != NULL) {
      VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, NULL, 0);
    }
    /* call select-menu */
    erg = VG3_input_select(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, VG3_multilang_get(gmain->mlang, "network-game"), selarray, 0);
    if (erg < 0) {
      if (erg == -1) { fprintf(stderr, "%s\n", VG3_error()); }
      erg = 1;
      break;
    } else if (erg == 0) {
      if (VG3_sysmenu_exec(gmain->sysm) > 0) { erg = 1; break; }
    } else {
      if (erg == 2) { *master = 1; } else { *master = 0; }  /* interpret selected option */
      erg = 0;
      break;
    }
  }

  /* free array */
  free(selarray);

  return erg;
}


/* connect via broad-/multicast or to a specific host?
 * @return  0 = ok, 1 = exit
 */
static int
ask_client(struct g_main *gmain, char *rhost, size_t hsize)
{
  char **selarray;
  int erg;

  if (gmain == NULL || rhost == NULL || hsize == 0) { return 0; }

  selarray = calloc(3, sizeof(*selarray));
  if (selarray == NULL) { fprintf(stderr, "%s\n", strerror(errno)); return 1; }
  selarray[0] = (char *)VG3_multilang_get(gmain->mlang, "yes-usebc");
  selarray[1] = (char *)VG3_multilang_get(gmain->mlang, "no-usebc");
  selarray[2] = NULL;

  /* loop: select from options */
  *rhost = '\0';
  for (;;) {
    /* draw background */
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    if (bgimage != NULL) {
      VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, NULL, 0);
    }
    /* call select-menu */
    erg = VG3_input_select(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, VG3_multilang_get(gmain->mlang, "sel-nwserver"), selarray, 0);
    if (erg < 0) {
      if (erg == -1) { fprintf(stderr, "%s\n", VG3_error()); }
      erg = 1;
      break;
    } else if (erg == 0) {
      if (VG3_sysmenu_exec(gmain->sysm) > 0) { erg = 1; break; }
    } else {
      if (erg == 2) {  /* input host name */
        erg = VG3_input_box(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE, VG3_multilang_get(gmain->mlang, "no-usebc"), rhost, hsize);
        if (erg < 0) {
          if (erg == -1) { fprintf(stderr, "%s\n", VG3_error()); }
          erg = 1;
        } else if (erg > 0) {
          erg = 0;
        } else {
          continue;
        }
      } else {
        erg = 0;
      }
      break;
    }
  }

  /* free array */
  free(selarray);

  return erg;
}


static int
infotext(struct g_main *gmain, int fcolor, const char *text)
{
  struct vg3_rect rect;
  struct vg3_text stxt;

  if (gmain == NULL || text == NULL || *text == '\0') { return 0; }

  if (VG3_inputevent_update(gmain->wstruct)) { return 1; }

  /* draw background */
  VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
  if (bgimage != NULL) {
    VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, &bgattr, 0);
  }

  /* draw info centered */

  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, text);

  rect.x = rect.y = 0;
  rect.w = gmain->winw; rect.h = gmain->winh;
  rect = VG3_draw_text(gmain->wstruct, NULL, &rect, 0, &stxt, fcolor, VGAG3_COLOR_TRANSPARENT, 1);

  rect.x = gmain->winw / 2 - rect.w / 2; rect.y = gmain->winh / 2 - rect.h / 2;
  VG3_draw_text(gmain->wstruct, NULL, &rect, 0, &stxt, fcolor, VGAG3_COLOR_TRANSPARENT, 0);

  rect.x--; rect.y--; rect.w += 2; rect.h += 2;
  VG3_draw_rect(gmain->wstruct, NULL, &rect, 0, VGAG3_COLOR_YELLOW);

  VG3_window_update(gmain->wstruct, 0, 0);
  return 0;
}


/* wait sek seconds, check for Key */
static int
waitit(struct g_main *gmain, int sek)
{
  if (gmain == NULL) { return 0; }

  VG3_discard_input(gmain->wstruct);
  sek *= 1000;
  for (; sek > 0; sek -= 100) {
    if (VG3_inputevent_update(gmain->wstruct)) { return 1; }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) { break; }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_ENTER, VGAG3_IS_NEW_PRESSED)) { break; }
    VG3_wait_time(100);
  }
  VG3_discard_input(gmain->wstruct);

  return 0;
}
