/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-ground.h"
#include "obj-wall.h"

void getofc_ground(struct vg3_ofunc_objfunc *);
void getofc_wall(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new_ground(void *, unsigned int, ...);
static void f_free_ground(void *, struct vg3_ofunc_object *);
static void f_draw_ground(void *, struct vg3_ofunc_object *);
static struct vg3_ofunc_object * f_new_wall(void *, unsigned int, ...);
static void f_free_wall(void *, struct vg3_ofunc_object *);

/* positions of walls */
#define MAX_GR_WALL 12
#define MAX_GR_WALLELEM 8
static struct {
  int direction;    /* 0 = unused, 1 = left - right, 2 = top - down */
  int x, y;         /* start-position: from 0 to 15 */
  int elements;     /* number of wall-elements */
} walls[MAX_GR_WALL][MAX_GR_WALLELEM] = {
  {
    { 1, 4, 2, 8 },
    { 2, 1, 5, 4 },
    { 1, 2, 8, 3 },
    { 1, 10, 9, 2 },
    { 2, 11, 10, 4 },
    { 2, 6, 11, 3 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 7, 8, 6 },
    { 2, 12, 9, 3 },
    { 2, 2, 11, 4 },
    { 1, 3, 14, 4 },
    { 2, 5, 2, 5 },
    { 2, 13, 3, 2 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 4, 3, 4 },
    { 2, 4, 4, 3 },
    { 2, 7, 4, 3 },
    { 1, 4, 7, 4 },
    { 1, 9, 13, 5 },
    { 2, 13, 11, 2 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 2, 4, 5, 8 },
    { 2, 7, 5, 8 },
    { 1, 11, 3, 2 },
    { 2, 13, 3, 5 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 2, 1, 1 },
    { 2, 4, 5, 4 },
    { 1, 9, 7, 5 },
    { 1, 7, 13, 4 },
    { 2, 10, 11, 2 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 4, 4, 2 },
    { 1, 4, 5, 2 },
    { 1, 7, 10, 2 },
    { 1, 7, 11, 2 },
    { 1, 11, 2, 4 },
    { 2, 11, 3, 2 },
    { 2, 14, 3, 2 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 2, 5, 9 },
    { 1, 2, 9, 9 },
    { 2, 6, 10, 4 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 3, 2, 1 },
    { 2, 12, 7, 5 },
    { 2, 13, 7, 5 },
    { 1, 2, 14, 3 },
    { 2, 4, 9, 5 },
    { 1, 4, 8, 2 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 5, 5, 5 },
    { 1, 5, 10, 5 },
    { 2, 5, 6, 4 },
    { 2, 9, 6, 4 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 2, 12, 3 },
    { 2, 2, 13, 2 },
    { 1, 10, 5, 3 },
    { 2, 12, 6, 3 },
    { 1, 3, 2, 5 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 8, 11, 5 },
    { 2, 12, 8, 3 },
    { 1, 10, 8, 2 },
    { 2, 2, 2, 6 },
    { 1, 3, 4, 3 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  }, {
    { 1, 2, 1, 5 },
    { 2, 6, 4, 2 },
    { 2, 6, 8, 2 },
    { 2, 6, 12, 2 },
    { 1, 7, 13, 3 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
    { 0, 0, 0, 0 },
  },
};


/* +++ get-functions +++ */

/* get-function for ground-object */
void
getofc_ground(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_GROUND));
  ofc->f_new = f_new_ground;
  ofc->f_free = f_free_ground;
  ofc->f_draw = f_draw_ground;
}


/* get-function for wall-object */
void
getofc_wall(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_WALL));
  ofc->f_new = f_new_wall;
  ofc->f_free = f_free_wall;
}


/* +++ object-functions for ground-object +++ */

/* new-function for ground-object
 * variable parameter: int wsize = width in 128-pixel-blocks
 *                     int hsize = height in 128-pixel-blocks
 */
static struct vg3_ofunc_object *
f_new_ground(void *vmain, unsigned int iparent, ...)
{
  const int sizemulti = 4;
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_ground *gobj;
  struct vg3_rect rect;
  struct vg3_image *img_ground1, *img_ground2, *imgptr, *img_wall1, *img_wall2;
  int wsize, hsize;
  int groundw, groundh, wallw, wallh, w1, h1;
  int i1, i2, i3, i4, i5;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments: wsize:int, hsize:int */
  va_start(ap, iparent);
  wsize = va_arg(ap, int);
  hsize = va_arg(ap, int);
  va_end(ap);

  if (wsize <= 0) { wsize = 3; }
  if (hsize <= 0) { hsize = 2; }

  /* load ground elements */

  img_ground1 = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/ground1.bmp", 1);
  if (img_ground1 == NULL) { return NULL; }
  VG3_image_getsize(gmain->wstruct, img_ground1, NULL, &groundw, &groundh);

  img_ground2 = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/ground2.bmp", 1);
  if (img_ground2 == NULL) { return NULL; }
  VG3_image_getsize(gmain->wstruct, img_ground2, NULL, &w1, &h1);

  if (groundw != w1 || groundh != h1) {
    VG3_seterror(EINVAL, "ground1.bmp and ground2.bmp have different sizes");
    return NULL;
  }

  /* load wall elements */

  img_wall1 = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/wall1.bmp", 1);
  if (img_wall1 == NULL) { return NULL; }
  VG3_image_getsize(gmain->wstruct, img_wall1, NULL, &wallw, &wallh);

  img_wall2 = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/wall2.bmp", 1);
  if (img_wall2 == NULL) { return NULL; }
  VG3_image_getsize(gmain->wstruct, img_wall2, NULL, &w1, &h1);

  if (wallw != w1 || wallh != h1) {
    VG3_seterror(EINVAL, "wall1.bmp and wall2.bmp have different sizes");
    return NULL;
  }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* create ground-image und set size of image into gmain->hgrect */
  gobj->img = VG3_image_create(gmain->wstruct, groundw * sizemulti * wsize, groundh * sizemulti * hsize);
  if (gobj->img == NULL) { return NULL; }
  gmain->hgrect.x = gmain->hgrect.y = 0;
  VG3_image_getsize(gmain->wstruct, gobj->img, NULL, &gmain->hgrect.w, &gmain->hgrect.h);

  /* resize quadtree */
  VG3_coll_q_free(gmain->qdtr);
  gmain->qdtr = VG3_coll_q_new(&gmain->hgrect, 0, 0);
  if (gmain->qdtr == NULL) { return NULL; }

  /* fill ground with ground images */
  for (i1 = 0; i1 < sizemulti * hsize; i1++) {
    for (i2 = 0; i2 < sizemulti * wsize; i2++) {
      unsigned int u1 = VG3_nw_get_random(1, 3);
      if (u1 == 1) { imgptr = img_ground2; } else { imgptr = img_ground1; }
      VG3_image_copy(gmain->wstruct, gobj->img, imgptr, groundw / 2 + groundw * i2, groundh / 2 + groundh * i1, NULL, 0);
    }
  }

  /* create walls and put them onto the ground */
  i1 = (int)VG3_nw_get_random(1, MAX_GR_WALL - 1);
  for (i2 = 0; i2 < hsize; i2++) {
    for (i3 = 0; i3 < wsize; i3++) {
      for (i4 = 0; i4 < MAX_GR_WALLELEM; i4++) {
        w1 = i3 * groundw * sizemulti + walls[i1][i4].x * wallw + (wallw / 2);
        h1 = i2 * groundh * sizemulti + walls[i1][i4].y * wallh + (wallh / 2);
        rect.x = w1 - (wallw / 2);
        rect.y = h1 - (wallh / 2);
        rect.w = rect.h = 0;
        for (i5 = 0; i5 < walls[i1][i4].elements; i5++) {  /* elements of the wall */
          if (walls[i1][i4].direction <= 0) { break; }
          if (i5 % 2 == 0) { imgptr = img_wall1; } else { imgptr = img_wall2; }
          VG3_image_copy(gmain->wstruct, gobj->img, imgptr, w1, h1, NULL, 0);
          if (walls[i1][i4].direction == 1) {
            w1 += wallw;
            rect.w += wallw;
            rect.h = wallh;
          } else if (walls[i1][i4].direction == 2) {
            h1 += wallh;
            rect.h += wallh;
            rect.w = wallw;
          } else {
            break;
          }
        }
        /* create single wall-instance with all the elements */
        f_new_wall(gmain, 0, rect.x, rect.y, rect.w, rect.h);
      }
      if (++i1 == MAX_GR_WALL) { i1 = 0; }
    }
  }

  /* create nearly invisible 1-pixel boundaries for background */
  f_new_wall(gmain, 0, gmain->hgrect.x, gmain->hgrect.y, gmain->hgrect.w, 1);
  f_new_wall(gmain, 0, gmain->hgrect.x, gmain->hgrect.y, 1, gmain->hgrect.h);
  f_new_wall(gmain, 0, gmain->hgrect.x, gmain->hgrect.y + gmain->hgrect.h - 1, gmain->hgrect.w, 1);
  f_new_wall(gmain, 0, gmain->hgrect.x + gmain->hgrect.w - 1, gmain->hgrect.y, 1, gmain->hgrect.h);
  VG3_draw_rect(gmain->wstruct, gobj->img, &gmain->hgrect, 0, VGAG3_COLOR_RED);

  VG3_image_unload(gmain->wstruct, img_ground1);
  VG3_image_unload(gmain->wstruct, img_ground2);
  VG3_image_unload(gmain->wstruct, img_wall1);
  VG3_image_unload(gmain->wstruct, img_wall2);
#ifdef SET_MARKER
  /* show collision rectangles for walls */
  VG3_coll_q_mark(gmain->qdtr, gmain->wstruct, gobj->img, VGAG3_COLOR_RED, NULL);
#endif

  /* initialize struct for positioning window */
  VG3_init_windowpos_on_bgimage(&gobj->wbpos, gmain->hgrect.w, gmain->hgrect.h, gmain->winw, gmain->winh, 2, 60);

  /* create and fill ground-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_GROUND));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert ground-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* don't insert ground-instance into quadtree */

  return objp;
}


/* free-function for ground-object */
static void
f_free_ground(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_ground *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  /* ground-instance is not in the quadtree */

  /* remove ground-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free created wall-instances */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_WALL));

  /* free ground-instance */
  gobj = (struct g_obj_ground *)objp->ostruct;
  VG3_image_unload(gmain->wstruct, gobj->img);
  free(gobj);
  free(objp);
}


/* draw-function for ground-object */
static void
f_draw_ground(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_ground *gobj;
  struct vg3_ofunc_object *me_objp;
  int i1, atonce;

  gobj = (struct g_obj_ground *)objp->ostruct;

  /* get position of followplayer to follow with the window */

  me_objp = NULL;
  atonce = 0;

  if (gmain->followplayer >= 0) {
    /* get object-instance of followplayer's instanceid */
    me_objp = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gmain->player[gmain->followplayer].instanceid);
    if (me_objp == NULL || !gmain->player[gmain->followplayer].alive) {  /* player is dead */
      gmain->followplayer = -1;
    }
  }

  if (gmain->followplayer < 0) {
    atonce = 1;
    /* search for next valid local player to follow and get object-instance */
    for (i1 = 0; i1 < MAXPLAYER; i1++) {
      if (gmain->player[i1].ply == PLY_IS_LOCAL && gmain->player[i1].alive) {
        me_objp = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gmain->player[i1].instanceid);
        if (me_objp != NULL) { gmain->followplayer = i1; break; }
      }
    }
  }

  if (gmain->followplayer < 0) {
    /* search for next valid any player to follow and get object-instance */
    for (i1 = 0; i1 < MAXPLAYER; i1++) {
      if (gmain->player[i1].alive) {
        me_objp = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gmain->player[i1].instanceid);
        if (me_objp != NULL) { gmain->followplayer = i1; break; }
      }
    }
  }

  if (me_objp != NULL) {  /* valid object-instance */
    /* get object-functions of object-instance me_objp and call function f_data() for position */
    const struct vg3_ofunc_objfunc *ofc;
    ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, me_objp->oid);
    if (ofc != NULL && ofc->f_data != NULL) {
      /* set window coordinates according to object-instance position */
      struct vg3_rect opos;
      ofc->f_data(gmain, me_objp, &opos);
      VG3_get_windowpos_on_bgimage(&gobj->wbpos, opos.x + opos.w / 2, opos.y + opos.h / 2, &gmain->hgrect.x, &gmain->hgrect.y, atonce);
    }
  }

  /* copy ground-image to window */
  VG3_image_copy(gmain->wstruct, NULL, gobj->img, gmain->hgrect.x + gmain->hgrect.w / 2, gmain->hgrect.y + gmain->hgrect.h / 2, NULL, 0);
#ifdef SET_MARKER
  { /* show position rectangle for background image */
    struct vg3_rect rc;
    rc.x = (gmain->winw - gobj->wbpos.border_w) / 2;
    rc.y = (gmain->winh - gobj->wbpos.border_h) / 2;
    rc.w = gobj->wbpos.border_w;
    rc.h = gobj->wbpos.border_h;
    VG3_draw_rect(gmain->wstruct, NULL, &rc, 0, VGAG3_COLOR_BLUE);
  }
#endif
}


/* +++ object-functions for wall-object +++ */

/* new-function for wall-object
 * variable parameter: int x = x-position (left upper corner)
 *                     int y = y-position (left upper corner)
 *                     int w = width
 *                     int h = height
 */
static struct vg3_ofunc_object *
f_new_wall(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_wall *gobj;
  struct vg3_coll coll;
  int x, y, w, h;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments: x:int, y:int, w:int, h:int */
  va_start(ap, iparent);
  x = va_arg(ap, int);
  y = va_arg(ap, int);
  w = va_arg(ap, int);
  h = va_arg(ap, int);
  va_end(ap);

  if (w <= 0 || h <= 0) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->rect.x = x;
  gobj->rect.y = y;
  gobj->rect.w = w;
  gobj->rect.h = h;

  /* create and fill wall-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_WALL));
  objp->drawlevel = 2;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert wall-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* insert wall-instance into quadtree */
  memset(&coll, 0, sizeof(coll));
  coll.rect.x = gobj->rect.x;
  coll.rect.y = gobj->rect.y;
  coll.rect.w = gobj->rect.w;
  coll.rect.h = gobj->rect.h;
  snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_WALL));
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);

  return objp;
}


/* wall-object needs no run-function */


/* free-function for wall-object */
static void
f_free_wall(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_wall *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  /* remove wall-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove wall-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free wall-instance */
  gobj = (struct g_obj_wall *)objp->ostruct;
  free(gobj);
  free(objp);
}
