/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-shot.h"

/* after how many loops a shot is removed */
const int max_run_loops = 150;


void getofc_shot(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);
static void f_explode(void *, struct vg3_ofunc_object *);

/* struct with individual functions */
static struct g_obj_shot_privfunc privfunc;


/* +++ get-function +++ */

void
getofc_shot(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_SHOT));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
  ofc->f_data = f_data;
  /* individual functions */
  ofc->vpriv = &privfunc;
  privfunc.f_explode = f_explode;
}


/* +++ object-functions +++ */

/* new-function; x and y are the position of the middle of the image
 * variable parameter: int xm = x-position (middle)
 *                     int ym = y-position (middle)
 *                     int angle = angle in degrees
 *                     int plcolor = player color
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_shot *gobj;
  struct vg3_image *vimg;
  char buf[128];
  int xm, ym, angle, plcolor;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments: xm:int, ym:int, angle:int, plcolor:int */
  va_start(ap, iparent);
  xm = va_arg(ap, int);
  ym = va_arg(ap, int);
  angle = va_arg(ap, int);
  plcolor = va_arg(ap, int);
  va_end(ap);

  /* load image */
  snprintf(buf, sizeof(buf), "%s/bmp/shot-%s.bmp", FILES_DIR, get_player_color(plcolor));
  vimg = VG3_image_load(gmain->wstruct, buf, 1);
  if (vimg == NULL) { return NULL; }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->loop = max_run_loops;
  gobj->destroy_me = 0;
  gobj->img = vimg;
  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gobj->iattr);
  gobj->iattr.rotate = VG3_get_xy_direction_from_angle(angle, &gobj->xdelta, &gobj->ydelta);
  gobj->rect = VG3_image_copy(gmain->wstruct, NULL, gobj->img, xm, ym, &gobj->iattr, 1);
  gobj->iparent = iparent;
  gobj->expl.sprt = NULL;

  /* create and fill shot-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_SHOT));
  objp->drawlevel = 2;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert shot-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_shot *)objp->ostruct;

  /* inform parent object-instance */
  if (gobj->iparent > 0) {
    struct vg3_ofunc_object *oparent = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gobj->iparent);
    if (oparent != NULL) {
      const struct vg3_ofunc_objobjfunc *oofc;
      oofc = VG3_ofunc_get_objobjfunc(gmain->ofstruct, objp->oid, oparent->oid);
      if (oofc != NULL && oofc->f_quit != NULL) {
        oofc->f_quit(gmain, objp, oparent);
      }
    }
  }

  /* remove shot-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove shot-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free shot-instance */
  if (gobj->expl.sprt != NULL) { VG3_sprite_free(gobj->expl.sprt); }
  VG3_image_unload(gmain->wstruct, gobj->img);
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 30;
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj;
  struct vg3_coll coll;
  int erg, angle;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_shot *)objp->ostruct;

  /* remove shot-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* if shot is exploding, just get new sprite-image and return */
  if (gobj->expl.sprt != NULL) {
    erg = VG3_sprite_get(gobj->expl.sprt, &gobj->expl.img, &gobj->expl.iattr, NULL);
    if (erg == 0) { f_free(gmain, objp); }  /* sprite ended */
    return;
  }

  /* if run-time is over, destroy shot-instance */
  if (--gobj->loop <= 0) {
    f_free(gmain, objp);
    return;
  }

  /* move shot-instance and check for collisions */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->rect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: destroy shot-instance */
    f_free(gmain, objp);
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }
  if (erg == VGAG3_COLL_RETURN_CATCHED) {  /* destroy shot-instance */
    f_free(gmain, objp);
    return;
  }

  /* get angle from (modified) xdelta/ydelta */
  angle = VG3_get_angle_from_xy_direction(gobj->xdelta, gobj->ydelta);

  /* update gobj->iattr for drawing */
  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gobj->iattr);
  gobj->iattr.rotate = angle;

  /* insert shot-instance into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->rect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_SHOT));
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_shot *)objp->ostruct;

  /* draw shot-instance */
  if (gobj->expl.sprt != NULL) {  /* draw exploding sprite instead of shot */
    VG3_image_copy(gmain->wstruct, NULL, gobj->expl.img, gobj->rect.x + gobj->rect.w / 2 + gmain->hgrect.x, gobj->rect.y + gobj->rect.h / 2 + gmain->hgrect.y, &gobj->expl.iattr, 0);
  } else {  /* draw shot */
    VG3_image_copy(gmain->wstruct, NULL, gobj->img, gobj->rect.x + gobj->rect.w / 2 + gmain->hgrect.x, gobj->rect.y + gobj->rect.h / 2 + gmain->hgrect.y, &gobj->iattr, 0);
#ifdef SET_MARKER
    { /* show collision (and image) rectangle */
      struct vg3_coll coll;
      if (VG3_coll_q_getobj(gmain->qdtr, objp, &coll)) {
        coll.rect.x += gmain->hgrect.x;
        coll.rect.y += gmain->hgrect.y;
        VG3_draw_rect(gmain->wstruct, NULL, &coll.rect, 0, VGAG3_COLOR_BLUE);
      }
    }
#endif
  }
}


/* function for returning position */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *vdata)
{
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj;
  struct vg3_rect *rect = (struct vg3_rect *)vdata;

  if (gmain == NULL || objp == NULL || rect == NULL) { return 0; }

  gobj = (struct g_obj_shot *)objp->ostruct;
  (void)gmain;

  memmove(rect, &gobj->rect, sizeof(*rect));
  return 0;
}


/* individual function for creating exploding sprite */
static void
f_explode(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_shot *)objp->ostruct;

  if (gobj->expl.sprt == NULL) {
    int dfl_loop = 3;  /* default loop number */
    int lifetime = 1;  /* life time of sprite */
    char buf[128];
    int i1;
    gobj->expl.sprt = VG3_sprite_new(gmain->wstruct);
    if (gobj->expl.sprt == NULL) { return; }
    VG3_sprite_setattr(gobj->expl.sprt, &lifetime, &dfl_loop, NULL);
    /* add sound to sprite */
    snprintf(buf, sizeof(buf), "%s/sound/shothit.wav", FILES_DIR);
    VG3_sprite_addelem(gobj->expl.sprt, 0, NULL, NULL, buf, 100);
    /* add images to sprite */
    for (i1 = 4; i1 <= 5; i1++) {
      snprintf(buf, sizeof(buf), "%s/bmp/tank-explode%d.bmp", FILES_DIR, i1);
      VG3_sprite_addelem(gobj->expl.sprt, 0, buf, NULL, NULL, 0);
    }
  }
}
