/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-tank.h"


void getofc_tank(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static int run_moving(struct g_main *, struct vg3_ofunc_object *, const int);
static void f_draw(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);
static void fire_shot(struct g_main *, unsigned int, struct g_obj_tank *);

const int health_max_local = 3;
const int health_max_computer = 5;
const int shots_max = 3;


/* +++ get-function +++ */

void
getofc_tank(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_TANK));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
  ofc->f_data = f_data;
}


/* +++ object-functions +++ */

/* new-function; x and y are the position of the middle of the image
 * variable parameter: int xm = x-position (middle)
 *                     int ym = y-position (middle)
 *                     int angle = angle in degrees
 *                     int plnumber = player number (1 to MAXPLAYER)

 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_tank *gobj;
  struct vg3_sprite *sprt[3];
  char buf[128];
  int xm, ym, angle, plnumber;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments: xm:int, ym:int, angle:int, plnumber:int */
  va_start(ap, iparent);
  xm = va_arg(ap, int);
  ym = va_arg(ap, int);
  angle = va_arg(ap, int);
  plnumber = va_arg(ap, int);
  va_end(ap);

  if (plnumber < 1 || plnumber > MAXPLAYER) { VG3_seterror(EINVAL, "player number invalid"); return NULL; }

  /* create sprites */

  /* default sprite */
  { int dfl_loop = 5;  /* default loop number */
    int i1;
    sprt[TANK_IS_NORMAL] = VG3_sprite_new(gmain->wstruct);
    if (sprt[TANK_IS_NORMAL] == NULL) { return NULL; }
    VG3_sprite_setattr(sprt[TANK_IS_NORMAL], NULL, &dfl_loop, NULL);
    /* add images to sprite */
    for (i1 = 1; i1 <= 2; i1++) {
      snprintf(buf, sizeof(buf), "%s/bmp/tank-%s%d.bmp", FILES_DIR, get_player_color(gmain->player[plnumber - 1].color), i1);
      VG3_sprite_addelem(sprt[TANK_IS_NORMAL], 0, buf, NULL, NULL, 0);
    }
  }

  /* shooting sprite */
  { int dfl_loop = 10;  /* default loop number */
    int lifetime = 1;  /* life time of sprite */
    char bsound[128];
    sprt[TANK_IS_SHOOTING] = VG3_sprite_new(gmain->wstruct);
    if (sprt[TANK_IS_SHOOTING] == NULL) { return NULL; }
    VG3_sprite_setattr(sprt[TANK_IS_SHOOTING], &lifetime, &dfl_loop, NULL);
    /* add image and sound to sprite */
    snprintf(buf, sizeof(buf), "%s/bmp/tank-%sshot.bmp", FILES_DIR, get_player_color(gmain->player[plnumber - 1].color));
    snprintf(bsound, sizeof(bsound), "%s/sound/shot.wav", FILES_DIR);
    VG3_sprite_addelem(sprt[TANK_IS_SHOOTING], 0, buf, NULL, bsound, 100);
  }

  /* exploding sprite */
  { int dfl_loop = 3;  /* default loop number */
    int lifetime = 3;  /* life time of sprite */
    int i1;
    sprt[TANK_IS_EXPLODING] = VG3_sprite_new(gmain->wstruct);
    if (sprt[TANK_IS_EXPLODING] == NULL) { return NULL; }
    VG3_sprite_setattr(sprt[TANK_IS_EXPLODING], &lifetime, &dfl_loop, NULL);
    /* add sound to sprite */
    snprintf(buf, sizeof(buf), "%s/sound/tankhit.wav", FILES_DIR);
    VG3_sprite_addelem(sprt[TANK_IS_EXPLODING], 0, NULL, NULL, buf, 100);
    /* add images to sprite */
    for (i1 = 1; i1 <= 5; i1++) {
      snprintf(buf, sizeof(buf), "%s/bmp/tank-explode%d.bmp", FILES_DIR, i1);
      VG3_sprite_addelem(sprt[TANK_IS_EXPLODING], 0, buf, NULL, NULL, 0);
    }
  }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->plnumber = plnumber;
  gobj->shots_running = 0;
  if (gmain->player[gobj->plnumber - 1].ply == PLY_IS_COMPUTER) {  /* computer player */
    gobj->health_max = health_max_computer;
  } else {  /* local or remote player */
    int i1;
    for (i1 = 0; i1 < MAXPLAYER; i1++) {
      if (i1 != gobj->plnumber - 1 && (gmain->player[i1].ply == PLY_IS_LOCAL || gmain->player[i1].ply == PLY_IS_REMOTE)) { break; }
    }
    if (i1 < MAXPLAYER) {  /* two or more local or remote players */
      gobj->health_max = health_max_local;
    } else {  /* only one player */
      gobj->health_max = 1;
    }
  }
  gobj->health = gobj->health_max;
  gobj->angle = VG3_get_xy_direction_from_angle(angle, &gobj->xdelta, &gobj->ydelta);
  gobj->immun = 0;
  gobj->i.sprt_no = TANK_IS_NORMAL;  /* active sprite index */
  gobj->i.sprt[TANK_IS_NORMAL] = sprt[TANK_IS_NORMAL];  /* default sprite */
  gobj->i.sprt[TANK_IS_SHOOTING] = sprt[TANK_IS_SHOOTING];  /* shooting sprite */
  gobj->i.sprt[TANK_IS_EXPLODING] = sprt[TANK_IS_EXPLODING];  /* exploding sprite */
  gobj->i.img = NULL;

  /* set position of image:
   * at this time we don't want to know the size of the image
   * as it is corrected later in f_run with VG3_correct_imageposition()
   * so we now assume an image-rectangle consisting of just a pixel (or less)
   */
  gobj->rect.x = xm;
  gobj->rect.y = ym;
  gobj->rect.w = gobj->rect.h = 0;

  /* set collsion rectangle with 90% size of first image of default sprite */
  { int w1, h1;
    gobj->crect = gobj->rect;
    VG3_sprite_imagesize(sprt[gobj->i.sprt_no], NULL, 90, &w1, &h1);
    gobj->crect = VG3_correct_imageposition(&gobj->crect, w1, h1);
  }

  /* initialize struct for computer player (if needed or not) */
  gobj->cp.last_coll_status = VGAG3_COLL_RETURN_NOOP;
  gobj->cp.movingtype = 0;
  gobj->cp.mvsteps = 0;
  gobj->cp.turning_angle = 0;
  gobj->cp.shoot_rand = gobj->cp.shoot_total = 0;
  gobj->cp.shoot_number = 1000;

  /* create and fill tank-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_TANK));
  objp->drawlevel = 3;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert tank-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_tank *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_tank *)objp->ostruct;

  /* remove tank-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove tank-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free tank-instance */
  VG3_sprite_free(gobj->i.sprt[TANK_IS_NORMAL]);
  VG3_sprite_free(gobj->i.sprt[TANK_IS_SHOOTING]);
  VG3_sprite_free(gobj->i.sprt[TANK_IS_EXPLODING]);
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 15;
  const int turning_factor = 10;
  struct g_main *gmain = vmain;
  struct g_obj_tank *gobj;
  struct vg3_coll coll;
  int do_move, erg, nwidth, nheight;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_tank *)objp->ostruct;

  /* is player runnable? */
  if (gmain->player[gobj->plnumber - 1].ply == PLY_IS_NOTUSED) { return; }  /* should not be */
  if (!gmain->player[gobj->plnumber - 1].alive) { return; }  /* player is dead */
  
  /* remove tank-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* check if network-player is still connected */
  if (!network_is_client_connected(gmain, gobj->plnumber)) {
    gmain->player[gobj->plnumber - 1].alive = 0;
    return;
  }

  /* check for pressed keys when real player, else calculate moving and shooting */
  do_move = run_moving(gmain, objp, turning_factor);

  /* get next image from actual sprite */
  erg = 1;
  if (gobj->i.img == NULL || gobj->i.sprt_no != TANK_IS_NORMAL || do_move) {
    erg = VG3_sprite_get(gobj->i.sprt[gobj->i.sprt_no], &gobj->i.img, &gobj->i.iattr, NULL);
  }

  if (erg == 0) {  /* sprite ended */
    if (gobj->i.sprt_no == TANK_IS_EXPLODING) {
      if (gobj->health < 1) {  /* set player dead */
        gmain->player[gobj->plnumber - 1].alive = 0;
        return;
      }
    }
    gobj->i.sprt_no = TANK_IS_NORMAL;
    VG3_sprite_rewind(gobj->i.sprt[gobj->i.sprt_no]);
    VG3_sprite_get(gobj->i.sprt[gobj->i.sprt_no], &gobj->i.img, &gobj->i.iattr, NULL);
  }

  /* decrease gobj->immun until 0, if not exploding */
  if (gobj->immun > 0 && gobj->i.sprt_no != TANK_IS_EXPLODING) { gobj->immun--; }

  /* update gobj->iattr for drawing (rotating and image-attributes of actual image) */
  { struct vg3_image_attributes iattr;
    VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
    iattr.rotate = gobj->angle;
    VG3_image_attr_sum(&gobj->iattr, &gobj->i.iattr, &iattr);
  }

  /* get width and height of new sprite-image and correct x/y-position for same central point of the tank
   * we don't change gobj->crect, it could lead to unexpected collisions (without moving)
   */
  if (gobj->i.img != NULL) {
    VG3_image_getsize(gmain->wstruct, gobj->i.img, &gobj->iattr, &nwidth, &nheight);
    gobj->rect = VG3_correct_imageposition(&gobj->rect, nwidth, nheight);
  }

  /* move tank-instance and check for collisions with gobj->crect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->crect, moving_factor * do_move, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: do nothing */
    fprintf(stderr, "moving %s tank: %s\n", get_player_color(gmain->player[gobj->plnumber - 1].color), VG3_error());
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }
  /* VGAG3_COLL_RETURN_CATCHED: ignore */

  /* computer player: save status */
  if (gmain->player[gobj->plnumber - 1].ply == PLY_IS_COMPUTER) { gobj->cp.last_coll_status = erg; }

  /* x/y-value of gobj->crect could be modified, correct it also in gobj->rect */
  gobj->rect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w, gobj->rect.h);

  /* insert tank-instance into quadtree again
   * we use for collision check the same as above: gobj->crect
   */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_TANK));
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* run subfunction: check for moving or turning player */
static int
run_moving(struct g_main *gmain, struct vg3_ofunc_object *objp, const int turning_factor)
{
  int do_move = 0;
  struct g_obj_tank *gobj;

  if (gmain == NULL || objp == NULL) { return 0; }

  gobj = (struct g_obj_tank *)objp->ostruct;

  if (gobj->i.sprt_no == TANK_IS_EXPLODING) { return 0; }  /* player cannot move */

  if (gmain->player[gobj->plnumber - 1].ply != PLY_IS_COMPUTER) {  /* real player (local or remote) */

    /* move forward */
    if (key_ispressed(gmain, gobj->plnumber, KEYDEF_GO_FORWARD, VGAG3_IS_PRESSED)) {
      do_move++;
    }

    /* move backward */
    if (key_ispressed(gmain, gobj->plnumber, KEYDEF_GO_BACKWARD, VGAG3_IS_PRESSED)) {
      do_move--;
    }

    /* turn right */
    if (key_ispressed(gmain, gobj->plnumber, KEYDEF_TURN_RIGHT, VGAG3_IS_PRESSED)) {
      gobj->angle += turning_factor;
    }

    /* turn left */
    if (key_ispressed(gmain, gobj->plnumber, KEYDEF_TURN_LEFT, VGAG3_IS_PRESSED)) {
      gobj->angle -= turning_factor;
    }

    /* fire gun */
    if (key_ispressed(gmain, gobj->plnumber, KEYDEF_FIRE, VGAG3_IS_NEW_PRESSED)) {
      if (gobj->shots_running < shots_max) {  /* player may still fire a shot */
        fire_shot(gmain, objp->instanceid, gobj);
      }
    }

  } else {  /* computer player */
    int noshot = 0;

    if (gobj->cp.last_coll_status == VGAG3_COLL_RETURN_FULLSTOP) {  /* tank has been stopped */
      int negative = VG3_nw_get_random(0, 1);
      if (gobj->cp.movingtype == 0) {  /* move backwards */
        gobj->cp.mvsteps = VG3_nw_get_random(0, 15);
      } else {  /* stop moving backwards */
        gobj->cp.mvsteps = 0;
      }
      /* set another angle to turn to */
      gobj->cp.turning_angle = VG3_nw_get_random(0, 180 / turning_factor);
      if (negative) { gobj->cp.turning_angle = -gobj->cp.turning_angle; }
      gobj->cp.movingtype = 1;
    } else if (gobj->cp.last_coll_status == VGAG3_COLL_RETURN_HALFSTOP) {  /* tank scratches */
      if (gobj->cp.mvsteps >= 40) {  /* reduce moving steps */
        gobj->cp.mvsteps = VG3_nw_get_random(10, 39);
      }
      if (gobj->cp.shoot_rand > 0) { gobj->cp.shoot_rand--; }
      noshot = 1;
    }

    if (gobj->cp.movingtype == 1) {  /* move away from obstacle */
      if (gobj->cp.mvsteps > 0) {  /* move backwards */
        gobj->cp.mvsteps--;
        do_move = -1;
      } else if (gobj->cp.turning_angle != 0) {  /* turn */
        if (gobj->cp.turning_angle > 0) {
          gobj->angle += turning_factor;
          gobj->cp.turning_angle--;
        } else {
          gobj->angle -= turning_factor;
          gobj->cp.turning_angle++;
        }
      } else {  /* ready, switch to normal moving */
        gobj->cp.mvsteps = 0;
        gobj->cp.movingtype = 0;
      }
      if (gobj->cp.shoot_rand > 0) { gobj->cp.shoot_rand--; }
      noshot = 1;

    } else if (gobj->cp.movingtype == 2) {  /* fire a double-shot */
      if (gobj->cp.turning_angle != 0) {  /* turn */
        if (gobj->cp.turning_angle > 0) {
          gobj->angle += turning_factor;
          gobj->cp.turning_angle--;
        } else {
          gobj->angle -= turning_factor;
          gobj->cp.turning_angle++;
        }
      } else {  /* ready, fire second shot and switch to normal moving */
        fire_shot(gmain, objp->instanceid, gobj);
        gobj->cp.mvsteps = 0;
        gobj->cp.movingtype = 0;
      }
      if (gobj->cp.shoot_rand > 0) { gobj->cp.shoot_rand--; }
      noshot = 1;

    } else {  /* move normal */
      if (gobj->cp.mvsteps == 0) {  /* get new moving */
        int dontstop = VG3_nw_get_random(0, 5);
        int negative = VG3_nw_get_random(0, 1);
        if (dontstop) {
          gobj->cp.mvsteps = VG3_nw_get_random(20, 80);
        } else {
          gobj->cp.mvsteps = -VG3_nw_get_random(1, 10);
        }
        /* set another angle to turn to */
        gobj->cp.turning_angle = VG3_nw_get_random(0, 90 / turning_factor);
        if (negative) { gobj->cp.turning_angle = -gobj->cp.turning_angle; }
      }
      if (gobj->cp.mvsteps < 0) { gobj->cp.mvsteps++; } else { gobj->cp.mvsteps--; do_move = 1; }
      if (gobj->cp.turning_angle != 0) {  /* turn */
        if (gobj->cp.turning_angle > 0) {
          gobj->angle += turning_factor;
          gobj->cp.turning_angle--;
        } else {
          gobj->angle -= turning_factor;
          gobj->cp.turning_angle++;
        }
      }
    }

    /* fire shot? */
    if (!noshot && gobj->shots_running < shots_max && ++gobj->cp.shoot_rand > 0) {
      gobj->cp.shoot_total += VG3_nw_get_random(0, gobj->cp.shoot_rand);
      if (gobj->cp.shoot_total >= gobj->cp.shoot_number) {  /* fire shot */
        fire_shot(gmain, objp->instanceid, gobj);
        gobj->cp.shoot_rand = gobj->cp.shoot_total = 0;
        gobj->cp.shoot_number = VG3_nw_get_random(100, 1500);
        /* fire a double-shot? */
        if (gobj->shots_running < shots_max && VG3_nw_get_random(0, 1) > 0) {
          int negative = VG3_nw_get_random(0, 1);
          gobj->cp.turning_angle = VG3_nw_get_random(30 / turning_factor, 90 / turning_factor);
          if (negative) { gobj->cp.turning_angle = -gobj->cp.turning_angle; }
          gobj->cp.movingtype = 2;
        }
      }
    }
  }

  /* correct angle and update xdelta/ydelta from angle */
  gobj->angle = VG3_get_xy_direction_from_angle(gobj->angle, &gobj->xdelta, &gobj->ydelta);

  return do_move;
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_tank *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_tank *)objp->ostruct;

  if (gmain->player[gobj->plnumber - 1].ply == PLY_IS_NOTUSED) { return; }  /* should not be */
  if (!gmain->player[gobj->plnumber - 1].alive) { return; }  /* player is dead */

  /* if tank is immun, do blinking */
  if (gobj->immun > 0 && gobj->i.sprt_no != TANK_IS_EXPLODING) {
    if ((gobj->immun / 5) % 2 == 0) { return; }
  }

  /* draw tank-instance */
  if (gobj->i.img != NULL) {
    VG3_image_copy(gmain->wstruct, NULL, gobj->i.img, gobj->rect.x + gobj->rect.w / 2 + gmain->hgrect.x, gobj->rect.y + gobj->rect.h / 2 + gmain->hgrect.y, &gobj->iattr, 0);
  }

  /* show player-name? */
  if (gmain->shownames > 0 && *gmain->player[gobj->plnumber - 1].name != '\0') {
    struct vg3_text stxt;
    struct vg3_rect trect;
    char buf[64];
    int fcolor = VGAG3_COLOR_WHITE;
    snprintf(buf, sizeof(buf), "%s:[%d%%]", gmain->player[gobj->plnumber - 1].name, gobj->health * 100 / gobj->health_max);
    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, 0, 0, buf);
    trect.x = trect.y = 0; trect.w = gmain->winw; trect.h = gmain->winh;
    trect = VG3_draw_text(gmain->wstruct, NULL, &trect, 0, &stxt, fcolor, VGAG3_COLOR_TRANSPARENT, 1);
    trect.x = gobj->rect.x + gmain->hgrect.x + (gobj->rect.w - trect.w) / 2;
    trect.y = gobj->rect.y + gmain->hgrect.y + (gobj->rect.h - trect.h) / 2;
    VG3_draw_text(gmain->wstruct, NULL, &trect, 0, &stxt, fcolor, VGAG3_COLOR_TRANSPARENT, 0);
  }

#ifdef SET_MARKER
  { /* show collision rectangle */
    struct vg3_coll coll;
    if (VG3_coll_q_getobj(gmain->qdtr, objp, &coll)) {
      coll.rect.x += gmain->hgrect.x;
      coll.rect.y += gmain->hgrect.y;
      VG3_draw_rect(gmain->wstruct, NULL, &coll.rect, 0, VGAG3_COLOR_BLUE);
    }
  }
  { /* show image rectangle */
    struct vg3_rect rct = gobj->rect;
    rct.x += gmain->hgrect.x;
    rct.y += gmain->hgrect.y;
    VG3_draw_rect(gmain->wstruct, NULL, &rct, 0, VGAG3_COLOR_GREEN);
  }
#endif
}


/* function for returning position */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *vdata)
{
  struct g_main *gmain = vmain;
  struct g_obj_tank *gobj;
  struct vg3_rect *rect = (struct vg3_rect *)vdata;

  if (gmain == NULL || objp == NULL || rect == NULL) { return 0; }

  gobj = (struct g_obj_tank *)objp->ostruct;
  (void)gmain;

  memmove(rect, &gobj->rect, sizeof(*rect));
  return 0;
}


/* tank fires a shot */
static void
fire_shot(struct g_main *gmain, unsigned int instanceid, struct g_obj_tank *gobj)
{
  const struct vg3_ofunc_objfunc *ofc;

  if (gmain == NULL || instanceid == 0 || gobj == NULL) { return; }

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_SHOT));  /* get shot's functions */

  if (ofc != NULL) {
    int shot_x, shot_y, shot_color;
    shot_x = gobj->rect.x + gobj->rect.w / 2;
    shot_y = gobj->rect.y + gobj->rect.h / 2;
    shot_color = gmain->player[gobj->plnumber - 1].color;
    if (ofc->f_new(gmain, instanceid, shot_x, shot_y, gobj->angle, shot_color) != NULL) {  /* create shot */
      gobj->shots_running++;
      gobj->i.sprt_no = TANK_IS_SHOOTING;
      VG3_sprite_rewind(gobj->i.sprt[gobj->i.sprt_no]);
    }
  }
}
