/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-shot.h"
#include "obj-tank.h"

void getoofc_shot_tank(struct vg3_ofunc_objobjfunc *);

static int f_collision(void *, struct vg3_ofunc_object *, struct vg3_ofunc_object *, struct vg3_coll_ret *);
static void f_quit(void *, struct vg3_ofunc_object *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getoofc_shot_tank(struct vg3_ofunc_objobjfunc *oofc)
{
  snprintf(oofc->oid1, sizeof(oofc->oid1), "%s", get_oid_name(OID_NAME_SHOT));
  snprintf(oofc->oid2, sizeof(oofc->oid2), "%s", get_oid_name(OID_NAME_TANK));

  oofc->f_collision = f_collision;
  oofc->f_quit = f_quit;
}


/* +++ object-to-object-functions +++ */

/* collision-function */
static int
f_collision(void *vmain, struct vg3_ofunc_object *objp1, struct vg3_ofunc_object *objp2, struct vg3_coll_ret *collret)
{
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj_shot;
  struct g_obj_tank *gobj_tank;
  const struct vg3_ofunc_objfunc *ofc_shot;
  struct vg3_ofunc_object *objp_shot, *objp_tank;
  int retw;

  if (gmain == NULL || objp1 == NULL || objp2 == NULL || collret == NULL) { return VGAG3_COLL_RETURN_NOOP; }

  /* get private structs */
  if (strcmp(objp1->oid, get_oid_name(OID_NAME_SHOT)) == 0) {
    objp_shot = objp1;
    objp_tank = objp2;
    retw = VGAG3_COLL_RETURN_DEAD;
  } else {
    objp_shot = objp2;
    objp_tank = objp1;
    retw = VGAG3_COLL_RETURN_HIT;
  }
  gobj_shot = (struct g_obj_shot *)objp_shot->ostruct;
  gobj_tank = (struct g_obj_tank *)objp_tank->ostruct;
  ofc_shot = VG3_ofunc_get_objfunc(gmain->ofstruct, objp_shot->oid);

  /* if shot cannot yet destroy own tank, return without collision */
  if (!gobj_shot->destroy_me && gobj_shot->iparent == objp_tank->instanceid) {
    return VGAG3_COLL_RETURN_NOOP;
  }

  /* if tank is immun, return without collision */
  if (gobj_tank->immun > 0) {
    return VGAG3_COLL_RETURN_NOOP;
  }

  /* increment hit-number for parent of shot unless shot has hit own tank */
  if (gobj_shot->iparent != objp_tank->instanceid) {  /* enemy tank */
    struct vg3_ofunc_object *opar = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gobj_shot->iparent);
    if (opar != NULL) {
      struct g_obj_tank *gpar = (struct g_obj_tank *)opar->ostruct;
      gmain->player[gpar->plnumber - 1].hits++;
    }
  }

  /* destroy shot */
  if (ofc_shot != NULL && ofc_shot->f_free != NULL) { ofc_shot->f_free(gmain, objp_shot); }

  /* explode tank */
  gobj_tank->health--;
  gobj_tank->i.sprt_no = TANK_IS_EXPLODING;
  VG3_sprite_rewind(gobj_tank->i.sprt[gobj_tank->i.sprt_no]);
  gobj_tank->immun = 80;

  /* return HIT or DEAD-HIT of object (objp1) */
  return retw;
}


static void
f_quit(void *vmain, struct vg3_ofunc_object *objp, struct vg3_ofunc_object *oparent)
{
  struct g_main *gmain = vmain;
  struct g_obj_tank *gobj_tank;

  if (gmain == NULL || objp == NULL || oparent == NULL) { return; }

  /* valid only when objp is shot and oparent is tank */
  if (strcmp(objp->oid, get_oid_name(OID_NAME_SHOT)) != 0) { return; }
  if (strcmp(oparent->oid, get_oid_name(OID_NAME_TANK)) != 0) { return; }

  /* tank's shot is gone, tank may fire another shot instead */
  gobj_tank = (struct g_obj_tank *)oparent->ostruct;
  gobj_tank->shots_running--;
}
