/* Copyright 2017-2020 Kurt Nienhaus
 *
 * This file is part of VgaGames3.
 * VgaGames3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

int player_select(struct g_main *, int *, int *);

static int ask_nw(struct g_main *, int *);
static int ask_plnumber(struct g_main *, int *, int *);
static int ask_inpdev(struct g_main *, int *, int, int *);

static char *bgfile = "files/bmp/tank-bg.bmp";
static struct vg3_image *bgimage = NULL;


/* ask whether network or local, number of players and their input-devices
 * @return  0 = ok, 1 = exit
 */
int
player_select(struct g_main *gmain, int *isnetwork, int *nw_jid)
{
  int retw, *jidf, jid, plnumber, i1;

  if (gmain == NULL || isnetwork == NULL || nw_jid == NULL) { return 1; }

  /* add multi-language help-file for player selection */
  VG3_multilang_add(gmain->mlang, FILES_DIR "/mlang/plysel");

  for (i1 = 0; i1 < MAXPLAYER; i1++) { gmain->player[i1].ply = PLY_IS_NOTUSED; }

  retw = 0;

  /* load background image */
  bgimage = VG3_image_load(gmain->wstruct, bgfile, 1);

  /* network game? */
  jidf = NULL;
  *nw_jid = -1;
  if (ask_nw(gmain, isnetwork) > 0) { retw = 1; goto endpl; }

  /* get gamecontroller-/joystick-devices */
  VG3_gamecontroller_getall(gmain->wstruct, &jidf);
  if (jidf[0] > MAXPLAYER) { jidf[0] = MAXPLAYER; }

  /* network game */
  if (*isnetwork) {
    *nw_jid = 0;
    if (jidf[0] > 0) {
      /* only one input-device is allowed to be used because of sending it via network */
      if (ask_inpdev(gmain, jidf, 1, nw_jid) > 0) { retw = 1; goto endpl; }
    }
    goto endpl;
  }

  /* number of players? */
  if (ask_plnumber(gmain, jidf, &plnumber) > 0) { retw = 1; goto endpl; }

  /* single-player game: use all input-devices */
  if (plnumber == 1) {
    gmain->player[0].ply = PLY_IS_LOCAL;
    gmain->player[0].jid = -1;
    goto endpl;
  }

  /* for each player: get input-device */
  for (i1 = 1; i1 <= plnumber; i1++) {
    if (jidf[0] > plnumber) {  /* too much input-devices: ask each player */
      if (ask_inpdev(gmain, jidf, i1, &jid) > 0) { retw = 1; goto endpl; }
    } else if (jidf[0] == plnumber) {  /* each player is mapped to a gamecontroller/joystick */
      jid = jidf[i1];
    } else {  /* first player is mapped to keyboard, all other are mapped to a gamecontroller/joystick */
      if (i1 == 1) {  /* first player: keyboard */
        jid = 0;
      } else {  /* next players: gamecontroller/joystick */
        jid = jidf[i1 - 1];
      }
    }
    gmain->player[i1 - 1].ply = PLY_IS_LOCAL;
    gmain->player[i1 - 1].jid = jid;
  }

endpl:
  if (jidf != NULL) { free(jidf); }
  if (bgimage != NULL) { VG3_image_unload(gmain->wstruct, bgimage); }

  return retw;
}


/* network game?
 * @return  0 = ok, 1 = exit
 */
static int
ask_nw(struct g_main *gmain, int *isnetwork)
{
  char **selarray;
  int erg;

  if (gmain == NULL || isnetwork == NULL) { return 1; }

  selarray = calloc(3, sizeof(*selarray));
  if (selarray == NULL) { fprintf(stderr, "%s\n", strerror(errno)); return 1; }
  selarray[0] = (char *)VG3_multilang_get(gmain->mlang, "local-game");
  selarray[1] = (char *)VG3_multilang_get(gmain->mlang, "network-game");
  selarray[2] = NULL;

  /* loop: select from options */
  *isnetwork = 0;
  for (;;) {
    /* draw background */
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    if (bgimage != NULL) {
      VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, NULL, 0);
    }
    /* call select-menu */
    erg = VG3_input_select(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, VG3_multilang_get(gmain->mlang, "network-headertext"), selarray, 0);
    if (erg < 0) {
      if (erg == -1) { fprintf(stderr, "%s\n", VG3_error()); }
      erg = 1;
      break;
    } else if (erg == 0) {
      if (VG3_sysmenu_exec(gmain->sysm) > 0) { erg = 1; break; }
    } else {
      if (erg == 2) { *isnetwork = 1; } else { *isnetwork = 0; }  /* interpret selected option */
      erg = 0;
      break;
    }
  }

  /* free array */
  free(selarray);

  return erg;
}


/* ask for number of players
 * @return  0 = ok, 1 = exit
 */
static int
ask_plnumber(struct g_main *gmain, int *jidf, int *plnumber)
{
  char **selarray, btext[64];
  int erg, maxplayer, plyno;

  if (gmain == NULL || jidf == NULL || plnumber == NULL) { return 1; }

  maxplayer = jidf[0] + 1;
  if (maxplayer == 1) { *plnumber = 1; return 0; }  /* no gamecontroller/joystick found: one player */

  selarray = calloc(maxplayer + 1, sizeof(*selarray));
  if (selarray == NULL) { fprintf(stderr, "%s\n", strerror(errno)); return 1; }
  for (plyno = 1; plyno <= maxplayer; plyno++) {
    snprintf(btext, sizeof(btext), "%d %s", plyno, VG3_multilang_get(gmain->mlang, "player"));
    selarray[plyno - 1] = strdup(btext);
  }
  selarray[maxplayer] = NULL;

  /* loop: select number of players */
  *plnumber = 1;
  for (;;) {
    /* draw background */
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    if (bgimage != NULL) {
      VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, NULL, 0);
    }
    /* call select-menu */
    erg = VG3_input_select(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, VG3_multilang_get(gmain->mlang, "plyno-headertext"), selarray, 0);
    if (erg < 0) {
      if (erg == -1) { fprintf(stderr, "%s\n", VG3_error()); }
      erg = 1;
      break;
    } else if (erg == 0) {
      if (VG3_sysmenu_exec(gmain->sysm) > 0) { erg = 1; break; }
    } else {
      *plnumber = erg;  /* set number of players from selected option */
      erg = 0;
      break;
    }
  }

  /* free array */
  for (plyno = 1; plyno <= maxplayer; plyno++) {
    if (selarray[plyno - 1] != NULL) { free(selarray[plyno - 1]); }
  }
  free(selarray);

  return erg;
}


/* ask for input-device of player
 * @return  0 = ok, 1 = exit
 */
static int
ask_inpdev(struct g_main *gmain, int *jidf, int plyno, int *jid)
{
  static char *header_fontfile = "10x17.font";  /* font-file for header */
  static char *item_fontfile = NULL;  /* font-file for entries */
  static int fg_color = VGAG3_COLOR_YELLOW;  /* foreground color */
  static int bg_color = VGAG3_COLOR_BLUE;  /* background color */
  int retw, fcolor, bcolor, jno, i1, loopno;
  struct vg3_text stxt;
  struct vg3_rect brect;
  char btext[64], devname[128];
  struct vg3_gamecontroller gcs;

  if (gmain == NULL || jidf == NULL || plyno < 1 || plyno > MAXPLAYER || jid == NULL) { return 1; }

  /* loop: discover input-device of player */

  VG3_discard_input(gmain->wstruct);

  retw = 0;
  *jid = -1;
  *devname = '\0';
  loopno = 0;
  for (;;) {
    if (VG3_inputevent_update(gmain->wstruct)) { retw = 1; break; }

    /* ESC: system-menu */
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) {
      if (VG3_sysmenu_exec(gmain->sysm) > 0) { retw = 1; break; }
    }

    /* ALT+Q: exit */
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
      && VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) { retw = 1; break; }

    if (*jid == -1) {
      /* is a button/axis of a gamecontroller/joystick pressed? */
      for (jno = 1; jno <= jidf[0]; jno++) {  /* all input-devices */
        for (i1 = 0; i1 < MAXPLAYER; i1++) {  /* is input-device already selected? */
          if (gmain->player[i1].ply == PLY_IS_LOCAL && gmain->player[i1].jid == jidf[jno]) { break; }
        }
        if (i1 < MAXPLAYER) { continue; }  /* input-device is already selected */
        if (VG3_gamecontroller_getpressed(gmain->wstruct, jidf[jno]) != VGAG3_GC_NOKEY) {  /* pressed */
          *jid = jidf[jno];
          break;
        }
      }
    }

    if (*jid == -1) {
      /* is a key of the keyboard pressed? */
      for (i1 = 0; i1 < MAXPLAYER; i1++) {  /* is input-device already selected? */
        if (gmain->player[i1].ply == PLY_IS_LOCAL && gmain->player[i1].jid == 0) { break; }
      }
      if (i1 == MAXPLAYER) {
        if (VG3_key_getpressed(gmain->wstruct) != VGAG3_KEY_NOKEY) {
          *jid = 0;
        }
      }
    }

    /* once a button/key is pressed, get name of device */
    if (*devname == '\0' && *jid >= 0) {
      if (*jid > 0) {
        VG3_gamecontroller_getinfo(gmain->wstruct, *jid, &gcs);
        snprintf(devname, sizeof(devname), "%s", gcs.name);
      } else {
        snprintf(devname, sizeof(devname), "%s", VG3_multilang_get(gmain->mlang, "keyboard"));
      }
      loopno = 30;  /* number of loops to show device-name */
    }

    /* set borders for text */
    brect.x = brect.y = 10;
    brect.w = gmain->winw - 20;
    brect.h = gmain->winh - 20;

    /* draw background */
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    if (bgimage != NULL) {
      struct vg3_image_attributes bgattr;
      VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&bgattr);
      bgattr.bright = 128;
      VG3_image_copy(gmain->wstruct, NULL, bgimage, gmain->winw / 2, gmain->winh / 2, &bgattr, 0);
    }

    /* draw header-text centered */
    snprintf(btext, sizeof(btext), "%s %d", VG3_multilang_get(gmain->mlang, "player"), plyno);
    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, header_fontfile, '\n', 0, btext);
    VG3_draw_textline(gmain->wstruct, NULL, &brect, ' ', &stxt, fg_color, VGAG3_COLOR_TRANSPARENT, 0, 0, 1);

    /* draw some empty lines */
    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, header_fontfile, '\n', 0, "\n\n");
    VG3_draw_textline(gmain->wstruct, NULL, &brect, ' ', &stxt, fg_color, VGAG3_COLOR_TRANSPARENT, 0, 0, 0);

    /* draw text */
    if (*devname != '\0') {
      fcolor = VG3_color_brightness(fg_color, 50);
      bcolor = VGAG3_COLOR_TRANSPARENT;
    } else {
      fcolor = fg_color;
      bcolor = VGAG3_COLOR_TRANSPARENT;
    }
    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, item_fontfile, '\n', 0, VG3_multilang_get(gmain->mlang, "press-key"));
    VG3_draw_textline(gmain->wstruct, NULL, &brect, ' ', &stxt, fcolor, bcolor, 0, 0, 0);

    /* if device-name is set, draw it */
    if (*devname != '\0') {
      VGAG3_TEXT_ATTRIBUTES_SET(&stxt, item_fontfile, '\n', 0, "\n\n");
      VG3_draw_textline(gmain->wstruct, NULL, &brect, ' ', &stxt, fg_color, VGAG3_COLOR_TRANSPARENT, 0, 0, 0);
      bcolor = VG3_color_brightness(bg_color, 50);
      VGAG3_TEXT_ATTRIBUTES_SET(&stxt, item_fontfile, '\n', 0, devname);
      VG3_draw_textline(gmain->wstruct, NULL, &brect, ' ', &stxt, fg_color, bcolor, 0, 0, 1);
    }

    /* draw last line */
    draw_lastline(gmain, item_fontfile, brect, fg_color);

    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(80);

    if (loopno > 0 && --loopno == 0) { break; }
  }

  VG3_discard_input(gmain->wstruct);

  return retw;
}
