#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

const char * get_oid_name(int);

int gameskel_intro(struct vg3_window *);
void gameskel_setkeys(void *, struct vg3_sysmenu_submenu *, struct vg3_sysmenu_submenu *);
void gameskel_beforelevels(void *, struct menu_helper *);
void gameskel_levelstart(void *);
int gameskel_leveldone(void *);
void gameskel_levelstop(void *);
void gameskel_afterlevels(void *, struct menu_helper *);
void gameskel_settingsread(void *, FILE *);
void gameskel_settingssave(void *, FILE *);
void gameskel_settingsreset(void *);

void check_direction(void *, int, int, int, int *, int *);


/* return target steps (left and right) of moving smurf */
void
check_direction(void *vmain, int pfstep, int steps, int species, int *targetstep_left, int *targetstep_right)
{
  struct g_main *gmain = (struct g_main *)vmain;
  int targetstep[2], tidx;

  if (targetstep_left != NULL) { *targetstep_left = -1; }
  if (targetstep_right != NULL) { *targetstep_right = -1; }
  if (gmain == NULL || targetstep_left == NULL || targetstep_right == NULL) { return; }
  if (pfstep < 0 || steps < 1) { return; }

  for (targetstep[0] = targetstep[1] = pfstep; steps > 0; steps--) {
    /* get next step left and right */
    if (targetstep[0] >= 0) {
      targetstep[0] = gmain->game.pfmap[targetstep[0]].next[0];
    }
    if (targetstep[1] >= 0) {
      if (gmain->game.pfmap[targetstep[1]].next[1] >= 0) {
        targetstep[1] = gmain->game.pfmap[targetstep[1]].next[1];
      } else {
        targetstep[1] = gmain->game.pfmap[targetstep[1]].next[0];
      }
    }
    /* check next step left and right */
    for (tidx = 0; tidx <= 1; tidx++) {
      if (targetstep[tidx] < 0) { continue; }
      if (gmain->game.pfmap[targetstep[tidx]].village) {
        if (steps > 1 || species != SPECIES_SMURF) { targetstep[tidx] = -1; continue; }
      }
      if (gmain->game.pfmap[targetstep[tidx]].sarsaparille) {
        if (steps == 1 && species == SPECIES_FLY) { targetstep[tidx] = -1; continue; }
      }
      if (gmain->game.pfmap[targetstep[tidx]].river) {
        if (steps == 1 && species != SPECIES_FLY) { targetstep[tidx] = -1; continue; }
      }
    }
  }

  *targetstep_left = targetstep[0];
  *targetstep_right = targetstep[1];
}


/* get_oid_name:
 * returns object-ID as a string
 * @param oidn  object-ID (OID_NAMES)
 * @return      object-ID as string (or empty)
 */
const char *
get_oid_name(int oidn)
{
  switch(oidn) {
    case OID_NAME_PLAYFIELD:  return "playfield";
    case OID_NAME_SMURF:      return "smurf";
    case OID_NAME_DIRSEL:     return "select-direction";
    case OID_NAME_MOUSESEL:   return "mouse-select";
    case OID_NAME_DICE:       return "dice";
    case OID_NAME_ANIMATION:  return "animation";
    case OID_NAME_MUSIC:      return "music";
  }
  return "";
}


/* gameskel_intro:
 * play intro(-film) before main-menu
 * (once called)
 * @param wstruct  window struct
 * @return         0 = OK or 1 = exit
 */
int
gameskel_intro(struct vg3_window *wstruct)
{
  int erg;
  
  if (wstruct == NULL) { return 0; }
  
  erg = VG3_film_play(wstruct, NULL, FILES_DIR "/intro/intro.film", 1);
  if (erg != 1) { erg = 0; }
  
  return erg;
}


/* gameskel_setkeys:
 * install keys
 * (called before showing the main-menu)
 * @param vmain      main game struct (struct g_main)
 * @param subm_keyb  submenu for keyboard keys, or NULL = no keys
 * @param subm_gc    submenu for gamecontroller/joystick keys, or NULL = no keys
 */
void
gameskel_setkeys(void *vmain, struct vg3_sysmenu_submenu *subm_keyb, struct vg3_sysmenu_submenu *subm_gc)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL || gmain->skeys == NULL) { return; }

  /* load multilanguage for game */
  if (gmain->mlang != NULL) {
    VG3_multilang_add(gmain->mlang, FILES_DIR "/mlang/game");
  }

  /* set keys */

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_GO_RIGHT, "right", "go-right",
                       subm_keyb, VGAG3_KEY_RCURS,
                       NULL, VGAG3_GC_NOKEY, NULL);

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_GO_LEFT, "left", "go-left",
                       subm_keyb, VGAG3_KEY_LCURS,
                       NULL, VGAG3_GC_NOKEY, NULL);

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_STOP, "stop", "dice-stop",
                       subm_keyb, VGAG3_KEY_SPACE,
                       NULL, VGAG3_GC_NOKEY, NULL);

  (void)subm_gc;
}


/* gameskel_beforelevels:
 * do anything before first level of game starts,
 * set element "exit" of main game struct with a value of EXITDEFS
 * (called just before entering the game-level loop)
 * @param vmain  main game struct (struct g_main)
 * @param mhelp  menu_helper
 *
 * EXITDEFS:
 * - EXITDEF_NOOP:           OK, go on
 * - EXITDEF_GOTO_MAINMENU:  quit and go to main-menu
 * - EXITDEF_EXIT_OK:        exit game with no error
 * - EXITDEF_EXIT_FAILURE:   exit game with error
 */
void
gameskel_beforelevels(void *vmain, struct menu_helper *mhelp)
{
  struct g_main *gmain = (struct g_main *)vmain;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_ofunc_object *objp;
  int mret;

  /* select difficulty */
  mhelp->sel.title = (char *)VG3_multilang_get(gmain->mlang, "game_difficulty");
  mhelp->sel.item[0] = (char *)VG3_multilang_get(gmain->mlang, "game_easy");
  mhelp->sel.item[1] = (char *)VG3_multilang_get(gmain->mlang, "game_difficult");
  mhelp->sel.item[2] = NULL;
  mret = mhelp->func.show_menu(mhelp);
  if (mret == -2) { gmain->exit = EXITDEF_EXIT_OK; return; }
  if (mret == -1) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }
  if (mret == 0) { gmain->exit = EXITDEF_GOTO_MAINMENU; return; }
  if (mret == 1) { mret = 0; } else { mret = 1; }

  /* create playfield-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYFIELD));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_PLAYFIELD));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }
  objp = ofc->f_new(gmain, 0, mret);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }

  gmain->exit = EXITDEF_NOOP;
}


/* gameskel_levelstart:
 * initialize for actual gamelevel,
 * set element "exit" of main game struct with a value of EXITDEFS
 * (called for each level just before entering the game loop)
 * @param vmain  main game struct (struct g_main)
 *
 * EXITDEFS:
 * - EXITDEF_NOOP:           OK, go on
 * - EXITDEF_GOTO_MAINMENU:  quit loop and go to main-menu
 * - EXITDEF_EXIT_OK:        exit game with no error
 * - EXITDEF_EXIT_FAILURE:   exit game with error
 */
void
gameskel_levelstart(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_ofunc_object *objp;

  if (gmain == NULL || gmain->gamelevel < 1) {
    fprintf(stderr, "gameskel_levelstart: %s\n", strerror(EINVAL));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }

  if (gmain->gamelevel > 1) {  /* no more game level */
    gmain->exit = EXITDEF_GOTO_MAINMENU;
    return;
  }

  /* activate smurf-object-management */
  if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_SMURF), 20, 7) != 0) {
    fprintf(stderr, "%s\n", VG3_error());
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }

  /* create papasmurf- and fly-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_SMURF));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }
  objp = ofc->f_new(gmain, 0, SPECIES_PAPASMURF);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }
  objp = ofc->f_new(gmain, 0, SPECIES_FLY);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }

  /* create dirsel-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_DIRSEL));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_DIRSEL));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }
  objp = ofc->f_new(gmain, 0);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }

  /* create mousesel-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_MOUSESEL));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_MOUSESEL));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }
  objp = ofc->f_new(gmain, 0);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }

  /* create dice-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_DICE));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_DICE));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }
  objp = ofc->f_new(gmain, 0);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }

  /* create animation-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_ANIMATION));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_ANIMATION));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }
  objp = ofc->f_new(gmain, 0);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }

  /* create music-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_MUSIC));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_MUSIC));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }
  objp = ofc->f_new(gmain, 0);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }

  /* create actionstack and push starting animation */
  gmain->game.astck = VG3_actionstack_new();
  { struct vg3_hash *hparm = VG3_hash_new();
    VG3_hash_setint(hparm, "num", sizeof("num"), ANIMATION_STARTING);
    VG3_actionstack_push(gmain->game.astck, UART_ANIMATION, hparm, 0);
    VG3_hash_free(hparm);
  }

  gmain->exit = EXITDEF_NOOP;
}


/* gameskel_leveldone:
 * check whether quit game-loop,
 * may evaluate element "exit" of main game struct,
 * set eventually element "exit" of main game struct with a value of EXITDEFS
 * (called repeatedly in the game loop)
 * @param vmain  main game struct (struct g_main)
 * @return       0 = OK
 *               1 = quit game-loop and set EXITDEFS
 *
 * EXITDEFS (on return = 1):
 * - EXITDEF_NOOP:           OK, go to next level
 * - EXITDEF_REDO_LEVEL:     restart actual game-level
 * - EXITDEF_GOTO_MAINMENU:  quit loop and go to main-menu
 * - EXITDEF_EXIT_OK:        exit game with no error
 * - EXITDEF_EXIT_FAILURE:   exit game with error
 */
int
gameskel_leveldone(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL || gmain->gamelevel < 1) {
    fprintf(stderr, "gameskel_leveldone: %s\n", strerror(EINVAL));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return 1;
  }

  /* quit if exit-request was set */
  if (gmain->exit != EXITDEF_NOOP) { return 1; }

  return 0;
}


/* gameskel_levelstop:
 * destruct for actual gamelevel,
 * element "exit" of main game struct contains the exit-value (of EXITDEFS) from gameskel_leveldone()
 * (called for each level just after quitting the game loop)
 * @param vmain  main game struct (struct g_main)
 */
void
gameskel_levelstop(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL || gmain->gamelevel < 1) { return; }

  /* deactivate smurf-object-management (and separate created papa-smurf and fly) */
  VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_SMURF));

  /* destroy dirsel-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_DIRSEL));

  /* destroy mousesel-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_MOUSESEL));

  /* destroy dice-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_DICE));

  /* destroy animation-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_ANIMATION));

  /* destroy music-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_MUSIC));

  /* destroy actionstack */
  if (gmain->game.astck != NULL) { VG3_actionstack_free(gmain->game.astck); gmain->game.astck = NULL; }
}


/* gameskel_afterlevels:
 * do anything and clean up rest of game-variables,
 * element "exit" of main game struct contains the exit-value (of EXITDEFS) from gameskel_leveldone()
 * (called just after quitting the game-level loop, and the window has not been cleared)
 * @param vmain   main game struct (struct g_main)
 * @param mhelp   menu_helper
 */
void
gameskel_afterlevels(void *vmain, struct menu_helper *mhelp)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL) { return; }

  (void)mhelp;

  if (gmain->game.showending) {
    /* show number of saved smurfs as animation */
    const struct vg3_ofunc_objfunc *ofc;
    struct vg3_ofunc_object *objp;
    struct vg3_hash *hparm;
    char buf[64];
    ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_ANIMATION));
    if (ofc == NULL) { return; }
    objp = ofc->f_new(gmain, 0);
    if (objp == NULL) { VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_ANIMATION)); return; }
    hparm = VG3_hash_new();
    VG3_hash_setint(hparm, "num", sizeof("num"), ANIMATION_ENDING);
    snprintf(buf, sizeof(buf), "%d / %d\n\n%s", gmain->game.smurf_saved, gmain->game.smurf_total, VG3_multilang_get(gmain->mlang, "smurf-saved"));
    VG3_hash_set(hparm, "text", sizeof("text"), buf, strlen(buf) + 1);
    gmain->game.astck = VG3_actionstack_new();
    VG3_actionstack_push(gmain->game.astck, UART_ANIMATION, hparm, 0);
    { struct vg3_actionstack_elem actionp;
      struct vg3_image *bgimg = VG3_image_clone(gmain->wstruct, NULL, NULL, NULL, NULL);
      for (;;) {
        if (VG3_inputevent_update(gmain->wstruct)) { break; }
        if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_PRESSED)) { break; }
        actionp = VG3_actionstack_get(gmain->game.astck);
        if (actionp.stack_id == 0) { break; }
        if (actionp.name_id != UART_ANIMATION) { break; }
        if (bgimg != NULL) {
          VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
          VG3_image_copy(gmain->wstruct, NULL, bgimg, gmain->winw / 2, gmain->winh / 2, NULL, 0);
        }
        VG3_ofunc_objlist_call_run(gmain->ofstruct, gmain);
        VG3_ofunc_objlist_call_draw(gmain->ofstruct, gmain);
        VG3_window_update(gmain->wstruct, 0, 0);
        VG3_wait_time(LOOP_TIME);
      }
      if (bgimg != NULL) { VG3_image_unload(gmain->wstruct, bgimg); }
    }
    if (gmain->game.astck != NULL) { VG3_actionstack_free(gmain->game.astck); gmain->game.astck = NULL; }
    VG3_hash_free(hparm);
    VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_ANIMATION));
  }

  /* destroy playfield-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_PLAYFIELD));
}


/* gameskel_settingsread:
 * read individual settings from opened settings-file
 * (called by settings_readfromfile())
 * @param vmain   main game struct (struct g_main)
 * @param ffp     opened settings-file
 */
void
gameskel_settingsread(void *vmain, FILE *ffp)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL) { return; }

  (void)ffp;
}


/* gameskel_settingssave:
 * write individual settings to opened settings-file
 * (called by settings_savetofile())
 * @param vmain   main game struct (struct g_main)
 * @param ffp     opened settings-file
 */
void
gameskel_settingssave(void *vmain, FILE *ffp)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL) { return; }

  (void)ffp;
}


/* gameskel_settingsreset:
 * reset individual settings for starting with gamelevel=1
 * (called when selecting "new game")
 * @param vmain   main game struct (struct g_main)
 */
void
gameskel_settingsreset(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL) { return; }
}
