#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-animation.h"

void getofc_animation(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_animation(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_ANIMATION));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameter: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_animation *gobj;
  va_list ap;
  int i1;
  char buf[128];

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->num = 0;

  for (i1 = 0; i1 < ANIMATION_ENDING; i1++) {
    snprintf(buf, sizeof(buf), "%s/bmp/animation/num%d.sprite", FILES_DIR, i1 + 1);
    gobj->a[i1].sprt = VG3_sprite_load(gmain->wstruct, buf);
    if (gobj->a[i1].sprt == NULL) { return NULL; }
  }

  /* create and fill animation-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_ANIMATION));
  objp->drawlevel = 4;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert animation-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_animation *gobj;
  int i1;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_animation *)objp->ostruct;

  /* remove animation-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free animation-instance */
  for (i1 = 0; i1 < ANIMATION_ENDING; i1++) {
    if (gobj->a[i1].sprt != NULL) { VG3_sprite_free(gobj->a[i1].sprt); }
  }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_animation *gobj;
  struct vg3_actionstack_elem actionp;

  if (gmain == NULL || objp == NULL) { return; }

  actionp = VG3_actionstack_get(gmain->game.astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != UART_ANIMATION) { return; }

  gobj = (struct g_obj_animation *)objp->ostruct;

  if (gobj->num == 0) {  /* new animation */
    gobj->num = VG3_hash_getint(actionp.hparm, "num", sizeof("num"));
    if (gobj->num < 1 || gobj->num > ANIMATION_ENDING) { gobj->num = 0; return; }
    gobj->text = VG3_hash_get(actionp.hparm, "text", sizeof("text"), NULL);
    if (gobj->text != NULL && *gobj->text == '\0') { gobj->text = NULL; }
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_animation *gobj;
  struct vg3_actionstack_elem actionp;
  struct vg3_image *imgp;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  actionp = VG3_actionstack_get(gmain->game.astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != UART_ANIMATION) { return; }

  gobj = (struct g_obj_animation *)objp->ostruct;

  /* draw animation-instance */
  if (gobj->num == 0) { return; }
  if (VG3_sprite_get(gobj->a[gobj->num - 1].sprt, &imgp, &iattr, NULL) == 0) {
    VG3_sprite_rewind(gobj->a[gobj->num - 1].sprt);
    gobj->num = 0;
    VG3_actionstack_pop(gmain->game.astck);
    return;
  }
  VG3_image_copy(gmain->wstruct, NULL, imgp, gmain->winw / 2, gmain->winh / 2, &iattr, 0);
  if (gobj->text != NULL) {
    VG3_text_simpledraw(gmain->wstruct, NULL, "8x8.font+", gmain->winw / 2, gmain->winh / 2, gobj->text, VGAG3_COLOR_BLACK, VGAG3_COLOR_TRANSPARENT, 1);
  }
}
