#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-dice.h"

void getofc_dice(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);

static int number_black[] = { 5, 1, 2, 3, 4, 5 };
static int number_blue[] = { 14, 1, 0, 2, 3, 4, 0, 5, 1, 2, 0, 3, 0, 4, 5 };


/* +++ get-function +++ */

void
getofc_dice(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_DICE));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameter: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_dice *gobj;
  va_list ap;
  int i1;
  char buf[128];

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->img_dice_blue = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/dice/dice-blue.bmp", 1);
  if (gobj->img_dice_blue == NULL) { return NULL; }
  gobj->img_dice_black = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/dice/dice-black.bmp", 1);
  if (gobj->img_dice_black == NULL) { return NULL; }
  gobj->img_dice_shadow = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/dice/dice-shadow.bmp", 1);
  if (gobj->img_dice_shadow == NULL) { return NULL; }
  for (i1 = 0; i1 < MAX_DICE_WFW; i1++) {
    snprintf(buf, sizeof(buf), "%s/bmp/dice/wfw%d.bmp", FILES_DIR, i1 + 1);
    gobj->img_wfw[i1] = VG3_image_load(gmain->wstruct, buf, 1);
    if (gobj->img_wfw[i1] == NULL) { return NULL; }
  }
  for (i1 = 0; i1 < MAX_DICE_NUMBER; i1++) {
    snprintf(buf, sizeof(buf), "%s/bmp/dice/wfz-%d.bmp", FILES_DIR, i1 + 1);
    gobj->img_wfz[i1] = VG3_image_load(gmain->wstruct, buf, 1);
    if (gobj->img_wfz[i1] == NULL) { return NULL; }
  }
  gobj->img_wfz_fly = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/dice/wfz-fly.bmp", 1);
  if (gobj->img_wfz_fly == NULL) { return NULL; }
  gobj->snd_falling = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/dice.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);

  /* create and fill dice-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_DICE));
  objp->drawlevel = 4;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert dice-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_dice *gobj;
  int i1;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_dice *)objp->ostruct;

  /* remove dice-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free dice-instance */
  if (gobj->img_dice_blue != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_dice_blue); }
  if (gobj->img_dice_black != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_dice_black); }
  if (gobj->img_dice_shadow != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_dice_shadow); }
  for (i1 = 0; i1 < MAX_DICE_WFW; i1++) {
    if (gobj->img_wfw[i1] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_wfw[i1]); }
  }
  for (i1 = 0; i1 < MAX_DICE_NUMBER; i1++) {
    if (gobj->img_wfz[i1] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_wfz[i1]); }
  }
  if (gobj->img_wfz_fly != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_wfz_fly); }
  if (gobj->snd_falling > 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_falling); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_dice *gobj;
  struct vg3_actionstack_elem actionp;
  const struct vg3_ofunc_objfunc *ofc;

  if (gmain == NULL || objp == NULL) { return; }

  actionp = VG3_actionstack_get(gmain->game.astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != UART_DICE) { return; }

  gobj = (struct g_obj_dice *)objp->ostruct;

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
  if (ofc == NULL || ofc->f_data == NULL) {
    fprintf(stderr, "Object \"%s\" or it's f_data() not found\n", get_oid_name(OID_NAME_SMURF));
    abort();
  }

  if (gobj->anim_max2 == 0) {  /* initialize */
    gobj->number_row = number_blue;
    gobj->anim_max1 = gobj->number_row[0] + ZUFALL(0, gobj->number_row[0] - 1);
    gobj->anim_max2 = gobj->anim_max1 + 10;
    gobj->number_idx = ZUFALL(0, gobj->number_row[0] - 1);
    gobj->anim_mom = 0;
    gobj->showdice = 0;
    gobj->dice_idx = ZUFALL(0, MAX_DICE_WFW - 1);
    gobj->steps = 0;
    gobj->isblack = 0;
    if (VG3_ofunc_objlist_find_subobj(gmain->ofstruct, get_oid_name(OID_NAME_SMURF), SPECIES_SMURF, NULL) == 0
        && VG3_ofunc_objlist_find_subobj(gmain->ofstruct, get_oid_name(OID_NAME_SMURF), SPECIES_PAPASMURF, NULL) == 0
       ) {  /* only black smurfs existing: inhibit random fly */
      gobj->isblack = 1;
      gobj->number_row = number_black;
      gobj->anim_max1 = gobj->number_row[0] + ZUFALL(0, gobj->number_row[0] - 1);
      gobj->anim_max2 = gobj->anim_max1 + 10;
      gobj->number_idx = ZUFALL(0, gobj->number_row[0] - 1);
    } else if (VG3_ofunc_objlist_find_subobj(gmain->ofstruct, get_oid_name(OID_NAME_SMURF), SPECIES_SMURF_BLACK, NULL) == 0
        && VG3_ofunc_objlist_find_subobj(gmain->ofstruct, get_oid_name(OID_NAME_SMURF), SPECIES_PAPASMURF_BLACK, NULL) == 0
       ) {  /* no black smurf existing: inhibit random fly */
      gobj->number_row = number_black;
      gobj->number_idx = ZUFALL(0, gobj->number_row[0] - 1);
    }
    VG3_discard_input(gmain->wstruct);
  }

  if (gobj->showdice > 0) {  /* show thrown dice */
    if (--gobj->showdice == 0) {
      if (gobj->steps > 0 && !gobj->isblack) {
        struct vg3_ofunc_objsnap *osnap;
        struct vg3_ofunc_object *objp2;
        int pfstep, pfstep1, pfstep2;
        osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
        while ((objp2 = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
          if (objp2->subid != SPECIES_SMURF && objp2->subid != SPECIES_PAPASMURF) { continue; }
          ofc->f_data(gmain, objp2, &pfstep);
          check_direction(gmain, pfstep, gobj->steps, objp2->subid, &pfstep1, &pfstep2);
          if (pfstep1 >= 0 || pfstep2 >= 0) { break; }
        }
        VG3_ofunc_objlist_freelist(osnap);
        if (objp2 == NULL) { gobj->steps = 0; }
      }
      if (gobj->steps > 0) {
        struct vg3_hash *hparm = VG3_hash_new();
        VG3_actionstack_pop(gmain->game.astck);

        if (gobj->number_row == number_black) {  /* fly, or no existing black smurf: insert additional fly */
          struct vg3_ofunc_object *objp2;
          int pfstep, pfstep1, pfstep2, steps;
          if (!gobj->isblack) { steps = ZUFALL(1, MAX_DICE_NUMBER); } else { steps = gobj->steps; }
          if (VG3_ofunc_objlist_find_subobj(gmain->ofstruct, get_oid_name(OID_NAME_SMURF), SPECIES_FLY, &objp2) > 0) {
            ofc->f_data(gmain, objp2, &pfstep);
            check_direction(gmain, pfstep, steps, objp2->subid, &pfstep1, &pfstep2);
            if (pfstep1 >= 0 || pfstep2 >= 0) {
              VG3_hash_setint(hparm, "instanceid", sizeof("instanceid"), objp2->instanceid);
              VG3_hash_setint(hparm, "steps", sizeof("steps"), steps);
              VG3_hash_setint(hparm, "direction", sizeof("direction"), 0);
              VG3_actionstack_push(gmain->game.astck, UART_MOVSMURF, hparm, 0);
            }
          }
        }
        if (!gobj->isblack) {  /* insert smurf */
          VG3_hash_clear(hparm);
          VG3_hash_setint(hparm, "steps", sizeof("steps"), gobj->steps);
          VG3_actionstack_push(gmain->game.astck, UART_SELSMURF, hparm, 0);
        }
        VG3_hash_free(hparm);
        actionp = VG3_actionstack_get(gmain->game.astck);
        gobj->anim_max1 = gobj->anim_max2 = 0;
      } else {
        gobj->isblack = 1;
        gobj->number_row = number_black;
        gobj->number_idx = ZUFALL(0, gobj->number_row[0] - 1);
        gobj->steps = ZUFALL(1, MAX_DICE_NUMBER);
        gobj->anim_max1 = gobj->number_row[0] + ZUFALL(0, gobj->number_row[0] - 1);
        gobj->anim_max2 = gobj->anim_max1 + 10;
        gobj->anim_mom = 0;
        VG3_discard_input(gmain->wstruct);
      }
    }

  } else {  /* show falling dice */
    if (gobj->anim_mom < gobj->anim_max1) {
      if (gobj->snd_falling > 0 && gobj->anim_mom == 0) { VG3_audio_play(gmain->wstruct, gobj->snd_falling, 0, 0); }
      if (++gobj->number_idx == gobj->number_row[0]) { gobj->number_idx = 1; }
      gobj->steps = gobj->number_row[gobj->number_idx];
      gobj->anim_mom++;
      if (VG3_keys_key_ispressed(gmain->skeys, gmain->nwptr, 0, 0, KEYDEF_STOP, VGAG3_IS_NEW_PRESSED)) {
        gobj->anim_mom = gobj->anim_max1;
      }
    } else if (++gobj->anim_mom == gobj->anim_max2) {
      if (gobj->steps == 0) {
        gobj->isblack = 1;
        gobj->number_row = number_black;
        gobj->number_idx = ZUFALL(0, gobj->number_row[0] - 1);
      }
      gobj->showdice = 20;
    }
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_dice *gobj;
  struct vg3_actionstack_elem actionp;

  if (gmain == NULL || objp == NULL) { return; }

  actionp = VG3_actionstack_get(gmain->game.astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != UART_DICE) { return; }

  gobj = (struct g_obj_dice *)objp->ostruct;

  /* draw dice-instance */

  VG3_image_copy(gmain->wstruct, NULL, gobj->img_dice_shadow, gmain->winw / 2 + 15, gmain->winh / 2 + 15, NULL, 0);
  if (gobj->isblack) {
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_dice_black, gmain->winw / 2, gmain->winh / 2, NULL, 0);
  } else {
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_dice_blue, gmain->winw / 2, gmain->winh / 2, NULL, 0);
  }

  if (gobj->showdice) {
    if (gobj->steps > 0) {
      VG3_image_copy(gmain->wstruct, NULL, gobj->img_wfz[gobj->steps - 1], gmain->winw / 2, gmain->winh / 2, NULL, 0);
    } else {
      VG3_image_copy(gmain->wstruct, NULL, gobj->img_wfz_fly, gmain->winw / 2, gmain->winh / 2, NULL, 0);
    }
  } else {
    if (gobj->anim_mom > gobj->anim_max1) {
      if (gobj->anim_max2 - gobj->anim_mom == 8
          || gobj->anim_max2 - gobj->anim_mom == 5
          || gobj->anim_max2 - gobj->anim_mom == 2) {
        if (++gobj->dice_idx == MAX_DICE_WFW) { gobj->dice_idx = 0; }
      }
    } else {
      if (++gobj->dice_idx == MAX_DICE_WFW) { gobj->dice_idx = 0; }
    }
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_wfw[gobj->dice_idx], gmain->winw / 2, gmain->winh / 2, NULL, 0);
  }
}
