#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-mousesel.h"

void getofc_mousesel(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_mousesel(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_MOUSESEL));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameter: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_mousesel *gobj;
  va_list ap;
  int i1;
  char buf[128];

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->img = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/mouse/mouse.bmp", 1);
  if (gobj->img == NULL) { return NULL; }

  gobj->sprt = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/mouse/mouse-click.sprite");
  if (gobj->sprt == NULL) { return NULL; }

  for (i1 = 0; i1 < MAX_DICE_NUMBER; i1++) {
    snprintf(buf, sizeof(buf), "%s/bmp/dice/wfz-%d.bmp", FILES_DIR, i1 + 1);
    gobj->img_wfz[i1] = VG3_image_load(gmain->wstruct, buf, 1);
    if (gobj->img_wfz[i1] == NULL) { return NULL; }
  }

  VG3_image_getsize(gmain->wstruct, gobj->img, NULL, &gobj->xwidth, &gobj->ywidth);

  /* create and fill mousesel-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_MOUSESEL));
  objp->drawlevel = 5;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert mousesel-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_mousesel *gobj;
  int i1;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_mousesel *)objp->ostruct;

  /* remove mousesel-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free mousesel-instance */
  if (gobj->img != NULL) { VG3_image_unload(gmain->wstruct, gobj->img); }
  if (gobj->sprt != NULL) { VG3_sprite_free(gobj->sprt); }
  for (i1 = 0; i1 < MAX_DICE_NUMBER; i1++) {
    if (gobj->img_wfz[i1] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_wfz[i1]); }
  }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_mousesel *gobj;
  struct vg3_actionstack_elem actionp;
  struct vg3_rect rect;
  struct vg3_coll *collp;
  int anzcoll, idxcoll, collretw, pfstep, pfstep1, pfstep2;
  const struct vg3_ofunc_objobjfunc *oofc;
  const struct vg3_ofunc_objfunc *ofc;

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
  if (ofc == NULL || ofc->f_data == NULL) {
    fprintf(stderr, "Object \"%s\" or it's f_data() not found\n", get_oid_name(OID_NAME_SMURF));
    abort();
  }

  if (gmain == NULL || objp == NULL) { return; }

  actionp = VG3_actionstack_get(gmain->game.astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != UART_MOUSESEL) { return; }

  gobj = (struct g_obj_mousesel *)objp->ostruct;
  if (gobj->instanceid > 0) { return; }

  /* check if mouse is over a selectable smurf */

  VG3_mouse_position(gmain->wstruct, &rect.x, &rect.y);
  rect.w = rect.h = 1;

  anzcoll = VG3_coll_q_find(gmain->qdtr, &rect, NULL, &collp);
  if (anzcoll <= 0) { return; }
  for (idxcoll = 0; idxcoll < anzcoll; idxcoll++) {
    if (strcmp(collp[0].oid, get_oid_name(OID_NAME_SMURF)) == 0) { break; }
  }
  if (idxcoll == anzcoll) { free(collp); return; }

  /* check if smurf is movable given steps */
  ofc->f_data(gmain, (struct vg3_ofunc_object *)collp[idxcoll].optr, &pfstep);
  if (objp->subid == SPECIES_SMURF_BLACK || objp->subid == SPECIES_PAPASMURF_BLACK || objp->subid == SPECIES_FLY) {
    free(collp);
    return;
  }
  check_direction(gmain, pfstep, VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")), objp->subid, &pfstep1, &pfstep2);
  if (pfstep1 < 0 && pfstep2 < 0) { free(collp); return; }  /* moving not possible */

  /* call collision function to mouse-over or select this smurf */
  oofc = VG3_ofunc_get_objobjfunc(gmain->ofstruct, objp->oid, collp[idxcoll].oid);
  if (oofc == NULL || oofc->f_collision == NULL) { free(collp); return; }
  collretw = oofc->f_collision(gmain, objp, collp[idxcoll].optr, &collp[idxcoll].ret);
  if (collretw < 0) { free(collp); return; }
  free(collp);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_mousesel *gobj;
  struct vg3_actionstack_elem actionp;
  int xpos, ypos;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  actionp = VG3_actionstack_get(gmain->game.astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != UART_MOUSESEL) { return; }

  gobj = (struct g_obj_mousesel *)objp->ostruct;

  VG3_mouse_position(gmain->wstruct, &xpos, &ypos);

  /* draw mousesel-instance */
  if (gobj->instanceid > 0) {
    struct vg3_image *img;
    struct vg3_image_attributes iattr;
    if (!VG3_sprite_get(gobj->sprt, &img, &iattr, 0)) {
      VG3_sprite_rewind(gobj->sprt);
      actionp = VG3_actionstack_parent(gmain->game.astck);
      if (actionp.stack_id > 0 && actionp.name_id == UART_SELSMURF) { 
        VG3_hash_setint(actionp.hparm, "instanceid", sizeof("instanceid"), gobj->instanceid);
        VG3_actionstack_pop(gmain->game.astck);
        gobj->instanceid = 0;
        return;
      }
    }
    if (img != NULL) { VG3_image_copy(gmain->wstruct, NULL, img, xpos, ypos, &iattr, 0); }
  } else {
    int steps = VG3_hash_getint(actionp.hparm, "steps", sizeof("steps"));
    VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
    iattr.alpha = 150;
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_wfz[steps - 1], gmain->winw / 2, gmain->winh / 2, &iattr, 0);
    VG3_image_copy(gmain->wstruct, NULL, gobj->img, xpos, ypos, NULL, 0);
  }
}
