#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-playfield.h"

void getofc_playfield(struct vg3_ofunc_objfunc *);

static void set_map(struct g_main *, int);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_playfield(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_PLAYFIELD));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_draw = f_draw;
}


/* +++ internal functions +++ */

static void
set_map(struct g_main *gmain, int isdiffi)
{
  static struct {
    int xpos, ypos, starting, mirrordraw;
  } mappos[PLAYFIELD_STEPS] = {
    { 320, 260, 0, 0 },  /* village */
    { 321, 431, 0, 0 },  /* divergency to village */
    { 264, 429, 0, 0 },
    { 213, 412, 0, 0 },
    { 159, 408, 0, 0 },
    { 114, 384, 1, 0 },  /* starting for smurfs */
    { 105, 325, 0, 0 },  /* divergency to sarsaparille */
    { 74, 275, 0, 0 },
    { 49, 227, 0, 0 },
    { 36, 168, 0, 0 },
    { 40, 115, 0, 0 },
    { 58, 66, 0, 1 },    /* in sarsaparille */
    { 103, 35, 2, 1 },   /* in sarsaparille and starting for papa-smurf */
    { 163, 36, 0, 1 },
    { 226, 45, 0, 1 },
    { 286, 44, 0, 1 },
    { 346, 59, 0, 1 },   /* in river */
    { 402, 60, 0, 1 },
    { 462, 55, 0, 1 },
    { 522, 46, 0, 1 },
    { 578, 61, 3, 1 },   /* starting for the fly */
    { 591, 116, 0, 0 },
    { 568, 167, 0, 0 },
    { 546, 224, 0, 0 },
    { 516, 280, 0, 0 },
    { 492, 329, 0, 0 },  /* in river */
    { 480, 379, 0, 0 },  /* in river */
    { 444, 426, 0, 0 },
    { 384, 420, 0, 0 },  /* little sarsaparille */
    { 131, 253, 0, 1 },  /* from divergency to sarsaparille */
    { 145, 189, 0, 1 },
    { 166, 130, 0, 1 },
    { 187, 78, 0, 1 },
  };
  int step;

  if (gmain == NULL) { abort(); }

  memset(gmain->game.pfmap, 0, sizeof(gmain->game.pfmap));
  for (step = 0; step < PLAYFIELD_STEPS; step++) {
    gmain->game.pfmap[step].xpos = mappos[step].xpos;
    gmain->game.pfmap[step].ypos = mappos[step].ypos;
    gmain->game.pfmap[step].starting = mappos[step].starting;
    gmain->game.pfmap[step].mirrordraw = mappos[step].mirrordraw;
    gmain->game.pfmap[step].next[0] = step + 1;
    gmain->game.pfmap[step].next[1] = -1;
  }

  gmain->game.pfmap[0].village = 1;
  gmain->game.pfmap[1].next[1] = 0;
  gmain->game.pfmap[6].next[1] = 29;
  gmain->game.pfmap[16].river = 1;
  gmain->game.pfmap[25].river = 1;
  gmain->game.pfmap[26].river = 1;
  gmain->game.pfmap[11].sarsaparille = 1;
  gmain->game.pfmap[12].sarsaparille = 1;
  gmain->game.pfmap[32].next[0] = 14;
  if (!isdiffi) {
    gmain->game.pfmap[28].sarsaparille = 1;
  }
  gmain->game.pfmap[28].next[0] = 1;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameter:
 *  - int: whether difficult (0 = easy, 1 = difficult)
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_playfield *gobj;
  va_list ap;
  int isdiffi;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  isdiffi = va_arg(ap, int);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  if (isdiffi) {
    gobj->img = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/playfield-difficult.bmp", 1);
  } else {
    gobj->img = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/playfield-easy.bmp", 1);
  }
  if (gobj->img == NULL) { return NULL; }
  gobj->img_smurf_pending = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/smurf_pending.bmp", 1);
  if (gobj->img_smurf_pending == NULL) { return NULL; }
  gobj->img_smurf_saved = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/smurf_saved.bmp", 1);
  if (gobj->img_smurf_saved == NULL) { return NULL; }

  /* create and fill playfield-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_PLAYFIELD));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* set playfield-map according to playfield-image */
  set_map(gmain, isdiffi);

  /* insert playfield-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_playfield *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_playfield *)objp->ostruct;

  /* remove playfield-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free playfield-instance */
  if (gobj->img != NULL) { VG3_image_unload(gmain->wstruct, gobj->img); }
  if (gobj->img_smurf_pending != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_smurf_pending); }
  if (gobj->img_smurf_saved != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_smurf_saved); }
  free(gobj);
  free(objp);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_playfield *gobj;
  char buf[8];

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_playfield *)objp->ostruct;

  /* draw playfield-instance */
  VG3_image_copy(gmain->wstruct, NULL, gobj->img, gmain->winw / 2, gmain->winh / 2, NULL, 0);
  if (gmain->game.smurf_pending > 0) {
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_smurf_pending, 100, 420, NULL, 0);
    snprintf(buf, sizeof(buf), "%d", gmain->game.smurf_pending);
    VG3_text_simpledraw(gmain->wstruct, NULL, NULL, 113, 417, buf, VGAG3_COLOR_BLACK, VGAG3_COLOR_TRANSPARENT, 0);
  }
  if (gmain->game.smurf_saved > 0) {
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_smurf_saved, 375, 262, NULL, 0);
    snprintf(buf, sizeof(buf), "%d", gmain->game.smurf_saved);
    VG3_text_simpledraw(gmain->wstruct, NULL, NULL, 375, 280, buf, VGAG3_COLOR_BLACK, VGAG3_COLOR_TRANSPARENT, 0);
  }

#if 0
  { int step, fcolor;
    char buf[32];
    for (step = 0; step < PLAYFIELD_STEPS; step++) {
      snprintf(buf, sizeof(buf), "%d:\n%02d,%02d", step, gmain->game.pfmap[step].next[0], gmain->game.pfmap[step].next[1]);
      if (gmain->game.pfmap[step].village) { fcolor = VGAG3_COLOR_RED; }
      else if (gmain->game.pfmap[step].sarsaparille) { fcolor = VGAG3_COLOR_YELLOW; }
      else if (gmain->game.pfmap[step].river) { fcolor = VGAG3_COLOR_GREEN; }
      else { fcolor = VGAG3_COLOR_TURQUOISE; }
      VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->game.pfmap[step].xpos, gmain->game.pfmap[step].ypos, buf, fcolor, VGAG3_COLOR_BLACK, 1);
    }
  }
#endif
}
