#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-smurf.h"

void getofc_smurf(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);

#define MAX_FUEL 5
#define MAX_VANISH 50


/* +++ get-function +++ */

void
getofc_smurf(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_SMURF));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
  ofc->f_data = f_data;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameter:
 *  - int: species (SPECIES_*)
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_smurf *gobj;
  struct vg3_coll coll;
  va_list ap;
  int species, step;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  species = va_arg(ap, int);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* load images */
  if (species == SPECIES_FLY) {
    gobj->img_normal = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/fly.bmp", 1);
    gobj->papa.img_empty = gobj->img_black = gobj->img_mouseover = NULL;
    gobj->snd_flymoving = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/fly.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  } else if (species == SPECIES_PAPASMURF || species == SPECIES_PAPASMURF_BLACK) {
    gobj->img_normal = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/papasmurf-blue.bmp", 1);
    if (gobj->img_normal == NULL) { return NULL; }
    gobj->papa.img_empty = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/papasmurf-blue-empty.bmp", 1);
    if (gobj->papa.img_empty == NULL) { return NULL; }
    gobj->img_mouseover = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/papasmurf-blue-mouseover.bmp", 1);
    if (gobj->img_mouseover == NULL) { return NULL; }
    gobj->img_black = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/papasmurf-black.bmp", 1);
    if (gobj->img_black == NULL) { return NULL; }
    gobj->papa.fuel = MAX_FUEL;
  } else {
    gobj->img_normal = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/smurf-blue.bmp", 1);
    if (gobj->img_normal == NULL) { return NULL; }
    gobj->img_mouseover = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/smurf-blue-mouseover.bmp", 1);
    if (gobj->img_mouseover == NULL) { return NULL; }
    gobj->img_black = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/smurfs/smurf-black.bmp", 1);
    if (gobj->img_black == NULL) { return NULL; }
    gobj->papa.img_empty = NULL;
  }

  /* select starting field */
  gobj->pfstep = -1;
  for (step = 0; step < PLAYFIELD_STEPS; step++) {
    if (gmain->game.pfmap[step].starting > 0) {
      if (species == SPECIES_FLY) {
        if (gmain->game.pfmap[step].starting == 3) { gobj->pfstep = step; }
      } else if (species == SPECIES_PAPASMURF || species == SPECIES_PAPASMURF_BLACK) {
        if (gmain->game.pfmap[step].starting == 2) { gobj->pfstep = step; }
      } else {
        if (gmain->game.pfmap[step].starting == 1) { gobj->pfstep = step; }
      }
    }
  }

  VG3_image_getsize(gmain->wstruct, gobj->img_normal, NULL, &gobj->rect.w, &gobj->rect.h);
  gobj->rect.x = gmain->game.pfmap[gobj->pfstep].xpos - gobj->rect.w / 2;
  gobj->rect.y = gmain->game.pfmap[gobj->pfstep].ypos - gobj->rect.h / 2;
  /* set collision rectangle with 90% of width and height of image-rectangle */
  gobj->crect = gobj->rect;
  gobj->crect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w * 90 / 100, gobj->rect.h * 90 / 100);

  /* create and fill smurf-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_SMURF));
  objp->subid = species;
  objp->drawlevel = 2;
  if (species == SPECIES_FLY) { objp->drawlevel = 3; }
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert smurf-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  if (objp->subid == SPECIES_SMURF || objp->subid == SPECIES_PAPASMURF) {
    /* insert smurf-instance (collision-rectangle) into quadtree */
    memset(&coll, 0, sizeof(coll));
    coll.rect = gobj->crect;
    snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
    coll.optr = objp;
    VG3_coll_q_insert(gmain->qdtr, &coll);
  }

  /* is field is occupated, move it one step */
  { struct vg3_ofunc_objsnap *osnap;
    struct vg3_ofunc_object *objp2;
    struct g_obj_smurf *gobj2;
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
    while ((objp2 = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (objp2 == objp) { continue; }
      gobj2 = (struct g_obj_smurf *)objp2->ostruct;
      if (gobj->pfstep == gobj2->pfstep) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);

    if (objp2 != NULL) {  /* move objp2 one step */
      struct vg3_hash *hparm = VG3_hash_new();
      VG3_hash_setint(hparm, "instanceid", sizeof("instanceid"), objp2->instanceid);
      VG3_hash_setint(hparm, "steps", sizeof("steps"), 1);
      VG3_hash_setint(hparm, "direction", sizeof("direction"), 0);
      VG3_actionstack_push(gmain->game.astck, UART_MOVSMURF, hparm, 0);
    }
  }

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_smurf *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_smurf *)objp->ostruct;

  /* remove smurf-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove smurf-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free smurf-instance */
  if (gobj->img_normal != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_normal); }
  if (gobj->papa.img_empty != NULL) { VG3_image_unload(gmain->wstruct, gobj->papa.img_empty); }
  if (gobj->img_mouseover != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_mouseover); }
  if (gobj->img_black != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_black); }
  if (gobj->snd_flymoving > 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_flymoving); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_smurf *gobj;
  struct vg3_coll coll;
  struct vg3_actionstack_elem actionp;
  struct vg3_rect rct1, rct2;
  int pfstep1, pfstep2;

  if (gmain == NULL || objp == NULL) { return; }

  actionp = VG3_actionstack_get(gmain->game.astck);

  /* relevant? */
  if (actionp.stack_id == 0) { return; }
  if (actionp.name_id != UART_MOVSMURF) { return; }
  if ((unsigned int)VG3_hash_getint(actionp.hparm, "instanceid", sizeof("instanceid")) != objp->instanceid) { return; }

  gobj = (struct g_obj_smurf *)objp->ostruct;

  /* remove smurf-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* move smurf-instance */
  while (gobj->pos.movmax == 0 || gobj->pos.movmom == gobj->pos.movmax) {
    if (gobj->pos.rectmov != NULL) { free(gobj->pos.rectmov); gobj->pos.rectmov = NULL; }
    /* target reached? */
    if (VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")) == 0) {
      VG3_actionstack_pop(gmain->game.astck);
      actionp.stack_id = 0;
      if (gmain->game.pfmap[gobj->pfstep].village) { f_free(vmain, objp); gmain->game.smurf_saved++; return; }  /* village reached */
      /* fill bellows if papasmurf is in sarsaparille */
      if ((objp->subid == SPECIES_PAPASMURF || objp->subid == SPECIES_PAPASMURF_BLACK)
          && gmain->game.pfmap[gobj->pfstep].sarsaparille
         ) {
        gobj->papa.fuel = MAX_FUEL;
      }
      break;
    }
    if (objp->subid == SPECIES_FLY) {
      if (gobj->snd_flymoving > 0 && !VG3_audio_isplaying(gmain->wstruct, gobj->snd_flymoving)) {
        VG3_audio_play(gmain->wstruct, gobj->snd_flymoving, 1, 0);
      }
    }
    /* select direction? */
    if (VG3_hash_getint(actionp.hparm, "direction", sizeof("direction")) == 0 && gmain->game.pfmap[gobj->pfstep].next[1] >= 0) {
      check_direction(gmain, gobj->pfstep, VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")), objp->subid, &pfstep1, &pfstep2);
      if (pfstep1 >= 0 && pfstep2 >= 0) {
        if (objp->subid == SPECIES_SMURF_BLACK || objp->subid == SPECIES_PAPASMURF_BLACK || objp->subid == SPECIES_FLY) {
          struct vg3_ofunc_objsnap *osnap;
          struct vg3_ofunc_object *objp2;
          struct g_obj_smurf *gobj2;
          int pf1, pf2;
          pf1 = pf2 = -1;
          osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
          while ((objp2 = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
            if (objp2 == objp) { continue; }
            if (objp2->subid != SPECIES_SMURF && objp2->subid != SPECIES_PAPASMURF) { continue; }
            gobj2 = (struct g_obj_smurf *)objp2->ostruct;
            if (pfstep1 == gobj2->pfstep) { pf1 = pfstep1; }
            if (pfstep2 == gobj2->pfstep) { pf2 = pfstep2; }
          }
          VG3_ofunc_objlist_freelist(osnap);
          if (pf1 >= 0 && pf2 >= 0) {
            VG3_hash_setint(actionp.hparm, "direction", sizeof("direction"), ZUFALL(1, 2));
          } else if (pf1 >= 0) {
            VG3_hash_setint(actionp.hparm, "direction", sizeof("direction"), 1);
          } else if (pf2 >= 0) {
            VG3_hash_setint(actionp.hparm, "direction", sizeof("direction"), 2);
          } else {
            VG3_hash_setint(actionp.hparm, "direction", sizeof("direction"), ZUFALL(1, 2));
          }
        } else {
          struct vg3_hash *hparm = VG3_hash_new();
          VG3_hash_setint(hparm, "instanceid", sizeof("instanceid"), objp->instanceid);
          VG3_actionstack_push(gmain->game.astck, UART_DIRSEL, hparm, actionp.stack_id);
          VG3_hash_free(hparm);
          return;
        }
      } else if (pfstep1 >= 0) {  /* left */
        VG3_hash_setint(actionp.hparm, "direction", sizeof("direction"), 1);
      } else if (pfstep2 >= 0) {  /* right */
        VG3_hash_setint(actionp.hparm, "direction", sizeof("direction"), 2);
      } else {  /* impossible */
        VG3_hash_setint(actionp.hparm, "steps", sizeof("steps"), 0);
        continue;
      }
    }
    /* prepare moving to the next step */
    VG3_hash_setint(actionp.hparm, "steps", sizeof("steps"), VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")) - 1);
    rct1.x = gmain->game.pfmap[gobj->pfstep].xpos; rct1.w = 1;
    rct1.y = gmain->game.pfmap[gobj->pfstep].ypos; rct1.h = 1;
    if (VG3_hash_getint(actionp.hparm, "direction", sizeof("direction")) == 2) {  /* to the right */
      gobj->pfstep = gmain->game.pfmap[gobj->pfstep].next[1];
    } else {  /* default or to the left */
      gobj->pfstep = gmain->game.pfmap[gobj->pfstep].next[0];
    }
    VG3_hash_setint(actionp.hparm, "direction", sizeof("direction"), 0);
    rct2.x = gmain->game.pfmap[gobj->pfstep].xpos; rct2.w = 1;
    rct2.y = gmain->game.pfmap[gobj->pfstep].ypos; rct2.h = 1;
    gobj->pos.movmax = VG3_line_positions(&rct1, &rct2, 5, &gobj->pos.rectmov);
    gobj->pos.movmom = 0;
  }

  if (actionp.stack_id > 0) {  /* move */
    gobj->rect.x = gobj->pos.rectmov[gobj->pos.movmom].x - gobj->rect.w / 2;
    gobj->rect.y = gobj->pos.rectmov[gobj->pos.movmom].y - gobj->rect.h / 2;
    gobj->crect = VG3_correct_imageposition(&gobj->rect, gobj->crect.w, gobj->crect.h);
    gobj->pos.movmom++;

  } else {  /* check if the target position is occupated */
    struct vg3_ofunc_objsnap *osnap;
    struct vg3_ofunc_object *objp2, *objp_smurf, *objp_fly;
    struct g_obj_smurf *gobj2;
    int is_sarsaparille;

    if (objp->subid == SPECIES_FLY && gobj->pos.movmax > 0) { VG3_audio_stop(gmain->wstruct, gobj->snd_flymoving, 0); }
    objp_smurf = objp_fly = NULL;
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
    while ((objp2 = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (objp2 == objp) { continue; }
      gobj2 = (struct g_obj_smurf *)objp2->ostruct;
      if (gobj->pfstep == gobj2->pfstep) {  /* occupated */
        if (objp2->subid == SPECIES_FLY) { objp_fly = objp2; } else { objp_smurf = objp2; }
      }
    }
    VG3_ofunc_objlist_freelist(osnap);

    is_sarsaparille = 0;
    if (gmain->game.pfmap[gobj->pfstep].sarsaparille) {
      if (objp->subid == SPECIES_SMURF_BLACK) {
        objp->subid = SPECIES_SMURF;
        is_sarsaparille = 1;
      } else if (objp->subid == SPECIES_PAPASMURF_BLACK) {
        objp->subid = SPECIES_PAPASMURF;
        is_sarsaparille = 1;
      }
    }

    objp2 = objp_smurf;
    if (objp2 == NULL) { objp2 = objp_fly; objp_fly = NULL; }
    if (objp2 != NULL) {  /* occupated by objp2 */
      struct vg3_hash *hparm = VG3_hash_new();
      int animation;

      gobj2 = (struct g_obj_smurf *)objp2->ostruct;

      /* move objp2 one step */
      if (objp->subid != SPECIES_FLY && objp2->subid != SPECIES_FLY) {
        VG3_hash_setint(hparm, "instanceid", sizeof("instanceid"), objp2->instanceid);
        VG3_hash_setint(hparm, "steps", sizeof("steps"), 1);
        VG3_hash_setint(hparm, "direction", sizeof("direction"), 0);
        VG3_actionstack_push(gmain->game.astck, UART_MOVSMURF, hparm, 0);
        actionp = VG3_actionstack_get(gmain->game.astck);
        for (;;) {  /* don't stop on invalid target step */
          check_direction(gmain, gobj2->pfstep, VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")), objp2->subid, &pfstep1, &pfstep2);
          if (pfstep1 >= 0 || pfstep2 >= 0) { break; }
          VG3_hash_setint(actionp.hparm, "steps", sizeof("steps"), VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")) + 1);
        }
      }

      /* change to blue, if in sarsaparille */
      if (is_sarsaparille) {
        if (objp->subid == SPECIES_PAPASMURF) {
          animation = ANIMATION_SMURF_BLUE;
        } else {
          animation = ANIMATION_SMURF_BLUE;
        }
        VG3_hash_clear(hparm);
        VG3_hash_setint(hparm, "num", sizeof("num"), animation);
        VG3_actionstack_push(gmain->game.astck, UART_ANIMATION, hparm, 0);
        actionp = VG3_actionstack_get(gmain->game.astck);
      }

      /* change one of the smurfs to black or blue? */
      animation = 0;
      if (objp->subid == SPECIES_PAPASMURF && objp_fly != NULL) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp->subid = SPECIES_PAPASMURF_BLACK;
      } else if (objp->subid == SPECIES_PAPASMURF && objp2->subid == SPECIES_SMURF_BLACK) {
        if (gobj->papa.fuel > 0) {
          animation = ANIMATION_SMURF_BLUE;
          objp2->subid = SPECIES_SMURF;
          gobj->papa.fuel--;
        } else {
          animation = ANIMATION_SMURF_BLACK;
          objp->subid = SPECIES_PAPASMURF_BLACK;
        }
      } else if (objp->subid == SPECIES_SMURF_BLACK && objp2->subid == SPECIES_PAPASMURF) {
        if (gobj2->papa.fuel > 0) {
          animation = ANIMATION_SMURF_BLUE;
          objp->subid = SPECIES_SMURF;
          gobj2->papa.fuel--;
        } else {
          animation = ANIMATION_SMURF_BLACK;
          objp2->subid = SPECIES_PAPASMURF_BLACK;
        }
      } else if (objp->subid == SPECIES_SMURF
                 && (objp2->subid == SPECIES_SMURF_BLACK || objp2->subid == SPECIES_PAPASMURF_BLACK)
                ) {
        animation = ANIMATION_SMURF_BLACK;
        objp->subid = SPECIES_SMURF_BLACK;
      } else if ((objp->subid == SPECIES_SMURF_BLACK || objp->subid == SPECIES_PAPASMURF_BLACK)
                 && objp2->subid == SPECIES_SMURF
                ) {
        animation = ANIMATION_SMURF_BLACK;
        objp2->subid = SPECIES_SMURF_BLACK;
      } else if (objp->subid == SPECIES_SMURF && objp2->subid == SPECIES_FLY) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp->subid = SPECIES_SMURF_BLACK;
      } else if (objp->subid == SPECIES_FLY && objp2->subid == SPECIES_SMURF) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp2->subid = SPECIES_SMURF_BLACK;
      } else if (objp->subid == SPECIES_PAPASMURF && objp2->subid == SPECIES_FLY) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp->subid = SPECIES_PAPASMURF_BLACK;
      } else if (objp->subid == SPECIES_FLY && objp2->subid == SPECIES_PAPASMURF) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp2->subid = SPECIES_PAPASMURF_BLACK;
      }
      if (animation > 0) {
        VG3_hash_clear(hparm);
        VG3_hash_setint(hparm, "num", sizeof("num"), animation);
        VG3_actionstack_push(gmain->game.astck, UART_ANIMATION, hparm, 0);
        actionp = VG3_actionstack_get(gmain->game.astck);
        VG3_hash_free(hparm);
      }

    } else if (is_sarsaparille) {  /* change to blue, if in sarsaparille */
      int animation;
      struct vg3_hash *hparm = VG3_hash_new();
      if (objp->subid == SPECIES_PAPASMURF) {
        animation = ANIMATION_SMURF_BLUE;
      } else {
        animation = ANIMATION_SMURF_BLUE;
      }
      VG3_hash_setint(hparm, "num", sizeof("num"), animation);
      VG3_actionstack_push(gmain->game.astck, UART_ANIMATION, hparm, 0);
      actionp = VG3_actionstack_get(gmain->game.astck);
      VG3_hash_free(hparm);
    }
  }

  if (objp->subid == SPECIES_SMURF || objp->subid == SPECIES_PAPASMURF) {
    /* insert smurf-instance (collision-rectangle) into quadtree again */
    memset(&coll, 0, sizeof(coll));
    coll.rect = gobj->crect;
    snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
    coll.optr = objp;
    VG3_coll_q_insert(gmain->qdtr, &coll);
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_smurf *gobj;
  struct vg3_image *imgp;
  struct vg3_rect rect;
  struct vg3_image_attributes iattr;
  struct vg3_actionstack_elem actionp;
  int dirsel, mouseover;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_smurf *)objp->ostruct;

  dirsel = 0;
  actionp = VG3_actionstack_get(gmain->game.astck);
  if (actionp.stack_id != 0 && actionp.name_id == UART_DIRSEL
    && (unsigned int)VG3_hash_getint(actionp.hparm, "instanceid", sizeof("instanceid")) == objp->instanceid) { dirsel = 1; }

  mouseover = 0;
  if (gobj->mouseover) { mouseover = 1; }

  /* draw smurf-instance */
  if (objp->subid == SPECIES_FLY) {
    imgp = gobj->img_normal;
  } else if (objp->subid == SPECIES_SMURF_BLACK || objp->subid == SPECIES_PAPASMURF_BLACK) {
    imgp = gobj->img_black;
  } else if (dirsel || mouseover) {
    imgp = gobj->img_mouseover;
    gobj->mouseover = 0;
  } else if (objp->subid == SPECIES_PAPASMURF && gobj->papa.fuel == 0) {
    imgp = gobj->papa.img_empty;
  } else {
    imgp = gobj->img_normal;
  }

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
  if (gmain->game.pfmap[gobj->pfstep].mirrordraw) {
    iattr.flip = 2;
  }

  if (gobj->vanish > 0) {
    if (--gobj->vanish == 0) { f_free(gmain, objp); return; }
    iattr.alpha = 255 * gobj->vanish / MAX_VANISH;
  }

  VG3_image_getsize(gmain->wstruct, imgp, NULL, &rect.w, &rect.h);
  rect = VG3_correct_imageposition(&gobj->rect, rect.w, rect.h);
  VG3_image_copy(gmain->wstruct, NULL, imgp, rect.x + rect.w / 2, rect.y + rect.h / 2, &iattr, 0);

  if (objp->subid == SPECIES_PAPASMURF && mouseover) {
    char buf[32];
    snprintf(buf, sizeof(buf), "%d%%", gobj->papa.fuel * 100 / MAX_FUEL);
    VG3_text_simpledraw(gmain->wstruct, NULL, NULL, rect.x + rect.w / 2, rect.y + rect.h, buf, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE, 0);
  }
}


/* data-exchanging-function */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *vdata)
{
  struct g_main *gmain = vmain;
  struct g_obj_smurf *gobj;
  int *pfstep = (int *)vdata;

  if (pfstep != NULL) { *pfstep = 0; }
  if (gmain == NULL || objp == NULL) { return 0; }

  gobj = (struct g_obj_smurf *)objp->ostruct;

  if (pfstep == NULL) {
    gobj->vanish = MAX_VANISH;
  } else {
    *pfstep = gobj->pfstep;
  }

  return 1;
}
