#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

void getofmgmt_smurf(struct vg3_ofunc_ofmgmt *);

/* static structure for this object-managing */
static struct {
  int smurf_active;    /* number of active smurfs */
} mgmt_smurf;

static int f_activate(void *, struct vg3_ofunc *, va_list);
static void f_deactivate(void *, struct vg3_ofunc *);
static void f_run(void *, struct vg3_ofunc *);


/* +++ get-function +++ */

/* fill out the passed structure for object-management-functions of this VgaGames3-object */
void
getofmgmt_smurf(struct vg3_ofunc_ofmgmt *ofm) {
  if (ofm == NULL) { return; }
  snprintf(ofm->oid, sizeof(ofm->oid), "%s", get_oid_name(OID_NAME_SMURF));
  ofm->f_activate = f_activate;
  ofm->f_deactivate = f_deactivate;
  ofm->f_run = f_run;
}


/* +++ object-management-functions +++ */

/* activate the managing of smurf-object
 * parameters in ap:
 * - integer: total number of smurfs
 * - integer: number of active smurfs
 */
static int
f_activate(void *vmain, struct vg3_ofunc *ofstruct, va_list ap)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return -1; }
  (void)gmain;

  memset(&mgmt_smurf, 0, sizeof(mgmt_smurf));

  /* get variadic parameters */
  gmain->game.smurf_pending = va_arg(ap, int);
  if (gmain->game.smurf_pending < 1) { gmain->game.smurf_pending = 1; }
  gmain->game.smurf_total = gmain->game.smurf_pending;
  mgmt_smurf.smurf_active = va_arg(ap, int);
  if (mgmt_smurf.smurf_active < 1) { mgmt_smurf.smurf_active = 1; }

  return 0;
}


/* deactivate the managing of smurf-object */
static void
f_deactivate(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;

  if (gmain == NULL || ofstruct == NULL) { return; }

  /* destroy all smurf-objects */
  VG3_ofunc_objlist_call_free(ofstruct, gmain, get_oid_name(OID_NAME_SMURF));

  /* clean up the static structure */
  memset(&mgmt_smurf, 0, sizeof(mgmt_smurf));
}


/* execute managing of smurf-instances */
static void
f_run(void *vmain, struct vg3_ofunc *ofstruct)
{
  struct g_main *gmain = vmain;
  struct vg3_actionstack_elem actionp;
  const struct vg3_ofunc_objfunc *ofc;

  if (gmain == NULL || ofstruct == NULL) { return; }

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
  if (ofc == NULL || ofc->f_data == NULL) {
    fprintf(stderr, "Object \"%s\" or it's f_data() not found\n", get_oid_name(OID_NAME_SMURF));
    abort();
  }

  actionp = VG3_actionstack_get(gmain->game.astck);

  /* handle no action */
  if (actionp.stack_id == 0) {
    struct vg3_ofunc_objsnap *osnap;
    struct vg3_ofunc_object *objp;
    int pfstep, startblue, anzblue, anzblack;

    /* find starting-field for normal smurf */
    for (startblue = 0; startblue < PLAYFIELD_STEPS; startblue++) {
      if (gmain->game.pfmap[startblue].starting == 1) { break; }
    }
    if (startblue == PLAYFIELD_STEPS) { startblue = -1; }

    /* count active blue and black smurfs and check whether starting-field is free */
    anzblue = anzblack = 0;
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      ofc->f_data(gmain, objp, &pfstep);
      if (objp->subid == SPECIES_SMURF_BLACK || objp->subid == SPECIES_PAPASMURF_BLACK || objp->subid == SPECIES_FLY) {
        if (startblue >= 0) {
          if (pfstep == startblue) { startblue = -1; }  /* starting-field black-occupated */
        }
      }
      if (objp->subid == SPECIES_SMURF) { anzblue++; }
      if (objp->subid == SPECIES_SMURF_BLACK) { anzblack++; }
    }
    VG3_ofunc_objlist_freelist(osnap);

    /* throw dice */
    VG3_actionstack_push(gmain->game.astck, UART_DICE, NULL, 0);

    /* if no blue smurf is active, all black smurfs vanish */
    if ((gmain->game.smurf_pending == 0 && anzblue == 0) || (anzblack == mgmt_smurf.smurf_active)) {
      if (gmain->game.smurf_pending == 0 && anzblack == 0) {
        /* exit if no blue smurf (except papa-smurf) is left */
        gmain->exit = EXITDEF_GOTO_MAINMENU;
        gmain->game.showending = 1;
        VG3_actionstack_pop(gmain->game.astck);  /* pop dice */
        return;
      } else {
        struct vg3_ofunc_objsnap *osnap;
        struct vg3_ofunc_object *objp;
        struct vg3_hash *hparm = VG3_hash_new();
        osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
        while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
          if (objp->subid == SPECIES_SMURF_BLACK) { ofc->f_data(gmain, objp, NULL); }
        }
        VG3_ofunc_objlist_freelist(osnap);
        if (gmain->game.smurf_pending == 0 && anzblue == 0) { VG3_actionstack_pop(gmain->game.astck); }  /* pop dice */
        VG3_hash_setint(hparm, "num", sizeof("num"), ANIMATION_BLACKSMURF_BYE);
        VG3_actionstack_push(gmain->game.astck, UART_ANIMATION, hparm, 0);
        VG3_hash_free(hparm);
        anzblack = 0;
      }
    }

    /* instantiate a new smurf? */
    if ((startblue >= 0 || anzblue == 0) && gmain->game.smurf_pending > 0 && mgmt_smurf.smurf_active > anzblue + anzblack) {
      if ((objp = ofc->f_new(gmain, 0, SPECIES_SMURF)) != NULL) {
        gmain->game.smurf_pending--;
      } else {
        fprintf(stderr, "%s\n", VG3_error());
        gmain->game.smurf_pending = 0;
      }
    }

    return;
  }

  /* relevant? */
  if (actionp.name_id != UART_SELSMURF) { return; }

  if (VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")) == 0) {
    /* move black smurf */
    struct vg3_ofunc_objsnap *osnap;
    struct vg3_ofunc_object *objp;
    int pfstep, pfstep1, pfstep2;
    int bluesteps[PLAYFIELD_STEPS];
    unsigned int black_mov_id[VG3_ofunc_objlist_find_obj(gmain->ofstruct, get_oid_name(OID_NAME_SMURF), NULL) + 1];
    unsigned int black_hit_id[VG3_ofunc_objlist_find_obj(gmain->ofstruct, get_oid_name(OID_NAME_SMURF), NULL) + 1];
    int blacksteps, black_mov_pos, black_hit_pos;

    blacksteps = ZUFALL(1, MAX_DICE_NUMBER);

    /* get playfield-steps occupated by blue smurfs */
    memset(bluesteps, 0, sizeof(bluesteps));
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (objp->subid == SPECIES_SMURF || objp->subid == SPECIES_PAPASMURF) {
        ofc->f_data(gmain, objp, &pfstep);
        bluesteps[pfstep] = 1;
      }
    }
    VG3_ofunc_objlist_freelist(osnap);

    /* check moving each black smurf */
    memset(black_mov_id, 0, sizeof(black_mov_id)); black_mov_pos = 0;
    memset(black_hit_id, 0, sizeof(black_hit_id)); black_hit_pos = 0;
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, get_oid_name(OID_NAME_SMURF));
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (objp->subid != SPECIES_SMURF_BLACK && objp->subid != SPECIES_PAPASMURF_BLACK) { continue; }
      ofc->f_data(gmain, objp, &pfstep);
      check_direction(gmain, pfstep, blacksteps, objp->subid, &pfstep1, &pfstep2);
      if (pfstep1 < 0 && pfstep2 < 0) { continue; }  /* moving not possible */
      black_mov_id[black_mov_pos++] = objp->instanceid;
      if (pfstep1 >= 0 && bluesteps[pfstep1]) {  /* hit moving left */
        black_hit_id[black_hit_pos++] = objp->instanceid;
      } else if (pfstep2 >= 0 && bluesteps[pfstep2]) {  /* hit moving right */
        black_hit_id[black_hit_pos++] = objp->instanceid;
      }
    }
    VG3_ofunc_objlist_freelist(osnap);
    VG3_actionstack_pop(gmain->game.astck);

    if (black_hit_pos > 0 || black_mov_pos > 0) {
      struct vg3_hash *hparm2 = VG3_hash_new();
      if (black_hit_pos > 0) {
        black_hit_pos = ZUFALL(0, black_hit_pos - 1);
        VG3_hash_setint(hparm2, "instanceid", sizeof("instanceid"), black_hit_id[black_hit_pos]);
      } else if (black_mov_pos > 0) {
        black_mov_pos = ZUFALL(0, black_mov_pos - 1);
        VG3_hash_setint(hparm2, "instanceid", sizeof("instanceid"), black_mov_id[black_mov_pos]);
      }
      VG3_hash_setint(hparm2, "steps", sizeof("steps"), blacksteps);
      VG3_hash_setint(hparm2, "direction", sizeof("direction"), 0);
      VG3_actionstack_push(gmain->game.astck, UART_MOVSMURF, hparm2, 0);
      VG3_hash_free(hparm2);
    }
    return;

  } else if (VG3_hash_getint(actionp.hparm, "instanceid", sizeof("instanceid")) > 0) {
    /* start moving blue smurf-instance, after then select black smurf for moving */
    struct vg3_hash *hparm = VG3_hash_new();
    VG3_hash_setint(hparm, "instanceid", sizeof("instanceid"), VG3_hash_getint(actionp.hparm, "instanceid", sizeof("instanceid")));
    VG3_hash_setint(hparm, "steps", sizeof("steps"), VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")));
    VG3_hash_setint(hparm, "direction", sizeof("direction"), 0);
    VG3_actionstack_pop(gmain->game.astck);
    VG3_actionstack_push(gmain->game.astck, UART_SELSMURF, NULL, 0);
    VG3_actionstack_push(gmain->game.astck, UART_MOVSMURF, hparm, 0);
    VG3_hash_free(hparm);
    return;
  }

  /* select blue smurf */
  { struct vg3_hash *hparm = VG3_hash_new();
    VG3_hash_setint(hparm, "steps", sizeof("steps"), VG3_hash_getint(actionp.hparm, "steps", sizeof("steps")));
    VG3_actionstack_push(gmain->game.astck, UART_MOUSESEL, hparm, actionp.stack_id);
    VG3_hash_free(hparm);
    actionp = VG3_actionstack_get(gmain->game.astck);
  }
}
