#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

void create_input_keys(struct g_main *);
int key_ispressed(struct g_main *, int, int);

static void set_abstractkeys(struct vg3_window *);


/* key definitions */
struct keydefs {
  int keyref;           /* abstract key definition */
  int kbdidx;           /* default keyboard key */
  int gcidx;            /* default gamecontroller key */
  const char *joykey;   /* name of default joystick key */
  const char *keyname;  /* name of key */
  const char *info;     /* key description in change-menu */
};

#define MAX_KEYS  5
static struct keydefs kdefs[MAX_KEYS] = {
  { KEYDEF_GO_UP, VGAG3_KEY_UCURS, VGAG3_GC_AXIS_RIGHTY_UP, "Axis-2:neg", "up", "go-up" },
  { KEYDEF_GO_DOWN, VGAG3_KEY_DCURS, VGAG3_GC_AXIS_RIGHTY_DOWN, "Axis-2:pos", "down", "go-down" },
  { KEYDEF_GO_RIGHT, VGAG3_KEY_RCURS, VGAG3_GC_AXIS_RIGHTX_RIGHT, "Axis-1:pos", "right", "go-right" },
  { KEYDEF_GO_LEFT, VGAG3_KEY_LCURS, VGAG3_GC_AXIS_RIGHTX_LEFT, "Axis-1:neg", "left", "go-left" },
  { KEYDEF_FIRE, VGAG3_KEY_SPACE, VGAG3_GC_BUTTON_LEFTSHOULDER, "Button-1", "poisonspit", "poisonspit" },
};

#define MAX_DEVICES  4
struct {  /* input-devices (indx: 0 = keyboard, 1 to MAX_DEVICES = gamecontroller/joystick) */
  int enabled;               /* whether input-device is enabled */
  int jid;                   /* input-device number: 0 = keyboard, >0 = gamecontroller/joystick */
  int keyidx[MAX_KEYS];      /* definitions for each abstract key */
} abstractkey[MAX_DEVICES];


/* set key definitions and corresponding values */
static void
set_abstractkeys(struct vg3_window *wstruct)
{
  int *jidf, ikey, idev;
  struct vg3_gamecontroller gcs;

  /* disable all input-devices */
  for (idev = 0; idev < MAX_DEVICES; idev++) {
    abstractkey[idev].enabled = 0;
  }

  if (wstruct == NULL) { return; }

  /* input-devices: 0 = reserved for keyboard, 1 to MAX_DEVICES-1 = gamecontroller/joysticks */
  VG3_gamecontroller_getall(wstruct, &jidf);
  if (jidf[0] > MAX_DEVICES - 1) { jidf[0] = MAX_DEVICES - 1; }

  /* 0 = keyboard; 1 to MAX_DEVICES-1 = gamecontroller/joystick */
  for (idev = 0; idev <= jidf[0]; idev++) {
    abstractkey[idev].enabled = 1;  /* enable input-device */

    if (idev > 0) {  /* get info of gamecontroller/joystick */
      VG3_gamecontroller_getinfo(wstruct, jidf[idev], &gcs);
    } else {
      gcs.is_gamecontroller = 0;
    }

    /* set 0=keyboard or gamecontroller-/joystick-device-number */
    if (idev == 0) {  /* keyboard */
      abstractkey[idev].jid = 0;
    } else {  /* gamecontroller/joystick */
      abstractkey[idev].jid = jidf[idev];
    }

    /* set default keys */
    for (ikey = 0; ikey < MAX_KEYS; ikey++) {
      if (idev == 0) {  /* keyboard */
        abstractkey[idev].keyidx[ikey] = kdefs[ikey].kbdidx;
      } else {  /* gamecontroller/joystick */
        if (gcs.is_gamecontroller) {
          abstractkey[idev].keyidx[ikey] = kdefs[ikey].gcidx;
        } else {
          abstractkey[idev].keyidx[ikey] = VG3_gamecontroller_name2num(wstruct, abstractkey[idev].jid, kdefs[ikey].joykey);
        }
      }
    }
  }

  free(jidf);
}


/* define keys for found input-devices and add them into the system-menu */
void
create_input_keys(struct g_main *gmain)
{
  struct vg3_sysmenu_submenu *subm;
  int ikey, idev;

  if (gmain == NULL) { return; }

  /* set key definitions */
  set_abstractkeys(gmain->wstruct);

  /* add menu-entry: keyboard-submenu (abstract key definition used as key reference) */
  if (abstractkey[0].enabled) {
    subm = VG3_sysmenu_simple_keyboardmenu(gmain->sysm, NULL);
    for (ikey = 0; ikey < MAX_KEYS; ikey++) {
      /* find saved setting */
      abstractkey[0].keyidx[ikey] = VG3_sysmenu_savestring_find(gmain->sysm, gmain->sysm_string, 0, kdefs[ikey].keyref, abstractkey[0].keyidx[ikey]);
      /* add entry */
      VG3_sysmenu_simple_keyboardentry(gmain->sysm, subm, kdefs[ikey].keyref,
        VG3_multilang_get(gmain->mlang, kdefs[ikey].keyname),
        abstractkey[0].keyidx[ikey],
        VG3_multilang_get(gmain->mlang, kdefs[ikey].info)
      );
    }
  }

  /* add menu-entry: gamecontroller-/joystick-submenu (abstract key definition used as key reference) */
  for (idev = 1; idev < MAX_DEVICES; idev++) {
    if (abstractkey[idev].enabled) { break; }
  }
  if (idev < MAX_DEVICES) {  /* input-device found */
    subm = VG3_sysmenu_simple_gcmenu(gmain->sysm, NULL);
    for (idev = 1; idev < MAX_DEVICES; idev++) {  /* for each gamecontroller/joystick */
      if (abstractkey[idev].enabled) {
        for (ikey = 0; ikey < MAX_KEYS; ikey++) {
          /* find saved setting */
          abstractkey[idev].keyidx[ikey] = VG3_sysmenu_savestring_find(gmain->sysm, gmain->sysm_string, abstractkey[idev].jid, kdefs[ikey].keyref, abstractkey[idev].keyidx[ikey]);
          /* add entry */
          VG3_sysmenu_simple_gcentry(gmain->sysm, subm, abstractkey[idev].jid, kdefs[ikey].keyref,
            VG3_multilang_get(gmain->mlang, kdefs[ikey].keyname),
            abstractkey[idev].keyidx[ikey],
            VG3_multilang_get(gmain->mlang, kdefs[ikey].info)
          );
        }
      }
    }
  }
}


/* check if a key is pressed
 * - keyref:  abstract key (struct keydefs)
 * - flag:    kind of pressing (VGAG3_PRESSED)
 * return:    0 = key is not pressed, 1 = key is (new) pressed
 */
int
key_ispressed(struct g_main *gmain, int keyref, int flag)
{
  static time_t lastcall = 0;
  int ispressed, keyidx, ikey, idev;

  if (gmain == NULL) { return 0; }

  /* was system-menu called since last time? */
  if (lastcall < VG3_sysmenu_lastcall(gmain->sysm)) {
    for (idev = 0; idev < MAX_DEVICES; idev++) {
      if (!abstractkey[idev].enabled) { continue; }
      for (ikey = 0; ikey < MAX_KEYS; ikey++) {
        keyidx = VG3_sysmenu_getkeyval(gmain->sysm, abstractkey[idev].jid, kdefs[ikey].keyref);
        if (keyidx != abstractkey[idev].keyidx[ikey]) { abstractkey[idev].keyidx[ikey] = keyidx; }
      }
    }
    lastcall = VG3_sysmenu_lastcall(gmain->sysm);
  }

  /* +++ get real key from keyref and check whether it is pressed +++ */

  ispressed = 0;

  /* for each input-device (0 = keyboard; 1 to MAX_DEVICES-1 = gamecontroller/joystick) */
  for (idev = 0; idev < MAX_DEVICES; idev++) {
    if (!abstractkey[idev].enabled) { continue; }

    /* find abstract key definition */
    for (ikey = 0; ikey < MAX_KEYS; ikey++) {
      if (keyref == kdefs[ikey].keyref) { break; }
    }
    if (ikey == MAX_KEYS) { continue; }  /* not found */

    /* get real key */
    keyidx = abstractkey[idev].keyidx[ikey];
    if (keyidx == VGAG3_KEY_NOKEY || keyidx == VGAG3_GC_NOKEY) { continue; }  /* not defined */

    /* check whether real key is pressed */
    if (abstractkey[idev].jid == 0) {
      ispressed = VG3_key_ispressed(gmain->wstruct, keyidx, flag);
    } else {
      ispressed = VG3_gamecontroller_ispressed(gmain->wstruct, abstractkey[idev].jid, keyidx, flag);
    }

    if (ispressed) { break; }
  }

  return ispressed;
}
