/* Copyright 2020 Kurt Nienhaus
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <vgagames3.h>
#include "main.h"
#include "level.h"

extern struct vg3_ofunc * ofunc_new(void);

const char * get_oid_name(int);

static struct vg3_image * create_ground(struct g_main *, struct g_level *);
static void draw_info(struct g_main *, int, int);
static int main_menu(struct g_main *);
static int help_menu(struct g_main *);


/* get_oid_name:
 * returns object-id as a string
 * @param oidn  object-id (OID_NAMES)
 * @return  object-id as string (or empty)
 */
const char *
get_oid_name(int oidn)
{
  switch(oidn) {
    case OID_NAME_WALL:    return "wall";
    case OID_NAME_ISNAKE:  return "my snake";
    case OID_NAME_ESNAKE:  return "enemy snake";
    case OID_NAME_SHOT:    return "shot";
    case OID_NAME_CAMHAT:  return "camouflage hat";
    case OID_NAME_BOTTLE:  return "poison bottle";
  }
  return "";
}


int main(int argc, char **argv) {
  struct vg3_window *wstruct;
  struct vg3_image *bgimage;
  struct g_main gmain;
  int retw, goend, scale, i1, bg_music;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_rect rect;
  struct g_level *level;
  int anz_level, esnake_remain;
  int isnake_health;
  char buf[256];
  struct vg3_multilang *mlang;
  struct vg3_ofunc_object *objisnake;

  scale = VGAG3_WINSCALE_NOSCALE;

  opterr = opterr ? opterr : 1;
  while ((i1 = getopt(argc, argv, "+s:h")) != -1) {
    switch(i1) {
      case 's':  if (strcmp(optarg, "no") == 0) {
                   scale = VGAG3_WINSCALE_NOSCALE;
                 } else if (strcmp(optarg, "best") == 0) {
                   scale = VGAG3_WINSCALE_BESTSCALE;
                 } else if (strcmp(optarg, "max") == 0) {
                   scale = VGAG3_WINSCALE_MAXIMIZE;
                 } else if (strcmp(optarg, "full") == 0) {
                   scale = VGAG3_WINSCALE_FULLSCREEN;
                 }
                 break;
      case 'h':
      default:  fprintf(stderr, "Usage: %s [<options>]\n", argv[0]);
                fprintf(stderr, "options:\n");
                fprintf(stderr, " -s <scaling>:   scale window according to <scaling>:\n");
                fprintf(stderr, "                 - no: no scaling\n");
                fprintf(stderr, "                 - best: best scaling\n");
                fprintf(stderr, "                 - max: maximize window\n");
                fprintf(stderr, "                 - full: fullscreen\n");
                exit(1);
    }
  }

  srand(time(NULL) % 1000);

  /* open window */
  wstruct = VG3_window_new(argv[0], VGAG3_VGAVERSION_LOW, scale);
  if (wstruct == NULL) { fprintf(stderr, "%s\n", VG3_error()); exit(1); }

  /* don't capture mouse */
  VG3_mouserelease(wstruct, VGAG3_KEY_NOKEY);

  /* show first title */
  { struct vg3_text stxt;
    struct vg3_image *imgtxt;
    struct vg3_image_attributes iattr;
    int cldef0, cldef1;
    memset(&gmain, 0, sizeof(gmain));
    gmain.wstruct = wstruct;
    settings_readfromfile(&gmain, 1);  /* initial settings */
    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, "10x17.font", 0, 0, "Snakes");
    bgimage = VG3_image_load(wstruct, FILES_DIR "/bmp/title.bmp", 1);
    if (bgimage == NULL) { fprintf(stderr, "%s\n", VG3_error()); VG3_window_free(wstruct); exit(1); }
    VG3_window_getsize(wstruct, &gmain.winw, &gmain.winh);
    imgtxt = VG3_image_text(wstruct, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT);
    if (imgtxt == NULL) { fprintf(stderr, "%s\n", VG3_error()); VG3_window_free(wstruct); exit(1); }
    VG3_window_attributes(wstruct, NULL, NULL, -1, &cldef0);
    VG3_discard_input(wstruct);
    for (i1 = 0; i1 < 300; i1++) {
      if (VG3_inputevent_update(wstruct)) { VG3_window_free(wstruct); exit(0); }
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_ENTER, VGAG3_IS_NEW_PRESSED)) { break; }
      if (VG3_key_ispressed(wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) { break; }
      VG3_draw_clear(wstruct, NULL, VGAG3_COLOR_BLACK);
      VG3_image_copy(wstruct, NULL, bgimage, gmain.winw / 2, gmain.winh / 2, NULL, 0);
      VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
      if (i1 < 36) { iattr.rotate = i1 * 10; iattr.zoom = 100 * i1 * 10 / 360; }
      else if (i1 < 100) { iattr.zoom = 100 * i1 * 10 / 360; }
      else { iattr.zoom = 100 * 100 * 10 / 360; }
      if (i1 == 120 || i1 == 123) {
        cldef1 = VGAG3_COLORDEF_INVERT;
        VG3_window_attributes(wstruct, NULL, NULL, cldef1, NULL);
      } else if (i1 == 121 || i1 == 124) {
        VG3_window_attributes(wstruct, NULL, NULL, cldef0, NULL);
      }
      VG3_image_copy(wstruct, NULL, imgtxt, gmain.winw / 2, gmain.winh / 2, &iattr, 0);
      VG3_window_update(wstruct, 0, 0);
      VG3_wait_time(50);
    }
    VG3_discard_input(wstruct);
    VG3_window_attributes(wstruct, NULL, NULL, -1, &cldef0);
    VG3_image_unload(wstruct, imgtxt);
    VG3_image_unload(wstruct, bgimage);
    VG3_draw_clear(wstruct, NULL, VGAG3_COLOR_BLACK);
    VG3_window_update(wstruct, 0, 0);
    VG3_wait_time(1000);
  }

gostart:
  retw = 1;
  level = NULL;
  goend = 0;

  /* initialize game-struct */
  memset(&gmain, 0, sizeof(gmain));
  gmain.wstruct = wstruct;
  VG3_window_getsize(gmain.wstruct, &gmain.winw, &gmain.winh);

  /* initialize multi-language */
  gmain.mlang = VG3_multilang_new(NULL, "en");

  /* initialize system-menu */
  gmain.sysm = VG3_sysmenu_new(gmain.wstruct, NULL, VG3_color_brightness(VGAG3_COLOR_GREEN, 50), VGAG3_COLOR_GREEN);
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/sysmenu");  /* add multi-language file for system-menu */

  /* add other multi-language files */
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/game");
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/help");
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/film");

  /* insert exit-menu, volume-menu and window-menu into system-menu */
  VG3_sysmenu_simple_exitmenu(gmain.sysm, NULL, VG3_multilang_get(gmain.mlang, "exit-game"));
  VG3_sysmenu_simple_volumemenu(gmain.sysm, NULL,
    VG3_multilang_get(gmain.mlang, "main-vol"),
    VG3_multilang_get(gmain.mlang, "sound-vol"),
    VG3_multilang_get(gmain.mlang, "music-vol"),
    NULL
  );
  VG3_sysmenu_simple_windowmenu(gmain.sysm, NULL, VG3_multilang_get(gmain.mlang, "window-change"));

  /* read in saved system-menu and other settings */
  settings_readfromfile(&gmain, 0);

  /* define keys for found input-devices and add them into the system-menu */
  create_input_keys(&gmain);

  /* show main menu */
  if (main_menu(&gmain)) { retw = 0; goend = 2; goto endgame; }

  /* create quadtree with size of window */
  rect.x = rect.y = 0;
  rect.w = gmain.winw; rect.h = gmain.winh;
  gmain.qdtr = VG3_coll_q_new(&rect, 0, 0); 
  if (gmain.qdtr == NULL) { fprintf(stderr, "%s\n", VG3_error()); goend = 2; goto endgame; }

  /* create main-struct for object-functions */
  gmain.ofstruct = ofunc_new();

  /* create my snake */
  ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_ISNAKE));
  if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_ISNAKE)); goend = 2; goto endgame; }
  if ((objisnake = ofc->f_new(&gmain, 0)) == NULL) { fprintf(stderr, "%s\n", VG3_error()); goend = 2; goto endgame; }

  /* load levels */
  { FILE *ffl = fopen(FILES_DIR "/levels.data", "r");
    if (ffl == NULL) { fprintf(stderr, "Cannot load levels.data\n"); goend = 2; goto endgame; }
    anz_level = 0;
    level = malloc((anz_level + 1) * sizeof(*level));
    if (level == NULL) { fprintf(stderr, "malloc: %s\n", strerror(errno)); goend = 2; goto endgame; }
    while (fread(&level[anz_level], sizeof(*level), 1, ffl) == 1) {
      anz_level++;
      level = realloc(level, (anz_level + 1) * sizeof(*level));
      if (level == NULL) { fprintf(stderr, "realloc: %s\n", strerror(errno)); goend = 2; goto endgame; }
    }
    fclose(ffl);
  }

  /* +++ level loop +++ */

  for (; gmain.mom_level < anz_level; gmain.mom_level++) {

    /* show intersection */
    snprintf(buf, sizeof(buf), "%s/film/zw_seq/zw_seq%d.film", FILES_DIR, gmain.mom_level + 1);
    if (VG3_film_play(gmain.wstruct, gmain.mlang, buf, 1) == 1) { goend = 2; goto endgame; }

    /* create ground image and wall-objects onto the ground image */
    bgimage = create_ground(&gmain, level);
    if (bgimage == NULL) { fprintf(stderr, "%s\n", VG3_error()); goend = 2; goto endgame; }

    /* reset my snake */
    ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_ISNAKE));
    if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_ISNAKE)); goend = 2; goto endgame; }
    if (ofc->f_data != NULL) { ofc->f_data(&gmain, objisnake, &gmain.mom_level); }
    isnake_health = gmain.isnake_health;

    /* activate enemy snake object-management */
    if (VG3_ofunc_mgmt_activate(gmain.ofstruct, &gmain, get_oid_name(OID_NAME_ESNAKE),
        level[gmain.mom_level].esnake_aktiv,
        level[gmain.mom_level].esnake_max,
        level[gmain.mom_level].loop_newsnake,
        level[gmain.mom_level].kind) != 0)
    {
      fprintf(stderr, "%s\n", VG3_error());
      goend = 2;
      goto endgame;
    }

    /* create camouflage hat */
    ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_CAMHAT));
    if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_CAMHAT)); goend = 2; goto endgame; }
    if (ofc->f_new(&gmain, 0, level[gmain.mom_level].camhat_no, level[gmain.mom_level].camhat_yes) == NULL) {
      fprintf(stderr, "%s\n", VG3_error());
      goend = 2;
      goto endgame;
    }

    /* create poison bottle */
    ofc = VG3_ofunc_get_objfunc(gmain.ofstruct, get_oid_name(OID_NAME_BOTTLE));
    if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_BOTTLE)); goend = 2; goto endgame; }
    if (ofc->f_new(&gmain, 0, level[gmain.mom_level].bottle_no, level[gmain.mom_level].bottle_yes) == NULL) {
      fprintf(stderr, "%s\n", VG3_error());
      goend = 2;
      goto endgame;
    }

    /* load and start background music */
    snprintf(buf, sizeof(buf), "%s/sound/m%d.mid", FILES_DIR, (gmain.mom_level / 2) % 5 + 1);
    bg_music = VG3_audio_load(gmain.wstruct, buf, 100, VGAG3_AUDIO_VOLUME_MUSIC);
    if (bg_music > 0) {
      VG3_audio_play(gmain.wstruct, bg_music, 1, 0);
    } else {
      fprintf(stderr, "loading %s: %s\n", buf, VG3_error());
    }

    /* game loop */
    VG3_discard_input(gmain.wstruct);
    for (;;) {
      if (VG3_inputevent_update(gmain.wstruct)) { goend = 2; break; }

      /* ALT+Q: exit */
      if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
        && VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) { goend = 1; break; }

      /* ESC: system-menu */
      if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) {
        if (VG3_sysmenu_exec(gmain.sysm) > 0) { goend = 2; break; }
      }

      /* P: pause game */
      if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_P, VGAG3_IS_NEW_PRESSED)) {
        if (VG3_pause(gmain.wstruct, NULL, VGAG3_KEY_NOKEY)) { goend = 2; break; }
      }

      /* get remaining esnakes */
      esnake_remain = VG3_ofunc_mgmt_data(gmain.ofstruct, &gmain, get_oid_name(OID_NAME_ESNAKE), NULL);

      VG3_draw_clear(gmain.wstruct, NULL, VGAG3_COLOR_BLACK);
      VG3_image_copy(gmain.wstruct, NULL, bgimage, gmain.winw / 2, gmain.winh / 2, NULL, 0);

      /* execute object-management */
      VG3_ofunc_mgmt_run(gmain.ofstruct, &gmain);

      /* execute run-function of loaded object-instances */
      VG3_ofunc_objlist_call_run(gmain.ofstruct, &gmain);

      /* execute draw-function of loaded object-instances */
      VG3_ofunc_objlist_call_draw(gmain.ofstruct, &gmain);

      draw_info(&gmain, gmain.isnake_health, esnake_remain);

#if 0
      VG3_coll_q_mark(gmain.qdtr, gmain.wstruct, NULL, VGAG3_COLOR_RED, NULL);
#endif

      VG3_window_update(gmain.wstruct, 0, 0);
      VG3_wait_time(40);

      /* game over? */
      if (VG3_ofunc_objlist_find_obj(gmain.ofstruct, get_oid_name(OID_NAME_ISNAKE), NULL) == 0) {
        struct vg3_text stxt;
        int endmusic;

        /* fade out background-music */
        if (bg_music > 0) {
          VG3_audio_stop(gmain.wstruct, bg_music, 1);
          for (;;) {
            VG3_window_update(gmain.wstruct, 0, 0);
            VG3_wait_time(50);
            if (!VG3_audio_isplaying(gmain.wstruct, bg_music)) { break; }
          }
        }

        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, "10x17.font", 0, 0, "Game over");
        rect.x = 0; rect.w = gmain.winw;
        rect.y = 0; rect.h = gmain.winh;
        rect = VG3_draw_text(gmain.wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, level[gmain.mom_level].bgcolor, 1);
        rect.x = (gmain.winw - rect.w) / 2;
        rect.y = (gmain.winh - rect.h) / 2;
        VG3_draw_text(gmain.wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, level[gmain.mom_level].bgcolor, 0);
        endmusic = VG3_audio_load(gmain.wstruct, FILES_DIR "/sound/gamov.wav", 100, VGAG3_AUDIO_VOLUME_MUSIC);
        if (endmusic > 0) {
          VG3_audio_play(gmain.wstruct, endmusic, 0, 0);
          for (;;) {
            VG3_window_update(gmain.wstruct, 0, 0);
            VG3_wait_time(50);
            if (!VG3_audio_isplaying(gmain.wstruct, endmusic)) { break; }
          }
          VG3_audio_unload(gmain.wstruct, endmusic);
        }
        goend = 1;
        break;
      }

      /* level clear? */
      if (esnake_remain == 0) {
        struct vg3_text stxt;
        VG3_audio_stop(gmain.wstruct, bg_music, 1);
        for (i1 = 0; i1 < 40; i1++) {
          VG3_window_update(gmain.wstruct, 0, 0);
          VG3_wait_time(100);
        }
        snprintf(buf, sizeof(buf), "Level %d clear", gmain.mom_level + 1);
        VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, 0, 0, buf);
        rect.x = 0; rect.w = gmain.winw;
        rect.y = 0; rect.h = gmain.winh;
        VG3_draw_clear(gmain.wstruct, NULL, VGAG3_COLOR_BLACK);
        rect = VG3_draw_text(gmain.wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 1);
        rect.x = (gmain.winw - rect.w) / 2;
        rect.y = (gmain.winh - rect.h) / 2;
        VG3_draw_text(gmain.wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 0);
        for (i1 = 0; i1 < 30; i1++) {
          VG3_window_update(gmain.wstruct, 0, 0);
          VG3_wait_time(100);
        }
        break;
      }
    }
    VG3_discard_input(gmain.wstruct);

    /* unload all except player */
    if (bg_music > 0) { VG3_audio_unload(gmain.wstruct, bg_music); }
    VG3_image_unload(gmain.wstruct, bgimage);
    VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, get_oid_name(OID_NAME_WALL));
    VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, get_oid_name(OID_NAME_ESNAKE));
    VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, get_oid_name(OID_NAME_SHOT));
    VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, get_oid_name(OID_NAME_CAMHAT));
    VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, get_oid_name(OID_NAME_BOTTLE));

    /* deactivate object-management */
    VG3_ofunc_mgmt_deactivate(gmain.ofstruct, &gmain, NULL);

    if (goend > 0) { gmain.isnake_health = isnake_health; gmain.mom_level--; }

    /* write out actual system-menu and other settings */
    { char *ptr;
      ptr = VG3_sysmenu_savestring_insert(gmain.sysm, gmain.sysm_string);
      if (gmain.sysm_string != NULL) { free(gmain.sysm_string); }
      gmain.sysm_string = ptr;
      settings_savetofile(&gmain);
    }

    if (goend > 0) { break; }
  }

  /* unload and free still existing object-instances */
  VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, NULL);
  objisnake = NULL;

  /* unload still existing images */
  VG3_image_unload(gmain.wstruct, NULL);

  /* free levels */
  free(level);

  /* free quadtree */
  VG3_coll_q_free(gmain.qdtr);

  /* free main-struct for object-functions */
  VG3_ofunc_free(gmain.ofstruct);

  retw = 0;
endgame:
  /* free system-menu */
  free(gmain.sysm_string);
  VG3_sysmenu_free(gmain.sysm);

  /* free multi-language */
  VG3_multilang_free(gmain.mlang);

  /* go to beginning? */
  if (goend < 2) {
    if (goend == 0) {
      mlang = VG3_multilang_new(NULL, "en");
      VG3_multilang_add(mlang, FILES_DIR "/mlang/film");
      if (VG3_film_play(gmain.wstruct, mlang, FILES_DIR "/film/extro/extro.film", 1) == 1) { goend = 2; }
      VG3_multilang_free(mlang);
    }
    if (goend < 2) { goto gostart; }
  }

  /* close window */
  VG3_window_free(gmain.wstruct);

  exit(retw);
}


/* create ground image, create wall-objects and draw them on the ground image, return it */
static struct vg3_image *
create_ground(struct g_main *gmain, struct g_level *level)
{
  const int wall_x = 7;  /* number of wall-columns in x-direction */
  const int wall_y = 4;  /* number of wall-columns in y-direction */
  const int nowall_x = 2;  /* space-wall-columns between wall-columns in x-direction */
  const int nowall_y = 2;  /* space-wall-columns between wall-columns in y-direction */
  const int wall_conn = wall_x * wall_y / 2;  /* number of connections between wall-columns = walls */
  int xwall[wall_x], ywall[wall_y], xpos, ypos, i1, i2;
  char buf[256];
  struct vg3_image_attributes iattr;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_image *wallimg, *bgimg, *groundimg;
  int wallsize;
  struct vg3_rect rect, wallrect;

  if (gmain == NULL || level == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  wallimg = bgimg = groundimg = NULL;

  /* background image */
  snprintf(buf, sizeof(buf), "%s/bmp/bgimage%d.bmp", FILES_DIR, level[gmain->mom_level].bgno);
  bgimg = VG3_image_load(gmain->wstruct, buf, 1);
  if (bgimg == NULL) { goto endgr; }

  /* wall image */
  snprintf(buf, sizeof(buf), "%s/bmp/wall%d.bmp", FILES_DIR, (gmain->mom_level / 2) % 5 + 1);
  wallimg = VG3_image_load(gmain->wstruct, buf, 1);
  if (wallimg == NULL) { goto endgr; }
  VG3_image_getsize(gmain->wstruct, wallimg, NULL, &wallrect.w, &wallrect.h);
  wallsize = (wallrect.w > wallrect.h ? wallrect.w : wallrect.h);

  /* ground image */
  groundimg = VG3_image_create(gmain->wstruct, gmain->winw, gmain->winh);
  if (groundimg == NULL) { goto endgr; }
  VG3_draw_clear(gmain->wstruct, groundimg, level[gmain->mom_level].bgcolor);

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr); iattr.bright = level[gmain->mom_level].bgbright;
  VG3_image_copy(gmain->wstruct, groundimg, bgimg, gmain->winw / 2, gmain->winh / 2, &iattr, 0);

  /* +++ create walls and draw them on the ground image +++ */

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_WALL));
  if (ofc == NULL) { fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_WALL)); goto endgr; }

  /* x-position of walls */
  for (xpos = 0; xpos < wall_x; xpos++) {
    xwall[xpos] = (gmain->winw - ((wall_x + 1) * (wallsize * (nowall_x + 1)) + wallsize)) / 2 + (xpos + 1) * (wallsize * (nowall_x + 1)) + wallsize / 2;
  }

  /* y-position of walls */
  for (ypos = 0; ypos < wall_y; ypos++) {
    ywall[ypos] = (ypos + 1) * (wallsize * (nowall_y + 1)) + wallsize / 2;
  }

  /* create and draw wall-columns */
  for (ypos = 0; ypos < wall_y; ypos++) {
    for (xpos = 0; xpos < wall_x; xpos++) {
      wallrect.x = xwall[xpos] - wallsize / 2;
      wallrect.y = ywall[ypos] - wallsize / 2;
      if (ofc->f_new(gmain, 0, &wallrect) == NULL) { goto endgr; }
      VG3_image_copy(gmain->wstruct, groundimg, wallimg, wallrect.x + wallsize / 2, wallrect.y + wallsize / 2, NULL, 0);
    }
  }

  /* create and draw horizontal walls */
  for (i1 = 0; i1 < wall_conn / 2; i1++) {
    xpos = ZUFALL(0, wall_x - 2);
    ypos = ZUFALL(0, wall_y - 1);
    for (i2 = 1; i2 <= nowall_x; i2++) {
      wallrect.x = xwall[xpos] + (i2 * wallsize) - wallsize / 2;
      wallrect.y = ywall[ypos] - wallsize / 2;
      if (ofc->f_new(gmain, 0, &wallrect) == NULL) { goto endgr; }
      VG3_image_copy(gmain->wstruct, groundimg, wallimg, wallrect.x + wallsize / 2, wallrect.y + wallsize / 2, NULL, 0);
    }
  }

  /* create and draw vertical walls */
  for (i1 = 0; i1 < wall_conn / 2; i1++) {
    xpos = ZUFALL(0, wall_x - 1);
    ypos = ZUFALL(0, wall_y - 2);
    for (i2 = 1; i2 <= nowall_y; i2++) {
      wallrect.x = xwall[xpos] - wallsize / 2;
      wallrect.y = ywall[ypos] + (i2 * wallsize) - wallsize / 2;
      if (ofc->f_new(gmain, 0, &wallrect) == NULL) { goto endgr; }
      VG3_image_copy(gmain->wstruct, groundimg, wallimg, wallrect.x + wallsize / 2, wallrect.y + wallsize / 2, NULL, 0);
    }
  }

  /* draw horizontal borders */
  for (i1 = 0; i1 < (wall_x * (nowall_x + 1)) + nowall_x + 2; i1++) {
    wallrect.x = xwall[0] - ((nowall_x + 1) * wallsize) + (i1 * wallsize) - wallsize / 2;
    wallrect.y = ywall[0] - ((nowall_y + 1) * wallsize) - wallsize / 2;
    if (ofc->f_new(gmain, 0, &wallrect) == NULL) { goto endgr; }
    VG3_image_copy(gmain->wstruct, groundimg, wallimg, wallrect.x + wallsize / 2, wallrect.y + wallsize / 2, NULL, 0);
    wallrect.x = xwall[0] - ((nowall_x + 1) * wallsize) + (i1 * wallsize) - wallsize / 2;
    wallrect.y = ywall[wall_y - 1] + (nowall_y + 1) * wallsize - wallsize / 2;
    if (ofc->f_new(gmain, 0, &wallrect) == NULL) { goto endgr; }
    VG3_image_copy(gmain->wstruct, groundimg, wallimg, wallrect.x + wallsize / 2, wallrect.y + wallsize / 2, NULL, 0);
  }

  /* draw vertical borders */
  for (i1 = 0; i1 < (wall_y * (nowall_y + 1)) + nowall_y + 2; i1++) {
    wallrect.x = xwall[0] - ((nowall_x + 1) * wallsize) - wallsize / 2;
    wallrect.y = ywall[0] - ((nowall_y + 1) * wallsize) + (i1 * wallsize) - wallsize / 2;
    if (ofc->f_new(gmain, 0, &wallrect) == NULL) { goto endgr; }
    VG3_image_copy(gmain->wstruct, groundimg, wallimg, wallrect.x + wallsize / 2, wallrect.y + wallsize / 2, NULL, 0);
    wallrect.x = xwall[wall_x - 1] + ((nowall_x + 1) * wallsize) - wallsize / 2;
    wallrect.y = ywall[0] - (nowall_y + 1) * wallsize + (i1 * wallsize) - wallsize / 2;
    if (ofc->f_new(gmain, 0, &wallrect) == NULL) { goto endgr; }
    VG3_image_copy(gmain->wstruct, groundimg, wallimg, wallrect.x + wallsize / 2, wallrect.y + wallsize / 2, NULL, 0);
  }

  /* +++ make outside of ground black +++ */

  rect.x = 0;
  rect.w = xwall[0] - ((nowall_x + 1) * wallsize) - (wallsize / 2);
  rect.y = 0;
  rect.h = gmain->winh;
  VG3_draw_rect(gmain->wstruct, groundimg, &rect, 1, VGAG3_COLOR_BLACK);

  rect.x = xwall[0] - ((nowall_x + 1) * wallsize) + (((wall_x * (nowall_x + 1)) + nowall_x + 2) * wallsize) - (wallsize / 2);
  rect.w = gmain->winw - rect.x;
  rect.y = 0;
  rect.h = gmain->winh;
  VG3_draw_rect(gmain->wstruct, groundimg, &rect, 1, VGAG3_COLOR_BLACK);

  rect.x = 0;
  rect.w = gmain->winw;
  rect.y = ywall[0] - (nowall_y + 1) * wallsize + (((wall_y * (nowall_y + 1)) + nowall_y + 2) * wallsize) - (wallsize / 2);
  rect.h = gmain->winh - rect.y;
  VG3_draw_rect(gmain->wstruct, groundimg, &rect, 1, VGAG3_COLOR_BLACK);

  /* rectangle of ground */
  gmain->grect.x = xwall[0] - ((nowall_x + 1) * wallsize) - (wallsize / 2) + wallsize;
  gmain->grect.w = (xwall[0] - ((nowall_x + 1) * wallsize) + (((wall_x * (nowall_x + 1)) + nowall_x + 2) * wallsize) - (wallsize / 2)) - gmain->grect.x - wallsize;
  gmain->grect.y = ywall[0] - ((nowall_y + 1) * wallsize - (wallsize / 2));
  gmain->grect.h = (ywall[0] - ((nowall_y + 1) * wallsize) + (((wall_y * (nowall_y + 1)) + nowall_y + 2) * wallsize) - (wallsize / 2)) - gmain->grect.y - wallsize;

  /* done */
  VG3_image_unload(gmain->wstruct, bgimg);
  VG3_image_unload(gmain->wstruct, wallimg);
  return groundimg;

endgr:
  if (bgimg != NULL) { VG3_image_unload(gmain->wstruct, bgimg); }
  if (wallimg != NULL) { VG3_image_unload(gmain->wstruct, wallimg); }
  if (groundimg != NULL) { VG3_image_unload(gmain->wstruct, groundimg); }

  return NULL;
}


/* draw info on the bottom */
static void
draw_info(struct g_main *gmain, int health, int esnake_remain)
{
  struct vg3_rect rect;
  struct vg3_text stxt;
  int fonth;
  char buf[256];

  if (gmain == NULL) { return; }

  VG3_fontsize(gmain->wstruct, NULL, NULL, &fonth, NULL, NULL);
  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, 0, 0, buf);

  rect.x = 0; rect.w = gmain->winw;
  rect.y = gmain->winh - fonth; rect.h = fonth;
  snprintf(buf, sizeof(buf), "%s=%02d", VG3_multilang_get(gmain->mlang, "snakes"), esnake_remain);
  rect = VG3_draw_text(gmain->wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 1);
  rect.x = gmain->winw - rect.w;
  VG3_draw_text(gmain->wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 0);

  rect.x = 0; rect.w = gmain->winw;
  rect.y = gmain->winh - fonth; rect.h = fonth;
  snprintf(buf, sizeof(buf), "%s=%d%%", VG3_multilang_get(gmain->mlang, "health"), health);
  VG3_draw_text(gmain->wstruct, NULL, &rect, 0, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 0);
}


/* show main menu */
static int
main_menu(struct g_main *gmain)
{
  char **felem;
  int erg, fcolor, bcolor;

  if (gmain == NULL) { return 1; }

  fcolor = VGAG3_COLOR_YELLOW;
  bcolor = VGAG3_COLOR_BLUE;

  felem = calloc(6 + 1, sizeof(*felem));
  if (felem == NULL) { fprintf(stderr, "calloc: %s\n", strerror(errno)); return 1; }

  felem[0] = (char *)VG3_multilang_get(gmain->mlang, "help-title");
  felem[1] = (char *)VG3_multilang_get(gmain->mlang, "new-game");
  if (gmain->mom_level > 0 && gmain->isnake_health > 0) {
    felem[2] = (char *)VG3_multilang_get(gmain->mlang, "last-game");
  } else {
    felem[2] = (char *)VG3_multilang_get(gmain->mlang, "no-last-game");
  }
  felem[3] = (char *)VG3_multilang_get(gmain->mlang, "to-sysmenu");
  felem[4] = "! ";  /* empty line */
  felem[5] = (char *)VG3_multilang_get(gmain->mlang, "quit");
  felem[6] = NULL;

  for (;;) {
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    erg = VG3_input_select(gmain->wstruct, fcolor, bcolor, VG3_multilang_get(gmain->mlang, "main-menu"), felem, 1);
    if (erg < 0) {
      if (erg == -1) { fprintf(stderr, "%s\n", VG3_error()); }
      erg = 1;
      break;
    } else if (erg == 1) {  /* help */
      if (help_menu(gmain) > 0) { erg = 1; break; }
    } else if (erg == 2) {  /* new game */
      gmain->isnake_health = 100;
      gmain->mom_level = 0;
      erg = 0;
      if (VG3_film_play(gmain->wstruct, gmain->mlang, FILES_DIR "/film/intro/intro.film", 1) == 1) { erg = 1; }
      break;
    } else if (erg == 3) {  /* last game */
      erg = 0;
      break;
    } else if (erg == 4) {  /* system-menu */
      if (VG3_sysmenu_exec(gmain->sysm) > 0) { erg = 1; break; }
    } else if (erg == 6) {  /* quit */
      erg = 1;
      break;
    }
  }

  free(felem);

  return erg;
}


/* show help */
static int
help_menu(struct g_main *gmain)
{
  int erg;

  if (gmain == NULL) { return 1; }
  erg = VG3_output_text(gmain->wstruct, VGAG3_COLOR_YELLOW, VGAG3_COLOR_BLUE, VG3_multilang_get(gmain->mlang, "help-text"));

  if (erg > 0) { return 1; }
  return 0;
}
