#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-bottle.h"

void getofc_bottle(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_bottle(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_BOTTLE));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-functions +++ */

/* new-function
 * variable parameter: int bottle_no  = number of loops poison-bottle is inactive
 *                     int bottle_yes = number of loops poison-bottle is active
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_bottle *gobj;
  struct vg3_sprite *sprt;
  int bottle_no, bottle_yes;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments: bottle_no:int, bottle_yes:int */
  va_start(ap, iparent);
  bottle_no = va_arg(ap, int);
  bottle_yes = va_arg(ap, int);
  va_end(ap);

  /* load sprite */
  sprt = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/giftfl.sprite");
  if (sprt == NULL) { return NULL; }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->sprt = sprt;
  gobj->bottle_no = bottle_no;
  gobj->bottle_yes = bottle_yes;
  gobj->bottle_loop = -gobj->bottle_no;
  VG3_sprite_imagesize(gobj->sprt, "avg", 100, &gobj->rect.w, &gobj->rect.h);

  /* load audios */
  gobj->snd_bottle = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/giftflasche.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  if (gobj->snd_bottle == 0) { fprintf(stderr, "loading giftflasche.wav: %s\n", VG3_error()); }

  /* create and fill bottle-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_BOTTLE));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert bottle-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_bottle *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_bottle *)objp->ostruct;

  /* remove bottle-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove bottle-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free bottle-instance */
  if (gobj->snd_bottle != 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_bottle); }
  VG3_sprite_free(gobj->sprt);
  free(gobj);
  free(objp);
}


/* run-function: not running but activating or deactivating */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_bottle *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_bottle *)objp->ostruct;

  gobj->bottle_loop++;
  if (gobj->bottle_loop == 0) {  /* activate */
    struct vg3_coll coll, *pcoll;
    int erg;
    for (;;) {  /* find an unused place */
      gobj->rect.x = ZUFALL(gmain->grect.x, gmain->grect.x + gmain->grect.w - gobj->rect.w);
      gobj->rect.y = ZUFALL(gmain->grect.y, gmain->grect.y + gmain->grect.h - gobj->rect.h);
      erg = VG3_coll_q_find(gmain->qdtr, &gobj->rect, NULL, &pcoll);
      if (erg < 0) { fprintf(stderr, "%s\n", VG3_error()); f_free(gmain, objp); return; }
      if (erg == 0) { break; }
      free(pcoll);
    }
    VG3_sprite_rewind(gobj->sprt);
    /* insert bottle-instance into quadtree */
    memset(&coll, 0, sizeof(coll));
    coll.rect = gobj->rect;
    snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_BOTTLE));
    coll.optr = objp;
    VG3_coll_q_insert(gmain->qdtr, &coll);

  } else if (gobj->bottle_loop == gobj->bottle_yes) {  /* deactivate */
    VG3_coll_q_remove(gmain->qdtr, objp);  /* remove bottle-instance from quadtree */
    gobj->bottle_loop = -gobj->bottle_no;
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_bottle *gobj;
  struct vg3_image *img;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_bottle *)objp->ostruct;

  if (gobj->bottle_loop < 0) { return; }

  /* draw bottle-instance */
  VG3_sprite_get(gobj->sprt, &img, &iattr, NULL);
  if (img != NULL) {
    VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
  }
}
