#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-camhat.h"

void getofc_camhat(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_camhat(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_CAMHAT));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-functions +++ */

/* new-function
 * variable parameter: int camhat_no  = number of loops camouflage hat is inactive
 *                     int camhat_yes = number of loops camouflage hat is active
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_camhat *gobj;
  struct vg3_sprite *sprt;
  int camhat_no, camhat_yes;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments: camhat_no:int, camhat_yes:int */
  va_start(ap, iparent);
  camhat_no = va_arg(ap, int);
  camhat_yes = va_arg(ap, int);
  va_end(ap);

  /* load sprite */
  sprt = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/tarnkp.sprite");
  if (sprt == NULL) { return NULL; }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->sprt = sprt;
  gobj->camhat_no = camhat_no;
  gobj->camhat_yes = camhat_yes;
  gobj->camhat_loop = -gobj->camhat_no;
  VG3_sprite_imagesize(gobj->sprt, "avg", 100, &gobj->rect.w, &gobj->rect.h);

  /* load audios */
  gobj->snd_camhat = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/tarnkappe.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  if (gobj->snd_camhat == 0) { fprintf(stderr, "loading tarnkappe.wav: %s\n", VG3_error()); }

  /* create and fill camhat-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_CAMHAT));
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert camhat-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_camhat *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_camhat *)objp->ostruct;

  /* remove camhat-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove camhat-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free camhat-instance */
  VG3_sprite_free(gobj->sprt);
  if (gobj->snd_camhat != 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_camhat); }
  free(gobj);
  free(objp);
}


/* run-function: not running but activating or deactivating */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_camhat *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_camhat *)objp->ostruct;

  if (gobj->camhat_yes < 1) { return; }

  gobj->camhat_loop++;
  if (gobj->camhat_loop == 0) {  /* activate */
    struct vg3_coll coll, *pcoll;
    int erg;
    for (;;) {  /* find an unused place */
      gobj->rect.x = ZUFALL(gmain->grect.x, gmain->grect.x + gmain->grect.w - gobj->rect.w);
      gobj->rect.y = ZUFALL(gmain->grect.y, gmain->grect.y + gmain->grect.h - gobj->rect.h);
      erg = VG3_coll_q_find(gmain->qdtr, &gobj->rect, NULL, &pcoll);
      if (erg < 0) { fprintf(stderr, "%s\n", VG3_error()); f_free(gmain, objp); return; }
      if (erg == 0) { break; }
      free(pcoll);
    }
    VG3_sprite_rewind(gobj->sprt);
    /* insert camhat-instance into quadtree */
    memset(&coll, 0, sizeof(coll));
    coll.rect = gobj->rect;
    snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_CAMHAT));
    coll.optr = objp;
    VG3_coll_q_insert(gmain->qdtr, &coll);

  } else if (gobj->camhat_loop == gobj->camhat_yes) {  /* deactivate */
    VG3_coll_q_remove(gmain->qdtr, objp);  /* remove camhat-instance from quadtree */
    gobj->camhat_loop = -gobj->camhat_no;
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_camhat *gobj;
  struct vg3_image *img;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_camhat *)objp->ostruct;

  if (gobj->camhat_loop < 0) { return; }

  /* draw camhat-instance */
  VG3_sprite_get(gobj->sprt, &img, &iattr, NULL);
  if (img != NULL) {
    VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
  }
}
