#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-esnake.h"

void getofc_esnake(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);
static int get_isnake_limbs(void *, struct g_obj_esnake *);

/* kinds of enemy snakes */
static struct es_kind es_kinds[] = {
  { 5, 1, 1, 7 },     /* Buschnatter */
  { 6, 2, 1, 9 },     /* gelbe Ringelotter */
  { 6, 2, 1, 9 },     /* gruene Ringelotter */
  { 7, 2, 1, 18 },    /* Nebelnatter */
  { 8, 3, 1, 11 },    /* gemeine Grabviper */
  { 9, 1, 2, 13 },    /* gescheckte Felsotter */
  { 12, 2, 2, 15 },   /* schmale Nadelviper */
};

/* +++ get-function +++ */

void
getofc_esnake(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_ESNAKE));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-functions +++ */

/* new-function
 * variable parameters: kind of enemy snake (integer)
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_esnake *gobj;
  struct vg3_rect rect;
  int kind;
  char buf[256];
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* variable parameters: kind of enemy */
  va_start(ap, iparent);
  kind = va_arg(ap, int);
  va_end(ap);
  if (kind < 1 || kind > (int)(sizeof(es_kinds) / sizeof(*es_kinds))) {
    VG3_seterror(EINVAL, "Kind of enemy snake invalid\n");
    return NULL;
  }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->kind = es_kinds[kind - 1];

  gobj->limb_anz = gobj->kind.limb_max;
  if (gobj->limb_anz > ESNAKE_LIMB_MAX) { gobj->limb_anz = ESNAKE_LIMB_MAX; }

  /* load images */
  gobj->img_head_short = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/ekopfshort.bmp", 1);
  if (gobj->img_head_short == NULL) { return NULL; }
  gobj->img_head_long = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/ekopflong.bmp", 1);
  if (gobj->img_head_long == NULL) { return NULL; }
  gobj->img_eye = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/eauge.bmp", 1);
  if (gobj->img_eye == NULL) { return NULL; }
  snprintf(buf, sizeof(buf), "%s/bmp/eglied%d.bmp", FILES_DIR, kind);
  gobj->img_limb = VG3_image_load(gmain->wstruct, buf, 1);
  if (gobj->img_limb == NULL) { return NULL; }

  /* create tongue images */
  /* up */
  gobj->img_tg[ES_MOVING_DIR_UP] = VG3_image_create(gmain->wstruct, 4, 5);
  if (gobj->img_tg[ES_MOVING_DIR_UP] == NULL) { return NULL; }
  VG3_draw_line(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_UP], 0, 0, 0, 2, VGAG3_COLOR_RED);
  VG3_draw_line(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_UP], 3, 0, 3, 2, VGAG3_COLOR_RED);
  rect.x = 1; rect.w = 2; rect.y = 2; rect.h = 3;
  VG3_draw_rect(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_UP], &rect, 1, VGAG3_COLOR_RED);
  /* right */
  gobj->img_tg[ES_MOVING_DIR_RIGHT] = VG3_image_create(gmain->wstruct, 5, 4);
  if (gobj->img_tg[ES_MOVING_DIR_RIGHT] == NULL) { return NULL; }
  VG3_draw_line(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_RIGHT], 2, 0, 4, 0, VGAG3_COLOR_RED);
  VG3_draw_line(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_RIGHT], 2, 3, 4, 3, VGAG3_COLOR_RED);
  rect.x = 0; rect.w = 3; rect.y = 1; rect.h = 2;
  VG3_draw_rect(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_RIGHT], &rect, 1, VGAG3_COLOR_RED);
  /* down */
  gobj->img_tg[ES_MOVING_DIR_DOWN] = VG3_image_create(gmain->wstruct, 4, 5);
  if (gobj->img_tg[ES_MOVING_DIR_DOWN] == NULL) { return NULL; }
  VG3_draw_line(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_DOWN], 0, 2, 0, 4, VGAG3_COLOR_RED);
  VG3_draw_line(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_DOWN], 3, 2, 3, 4, VGAG3_COLOR_RED);
  rect.x = 1; rect.w = 2; rect.y = 0; rect.h = 3;
  VG3_draw_rect(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_DOWN], &rect, 1, VGAG3_COLOR_RED);
  /* left */
  gobj->img_tg[ES_MOVING_DIR_LEFT] = VG3_image_create(gmain->wstruct, 5, 4);
  if (gobj->img_tg[ES_MOVING_DIR_LEFT] == NULL) { return NULL; }
  VG3_draw_line(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_LEFT], 0, 0, 2, 0, VGAG3_COLOR_RED);
  VG3_draw_line(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_LEFT], 0, 3, 2, 3, VGAG3_COLOR_RED);
  rect.x = 2; rect.w = 3; rect.y = 1; rect.h = 2;
  VG3_draw_rect(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_LEFT], &rect, 1, VGAG3_COLOR_RED);

  /* create limbstream and set limbidx */
  { int wks, hks, wkl, hkl, wk, wl, hl, lidx;
    VG3_image_getsize(gmain->wstruct, gobj->img_head_short, NULL, &wks, &hks);
    wks = (wks > hks ? wks : hks);
    VG3_image_getsize(gmain->wstruct, gobj->img_head_short, NULL, &wkl, &hkl);
    wkl = (wkl > hkl ? wkl : hkl);
    wk = (wkl > wks ? wkl : wks);
    VG3_image_getsize(gmain->wstruct, gobj->img_limb, NULL, &wl, &hl);
    wl = (wl > hl ? wl : hl);
    gobj->limbstream_size = gobj->kind.limb_max * wl - wl / 2 + 1 + wk / 2 + 1;
    gobj->limbstream = calloc(gobj->limbstream_size, sizeof(*gobj->limbstream));
    if (gobj->limbstream == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
    gobj->limbidx[0] = wk / 2 + wk % 2 + wl / 2;
    for (lidx = 1; lidx < gobj->kind.limb_max; lidx++) {
      gobj->limbidx[lidx] = gobj->limbidx[lidx - 1] + wl;
    }
  }

  /* load audios */
  gobj->snd_hit = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/es_giftgetroffen.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  if (gobj->snd_hit == 0) { fprintf(stderr, "loading es_giftgetroffen.wav: %s\n", VG3_error()); }
  gobj->snd_dead = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/es_gefressen.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  if (gobj->snd_dead == 0) { fprintf(stderr, "loading es_gefressen.wav: %s\n", VG3_error()); }
  gobj->snd_come = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/es_kommt.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  if (gobj->snd_come == 0) { fprintf(stderr, "loading es_kommt.wav: %s\n", VG3_error()); }

  /* create and fill esnake-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_ESNAKE));
  objp->drawlevel = 2;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* set rectangle and collsion rectangle with 80% size of head-image */
  { int wks, hks, wkl, hkl, ipos, iz1, iz2;

    VG3_image_getsize(gmain->wstruct, gobj->img_head_short, NULL, &wks, &hks);
    VG3_image_getsize(gmain->wstruct, gobj->img_head_short, NULL, &wkl, &hkl);
    gobj->rect.w = (wkl < wks ? wkl : wks);
    gobj->rect.h = (hkl < hks ? hkl : hks);
    iz1 = ZUFALL(1, 4);
    iz2 = ZUFALL(0, 1);
    if (iz1 == 1) {
      gobj->rect.x = gmain->grect.x + 1;
      gobj->rect.y = gmain->grect.y + 1;
      gobj->movdir = iz2 ? ES_MOVING_DIR_RIGHT : ES_MOVING_DIR_DOWN;
    } else if (iz1 == 2) {
      gobj->rect.x = gmain->grect.x + gmain->grect.w - gobj->rect.w - 1;
      gobj->rect.y = gmain->grect.y + 1;
      gobj->movdir = iz2 ? ES_MOVING_DIR_LEFT : ES_MOVING_DIR_DOWN;
    } else if (iz1 == 3) {
      gobj->rect.x = gmain->grect.x + 1;
      gobj->rect.y = gmain->grect.y + gmain->grect.h - gobj->rect.h - 1;
      gobj->movdir = iz2 ? ES_MOVING_DIR_RIGHT : ES_MOVING_DIR_UP;
    } else {
      gobj->rect.x = gmain->grect.x + gmain->grect.w - gobj->rect.w - 1;
      gobj->rect.y = gmain->grect.y + gmain->grect.h - gobj->rect.h - 1;
      gobj->movdir = iz2 ? ES_MOVING_DIR_LEFT : ES_MOVING_DIR_UP;
    }

    gobj->keep_moving = -90;
    gobj->rand_poison = 100 / gobj->kind.poison_max;
    gobj->crect = gobj->rect;
    gobj->crect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w * 80 / 100, gobj->rect.h * 80 / 100);

    for (ipos = 0; ipos < gobj->limbstream_size; ipos++) {
      gobj->limbstream[ipos].x = gobj->rect.x + gobj->rect.w / 2;
      gobj->limbstream[ipos].y = gobj->rect.y + gobj->rect.h / 2;
      gobj->limbstream[ipos].movdir = gobj->movdir;
    }
  }

  /* insert esnake-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* don't insert esnake-instance into quadtree now */

  /* play coming-sound */
  if (gobj->snd_come != 0) { VG3_audio_play(gmain->wstruct, gobj->snd_come, 0, 0); }

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_esnake *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_esnake *)objp->ostruct;

  /* remove esnake-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove esnake-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free esnake-instance */
  if (gobj->limbstream != NULL) { free(gobj->limbstream); }
  if (gobj->snd_hit != 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_hit); }
  if (gobj->snd_dead != 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_dead); }
  if (gobj->snd_come != 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_come); }
  if (gobj->img_head_short != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_head_short); }
  if (gobj->img_head_long != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_head_long); }
  if (gobj->img_eye != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_eye); }
  if (gobj->img_limb != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_limb); }
  if (gobj->img_tg[ES_MOVING_DIR_UP] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_UP]); }
  if (gobj->img_tg[ES_MOVING_DIR_RIGHT] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_RIGHT]); }
  if (gobj->img_tg[ES_MOVING_DIR_DOWN] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_DOWN]); }
  if (gobj->img_tg[ES_MOVING_DIR_LEFT] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_LEFT]); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_esnake *gobj;
  struct vg3_coll coll;
  int erg;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_esnake *)objp->ostruct;

  /* did not yet come out */
  if (gobj->keep_moving < 0) { gobj->keep_moving++; return; }

  /* destroy esnake, if it has no more limb and snd_dead-sound is over */
  if (gobj->limb_anz == 0) {
    if (!VG3_audio_isplaying(gmain->wstruct, gobj->snd_dead)) { f_free(gmain, objp); }
    return;
  }

  /* been eaten: no moving */
  if (gobj->dying > 0) {
    if (--gobj->dying == 0) {
      f_free(vmain, objp);
    } else {
      gobj->limb_anz = gobj->dying / 10 + 1;
    }
    return;
  }

  /* decrement camhat */
  if (gobj->camhat > 0) { gobj->camhat--; }

  /* remove esnake-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* change moving direction? */
  if (--gobj->keep_moving <= 0) {
    for (;;) {
      erg = ZUFALL(0, 3);
      if (erg != gobj->movdir) { gobj->movdir = erg; break; }
    }
    gobj->keep_moving = ZUFALL(30, 200);
    gobj->xremainder = gobj->yremainder = 0;
    if (gobj->movdir == ES_MOVING_DIR_UP) {
      gobj->xdelta = 0;
      gobj->ydelta = -100;
    } else if (gobj->movdir == ES_MOVING_DIR_RIGHT) {
      gobj->xdelta = 100;
      gobj->ydelta = 0;
    } else if (gobj->movdir == ES_MOVING_DIR_DOWN) {
      gobj->xdelta = 0;
      gobj->ydelta = 100;
    } else {
      gobj->xdelta = -100;
      gobj->ydelta = 0;
    }
  }

  /* spit poison? */
  if (gobj->rand_poison > 0) { gobj->rand_poison--; }
  if (ZUFALL(0, 30 + gobj->rand_poison * 5) <= 2) {
    if (gobj->poison_shots < gobj->kind.poison_max) {
      const struct vg3_ofunc_objfunc *ofc;
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_SHOT));
      if (ofc != NULL) {
        int shot_x, shot_y, angle;
        shot_x = gobj->rect.x + gobj->rect.w / 2;
        shot_y = gobj->rect.y + gobj->rect.h / 2;
        angle = 0;
        if (gobj->movdir == ES_MOVING_DIR_RIGHT) { angle = 90; }
        else if (gobj->movdir == ES_MOVING_DIR_DOWN) { angle = 180; }
        else if (gobj->movdir == ES_MOVING_DIR_LEFT) { angle = 270; }
        if (ofc->f_new(gmain, objp->instanceid, shot_x, shot_y, angle, gobj->kind.poison_strength) != NULL) {
          gobj->poison_shots++;
          gobj->rand_poison = 100 / gobj->kind.poison_max;
        }
      }
    }
  }

  /* move esnake-instance and check for collisions with gobj->crect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->crect, gobj->kind.moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: do nothing */
    fprintf(stderr, "moving esnake: %s\n", VG3_error());
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }
  if (erg == VGAG3_COLL_RETURN_CATCHED) { f_free(vmain, objp); return; }

  /* x/y-value of gobj->crect could be modified, correct it also in gobj->rect */
  gobj->rect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w, gobj->rect.h);

  /* move limbs */
  erg = gobj->rect.x + gobj->rect.w / 2 - gobj->limbstream[0].x;
  if (erg == 0) { erg = gobj->rect.y + gobj->rect.h / 2 - gobj->limbstream[0].y; }
  if (erg < 0) { erg = -erg; }
  for (; erg > 0; erg--) {
    memmove(&gobj->limbstream[1], &gobj->limbstream[0], sizeof(*gobj->limbstream) * (gobj->limbstream_size - 1));
    gobj->limbstream[0].x = gobj->rect.x + gobj->rect.w / 2;
    gobj->limbstream[0].y = gobj->rect.y + gobj->rect.h / 2;
    gobj->limbstream[0].movdir = gobj->movdir;
  }

  if (gobj->limb_anz == 0) { return; }  /* don't put dead snake into quadtree again */

  /* insert esnake-instance (just head) into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_ESNAKE));
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_esnake *gobj;
  int lidx, isize;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_esnake *)objp->ostruct;

  if (gobj->keep_moving < 0) {
    struct vg3_rect rect;
    int iprz = (-(gobj->keep_moving % 30) / 3 + 1) * 10;
    rect.w = gobj->rect.w * iprz / 100;
    rect.h = gobj->rect.h * iprz / 100;
    rect.x = gobj->rect.x + (gobj->rect.w - rect.w) / 2;
    rect.y = gobj->rect.y + (gobj->rect.h - rect.h) / 2;
    VG3_draw_rect(gmain->wstruct, NULL, &rect, 1, VGAG3_COLOR_BLACK);
    return;
  }

  if (gobj->camhat == 0 || (gobj->camhat < 100 && gobj->camhat % 20 < 5)) {
    /* draw limbs */
    for (lidx = gobj->limb_anz - 1; lidx >= 0; lidx--) {
      VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
      iattr.rotate = gobj->limbstream[gobj->limbidx[lidx]].movdir * 90;
      VG3_image_copy(gmain->wstruct, NULL, gobj->img_limb, gobj->limbstream[gobj->limbidx[lidx]].x, gobj->limbstream[gobj->limbidx[lidx]].y, &iattr, 0);
    }
  }

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
  iattr.rotate = gobj->movdir * 90;
  if (gobj->camhat == 0) {  /* draw head */
    int is_limbs = get_isnake_limbs(gmain, gobj);
    if (is_limbs < gobj->limb_anz) {  /* draw red head */
      VG3_image_copy(gmain->wstruct, NULL, gobj->img_head_long, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
    } else {  /* draw green head */
      VG3_image_copy(gmain->wstruct, NULL, gobj->img_head_short, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
    }
  } else {  /* draw eyes */
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_eye, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
  }

  if (gobj->camhat == 0 && gobj->limb_anz > 0) {
    /* draw tongue */
    if (ZUFALL(1, 12) == 1) {
      if (gobj->movdir == ES_MOVING_DIR_UP) {
        VG3_image_getsize(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_UP], NULL, NULL, &isize);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_tg[ES_MOVING_DIR_UP], gobj->rect.x + gobj->rect.w / 2, gobj->rect.y - isize / 2, NULL, 0);
      } else if (gobj->movdir == ES_MOVING_DIR_RIGHT) {
        VG3_image_getsize(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_RIGHT], NULL, &isize, NULL);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_tg[ES_MOVING_DIR_RIGHT], gobj->rect.x + gobj->rect.w + isize / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
      } else if (gobj->movdir == ES_MOVING_DIR_DOWN) {
        VG3_image_getsize(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_DOWN], NULL, NULL, &isize);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_tg[ES_MOVING_DIR_DOWN], gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h + isize / 2, NULL, 0);
      } else if (gobj->movdir == ES_MOVING_DIR_LEFT) {
        VG3_image_getsize(gmain->wstruct, gobj->img_tg[ES_MOVING_DIR_LEFT], NULL, &isize, NULL);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_tg[ES_MOVING_DIR_LEFT], gobj->rect.x - isize / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
      }
    }
  }
}


/* get actual number of limbs from isnake */
static int
get_isnake_limbs(void *vmain, struct g_obj_esnake *gobj)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp_isnake;
  int retw;

  /* if isnake's instance-ID is invalid, actualize it */
  if (gobj->isnake_iid == 0
  || (objp_isnake = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gobj->isnake_iid)) == NULL)
  {
    if (VG3_ofunc_objlist_find_obj(gmain->ofstruct, get_oid_name(OID_NAME_ISNAKE), &objp_isnake) > 0) {
      gobj->isnake_iid = objp_isnake->instanceid;
    }
  }

  /* call f_data() from isnake to return number of limbs */
  retw = gobj->limb_anz;
  if (gobj->isnake_iid > 0) {
    const struct vg3_ofunc_objfunc *ofc;
    ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_ISNAKE));
    if (ofc != NULL && ofc->f_data != NULL) {
      retw = ofc->f_data(gmain, objp_isnake, NULL);
    }
  }

  return retw;
}
