#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-isnake.h"

void getofc_isnake(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);
static int f_data(void *, struct vg3_ofunc_object *, void *);


/* +++ get-function +++ */

void
getofc_isnake(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_ISNAKE));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
  ofc->f_data = f_data;
}


/* +++ object-functions +++ */

/* new-function
 * no variable parameters
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_isnake *gobj;
  struct vg3_coll coll;
  struct vg3_rect rect;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* no variable parameters */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  gobj->limb_anz = (ISNAKE_LIMB_MAX * gmain->isnake_health + 50) / 100;
  if (gobj->limb_anz > ISNAKE_LIMB_MAX) { gobj->limb_anz = ISNAKE_LIMB_MAX; }

  /* load images */
  gobj->img_head = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/ikopf.bmp", 1);
  if (gobj->img_head == NULL) { return NULL; }
  gobj->img_dead = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/tkopf.bmp", 1);
  if (gobj->img_dead == NULL) { return NULL; }
  gobj->img_eye = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/iauge.bmp", 1);
  if (gobj->img_eye == NULL) { return NULL; }
  gobj->img_limb = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/iglied.bmp", 1);
  if (gobj->img_limb == NULL) { return NULL; }

  /* create tongue images */
  /* up */
  gobj->img_tg[IS_MOVING_DIR_UP] = VG3_image_create(gmain->wstruct, 4, 5);
  if (gobj->img_tg[IS_MOVING_DIR_UP] == NULL) { return NULL; }
  VG3_draw_line(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_UP], 0, 0, 0, 2, VGAG3_COLOR_RED);
  VG3_draw_line(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_UP], 3, 0, 3, 2, VGAG3_COLOR_RED);
  rect.x = 1; rect.w = 2; rect.y = 2; rect.h = 3;
  VG3_draw_rect(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_UP], &rect, 1, VGAG3_COLOR_RED);
  /* right */
  gobj->img_tg[IS_MOVING_DIR_RIGHT] = VG3_image_create(gmain->wstruct, 5, 4);
  if (gobj->img_tg[IS_MOVING_DIR_RIGHT] == NULL) { return NULL; }
  VG3_draw_line(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_RIGHT], 2, 0, 4, 0, VGAG3_COLOR_RED);
  VG3_draw_line(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_RIGHT], 2, 3, 4, 3, VGAG3_COLOR_RED);
  rect.x = 0; rect.w = 3; rect.y = 1; rect.h = 2;
  VG3_draw_rect(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_RIGHT], &rect, 1, VGAG3_COLOR_RED);
  /* down */
  gobj->img_tg[IS_MOVING_DIR_DOWN] = VG3_image_create(gmain->wstruct, 4, 5);
  if (gobj->img_tg[IS_MOVING_DIR_DOWN] == NULL) { return NULL; }
  VG3_draw_line(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_DOWN], 0, 2, 0, 4, VGAG3_COLOR_RED);
  VG3_draw_line(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_DOWN], 3, 2, 3, 4, VGAG3_COLOR_RED);
  rect.x = 1; rect.w = 2; rect.y = 0; rect.h = 3;
  VG3_draw_rect(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_DOWN], &rect, 1, VGAG3_COLOR_RED);
  /* left */
  gobj->img_tg[IS_MOVING_DIR_LEFT] = VG3_image_create(gmain->wstruct, 5, 4);
  if (gobj->img_tg[IS_MOVING_DIR_LEFT] == NULL) { return NULL; }
  VG3_draw_line(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_LEFT], 0, 0, 2, 0, VGAG3_COLOR_RED);
  VG3_draw_line(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_LEFT], 0, 3, 2, 3, VGAG3_COLOR_RED);
  rect.x = 2; rect.w = 3; rect.y = 1; rect.h = 2;
  VG3_draw_rect(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_LEFT], &rect, 1, VGAG3_COLOR_RED);

  /* create limbstream and set limbidx */
  { int wk, hk, wl, hl, lidx;
    VG3_image_getsize(gmain->wstruct, gobj->img_head, NULL, &wk, &hk);
    wk = (wk > hk ? wk : hk);
    VG3_image_getsize(gmain->wstruct, gobj->img_limb, NULL, &wl, &hl);
    wl = (wl > hl ? wl : hl);
    gobj->limbstream_size = ISNAKE_LIMB_MAX * wl - wl / 2 + 1 + wk / 2 + 1;
    gobj->limbstream = calloc(gobj->limbstream_size, sizeof(*gobj->limbstream));
    if (gobj->limbstream == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
    gobj->limbidx[0] = wk / 2 + wk % 2 + wl / 2;
    for (lidx = 1; lidx < ISNAKE_LIMB_MAX; lidx++) {
      gobj->limbidx[lidx] = gobj->limbidx[lidx - 1] + wl;
    }
  }

  /* load audios */
  gobj->snd_hit = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/is_giftgetroffen.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  if (gobj->snd_hit == 0) { fprintf(stderr, "loading is_giftgetroffen.wav: %s\n", VG3_error()); }
  gobj->snd_spit = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/is_giftspritzen.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  if (gobj->snd_spit == 0) { fprintf(stderr, "loading is_giftspritzen.wav: %s\n", VG3_error()); }

  /* create and fill isnake-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_ISNAKE));
  objp->drawlevel = 3;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* set position */
  f_data(gmain, objp, &gmain->mom_level);

  /* insert isnake-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* insert isnake-instance (just head) into quadtree */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_ISNAKE));
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_isnake *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_isnake *)objp->ostruct;

  /* remove isnake-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove isnake-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free isnake-instance */
  if (gobj->limbstream != NULL) { free(gobj->limbstream); }
  if (gobj->snd_hit != 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_hit); }
  if (gobj->snd_spit != 0) { VG3_audio_unload(gmain->wstruct, gobj->snd_spit); }
  if (gobj->img_head != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_head); }
  if (gobj->img_dead != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_dead); }
  if (gobj->img_eye != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_eye); }
  if (gobj->img_limb != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_limb); }
  if (gobj->img_tg[IS_MOVING_DIR_UP] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_UP]); }
  if (gobj->img_tg[IS_MOVING_DIR_RIGHT] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_RIGHT]); }
  if (gobj->img_tg[IS_MOVING_DIR_DOWN] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_DOWN]); }
  if (gobj->img_tg[IS_MOVING_DIR_LEFT] != NULL) { VG3_image_unload(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_LEFT]); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 10;
  struct g_main *gmain = vmain;
  struct g_obj_isnake *gobj;
  struct vg3_coll coll;
  int erg, movdir;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_isnake *)objp->ostruct;

  /* destroy esnake, if it has no more limb and snd_hit-sound is over */
  if (gobj->limb_anz == 0) {
    if (!VG3_audio_isplaying(gmain->wstruct, gobj->snd_hit)) { f_free(gmain, objp); }
    return;
  }

  /* eating an enemy snake: no moving */
  if (gobj->eating > 0) { gobj->eating--; return; }

  /* decrement camhat */
  if (gobj->camhat > 0) { gobj->camhat--; }

  /* remove isnake-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* check for keys-strokes */
  gobj->xdelta = gobj->ydelta = 0;
  movdir = gobj->movdir;
  if (key_ispressed(gmain, KEYDEF_GO_UP, VGAG3_IS_PRESSED)) {
    gobj->ydelta = -100;
    gobj->movdir = IS_MOVING_DIR_UP;
  }
  if (key_ispressed(gmain, KEYDEF_GO_DOWN, VGAG3_IS_PRESSED)) {
    gobj->ydelta = 100;
    gobj->movdir = IS_MOVING_DIR_DOWN;
  }
  if (key_ispressed(gmain, KEYDEF_GO_RIGHT, VGAG3_IS_PRESSED)) {
    gobj->xdelta = 100;
    gobj->movdir = IS_MOVING_DIR_RIGHT;
  }
  if (key_ispressed(gmain, KEYDEF_GO_LEFT, VGAG3_IS_PRESSED)) {
    gobj->xdelta = -100;
    gobj->movdir = IS_MOVING_DIR_LEFT;
  }
  if (gobj->xdelta != 0 && gobj->ydelta != 0) { gobj->xdelta = gobj->ydelta = 0; gobj->movdir = movdir; }

  if (key_ispressed(gmain, KEYDEF_FIRE, VGAG3_IS_NEW_PRESSED)) {
    if (!gobj->poison_shot) {
      const struct vg3_ofunc_objfunc *ofc;
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_SHOT));
      if (ofc != NULL) {
        int shot_x, shot_y, angle;
        shot_x = gobj->rect.x + gobj->rect.w / 2;
        shot_y = gobj->rect.y + gobj->rect.h / 2;
        angle = 0;
        if (gobj->movdir == IS_MOVING_DIR_RIGHT) { angle = 90; }
        else if (gobj->movdir == IS_MOVING_DIR_DOWN) { angle = 180; }
        else if (gobj->movdir == IS_MOVING_DIR_LEFT) { angle = 270; }
        if (ofc->f_new(gmain, objp->instanceid, shot_x, shot_y, angle, 0) != NULL) {
          gobj->poison_shot = 1;
          if (gobj->snd_spit != 0) { VG3_audio_play(gmain->wstruct, gobj->snd_spit, 0, 0); }
        }
      }
    }
  }

  /* move isnake-instance and check for collisions with gobj->crect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->crect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: do nothing */
    fprintf(stderr, "moving isnake: %s\n", VG3_error());
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }
  /* VGAG3_COLL_RETURN_CATCHED: ignore */

  /* x/y-value of gobj->crect could be modified, correct it also in gobj->rect */
  gobj->rect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w, gobj->rect.h);

  /* move limbs */
  if (gobj->rect.x + gobj->rect.w / 2 != gobj->limbstream[0].x || gobj->rect.y + gobj->rect.h / 2 != gobj->limbstream[0].y) {
    memmove(&gobj->limbstream[1], &gobj->limbstream[0], sizeof(*gobj->limbstream) * (gobj->limbstream_size - 1));
    gobj->limbstream[0].x = gobj->rect.x + gobj->rect.w / 2;
    gobj->limbstream[0].y = gobj->rect.y + gobj->rect.h / 2;
    gobj->limbstream[0].movdir = gobj->movdir;
  }

  /* actualize health value */
  gmain->isnake_health = gobj->limb_anz * 100 / ISNAKE_LIMB_MAX;
  if (gobj->limb_anz == 0) { return; }  /* isnake dead */

  /* insert isnake-instance (just head) into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_ISNAKE));
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_isnake *gobj;
  int lidx, isize;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_isnake *)objp->ostruct;

  if (gobj->camhat == 0 || (gobj->camhat < 100 && gobj->camhat % 20 < 5)) {
    /* draw limbs */
    for (lidx = gobj->limb_anz - 1; lidx >= 0; lidx--) {
      VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
      iattr.rotate = gobj->limbstream[gobj->limbidx[lidx]].movdir * 90;
      VG3_image_copy(gmain->wstruct, NULL, gobj->img_limb, gobj->limbstream[gobj->limbidx[lidx]].x, gobj->limbstream[gobj->limbidx[lidx]].y, &iattr, 0);
    }
  }

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
  iattr.rotate = gobj->movdir * 90;
  if (gobj->limb_anz == 0) {  /* draw dead-head */
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_dead, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
  } else if (gobj->camhat == 0) {  /* draw head */
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_head, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
  } else {  /* draw eyes */
    VG3_image_copy(gmain->wstruct, NULL, gobj->img_eye, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
  }

  if (gobj->camhat == 0) {
    /* draw tongue */
    if (ZUFALL(1, 12) == 1) {
      if (gobj->movdir == IS_MOVING_DIR_UP) {
        VG3_image_getsize(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_UP], NULL, NULL, &isize);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_tg[IS_MOVING_DIR_UP], gobj->rect.x + gobj->rect.w / 2, gobj->rect.y - isize / 2, NULL, 0);
      } else if (gobj->movdir == IS_MOVING_DIR_RIGHT) {
        VG3_image_getsize(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_RIGHT], NULL, &isize, NULL);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_tg[IS_MOVING_DIR_RIGHT], gobj->rect.x + gobj->rect.w + isize / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
      } else if (gobj->movdir == IS_MOVING_DIR_DOWN) {
        VG3_image_getsize(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_DOWN], NULL, NULL, &isize);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_tg[IS_MOVING_DIR_DOWN], gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h + isize / 2, NULL, 0);
      } else if (gobj->movdir == IS_MOVING_DIR_LEFT) {
        VG3_image_getsize(gmain->wstruct, gobj->img_tg[IS_MOVING_DIR_LEFT], NULL, &isize, NULL);
        VG3_image_copy(gmain->wstruct, NULL, gobj->img_tg[IS_MOVING_DIR_LEFT], gobj->rect.x - isize / 2, gobj->rect.y + gobj->rect.h / 2, NULL, 0);
      }
    }
  }
}


/* data-exchange-function */
static int
f_data(void *vmain, struct vg3_ofunc_object *objp, void *flag)
{
  struct g_main *gmain = vmain;
  struct g_obj_isnake *gobj;
  int ipos;

  if (gmain == NULL || objp == NULL) { return 0; }

  gobj = (struct g_obj_isnake *)objp->ostruct;

  if (flag != NULL) {
    /* set rectangle and collsion rectangle with 80% size of head-image */
    VG3_image_getsize(gmain->wstruct, gobj->img_head, NULL, &gobj->rect.w, &gobj->rect.h);
    gobj->rect.x = gmain->grect.x + 2;
    gobj->rect.y = (gmain->winh - gobj->rect.h) / 2;
    gobj->movdir = IS_MOVING_DIR_RIGHT;
    gobj->camhat = 0;
    gobj->eating = 0;
    gobj->crect = gobj->rect;
    gobj->crect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w * 80 / 100, gobj->rect.h * 80 / 100);

    for (ipos = 0; ipos < gobj->limbstream_size; ipos++) {
      gobj->limbstream[ipos].x = gobj->rect.x + gobj->rect.w / 2;
      gobj->limbstream[ipos].y = gobj->rect.y + gobj->rect.h / 2;
      gobj->limbstream[ipos].movdir = gobj->movdir;
    }

  } else {
    /* return actual number of limbs */
    return gobj->limb_anz;
  }

  return 0;
}
