#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-shot.h"

void getofc_shot(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_shot(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_SHOT));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-functions +++ */

/* new-function; x and y are the position of the middle of the image
 * variable parameter: int xm = x-position (middle)
 *                     int ym = y-position (middle)
 *                     int angle = angle in degrees (0, 90, 180, 270)
 *                     int shotimg = 0=ishot, 1=eshot1, 2=eshot2
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_shot *gobj;
  struct vg3_image *img;
  int xm, ym, angle, shotimg, poison_strength;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments: xm:int, ym:int, angle:int, shotimg:int */
  va_start(ap, iparent);
  xm = va_arg(ap, int);
  ym = va_arg(ap, int);
  angle = va_arg(ap, int);
  shotimg = va_arg(ap, int);
  va_end(ap);

  /* load image */
  if (shotimg == 0) {
    img = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/ishot.bmp", 1);
    poison_strength = 1;
  } else if (shotimg == 1) {
    img = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/eshot1.bmp", 1);
    poison_strength = 1;
  } else if (shotimg == 2) {
    img = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/eshot2.bmp", 1);
    poison_strength = 2;
  } else {
    img = NULL;
    VG3_seterror(EINVAL, strerror(EINVAL));
  }
  if (img == NULL) { return NULL; }

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  gobj->img = img;
  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&gobj->iattr);
  gobj->iattr.rotate = VG3_get_xy_direction_from_angle(angle, &gobj->xdelta, &gobj->ydelta);
  gobj->rect = VG3_image_copy(gmain->wstruct, NULL, gobj->img, xm, ym, &gobj->iattr, 1);
  gobj->iparent = iparent;
  gobj->poison_strength = poison_strength;

  /* create and fill shot-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_SHOT));
  objp->drawlevel = 4;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert shot-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_shot *)objp->ostruct;

  /* inform parent object-instance */
  if (gobj->iparent > 0) {
    struct vg3_ofunc_object *oparent = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gobj->iparent);
    if (oparent != NULL) {
      const struct vg3_ofunc_objobjfunc *oofc;
      oofc = VG3_ofunc_get_objobjfunc(gmain->ofstruct, objp->oid, oparent->oid);
      if (oofc != NULL && oofc->f_quit != NULL) {
        oofc->f_quit(gmain, objp, oparent);
      }
    }
  }

  /* remove shot-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove shot-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free shot-instance */
  VG3_image_unload(gmain->wstruct, gobj->img);
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 40;
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj;
  struct vg3_coll coll;
  int erg;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_shot *)objp->ostruct;

  /* remove shot-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* move shot-instance and check for collisions */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->rect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: destroy shot-instance */
    f_free(gmain, objp);
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }
  if (erg == VGAG3_COLL_RETURN_CATCHED) {  /* destroy shot-instance */
    f_free(gmain, objp);
    return;
  }

  if (gobj->fliploop == 0) {  /* flip horizontal */
    gobj->iattr.flip = !gobj->iattr.flip;
    gobj->fliploop = 5;
  }
  gobj->fliploop--;

  /* insert shot-instance into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->rect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", get_oid_name(OID_NAME_SHOT));
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_shot *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_shot *)objp->ostruct;

  /* draw shot-instance */
  VG3_image_copy(gmain->wstruct, NULL, gobj->img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &gobj->iattr, 0);
}
