#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

const char * get_oid_name(int);

int gameskel_intro(struct vg3_window *);
void gameskel_setkeys(void *, struct vg3_sysmenu_submenu *, struct vg3_sysmenu_submenu *);
void gameskel_beforelevels(void *, struct menu_helper *);
int gameskel_levelstart(void *);
int gameskel_leveldone(void *);
void gameskel_levelstop(void *);
void gameskel_afterlevels(void *, struct menu_helper *);
void gameskel_settingsread(void *, FILE *);
void gameskel_settingssave(void *, FILE *);
void gameskel_settingsreset(void *);


/* get_oid_name:
 * returns object-ID as a string
 * @param oidn  object-ID (OID_NAMES)
 * @return      object-ID as string (or empty)
 */
const char *
get_oid_name(int oidn)
{
  switch(oidn) {
    case OID_NAME_BG:              return "background";
    case OID_NAME_TEXT:            return "text";
    case OID_NAME_PLAYER:          return "player";
    case OID_NAME_PLAYERSHOT:      return "playershot";
    case OID_NAME_FIGHTER:         return "fighter";
    case OID_NAME_FIGHTERSHOT:     return "fightershot";
    case OID_NAME_RADAR:           return "radar";
    case OID_NAME_MINE:            return "mine";
    case OID_NAME_TANKER:          return "tanker";
    case OID_NAME_BOMBER:          return "bomber";
    case OID_NAME_ESCORT:          return "escort";
    case OID_NAME_CARGOSHIP:       return "cargoship";
    case OID_NAME_FACTORY:         return "factory";
    case OID_NAME_ASTEROID:        return "asteroid";
    case OID_NAME_BATTLESHIP:      return "battleship";
    case OID_NAME_MOTHERSHIP:      return "mothership";
    case OID_NAME_MOTHERSHOT:      return "mothershot";
  }
  return "";
}


/* gameskel_intro:
 * play intro(-film) before main-menu
 * (once called)
 * @param wstruct  window struct
 * @return         0 = OK or 1 = exit
 */
int
gameskel_intro(struct vg3_window *wstruct)
{
  int erg;

  if (wstruct == NULL) { return 0; }

  do {
    erg = VG3_film_play(wstruct, NULL, FILES_DIR "/intro/intro.film", 1);
  } while (erg == 0);

  if (erg != 1) { erg = 0; }
  return erg;
}


/* gameskel_setkeys:
 * install keys
 * (called before showing the main-menu)
 * @param vmain      main game struct (struct g_main)
 * @param subm_keyb  submenu for keyboard keys, or NULL = no keys
 * @param subm_gc    submenu for gamecontroller/joystick keys, or NULL = no keys
 */
void
gameskel_setkeys(void *vmain, struct vg3_sysmenu_submenu *subm_keyb, struct vg3_sysmenu_submenu *subm_gc)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL || gmain->skeys == NULL) { return; }

  /* load multilanguage for game */
  if (gmain->mlang != NULL) {
    VG3_multilang_add(gmain->mlang, FILES_DIR "/mlang/game");
  }

  /* set keys */

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_GO_FORWARD, "fwd", "go-fwd",
                       subm_keyb, VGAG3_KEY_RCURS,
                       subm_gc, VGAG3_GC_AXIS_RIGHTX_RIGHT, "Axis-1:pos");

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_GO_BACKWARD, "bwd", "go-bwd",
                       subm_keyb, VGAG3_KEY_LCURS,
                       subm_gc, VGAG3_GC_AXIS_RIGHTX_LEFT, "Axis-1:neg");

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_GO_UP, "up", "go-up",
                       subm_keyb, VGAG3_KEY_UCURS,
                       subm_gc, VGAG3_GC_AXIS_LEFTY_UP, "Axis-2:neg");

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_GO_DOWN, "down", "go-down",
                       subm_keyb, VGAG3_KEY_DCURS,
                       subm_gc, VGAG3_GC_AXIS_LEFTY_DOWN, "Axis-2:pos");

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_FIRE_LASER, "fire-laser", "fire-laser",
                       subm_keyb, VGAG3_KEY_X,
                       subm_gc, VGAG3_GC_BUTTON_RIGHTSHOULDER, "Button-1");

  VG3_keys_menu_insert(gmain->skeys, gmain->sysm_string, gmain->mlang,
                       KEYDEF_FIRE_TORPEDO, "fire-torpedo", "fire-torpedo",
                       subm_keyb, VGAG3_KEY_Y,
                       subm_gc, VGAG3_GC_BUTTON_LEFTSHOULDER, "Button-2");
}


/* gameskel_beforelevels:
 * do anything before first level of game starts,
 * set element "exit" of main game struct with a value of EXITDEFS
 * (called just before entering the game-level loop)
 * @param vmain  main game struct (struct g_main)
 * @param mhelp  menu_helper
 *
 * EXITDEFS:
 * - EXITDEF_NOOP:           OK, go on
 * - EXITDEF_GOTO_MAINMENU:  quit and go to main-menu
 * - EXITDEF_GOTO_INTRO:     quit and go to playing intro
 * - EXITDEF_EXIT_OK:        exit game with no error
 * - EXITDEF_EXIT_FAILURE:   exit game with error
 */
void
gameskel_beforelevels(void *vmain, struct menu_helper *mhelp)
{
  struct g_main *gmain = (struct g_main *)vmain;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_ofunc_object *objp;

  if (gmain == NULL || mhelp == NULL) {
    fprintf(stderr, "gameskel_beforelevels: %s\n", strerror(EINVAL));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }

  /* create player-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_PLAYER));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return;
  }
  objp = ofc->f_new(gmain, 0);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return; }
  gmain->game.instanceid.player = objp->instanceid;

  gmain->exit = EXITDEF_NOOP;
}


/* gameskel_levelstart:
 * initialize for actual gamelevel,
 * set element "exit" of main game struct with a value of EXITDEFS
 * (called for each level just before entering the game loop)
 * @param vmain  main game struct (struct g_main)
 * @return       0 = no more level
 *               1 = ok and "exit" set
 *
 * EXITDEFS:
 * - EXITDEF_NOOP:           OK, go on
 * - EXITDEF_GOTO_MAINMENU:  quit loop and go to main-menu
 * - EXITDEF_GOTO_INTRO:     quit and go to playing intro
 * - EXITDEF_EXIT_OK:        exit game with no error
 * - EXITDEF_EXIT_FAILURE:   exit game with error
 */
int
gameskel_levelstart(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_ofunc_object *objp;
  int erg;

  if (gmain == NULL || gmain->gamelevel < 1) {
    fprintf(stderr, "gameskel_levelstart: %s\n", strerror(EINVAL));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return 1;
  }

  /* check for current gamelevel */
  erg = level_act(vmain, LEVELACT_TEST);
  if (erg < 0) {
    fprintf(stderr, "%s\n", VG3_error());
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return 1;
  }
  if (erg == 0) {  /* no more levels */
    gmain->exit = EXITDEF_GOTO_INTRO;   /* EXITDEF_EXIT_OK or EXITDEF_GOTO_MAINMENU or EXITDEF_GOTO_INTRO */
    return 0;
  }

  /* reinitialize */
  gmain->game.level_retval = 0;
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
  if (ofc != NULL && ofc->f_data != NULL) {
    struct fdata_number fdnumber;
    memset(&fdnumber, 0, sizeof(fdnumber));
    fdnumber.flag = FDATA_REINIT;
    objp = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gmain->game.instanceid.player);
    if (objp != NULL) { ofc->f_data(gmain, objp, &fdnumber); }
  }

  /* show sequence between levels */
  if (show_zwseq(gmain)) { return 1; }

  /* act according to gamelevel */
  erg = level_act(vmain, LEVELACT_START);
  if (erg < 0) {
    fprintf(stderr, "%s\n", VG3_error());
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return 1;
  }

  /* create bg-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_BG));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_BG));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return 1;
  }
  objp = ofc->f_new(gmain, 0);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return 1; }

  /* create text-instance */
  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_TEXT));
  if (ofc == NULL) {
    fprintf(stderr, "Object \"%s\" not found\n", get_oid_name(OID_NAME_TEXT));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return 1;
  }
  objp = ofc->f_new(gmain, 0);
  if (objp == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain->exit = EXITDEF_EXIT_FAILURE; return 1; }
  gmain->game.instanceid.text = objp->instanceid;

  gmain->exit = EXITDEF_NOOP;
  return 1;
}


/* gameskel_leveldone:
 * check whether quit game-loop,
 * may evaluate element "exit" of main game struct,
 * set eventually element "exit" of main game struct with a value of EXITDEFS
 * (called repeatedly in the game loop)
 * @param vmain  main game struct (struct g_main)
 * @return       0 = OK
 *               1 = quit game-loop and set EXITDEFS
 *
 * EXITDEFS (on return = 1):
 * - EXITDEF_NOOP:           OK, go to next level
 * - EXITDEF_REDO_LEVEL:     restart actual game-level
 * - EXITDEF_GOTO_MAINMENU:  quit loop and go to main-menu
 * - EXITDEF_GOTO_INTRO:     quit and go to playing intro
 * - EXITDEF_EXIT_OK:        exit game with no error
 * - EXITDEF_EXIT_FAILURE:   exit game with error
 */
int
gameskel_leveldone(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL || gmain->gamelevel < 1) {
    fprintf(stderr, "gameskel_leveldone: %s\n", strerror(EINVAL));
    gmain->exit = EXITDEF_EXIT_FAILURE;
    return 1;
  }

  /* quit if exit-request was set */
  if (gmain->exit != EXITDEF_NOOP) { return 1; }

  /* check for level-done according to current level */
  if (gmain->game.level_retval == 0) {
    if (level_act(vmain, LEVELACT_DONE) <= 0) {
      fprintf(stderr, "%s\n", VG3_error());
      gmain->exit = EXITDEF_EXIT_FAILURE;
      return 1;
    }
  }

  /* quit if level ended */
  if (gmain->game.level_retval != 0) {
    if (gmain->game.level_retval > 0) {
      level_ending(vmain);
      if (gmain->game.level_retval == 1) {
        gmain->exit = EXITDEF_NOOP;  /* go to next level */
      } else {
        gmain->exit = EXITDEF_REDO_LEVEL;  /* restart actual game-level */
      }
    } else {
      gmain->exit = EXITDEF_GOTO_MAINMENU;
    }
    return 1;
  }

  return 0;
}


/* gameskel_levelstop:
 * destruct for actual gamelevel,
 * element "exit" of main game struct contains the exit-value (of EXITDEFS) from gameskel_leveldone()
 * (called for each level just after quitting the game loop)
 * @param vmain  main game struct (struct g_main)
 */
void
gameskel_levelstop(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL || gmain->gamelevel < 1) { return; }

  /* destroy text-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_TEXT));
  gmain->game.instanceid.text = 0;

  /* destroy bg-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_BG));

  /* act according to gamelevel */
  if (level_act(vmain, LEVELACT_STOP) <= 0) { return; }

  /* transfer ratios to save-values */
  if (gmain->exit == EXITDEF_NOOP || gmain->exit == EXITDEF_EXIT_OK || gmain->exit == EXITDEF_GOTO_MAINMENU || gmain->exit == EXITDEF_GOTO_INTRO) {
    gmain->game.hitratio_save += gmain->game.hitratio;
    gmain->game.hitdiv_save += gmain->game.hitdiv;
    gmain->game.hitratio  = gmain->game.hitdiv = 0;
  }
}


/* gameskel_afterlevels:
 * do anything and clean up rest of game-variables,
 * element "exit" of main game struct contains the exit-value (of EXITDEFS) from gameskel_leveldone()
 * (called just after quitting the game-level loop, and the window has not been cleared)
 * @param vmain   main game struct (struct g_main)
 * @param mhelp   menu_helper
 */
void
gameskel_afterlevels(void *vmain, struct menu_helper *mhelp)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL) { return; }

  /* do ending according to game.level_retval */
  if (gmain->game.level_retval == 1) {
    /* game completed: show fighter ratio */
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    if (gmain->game.hitdiv_save > 0) {  /* print hit-ratio */
      char buf[128];
      int i1;
      snprintf(buf, sizeof(buf), "%s: %d%%", VG3_multilang_get(gmain->mlang, "fighter_ratio"), gmain->game.hitratio_save / gmain->game.hitdiv_save);
      VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, gmain->winh / 2, buf, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 1);
      for (i1 = 0; i1 < 30; i1++) {
        VG3_window_update(gmain->wstruct, 0, 0);
        VG3_wait_time(100);
      }
    }
    if (gmain->game.hitdiv_save > 0) {  /* insert into top ten */
      topten_insert(gmain, gmain->game.hitratio_save / gmain->game.hitdiv_save);
    }
    VG3_film_play(gmain->wstruct, NULL, FILES_DIR "/extro/extro.film", 0);
  } else if (gmain->game.level_retval == -1) {
    /* game failed: fade window out */
    fade_out(gmain->wstruct, 100, 0);
  }

  /* destroy player-instance */
  VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_PLAYER));
  gmain->game.instanceid.player = 0;

  (void)mhelp;
}


/* gameskel_settingsread:
 * read individual settings from opened settings-file
 * (called by settings_readfromfile())
 * @param vmain   main game struct (struct g_main)
 * @param ffp     opened settings-file
 */
void
gameskel_settingsread(void *vmain, FILE *ffp)
{
  struct g_main *gmain = (struct g_main *)vmain;
  char buf[128];

  if (gmain == NULL) { return; }

  if (fgets(buf, sizeof(buf), ffp) == NULL) { return; }
  gmain->game.hitratio_save = atoi(buf);

  if (fgets(buf, sizeof(buf), ffp) == NULL) { return; }
  gmain->game.hitdiv_save = atoi(buf);
}


/* gameskel_settingssave:
 * write individual settings to opened settings-file
 * (called by settings_savetofile())
 * @param vmain   main game struct (struct g_main)
 * @param ffp     opened settings-file
 */
void
gameskel_settingssave(void *vmain, FILE *ffp)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL) { return; }

  fprintf(ffp, "%d\n", gmain->game.hitratio_save);
  fprintf(ffp, "%d\n", gmain->game.hitdiv_save);
}


/* gameskel_settingsreset:
 * reset individual settings for starting with gamelevel=1
 * (called when selecting "new game")
 * @param vmain   main game struct (struct g_main)
 */
void
gameskel_settingsreset(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;

  if (gmain == NULL) { return; }

  gmain->game.hitratio_save = gmain->game.hitdiv_save = 0;
}
