#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

int level_act(void *, int);

static void set_playertorpedo(struct g_main *, int);
static int get_hitratio(struct g_main *, int *, int *, int *);
static void update_hitratio(struct g_main *);
static int quadtree_longdist(struct g_main *, int);

static int levelact_1(struct g_main *, int);
static int levelact_2(struct g_main *, int);
static int levelact_3(struct g_main *, int);
static int levelact_4(struct g_main *, int);
static int levelact_5(struct g_main *, int);
static int levelact_6(struct g_main *, int);
static int levelact_7(struct g_main *, int);
static int levelact_8(struct g_main *, int);
static int levelact_9(struct g_main *, int);
static int levelact_10(struct g_main *, int);
static int levelact_11(struct g_main *, int);
static int levelact_12(struct g_main *, int);


/* set player-torpedo */
static void
set_playertorpedo(struct g_main *gmain, int strength)
{
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_ofunc_object *objp;

  if (gmain == NULL) { return; }
  if (strength < 0) { strength = 0; }

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_PLAYER));
  if (ofc != NULL && ofc->f_data != NULL) {
    struct fdata_number fdnumber;
    memset(&fdnumber, 0, sizeof(fdnumber));
    fdnumber.flag = FDATA_TORPEDO;
    fdnumber.number1 = strength;
    objp = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gmain->game.instanceid.player);
    if (objp != NULL) { ofc->f_data(gmain, objp, &fdnumber); }
  }
}


/* get fighter hit-ratio */
static int
get_hitratio(struct g_main *gmain, int *hitratio_green, int *hitratio_yellow, int *hitratio_red)
{
  struct fdata_number fdnumber;
  int hitdiv;

  if (gmain == NULL || hitratio_green == NULL || hitratio_yellow == NULL || hitratio_red == NULL) { return 0; }

  hitdiv = 0;

  memset(&fdnumber, 0, sizeof(fdnumber));
  fdnumber.flag = FDATA_RATIO;

  fdnumber.number1 = FIGHTER_GREEN;
  if (VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber)) {
    if (fdnumber.number1 > 0) { *hitratio_green = (fdnumber.number2 * 100 / fdnumber.number1); hitdiv++; }
  }

  fdnumber.number1 = FIGHTER_YELLOW;
  if (VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber)) {
    if (fdnumber.number1 > 0) { *hitratio_yellow = (fdnumber.number2 * 100 / fdnumber.number1); hitdiv++; }
  }

  fdnumber.number1 = FIGHTER_RED;
  if (VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber)) {
    if (fdnumber.number1 > 0) { *hitratio_red = (fdnumber.number2 * 100 / fdnumber.number1); hitdiv++; }
  }

  return hitdiv;
}


/* update fighter hit-ratio */
static void
update_hitratio(struct g_main *gmain)
{
  int hitdiv, hitratio_green, hitratio_yellow, hitratio_red;

  if (gmain == NULL) { return; }

  hitratio_green = hitratio_yellow = hitratio_red = -1;
  hitdiv = get_hitratio(gmain, &hitratio_green, &hitratio_yellow, &hitratio_red);

  if (hitdiv > 0) {
    struct vg3_image *img_green, *img_yellow, *img_red;
    struct vg3_sprite *sprt_green, *sprt_yellow, *sprt_red;
    char buf[128];
    int hitratio = 0, snd, inr, iloop, ypos, yadd;

    if (hitratio_green >= 0) { hitratio += hitratio_green; }
    if (hitratio_yellow >= 0) { hitratio += hitratio_yellow; }
    if (hitratio_red >= 0) { hitratio += hitratio_red; }
    gmain->game.hitratio += hitratio;
    gmain->game.hitdiv += hitdiv;

    /* get first sprite-image of all fighters */
    img_green = img_yellow = img_red = NULL;
    sprt_green = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/fighter1.sprite");
    if (sprt_green != NULL) {
      while (img_green == NULL) {
        if (!VG3_sprite_get(sprt_green, &img_green, NULL, &inr)) { break; }
      }
    }
    sprt_yellow = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/fighter2.sprite");
    if (sprt_yellow != NULL) {
      while (img_yellow == NULL) {
        if (!VG3_sprite_get(sprt_yellow, &img_yellow, NULL, &inr)) { break; }
      }
    }
    sprt_red = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/fighter3.sprite");
    if (sprt_red != NULL) {
      while (img_red == NULL) {
        if (!VG3_sprite_get(sprt_red, &img_red, NULL, &inr)) { break; }
      }
    }
    snd = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/ratio.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);

    ypos = 40; yadd = 25;
    for (inr = 1; inr <= 7; inr++) {
      if (inr <= 3 && snd > 0) { VG3_audio_play(gmain->wstruct, snd, 0, 0); }
      for (iloop = 10; iloop > 0; iloop--) {
        VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
        if (inr >= 1 && img_green != NULL) {
          VG3_image_copy(gmain->wstruct, NULL, img_green, 120, ypos + yadd * 1, NULL, 0);
          if (hitratio_green >= 0) {
            snprintf(buf, sizeof(buf), " - %3d%%", hitratio_green);
          } else {
            snprintf(buf, sizeof(buf), " -  [?]");
          }
          VG3_text_simpledraw(gmain->wstruct, NULL, NULL, 170, ypos + yadd * 1, buf, VGAG3_COLOR_GREEN, VGAG3_COLOR_TRANSPARENT, 1);
        }
        if (inr >= 2 && img_yellow != NULL) {
          VG3_image_copy(gmain->wstruct, NULL, img_yellow, 120, ypos + yadd * 2, NULL, 0);
          if (hitratio_yellow >= 0) {
            snprintf(buf, sizeof(buf), " - %3d%%", hitratio_yellow);
          } else {
            snprintf(buf, sizeof(buf), " -  [?]");
          }
          VG3_text_simpledraw(gmain->wstruct, NULL, NULL, 170, ypos + yadd * 2, buf, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 1);
        }
        if (inr >= 3 && img_red != NULL) {
          VG3_image_copy(gmain->wstruct, NULL, img_red, 120, ypos + yadd * 3, NULL, 0);
          if (hitratio_red >= 0) {
            snprintf(buf, sizeof(buf), " - %3d%%", hitratio_red);
          } else {
            snprintf(buf, sizeof(buf), " -  [?]");
          }
          VG3_text_simpledraw(gmain->wstruct, NULL, NULL, 170, ypos + yadd * 3, buf, VGAG3_COLOR_RED, VGAG3_COLOR_TRANSPARENT, 1);
        }
        if (inr >= 4) {
          int ratio, div;
          ratio = gmain->game.hitratio_save + gmain->game.hitratio;
          div = gmain->game.hitdiv_save + gmain->game.hitdiv;
          if (div > 0) {
            snprintf(buf, sizeof(buf), "%s: %d%%", VG3_multilang_get(gmain->mlang, "fighter_ratio"), ratio / div);
            VG3_text_simpledraw(gmain->wstruct, NULL, NULL, gmain->winw / 2, ypos + yadd * 4, buf, VGAG3_COLOR_WHITE, VGAG3_COLOR_TRANSPARENT, 1);
          }
        }
        VG3_window_update(gmain->wstruct, 0, 0);
        VG3_wait_time(100);
      }
    }
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    VG3_window_update(gmain->wstruct, 0, 0);

    if (snd > 0) { VG3_audio_unload(gmain->wstruct, snd); }
    if (sprt_green != NULL) { VG3_sprite_free(sprt_green); }
    if (sprt_yellow != NULL) { VG3_sprite_free(sprt_yellow); }
    if (sprt_red != NULL) { VG3_sprite_free(sprt_red); }
  }
}


/* create or destroy long-distance quadtree */
static int
quadtree_longdist(struct g_main *gmain, int flag)
{
  if (gmain == NULL) { return 0; }

  if (flag == 1) {  /* create quadtree for long-distance collision */
    struct vg3_rect rect;
    rect.x = rect.y = 0;
    rect.w = gmain->winw; rect.h = gmain->winh;
    gmain->game.qdtr_long = VG3_coll_q_new(&rect, 0, 0);
    if (gmain->game.qdtr_long == NULL) { return -1; }
    VG3_coll_q_tag(gmain->game.qdtr_long, 1);  /* tag quadtree with 1 */

  } else if (flag == 0) {  /* destroy long-distance quadtree */
    if (gmain->game.qdtr_long != NULL) {
      VG3_coll_q_free(gmain->game.qdtr_long);
      gmain->game.qdtr_long = NULL;
    }
  }

  return 0;
}


/* actions for levels
 * @param vmain   main game struct (struct g_main)
 * @param action  one of LEVELACT_*
 * @return        - LEVELACT_TEST: 1 = OK, 0 = no more levels, -1 = error
 *                - else: 1 = OK, -1 = error
 */
int
level_act(void *vmain, int action)
{
  struct g_main *gmain = (struct g_main *)vmain;
  char buf[128];
  int retw;

  if (gmain == NULL || gmain->gamelevel < 1) {
    snprintf(buf, sizeof(buf), "level_act: %s", strerror(EINVAL));
    VG3_seterror(EINVAL, buf);
    return -1;
  }

  switch(gmain->gamelevel) {
    case 1: retw = levelact_1(gmain, action); break;
    case 2: retw = levelact_2(gmain, action); break;
    case 3: retw = levelact_3(gmain, action); break;
    case 4: retw = levelact_4(gmain, action); break;
    case 5: retw = levelact_5(gmain, action); break;
    case 6: retw = levelact_6(gmain, action); break;
    case 7: retw = levelact_7(gmain, action); break;
    case 8: retw = levelact_8(gmain, action); break;
    case 9: retw = levelact_9(gmain, action); break;
    case 10: retw = levelact_10(gmain, action); break;
    case 11: retw = levelact_11(gmain, action); break;
    case 12: retw = levelact_12(gmain, action); break;
    default: retw = 0;
  }

  if (retw == 0) {
    snprintf(buf, sizeof(buf), "level_act: gamelevel %d not defined", gmain->gamelevel);
    VG3_seterror(EINVAL, buf);
  }

  return retw;
}


/* actions for level 1 */
static int
levelact_1(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 180;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate radar-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_RADAR), 50, 100) != 0) { return -1; }
    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 100, 0, 0, 150, 500, 20) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 0);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-01.mid", 100, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* increase fighters */
      if (gmain->game.loop_counter % 300 == 0) {
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_FREQ;
        fdnumber.number1 = -1;
        fdnumber.number2 = 2 + (gmain->game.loop_counter * 3 / 2 / 300);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
      }
      /* deactivate creating new radars and fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_RADAR), &fdnumber);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    /* wait until all radars, fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_RADAR)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);

    if (objp == NULL) {  /* set level_retval according to whether all instances were destroyed */
      struct fdata_number fdnumber;
      memset(&fdnumber, 0, sizeof(fdnumber));
      fdnumber.flag = FDATA_RATIO;
      if (VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_RADAR), &fdnumber)
          && fdnumber.number1 > fdnumber.number2
         ) {  /* some missed, redo level */
        gmain->game.level_retval = 2;
        show_failed(gmain, "failed-radar");
      } else {  /* all destroyed, go to next level */
        gmain->game.level_retval = 1;
      }
    }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* deactivate radar-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_RADAR));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 2 */
static int
levelact_2(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 120;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 70, 30, 0, 100, 50, 10) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 0);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-02.mid", 40, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* deactivate creating new fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    /* wait until all fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);
    if (objp == NULL) { gmain->game.level_retval = 1; }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 3 */
static int
levelact_3(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 200;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 50, 30, 20, 230, 75, 20) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 0);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-03.mid", 40, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* increase fighters */
      if (gmain->game.loop_counter % 400 == 0) {
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_FREQ;
        fdnumber.number1 = -1;
        fdnumber.number2 = 10 + (gmain->game.loop_counter / 400);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
      }
      /* deactivate creating new fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    /* wait until all fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);
    if (objp == NULL) { gmain->game.level_retval = 1; }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 4 */
static int
levelact_4(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 120;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* create long-distance quadtree */
    if (quadtree_longdist(gmain, 1) < 0) { return -1; }
    /* activate mine-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_MINE), 150, 30) != 0) { return -1; }
    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 80, 20, 0, 150, 100, 20) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 0);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-04a.mid", 40, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* increase fighters */
      if (gmain->game.loop_counter % 300 == 0) {
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_FREQ;
        fdnumber.number1 = -1;
        fdnumber.number2 = 10 + (gmain->game.loop_counter * 2 / 300);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
      }
      /* deactivate creating new mines and fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_MINE), &fdnumber);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    if (gmain->game.loop_counter == 0) { 
      /* wait until all mines, fighters and fightershots are gone */
      osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
      while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
        if (strcmp(objp->oid, get_oid_name(OID_NAME_MINE)) == 0) { break; }
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
      }
      VG3_ofunc_objlist_freelist(osnap);
      if (objp == NULL) { gmain->game.loop_counter = -1; }  /* all gone */
    }

    if (gmain->game.loop_counter == -1) { 
      /* reinit fighter-object-management */
      struct fdata_number fdnumber;
      memset(&fdnumber, 0, sizeof(fdnumber));
      fdnumber.flag = FDATA_REINIT;
      fdnumber.number1 = 0;
      fdnumber.number2 = 0;
      fdnumber.number3 = 0;
      fdnumber.number4 = 100;
      fdnumber.number5 = 80;
      fdnumber.number6 = 150;
      fdnumber.number7 = 20;
      VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);

      /* activate tanker-object-management */
      if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_TANKER), 4) != 0) { return -1; }
      /* set player-torpedo */
      set_playertorpedo(gmain, 3);
      /* unload music */
      if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }
      /* load and play music */
      gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-04b.mid", 30, VGAG3_AUDIO_VOLUME_MUSIC);
      if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

      activate_text(gmain, VG3_multilang_get(gmain->mlang, "tanker_approaching"), 0, 0, 1);
      gmain->game.loop_counter = -2;
    }

    /* deactivate creating new fighters if tankers are through */
    if (gmain->game.loop_counter == -2) { 
      struct fdata_number fdnumber;
      memset(&fdnumber, 0, sizeof(fdnumber));
      fdnumber.flag = FDATA_DONE;
      VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_TANKER), &fdnumber);
      if (fdnumber.number1) {  /* tankers are through */
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        gmain->game.loop_counter = -3;
      } else {
        snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", 100 - (fdnumber.number3 * 100 / fdnumber.number2));
      }
    }

    if (gmain->game.loop_counter == -3) { 
      /* wait until all tankers and fighters are gone */
      osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
      while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
        if (strcmp(objp->oid, get_oid_name(OID_NAME_TANKER)) == 0) { break; }
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
      }
      VG3_ofunc_objlist_freelist(osnap);

      if (objp == NULL) {  /* set level_retval according to whether all instances were destroyed */
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_RATIO;
        if (VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_TANKER), &fdnumber)
            && fdnumber.number1 > fdnumber.number2
           ) {  /* some missed, redo level */
          gmain->game.level_retval = 2;
          show_failed(gmain, "failed-tanker");
        } else {  /* all destroyed, go to next level */
          gmain->game.level_retval = 1;
        }
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
    }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* deactivate mine-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_MINE));
    /* deactivate tanker-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_TANKER));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
    /* destroy long-distance quadtree */
    quadtree_longdist(gmain, 0);
  }

  return 1;
}


/* actions for level 5 */
static int
levelact_5(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 180;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate bomber-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_BOMBER), 100, 120) != 0) { return -1; }
    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 80, 20, 0, 150, 50, 20) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 3);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-05.mid", 60, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* increase fighters */
      if (gmain->game.loop_counter % 300 == 0) {
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_FREQ;
        fdnumber.number1 = -1;
        fdnumber.number2 = 2 + (gmain->game.loop_counter * 3 / 2 / 300);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
      }
      /* deactivate creating new bombers and fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_BOMBER), &fdnumber);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    /* wait until all bombers, fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_BOMBER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);

    if (objp == NULL) {  /* set level_retval according to whether most instances were destroyed */
      struct fdata_number fdnumber;
      memset(&fdnumber, 0, sizeof(fdnumber));
      fdnumber.flag = FDATA_RATIO;
      if (VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_BOMBER), &fdnumber)
          && fdnumber.number2 < fdnumber.number1 * 2 / 3
         ) {  /* too many missed, redo level */
        gmain->game.level_retval = 2;
        show_failed(gmain, "failed-bomber");
      } else {  /* enough destroyed, go to next level */
        gmain->game.level_retval = 1;
      }
    }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* deactivate bomber-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_BOMBER));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 6 */
static int
levelact_6(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 115;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 1, 70, 30, 0, 200, 50, 15) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 0);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-06.mid", 50, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* deactivate creating new fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    /* wait until all fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);
    if (objp == NULL) { gmain->game.level_retval = 1; }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 7 */
static int
levelact_7(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 150;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 1, 70, 30, 0, 180, 200, 20) != 0) { return -1; }

    /* create escort-instance */
    { const struct vg3_ofunc_objfunc *ofc;
      struct vg3_ofunc_object *objp;
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_ESCORT));
      if (ofc == NULL) {
        char berr[128];
        snprintf(berr, sizeof(berr), "Object \"%s\" not found\n", get_oid_name(OID_NAME_ESCORT));
        VG3_seterror(EINVAL, berr);
        return -1;
      }
      objp = ofc->f_new(gmain, 0);
      if (objp == NULL) { return -1; }
    }

    /* set player-torpedo */
    set_playertorpedo(gmain, 0);

    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-07.mid", 50, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* increase fighters */
      if (gmain->game.loop_counter % 300 == 0) {
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_FREQ;
        fdnumber.number1 = -1;
        fdnumber.number2 = 2 + (gmain->game.loop_counter * 3 / 2 / 300);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
      }
      /* deactivate creating new fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    /* wait until all fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);
    if (objp == NULL) { gmain->game.level_retval = 1; }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* destroy escort-instance */
    VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_ESCORT));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 8 */
static int
levelact_8(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 150;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate cargoship-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_CARGOSHIP), 100, 150) != 0) { return -1; }
    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 70, 20, 10, 150, 50, 20) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 3);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-08.mid", 50, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* increase fighters */
      if (gmain->game.loop_counter % 300 == 0) {
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_FREQ;
        fdnumber.number1 = -1;
        fdnumber.number2 = 6 + (gmain->game.loop_counter * 3 / 2 / 300);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
      }
      /* deactivate creating new cargoships and fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_CARGOSHIP), &fdnumber);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    /* wait until all cargoships, fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_CARGOSHIP)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);

    if (objp == NULL) {  /* set level_retval according to whether most instances were destroyed */
      struct fdata_number fdnumber;
      memset(&fdnumber, 0, sizeof(fdnumber));
      fdnumber.flag = FDATA_RATIO;
      if (VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_CARGOSHIP), &fdnumber)
          && fdnumber.number2 < fdnumber.number1 * 2 / 3
         ) {  /* too many missed, redo level */
        gmain->game.level_retval = 2;
        show_failed(gmain, "failed-cargoship");
      } else {  /* enough destroyed, go to next level */
        gmain->game.level_retval = 1;
      }
    }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* deactivate cargoship-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_CARGOSHIP));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 9 */
static int
levelact_9(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 0;  /* 0 = active, -1 = wait for ending */
    gmain->game.loop_max = gmain->game.loop_counter = duration;

    /* activate factory-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FACTORY), 12) != 0) { return -1; }
    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 65, 20, 15, 200, 50, 20) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 5);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-09.mid", 40, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* deactivate creating new fighters if factories are through */
    if (gmain->game.loop_counter == 0) { 
      struct fdata_number fdnumber;
      memset(&fdnumber, 0, sizeof(fdnumber));
      fdnumber.flag = FDATA_DONE;
      VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FACTORY), &fdnumber);
      if (fdnumber.number1) {  /* factories are through */
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        gmain->game.loop_counter = -1;
      } else {
        snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", 100 - (fdnumber.number3 * 100 / fdnumber.number2));
      }
    }

    if (gmain->game.loop_counter == 0) { return 1; }

    /* wait until all factories, fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FACTORY)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);

    if (objp == NULL) {  /* set level_retval according to whether all instances were destroyed */
      struct fdata_number fdnumber;
      memset(&fdnumber, 0, sizeof(fdnumber));
      fdnumber.flag = FDATA_RATIO;
      if (VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FACTORY), &fdnumber)
          && fdnumber.number1 > fdnumber.number2
         ) {  /* some missed, redo level */
        gmain->game.level_retval = 2;
        show_failed(gmain, "failed-factory");
      } else {  /* enough destroyed, go to next level */
        gmain->game.level_retval = 1;
      }
      /* stop music */
      if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
    }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* deactivate factory-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FACTORY));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 10 */
static int
levelact_10(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 150;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate asteroid-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_ASTEROID), 300, 100) != 0) { return -1; }
    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 70, 20, 0, 150, 200, 10) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 0);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-10.mid", 50, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* deactivate creating new asteroids and fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_ASTEROID), &fdnumber);
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
      }
      return 1;
    }

    /* wait until all asteroids, fighters and fightershots are gone */
    osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
    while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
      if (strcmp(objp->oid, get_oid_name(OID_NAME_ASTEROID)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
      if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
    }
    VG3_ofunc_objlist_freelist(osnap);
    if (objp == NULL) { gmain->game.level_retval = 1; }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* deactivate asteroid-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_ASTEROID));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 11 */
static int
levelact_11(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 60;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 70, 30, 0, 250, 30, 15) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 0);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-11.mid", 80, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* deactivate creating new fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
      }
      return 1;
    }

    if (gmain->game.loop_counter == 0) { 
      /* wait until all fighters and fightershots are gone */
      osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
      while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
      }
      VG3_ofunc_objlist_freelist(osnap);
      if (objp == NULL) { gmain->game.loop_counter = -1; }  /* all gone */
    }

    if (gmain->game.loop_counter == -1) { 
      /* activate battleship-object-management */
      if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_BATTLESHIP), 3) != 0) { return -1; }
      /* set player-torpedo */
      set_playertorpedo(gmain, 5);

      activate_text(gmain, VG3_multilang_get(gmain->mlang, "battleship_approaching"), 3, 20, 1);
      gmain->game.loop_counter = -2;
    }

    /* check if battleships are through */
    if (gmain->game.loop_counter == -2) { 
      struct fdata_number fdnumber;
      memset(&fdnumber, 0, sizeof(fdnumber));
      fdnumber.flag = FDATA_DONE;
      VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_BATTLESHIP), &fdnumber);
      if (fdnumber.number1) {  /* battleships are through */
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
        gmain->game.loop_counter = -3;
      }
    }

    if (gmain->game.loop_counter == -3) { 
      /* wait until all battleships are gone */
      osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
      while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
        if (strcmp(objp->oid, get_oid_name(OID_NAME_BATTLESHIP)) == 0) { break; }
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
      }
      VG3_ofunc_objlist_freelist(osnap);
      if (objp == NULL) { gmain->game.level_retval = 1; }
    }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* deactivate battleship-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_BATTLESHIP));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}


/* actions for level 12 */
static int
levelact_12(struct g_main *gmain, int action)
{
  if (action == LEVELACT_START) {
    const int duration = 30;  /* in sec */
    gmain->game.loop_max = gmain->game.loop_counter = duration * 1000 / LOOP_TIME;

    /* activate fighter-object-management */
    if (VG3_ofunc_mgmt_activate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), 0, 0, 0, 100, 250, 50, 15) != 0) { return -1; }
    /* set player-torpedo */
    set_playertorpedo(gmain, 5);
    /* load and play music */
    gmain->game.muskz = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/mus-12.mid", 70, VGAG3_AUDIO_VOLUME_MUSIC);
    if (gmain->game.muskz > 0) { VG3_audio_play(gmain->wstruct, gmain->game.muskz, 2, 0); }

  } else if (action == LEVELACT_DONE) {
    struct vg3_ofunc_object *objp;
    struct vg3_ofunc_objsnap *osnap;

    *gmain->game.btim = '\0';

    /* run while loop_counter is > 0 */
    if (gmain->game.loop_counter > 0) {
      gmain->game.loop_counter--;
      snprintf(gmain->game.btim, sizeof(gmain->game.btim), "[%d%%]", (gmain->game.loop_counter * 100 / gmain->game.loop_max));
      /* deactivate creating new fighters */
      if (gmain->game.loop_counter == 0) { 
        struct fdata_number fdnumber;
        memset(&fdnumber, 0, sizeof(fdnumber));
        fdnumber.flag = FDATA_PAUSE;
        fdnumber.number1 = -1;
        VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
      }
      return 1;
    }

    if (gmain->game.loop_counter == 0) { 
      /* wait until all fighters and fightershots are gone */
      osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
      while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTER)) == 0) { break; }
        if (strcmp(objp->oid, get_oid_name(OID_NAME_FIGHTERSHOT)) == 0) { break; }
      }
      VG3_ofunc_objlist_freelist(osnap);
      if (objp == NULL) { gmain->game.loop_counter = -1; }  /* all gone */
    }

    if (gmain->game.loop_counter == -1) { 
      /* create mothership */
      const struct vg3_ofunc_objfunc *ofc;
      struct vg3_ofunc_object *objp;
      ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_MOTHERSHIP));
      if (ofc == NULL) {
        char berr[128];
        snprintf(berr, sizeof(berr), "Object \"%s\" not found\n", get_oid_name(OID_NAME_MOTHERSHIP));
        VG3_seterror(EINVAL, berr);
        return -1;
      }
      objp = ofc->f_new(gmain, 0);
      if (objp == NULL) { return -1; }

      activate_text(gmain, VG3_multilang_get(gmain->mlang, "mothership_approaching"), 3, 20, 1);
      gmain->game.loop_counter = -2;
    }

    /* check if mothership is destroyed */
    if (gmain->game.loop_counter == -2) { 
      int i1 = VG3_ofunc_objlist_find_obj(gmain->ofstruct, get_oid_name(OID_NAME_MOTHERSHIP), NULL);
      if (i1 == 0) {
        /* stop music */
        if (gmain->game.muskz > 0) { VG3_audio_stop(gmain->wstruct, gmain->game.muskz, 1); }
        gmain->game.loop_counter = -3;
      }
    }

    if (gmain->game.loop_counter == -3) { 
      /* wait until mothershot is gone */
      osnap = VG3_ofunc_objlist_newlist(gmain->ofstruct, NULL);
      while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
        if (strcmp(objp->oid, get_oid_name(OID_NAME_MOTHERSHOT)) == 0) { break; }
      }
      VG3_ofunc_objlist_freelist(osnap);
      if (objp == NULL) { gmain->game.level_retval = 1; }
    }

  } else if (action == LEVELACT_STOP) {
    const char *oidp_unless[] = {
      get_oid_name(OID_NAME_PLAYER),
      NULL
    };

    /* unload music */
    if (gmain->game.muskz > 0) { VG3_audio_unload(gmain->wstruct, gmain->game.muskz); gmain->game.muskz = 0; }

    /* update fighter hit-ratio */
    if (gmain->game.level_retval == 1) { update_hitratio(gmain); }

    /* deactivate fighter-object-management */
    VG3_ofunc_mgmt_deactivate(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER));
    /* destroy mothership-object-instance */
    VG3_ofunc_objlist_call_free(gmain->ofstruct, gmain, get_oid_name(OID_NAME_MOTHERSHIP));
    /* destroy all objects unless OID_NAME_PLAYER */
    VG3_ofunc_objlist_call_free_unless(gmain->ofstruct, gmain, oidp_unless);
  }

  return 1;
}
