/* Copyright 2022 Kurt Nienhaus
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

/* +++ needs not to be changed +++ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <vgagames3.h>
#include "main.h"

extern struct vg3_ofunc * ofunc_new(void);

void fade_out(struct vg3_window *, int, int);

static int hilfe(void *);


/* fade_out: fade out window,
 * if audio-descriptor is given, wait until it ends
 * @param wstruct    window-struct
 * @param loop_time  loop-time in msec
 * @param audkz      audio-descriptor or 0
 */
void
fade_out(struct vg3_window *wstruct, int loop_time, int audkz)
{
  struct vg3_image_attributes wattr;

  if (loop_time < 10) { loop_time = 10; }

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&wattr);
  for (;;) {
    if (audkz > 0) {
      if (!VG3_audio_isplaying(wstruct, audkz)) { break; }
    } else {
      if (wattr.bright == 0) { break; }
    }
    if (wattr.bright < 10) { wattr.bright = 0; } else { wattr.bright -= 10; }
    VG3_window_attributes(wstruct, &wattr, NULL, -1, NULL);
    VG3_window_update(wstruct, 0, 0);
    VG3_wait_time(loop_time);
  }

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&wattr);
  VG3_window_attributes(wstruct, &wattr, NULL, -1, NULL);
}


int main(int argc, char **argv) {
  struct g_main gmain;
  struct vg3_window *wstruct;
  struct vg3_sysmenu_submenu *subm_keyb, *subm_gc;
  struct vg3_rect rect;
  int i1, jid, dowait, use_nw, idone, more_levels;
  struct vg3_image *bgimage;
  struct menu_helper mhelp;
  struct vg3_hash *hmenu;
  const char *hmkey;

  opterr = opterr ? opterr : 1;
  while ((i1 = getopt(argc, argv, "+h")) != -1) {
    switch(i1) {
      case 'h':
      default:  fprintf(stderr, "Usage: %s [<options>]\n", argv[0]);
                fprintf(stderr, "options: none\n");
                exit(1);
    }
  }

  srand(time(NULL));
  bgimage = NULL;
  hmenu = VG3_hash_new();

  /* open window */
  wstruct = VG3_window_new(argv[0], GAMESKEL_VGAVERSION, GAMESKEL_SCALING);
  if (wstruct == NULL) { fprintf(stderr, "%s\n", VG3_error()); exit(1); }

playintro:
  /* play intro */
  memset(&gmain, 0, sizeof(gmain));
  gmain.wstruct = wstruct;
  settings_readfromfile(&gmain, 1);
  if (gameskel_intro(wstruct)) { gmain.exit = EXITDEF_EXIT_OK; goto endgame; }

startgame:
  memset(&gmain, 0, sizeof(gmain));
  gmain.wstruct = wstruct;
  gmain.exit = EXITDEF_EXIT_OK;

  /* initialize menu_helper */
  bgimage = NULL;
  if (GAMESKEL_GAMEMENU_BGFILE != NULL) {
    bgimage = VG3_image_load(gmain.wstruct, GAMESKEL_GAMEMENU_BGFILE, 1);
    if (bgimage == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain.exit = EXITDEF_EXIT_FAILURE; goto endgame; }
  }
  mh_init(&mhelp, FILES_DIR, gmain.wstruct, bgimage, GAMESKEL_GAMEMENU_BGBRIGHT, GAMESKEL_GAMEMENU_MENUFC, GAMESKEL_GAMEMENU_MENUBC, GAMESKEL_GAMEMENU_ONLEFT, GAMESKEL_GAMEMENU_TEXTFC, GAMESKEL_GAMEMENU_TEXTBC);

  /* get the size of the window */
  VG3_window_getsize(gmain.wstruct, &gmain.winw, &gmain.winh);

  /* create multilanguage */
  gmain.mlang = VG3_multilang_new(NULL, "en");
  VG3_multilang_add(gmain.mlang, FILES_DIR "/mlang/main");

  /* create system-menu */
  gmain.sysm = VG3_sysmenu_new(gmain.wstruct, NULL, GAMESKEL_SYSMENU_FCNORMAL, GAMESKEL_SYSMENU_FCSELECT);
  VG3_sysmenu_simple_exitmenu(gmain.sysm, NULL, VG3_multilang_get(gmain.mlang, "exit-game"));
  VG3_sysmenu_simple_volumemenu(gmain.sysm, NULL,
    VG3_multilang_get(gmain.mlang, "main-vol"),
    VG3_multilang_get(gmain.mlang, "sound-vol"),
    VG3_multilang_get(gmain.mlang, "music-vol"),
    NULL
  );
  VG3_sysmenu_simple_windowmenu(gmain.sysm, NULL, VG3_multilang_get(gmain.mlang, "window-change"));
  subm_keyb = VG3_sysmenu_simple_keyboardmenu(gmain.sysm, NULL);
  subm_gc = NULL;
  jid = 0;
  { int *jidf;
    VG3_gamecontroller_getall(gmain.wstruct, &jidf);
    if (jidf[0] > 0) {
      subm_gc = VG3_sysmenu_simple_gcmenu(gmain.sysm, NULL);
      jid = jidf[1];  /* use first found gamecontroller/joystick instead of keyboard */
    }
    free(jidf);
  }
  /* insert help into system-menu */
  { struct vg3_text skey;
    VGAG3_TEXT_ATTRIBUTES_SET(&skey, "10x17.font+", 0, 0, VG3_multilang_get(gmain.mlang, "help"));
    VG3_sysmenu_insert_callentry(gmain.sysm, NULL, &skey, hilfe, &gmain);
  }

  /* read saved values */
  settings_readfromfile(&gmain, 0);
  if (gmain.gamelevel < 1) { gmain.gamelevel = 1; }

  /* install keys */
  gmain.skeys = VG3_keys_new(gmain.wstruct, gmain.sysm);
  gameskel_setkeys(&gmain, subm_keyb, subm_gc);

  /* show main menu */
main_menu:
  VG3_hash_clear(hmenu);
  use_nw = 0;
  i1 = 0;
  mhelp.sel.title = (char *)VG3_multilang_get(gmain.mlang, "main-menu");

  hmkey = "help";
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;

  hmkey = "system-menu";
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;

#ifdef GAMESKEL_TOPTEN
  hmkey = "topten-list";
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;
#endif

#if GAMESKEL_STARTING == 1
  hmkey = "start-game";
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;

#elif GAMESKEL_STARTING == 2
  hmkey = "new-game";
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;
  if (gmain.gamelevel > 1) { hmkey = "last-game"; } else { hmkey = "no-last-game"; }
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;

#elif GAMESKEL_STARTING == 3
  hmkey = "single-game";
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;
  hmkey = "nw-game";
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;
#endif

  hmkey = "exit";
  mhelp.sel.item[i1] = (char *)VG3_multilang_get(gmain.mlang, hmkey);
  VG3_hash_setint(hmenu, hmkey, strlen(hmkey) + 1, i1 + 1);
  i1++;

  mhelp.sel.item[i1] = NULL;

  i1 = mhelp.func.show_menu(&mhelp);
  if (i1 == -2) { goto endgame; }
  if (i1 == -1) { fprintf(stderr, "%s\n", VG3_error()); gmain.exit = EXITDEF_EXIT_FAILURE; goto endgame; }
  if (i1 == 0) { goto main_menu; }

  if (i1 == VG3_hash_getint(hmenu, "help", sizeof("help"))) {  /* help */
    if (hilfe(&gmain)) { goto endgame; }
    goto main_menu;
  } else if (i1 == VG3_hash_getint(hmenu, "system-menu", sizeof("system-menu"))) {  /* system-menu */
    if (VG3_sysmenu_exec(gmain.sysm) > 0) { goto endgame; }
    goto main_menu;
  } else if (i1 == VG3_hash_getint(hmenu, "topten-list", sizeof("topten-list"))) {  /* topten-list */
    topten_show(&gmain, 0);
    goto main_menu;
  } else if (i1 == VG3_hash_getint(hmenu, "start-game", sizeof("start-game"))) {  /* start game */
    gmain.gamelevel = 1;
    use_nw = 0;
  } else if (i1 == VG3_hash_getint(hmenu, "new-game", sizeof("new-game"))) {  /* new game */
    gameskel_settingsreset(&gmain);
    gmain.gamelevel = 1;
    use_nw = 0;
  } else if (i1 == VG3_hash_getint(hmenu, "last-game", sizeof("last-game"))) {  /* last game */
    use_nw = 0;
  } else if (i1 == VG3_hash_getint(hmenu, "single-game", sizeof("single-game"))) {  /* single-user game */
    gmain.gamelevel = 1;
    use_nw = 0;
  } else if (i1 == VG3_hash_getint(hmenu, "nw-game", sizeof("nw-game"))) {  /* network-game */
    gmain.gamelevel = 1;
    use_nw = 1;
  } else if (i1 == VG3_hash_getint(hmenu, "exit", sizeof("exit"))) {  /* exit */
    goto endgame;
  }

  /* using network */
  gmain.clnr = 1;
  if (use_nw) {
    /* select network-server */
    snprintf(mhelp.nw.servername, sizeof(mhelp.nw.servername), "%s", gmain.nwserver);
    i1 = mhelp.func.sel_nwserver(&mhelp);
    if (i1 == -2) { goto endgame; }
    if (i1 == -1) { fprintf(stderr, "%s\n", VG3_error()); gmain.exit = EXITDEF_EXIT_FAILURE; goto endgame; }
    if (i1 == 0) { gmain.exit = EXITDEF_GOTO_MAINMENU; goto endgame; }

    /* connect to network-server */
    gmain.clnr = mhelp.func.open_network(&mhelp, jid, MAX_PLAYERS, gmain.skeys);
    if (gmain.clnr < 0) { fprintf(stderr, "%s\n", VG3_error()); gmain.exit = EXITDEF_EXIT_FAILURE; goto endgame; }
    if (gmain.clnr == 0) { goto endgame; }
    gmain.nwptr = mhelp.nwptr;
    snprintf(gmain.nwserver, sizeof(gmain.nwserver), "%s", mhelp.nw.servername);
    VG3_ofunc_reset_seeds(gmain.ofstruct);
  }

  /* create quadtree with size of window */
  rect.x = rect.y = 0;
  rect.w = gmain.winw; rect.h = gmain.winh;
  gmain.qdtr = VG3_coll_q_new(&rect, 0, 0);
  if (gmain.qdtr == NULL) { fprintf(stderr, "%s\n", VG3_error()); gmain.exit = EXITDEF_EXIT_FAILURE; goto endgame; }

  /* create main-struct for object-functions */
  gmain.ofstruct = ofunc_new();

  /* do anything before first level of game starts,
   * return a value of EXITDEFS in gmain.exit
   */
  gameskel_beforelevels(&gmain, &mhelp);
  switch(gmain.exit) {
    case EXITDEF_GOTO_MAINMENU: goto endgame;
    case EXITDEF_GOTO_INTRO: goto endgame;
    case EXITDEF_EXIT_OK: goto endgame;
    case EXITDEF_EXIT_FAILURE: goto endgame;
    default: break;
  }
  gmain.exit = EXITDEF_NOOP;


  /* +++ game-level loop +++ */

  more_levels = 0;
  for (;;) {
    /* initialize for actual gamelevel,
     * return a value of EXITDEFS in gmain.exit
     */
    more_levels = gameskel_levelstart(&gmain);
    if (!more_levels) {
      if (gmain.exit != EXITDEF_EXIT_OK && gmain.exit != EXITDEF_GOTO_MAINMENU && gmain.exit != EXITDEF_GOTO_INTRO) {
        gmain.exit = EXITDEF_GOTO_MAINMENU;
      }
    }
    switch(gmain.exit) {
      case EXITDEF_GOTO_MAINMENU: goto endloop;
      case EXITDEF_GOTO_INTRO: goto endloop;
      case EXITDEF_EXIT_OK: goto endloop;
      case EXITDEF_EXIT_FAILURE: goto endloop;
      default: break;
    }
    gmain.exit = EXITDEF_NOOP;

    /* +++ game loop +++ */

    idone = 0;
    VG3_discard_input(gmain.wstruct);
    for (;;) {
      if (gmain.nwptr != NULL) {
        if (VG3_nw_update(gmain.nwptr, &dowait)) { gmain.exit = EXITDEF_EXIT_OK; idone = 1; goto endlevel; }
      } else {
        if (VG3_inputevent_update(gmain.wstruct)) { gmain.exit = EXITDEF_EXIT_OK; idone = 1; goto endlevel; }
        dowait = 1;
      }

      /* ALT+Q: exit */
      if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)
          && VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_LALT, VGAG3_IS_PRESSED)) {
        gmain.exit = EXITDEF_EXIT_OK;
        idone = 1;
        goto endlevel;
      }

      /* ESC: system-menu */
      if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) {
        if (VG3_sysmenu_exec(gmain.sysm) > 0) { gmain.exit = EXITDEF_EXIT_OK; idone = 1; goto endlevel; }
      }

      /* P: pause game */
      if (VG3_key_ispressed(gmain.wstruct, VGAG3_KEY_P, VGAG3_IS_NEW_PRESSED)) {
        if (gmain.nwptr != NULL) {
          VG3_nw_pause(gmain.nwptr);
        } else {
          VG3_pause(gmain.wstruct, NULL, VGAG3_KEY_NOKEY);
        }
      }

      VG3_draw_clear(gmain.wstruct, NULL, VGAG3_COLOR_BLACK);

      VG3_ofunc_mgmt_run(gmain.ofstruct, &gmain);
      VG3_ofunc_objlist_call_run(gmain.ofstruct, &gmain);
      VG3_ofunc_objlist_call_draw(gmain.ofstruct, &gmain);

      VG3_window_update(gmain.wstruct, gmain.win_xdelta, gmain.win_ydelta);

      if (dowait) {
#if 0
        int iwt = VG3_wait_time(LOOP_TIME);
        if (iwt < LOOP_TIME / 2) { fprintf(stderr, "Slept-time low: %d msec of %d msec\n", iwt, LOOP_TIME); }
#else
        VG3_wait_time(LOOP_TIME);
#endif
      }

      /* check whether quit loop,
       * if yes, return a value of EXITDEFS in gmain.exit
       */
      if (gameskel_leveldone(&gmain)) { break; }
    }
    VG3_discard_input(gmain.wstruct);

    if (gmain.exit == EXITDEF_EXIT_FAILURE) { goto endloop; }

endlevel:
    /* destruct actual gamelevel, act according to exit-value of gameskel_leveldone() */
    i1 = gmain.exit;
    gameskel_levelstop(&gmain);
    gmain.exit = i1;
    if (idone) { goto endloop; }
    switch(gmain.exit) {
      case EXITDEF_REDO_LEVEL: break;
      case EXITDEF_GOTO_MAINMENU: goto endloop;
      case EXITDEF_GOTO_INTRO: goto endloop;
      case EXITDEF_EXIT_OK: goto endloop;
      case EXITDEF_EXIT_FAILURE: goto endloop;
      default: gmain.gamelevel++; break;
    }
    gmain.exit = EXITDEF_NOOP;
  }

endloop:
  if (gmain.exit != EXITDEF_EXIT_FAILURE) {
    /* do anything and clean up rest of game-variables */
    i1 = gmain.exit;
    gameskel_afterlevels(&gmain, &mhelp);
    gmain.exit = i1;
  }

  /* reset if game through */
  if (!more_levels) {
    gameskel_settingsreset(&gmain);
    gmain.gamelevel = 1;
  }

endgame:
  if (gmain.exit != EXITDEF_EXIT_FAILURE) {
    /* save values to file */
    if (gmain.sysm != NULL) {
      char *sysm_string_new = VG3_sysmenu_savestring_insert(gmain.sysm, gmain.sysm_string);
      if (gmain.sysm_string != NULL) { free(gmain.sysm_string); }
      gmain.sysm_string = sysm_string_new;
      settings_savetofile(&gmain);
    }

    /* reset window */
    { struct vg3_image_attributes wattr;
      VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&wattr);
      VG3_window_attributes(gmain.wstruct, &wattr, NULL, VGAG3_COLORDEF_DEFAULT, NULL);
      VG3_draw_clear(gmain.wstruct, NULL, VGAG3_COLOR_BLACK);
      VG3_window_update(gmain.wstruct, 0, 0);
    }

    /* clean up */
    if (gmain.nwptr != NULL) { VG3_nw_close(gmain.nwptr); }
    if (gmain.sysm_string != NULL) { free(gmain.sysm_string); }
    if (gmain.ofstruct != NULL) {
#if 0
      struct vg3_ofunc_object *objp;
      struct vg3_ofunc_objsnap *osnap;
      osnap = VG3_ofunc_objlist_newlist(gmain.ofstruct, NULL);
      printf("Objekt-Instance-Liste:\n");
      while ((objp = VG3_ofunc_objlist_nextlist(osnap)) != NULL) {
        printf(" - OID=%s, INSTANCE-ID=%u\n", objp->oid, objp->instanceid);
      }
      VG3_ofunc_objlist_freelist(osnap);
#endif
      VG3_ofunc_mgmt_deactivate(gmain.ofstruct, &gmain, NULL);
      VG3_ofunc_objlist_call_free(gmain.ofstruct, &gmain, NULL);
      VG3_ofunc_free(gmain.ofstruct);
    }
    if (gmain.qdtr != NULL) { VG3_coll_q_free(gmain.qdtr); }
    if (gmain.skeys != NULL) { VG3_keys_free(gmain.skeys); }
    if (gmain.sysm != NULL) { VG3_sysmenu_free(gmain.sysm); }
    if (gmain.mlang != NULL) { VG3_multilang_free(gmain.mlang); }
    if (bgimage != NULL) { VG3_image_unload(gmain.wstruct, bgimage); }
#if 0
    VG3_audio_unload(gmain.wstruct, VG3_audio_group(VGAG3_AUDIO_VOLUME_ALL));
    VG3_image_unload(gmain.wstruct, NULL);
#endif

    if (gmain.exit == EXITDEF_GOTO_MAINMENU) { goto startgame; }
    if (gmain.exit == EXITDEF_GOTO_INTRO) { goto playintro; }
  } else {
    if (gmain.nwptr != NULL) { VG3_nw_close(gmain.nwptr); }
  }

  /* close window and exit */
  VG3_hash_free(hmenu);
  VG3_window_free(gmain.wstruct);
  exit(gmain.exit == EXITDEF_EXIT_OK ? 0 : 1);
}


/* help
 * reads FILES_DIR "/help/help-text-<locale>.txt"
 * @return  0 = OK or 1 = exit request
 */
static int
hilfe(void *vmain)
{
  struct g_main *gmain = (struct g_main *)vmain;
  const char *locale;
  int retw, ypos, ysize, fcolor, bcolor;
  struct vg3_image *imgp;
  struct vg3_text stxt;
  char text[8192], path[256], *fontname, *ptr;
  size_t slen;
  FILE *ffp;

  if (gmain == NULL) { return 0; }

  fontname = NULL;
  fcolor = GAMESKEL_HELPMENU_FC;
  bcolor = GAMESKEL_HELPMENU_BC;

  locale = VG3_multilang_locale(gmain->mlang, 0);
  if (*locale == '\0') {
    locale = VG3_multilang_locale(gmain->mlang, 1);
  }

  snprintf(path, sizeof(path), "%s/help/help-text-%s.txt", FILES_DIR, locale);

  if ((ffp = fopen(path, "r")) == NULL) { fprintf(stderr, "%s not found\n", path); return 0; }
  slen = fread(text, 1, sizeof(text) - 1, ffp);
  fclose(ffp);
  if (slen == 0) { return 0; }
  text[slen] = '\0';

  if ((ptr = strrchr(path, '/')) != NULL) { *ptr = '\0'; } else { strcpy(path, "."); }

  VG3_utf8_to_iso(text, text, slen + 1);

  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, fontname, '\n', 0, text);
  imgp = VG3_text_with_images(gmain->wstruct, &stxt, path, fcolor, bcolor);
  if (imgp == NULL) { fprintf(stderr, "%s\n", VG3_error()); return 0; }
  VG3_image_getsize(gmain->wstruct, imgp, NULL, NULL, &ysize);

  retw = 0;
  ypos = (ysize - gmain->winh) / 2;
  VG3_discard_input(gmain->wstruct);
  for (;;) {
    if (VG3_inputevent_update(gmain->wstruct)) { retw = 1; break; }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_Q, VGAG3_IS_NEW_PRESSED)) { break; }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) { break; }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_UCURS, VGAG3_IS_PRESSED)) {
      if (ypos < (ysize - gmain->winh) / 2) {
        ypos += 8;
        if (ypos > (ysize - gmain->winh) / 2) { ypos = (ysize - gmain->winh) / 2; }
      }
    }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_DCURS, VGAG3_IS_PRESSED)) {
      if (ypos > -(ysize - gmain->winh) / 2) {
        ypos -= 8;
        if (ypos < -(ysize - gmain->winh) / 2) { ypos = -(ysize - gmain->winh) / 2; }
      }
    }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_PGUP, VGAG3_IS_NEW_PRESSED)
        || VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_BACKSPACE, VGAG3_IS_NEW_PRESSED)) {
      if (ypos < (ysize - gmain->winh) / 2) {
        ypos += (gmain->winh - 8);
        if (ypos > (ysize - gmain->winh) / 2) { ypos = (ysize - gmain->winh) / 2; }
      }
    }
    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_PGDOWN, VGAG3_IS_NEW_PRESSED)
        || VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) {
      if (ypos > -(ysize - gmain->winh) / 2) {
        ypos -= (gmain->winh - 8);
        if (ypos < -(ysize - gmain->winh) / 2) { ypos = -(ysize - gmain->winh) / 2; }
      }
    }

    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);

    VG3_image_copy(gmain->wstruct, NULL, imgp, gmain->winw / 2, ypos + gmain->winh / 2, NULL, 0);

    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(50);
  }
  VG3_discard_input(gmain->wstruct);

  VG3_image_unload(gmain->wstruct, imgp);
  VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
  VG3_window_update(gmain->wstruct, 0, 0);

  return retw;
}
