/* +++ needs not to be changed +++ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "menu_helper.h"

void mh_init(struct menu_helper *, const char *, struct vg3_window *, struct vg3_image *, int, int, int, int, int, int);

static void draw_bg(struct menu_helper *);
static void print_text(struct menu_helper *, const char *);
static int show_menu(struct menu_helper *);
static int sel_nwserver(struct menu_helper *);
static int open_network(struct menu_helper *, int, int, struct vg3_keys *);
static int recv_nwdata(struct menu_helper *, char *, size_t);
static int sel_imglist(struct menu_helper *, int, int *, int, int);


/* mh_init: initialize menu_helper-struct
 * @param mhelp          menu_helper-struct to initialize
 * @param wstruct        window-struct
 * @param bgimage        background image or NULL
 * @param bgbright       brightness of background (0 - 255), use window-content if bgbright>0 and bgimage=NULL
 * @param menu_fgcolor   foreground-color for menu
 * @param menu_bgcolor   background-color for menu
 * @param menu_onleft    orientation for menu selection elements (0 = centered, 1 = left)
 * @param text_fgcolor   foreground-color for info-text
 * @param text_bgcolor   background-color for info-text
 */
void
mh_init(struct menu_helper *mhelp, const char *files_dir, struct vg3_window *wstruct, struct vg3_image *bgimage, int bgbright, int menu_fgcolor, int menu_bgcolor, int menu_onleft, int text_fgcolor, int text_bgcolor)
{
  if (mhelp == NULL) { return; }

  memset(mhelp, 0, sizeof(*mhelp));
  mhelp->init.files_dir = files_dir;
  mhelp->init.wstruct = wstruct;
  mhelp->init.bgimage = bgimage;
  mhelp->init.bgbright = bgbright;
  mhelp->init.menu.fgcolor = menu_fgcolor;
  mhelp->init.menu.bgcolor = menu_bgcolor;
  mhelp->init.menu.onleft = menu_onleft;
  mhelp->init.text.fgcolor = text_fgcolor;
  mhelp->init.text.bgcolor = text_bgcolor;

  mhelp->func.show_menu = show_menu;
  mhelp->func.sel_nwserver = sel_nwserver;
  mhelp->func.open_network = open_network;
  mhelp->func.recv_nwdata = recv_nwdata;
  mhelp->func.sel_imglist = sel_imglist;
}


/* draw background with bgimage (or actual background if NULL) and brightness = bgbright,
 * just clear background if bgbright is 0
 * [mhelp: init]
 */
static void
draw_bg(struct menu_helper *mhelp)
{
  struct vg3_image *hgclone, *bgimage;

  if (mhelp == NULL || mhelp->init.wstruct == NULL) { return; }

  VG3_draw_clear(mhelp->init.wstruct, NULL, VGAG3_COLOR_BLACK);

  hgclone = NULL;
  bgimage = mhelp->init.bgimage;

  if (mhelp->init.bgbright > 0) {
    if (bgimage == NULL) { hgclone = VG3_image_clone(mhelp->init.wstruct, NULL, NULL, NULL, NULL); }
  } else {
    bgimage = NULL;
  }

  if (bgimage != NULL || hgclone != NULL) {
    int w0, h0;
    struct vg3_image_attributes iattr;

    VG3_window_getsize(mhelp->init.wstruct, &w0, &h0);
    VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
    iattr.bright = mhelp->init.bgbright;
    VG3_image_copy(mhelp->init.wstruct, NULL, (hgclone != NULL ? hgclone: bgimage), w0 / 2, h0 / 2, &iattr, 0);

    if (hgclone != NULL) { VG3_image_unload(mhelp->init.wstruct, hgclone); }
  }

  VG3_window_update(mhelp->init.wstruct, 0, 0);
}


/* output text centered onto the window with background
 * [mhelp: init]
 */
static void
print_text(struct menu_helper *mhelp, const char *text)
{
  if (mhelp == NULL || mhelp->init.wstruct == NULL) { return; }

  draw_bg(mhelp);
  if (text != NULL && *text != '\0') {
    struct vg3_rect rect;
    struct vg3_text stxt;
    int w0, h0;
    VG3_window_getsize(mhelp->init.wstruct, &w0, &h0);
    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, "8x8.font+", '\n', 0, text);
    rect.x = rect.y = 0; rect.w = w0; rect.h = h0;
    rect = VG3_draw_text(mhelp->init.wstruct, NULL, &rect, 0, &stxt, mhelp->init.text.fgcolor, mhelp->init.text.bgcolor, 1);
    rect.x = w0 / 2 - rect.w / 2; rect.y = h0 / 2 - rect.h / 2;
    VG3_draw_text(mhelp->init.wstruct, NULL, &rect, 0, &stxt, mhelp->init.text.fgcolor, mhelp->init.text.bgcolor, 0);
    rect.x -= 2; rect.y -= 2; rect.w += 4; rect.h += 4;
    VG3_draw_rect(mhelp->init.wstruct, NULL, &rect, 0, mhelp->init.text.fgcolor);
    VG3_window_update(mhelp->init.wstruct, 0, 0);
  }
}


/* show_menu: show menu
 * @param mhelp   menu_helper-struct
 * @return  return-value of VG3_input_select():
 *          - 1-?: selection
 *          -   0: cancelled with escape-key
 *          -  -1: failure
 *          -  -2: exit request
 * [mhelp: init, sel]
 */
static int
show_menu(struct menu_helper *mhelp)
{
  int erg, hdw;
  char **pselitem;

  if (mhelp == NULL || mhelp->init.wstruct == NULL) { return 0; }

  if (mhelp->sel.item[0] == NULL) { return 0; }
  pselitem = mhelp->sel.item;

  draw_bg(mhelp);

  hdw = VG3_font_highdouble(mhelp->init.wstruct, 1);
  erg = VG3_input_select(mhelp->init.wstruct, mhelp->init.menu.fgcolor, mhelp->init.menu.bgcolor, mhelp->sel.title, pselitem, mhelp->init.menu.onleft);
  VG3_font_highdouble(mhelp->init.wstruct, hdw);

  return erg;
}


/* sel_nwserver: select network-server
 * (returns selection in mhelp->nw)
 * @param mhelp   menu_helper-struct
 * @return  return-value of VG3_input_select() resp. VG3_input_box():
 *          - 1-?: selection
 *          -   0: cancelled with escape-key
 *          -  -1: failure
 *          -  -2: exit request
 * [mhelp: init, nw.servername, => nw]
 * uses multilanguage-file: mlang/menu_helper
 */
static int
sel_nwserver(struct menu_helper *mhelp)
{
  int erg, hdw;
  char buf[256], *selitem[4], **pselitem;
  struct vg3_multilang *mlang;

  if (mhelp == NULL || mhelp->init.files_dir == NULL || mhelp->init.wstruct == NULL) { return 0; }

  mhelp->nw.is_master = 0;

  mlang = VG3_multilang_new(NULL, "en");
  snprintf(buf, sizeof(buf), "%s/mlang/menu_helper", mhelp->init.files_dir);
  VG3_multilang_add(mlang, buf);

  draw_bg(mhelp);

  selitem[0] = (char *)VG3_multilang_get(mlang, "sel_nwserver-bcnw"),
  selitem[1] = (char *)VG3_multilang_get(mlang, "sel_nwserver-svnw"),
  selitem[2] = (char *)VG3_multilang_get(mlang, "sel_nwserver-start_sv"),
  selitem[3] = NULL;
  pselitem = selitem;

redonw:
  hdw = VG3_font_highdouble(mhelp->init.wstruct, 1);
  erg = VG3_input_select(mhelp->init.wstruct, mhelp->init.menu.fgcolor, mhelp->init.menu.bgcolor,
                         VG3_multilang_get(mlang, "sel_nwserver"), pselitem, mhelp->init.menu.onleft);
  VG3_font_highdouble(mhelp->init.wstruct, hdw);

  if (erg < 1) { VG3_multilang_free(mlang); return erg; }

  if (erg == 3) {
    mhelp->nw.is_master = 1;
    selitem[2] = (char *)VG3_multilang_get(mlang, "sel_nwserver-nostart_sv");
    goto redonw;
  }

  if (erg == 2) {  /* input server name */
    hdw = VG3_font_highdouble(mhelp->init.wstruct, 1);
    erg = VG3_input_box(mhelp->init.wstruct, mhelp->init.menu.fgcolor, mhelp->init.menu.bgcolor,
                        VG3_multilang_get(mlang, "sel_nwserver-enter_sv"),
                        mhelp->nw.servername, sizeof(mhelp->nw.servername));
    VG3_font_highdouble(mhelp->init.wstruct, hdw);
    VG3_multilang_free(mlang);
    if (erg < 1) { return erg; }
    return 2;
  }

  memset(mhelp->nw.servername, 0, sizeof(mhelp->nw.servername));
  VG3_multilang_free(mlang);
  return 1;
}


/* open_network: (start and) connect to network-server
 * (returns network-struct in mhelp->nwptr)
 * @param mhelp        menu_helper-struct
 * @param jid          for network-keys: joystick-ID or 0 = keyboard
 * @param max_players  maximum number of players
 * @param skeys        keys-struct
 * @return  >0 = client-number
 *           0 = exit request
 *          -1 = failure
 * [mhelp: init, nw, => nwptr]
 * uses multilanguage-file: mlang/menu_helper
 */
static int
open_network(struct menu_helper *mhelp, int jid, int max_players, struct vg3_keys *skeys)
{
  char buf[256], remhost[128], *ipsrv;
  int clnr, masterbyte;

  if (mhelp == NULL || mhelp->init.files_dir == NULL || mhelp->init.wstruct == NULL || max_players < 1 || skeys == NULL) {
    VG3_seterror(EINVAL, strerror(EINVAL));
    return -1;
  }

  clnr = 0;
  masterbyte = 1;

  /* master evtl. starts multicast/broadcast server and network server */
  if (mhelp->nw.is_master) {
    if (*mhelp->nw.servername == '\0') { VG3_nw_mbcast_start(); }
    VG3_nw_server(NULL);
  }

  /* retrieve ip of network server via multicast/broadcast */
  if (*mhelp->nw.servername == '\0') {
    struct vg3_multilang *mlang = VG3_multilang_new(NULL, "en");
    snprintf(buf, sizeof(buf), "%s/mlang/menu_helper", mhelp->init.files_dir);
    VG3_multilang_add(mlang, buf);

    if (VG3_inputevent_update(mhelp->init.wstruct)) { VG3_multilang_free(mlang); return 0; }
    print_text(mhelp, VG3_multilang_get(mlang, "open_network-searching"));
    ipsrv = VG3_nw_mbcast_getip(remhost, sizeof(remhost));
    if (ipsrv == NULL) { VG3_multilang_free(mlang); return -1; }

    /* show server name and ip */
    if (VG3_inputevent_update(mhelp->init.wstruct)) { VG3_multilang_free(mlang); return 0; }
    snprintf(buf, sizeof(buf), "%s:\n%s: %s", VG3_multilang_get(mlang, "open_network-found"), remhost, ipsrv);
    print_text(mhelp, buf);
    sleep(2);
    VG3_multilang_free(mlang);
  } else {
    ipsrv = strdup(mhelp->nw.servername);
  }

  /* get initialized network struct */
  mhelp->nwptr = VG3_nw_open(mhelp->init.wstruct);

  /* add keys installed into the system menu to network server */
  VG3_keys_nw_addkeys(skeys, mhelp->nwptr, jid);

  /* clear window and connect to network server */
  if (VG3_inputevent_update(mhelp->init.wstruct)) { return 0; }
  draw_bg(mhelp);
  clnr = VG3_nw_connect(mhelp->nwptr, ipsrv, NULL, NULL, max_players, &masterbyte);
  if (clnr <= 0) { return clnr; }
  if (VG3_inputevent_update(mhelp->init.wstruct)) { return 0; }
  free(ipsrv);

  /* master evtl. stops multicast/broadcast server */
  if (mhelp->nw.is_master && *mhelp->nw.servername == '\0') { VG3_nw_mbcast_stop(); }

  return clnr;
}


/* recv_nwdata: retrieve data-packet
 * @param mhelp    menu_helper-struct
 * @param nwdata   for returning data-packet
 * @param nwsize   sizeof(nwdata)
 * @return  1 = OK
 *          0 = exit request
 * [mhelp: init, nwptr, sel.title]
 *
 * Using example:
 *   char nwdata[1024];
 *   if (gmain.nwptr == NULL || clnr == 1) {
 *     mhelp.sel.title = (char *)VG3_multilang_get(gmain.mlang, "Any title");
 *     mhelp.sel.item[0] = (char *)VG3_multilang_get(gmain.mlang, "Selection1");
 *     mhelp.sel.item[1] = (char *)VG3_multilang_get(gmain.mlang, "Selection2");
 *     mhelp.sel.item[2] = NULL;
 *     erg = show_menu(&mhelp);
 *     if (erg < 1) { ...exit }
 *     snprintf(nwdata, sizeof(nwdata), "Selection %d", erg);
 *     if (gmain.nwptr != NULL) { VG3_nw_send_data(gmain.nwptr, nwdata); }
 *   } else {
 *     mhelp.sel.title = (char *)VG3_multilang_get(gmain.mlang, "receiving data");
 *     erg = mhelp.func.recv_nwdata(&mhelp, nwdata, sizeof(nwdata));
 *     if (!erg) { ...exit }
 *   }
 *   // now use nwdata
 */
static int
recv_nwdata(struct menu_helper *mhelp, char *nwdata, size_t nwsize)
{
  if (mhelp == NULL || mhelp->init.wstruct == NULL || mhelp->nwptr == NULL) { return 0; }
  if (nwdata == NULL || nwsize == 0) { return 0; }

  print_text(mhelp, mhelp->sel.title);

  VG3_discard_input(mhelp->init.wstruct);
  for (;;) {
    if (VG3_inputevent_update(mhelp->init.wstruct) > 0) { return 0; }

    /* ESC: exit */
    if (VG3_key_ispressed(mhelp->init.wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) { return 0; }

    /* check for and get data-packet */
    if (VG3_nw_recv_data(mhelp->nwptr)) { return 0; }
    if (VG3_nw_fetch_data(mhelp->nwptr, nwdata, nwsize) > 0) {
      draw_bg(mhelp);
      break;
    }

    VG3_window_update(mhelp->init.wstruct, 0, 0);
    VG3_wait_time(50);
  }
  VG3_discard_input(mhelp->init.wstruct);

  return 1;
}


/* sel_imglist: select one element from an image-list
 * @param mhelp   menu_helper-struct
 * @param clnr    own client-number
 * @param plysel  for returning selected element-numbers for each client
 *                (plysel[<client-number> - 1] = <selected element-number or 0>)
 * @param anzply  number of clients
 * @param zoom    whether zoom images:
 *                - 0: no
 *                - 1: zoom to the size of the window
 *                - 2 to 99: zoom from 2 to 99 percent to the size of the window
 * @return  return-value of VG3_input_image_select():
 *          - 1-?: selection
 *          -   0: cancelled with escape-key
 *          -  -1: failure
 *          -  -2: exit request
 * [mhelp: init, sel, nwptr]
 * uses multilanguage-file: mlang/menu_helper
 */
static int
sel_imglist(struct menu_helper *mhelp, int clnr, int *plysel, int anzply, int zoom)
{
  int erg, hdw, maxitem, inr, firstloop, datada;
  struct vg3_image *imgload[MH_MAX_SELITEM], *imgsel[MH_MAX_SELITEM], **pimgsel;
  char buf[256];
  struct vg3_multilang *mlang;

  if (mhelp == NULL || mhelp->init.files_dir == NULL || mhelp->init.wstruct == NULL) { return 0; }
  if (plysel == NULL || anzply <= 0) { return 0; }
  if (mhelp->nwptr == NULL) { clnr = 1; }

  for (maxitem = 0; mhelp->sel.item[maxitem] != NULL; maxitem++) {;}
  if (maxitem < 1 || anzply > maxitem || clnr < 1 || clnr > anzply) { return 0; }
  memset(plysel, 0, sizeof(*plysel) * anzply);

  mlang = VG3_multilang_new(NULL, "en");
  snprintf(buf, sizeof(buf), "%s/mlang/menu_helper", mhelp->init.files_dir);
  VG3_multilang_add(mlang, buf);

  /* load images */
  memset(imgload, 0, sizeof(imgload));
  for (inr = 0; inr < maxitem; inr++) {
    imgsel[inr] = imgload[inr] = VG3_image_load(mhelp->init.wstruct, mhelp->sel.item[inr], 1);
    if (imgload[inr] == NULL) { erg = -1; goto il_exit; }
  }
  pimgsel = imgsel;

il_redo:
  draw_bg(mhelp);

  hdw = VG3_font_highdouble(mhelp->init.wstruct, 1);
  erg = VG3_input_image_select(mhelp->init.wstruct, mhelp->sel.title, pimgsel, maxitem, zoom);
  VG3_font_highdouble(mhelp->init.wstruct, hdw);

  if (erg < 1) { goto il_exit; }

  /* set my selection */
  plysel[clnr - 1] = erg;
  imgsel[plysel[clnr - 1] - 1] = NULL;

  /* receive selections of all other players (and send my selection) */
  if (mhelp->nwptr != NULL) {
    print_text(mhelp, VG3_multilang_get(mlang, "sel_imglist-waiting"));

    VG3_discard_input(mhelp->init.wstruct);
    for (firstloop = 1;; firstloop = 0) {
      if (VG3_inputevent_update(mhelp->init.wstruct) > 0) { erg = -2; goto il_exit; }

      /* ESC: exit */
      if (VG3_key_ispressed(mhelp->init.wstruct, VGAG3_KEY_ESC, VGAG3_IS_NEW_PRESSED)) { erg = 0; goto il_exit; }

      /* check for and get other player-selections */
      if (VG3_nw_recv_data(mhelp->nwptr)) { erg = -2; goto il_exit; }
      datada = 0;
      if (VG3_nw_fetch_data(mhelp->nwptr, NULL, 0) > 0) { datada = 1; }  /* data available */

      if (datada) {
        while ((inr = VG3_nw_fetch_data(mhelp->nwptr, buf, sizeof(buf))) > 0) {
          if (inr > maxitem) { continue; }
          if (buf[0] == '0') {  /* player quit */
            if (plysel[inr - 1] > 0) { imgsel[plysel[inr - 1] - 1] = imgload[plysel[inr - 1] - 1]; }
            plysel[inr - 1] = -1;
          } else {  /* set player's selection */
            plysel[inr - 1] = atoi(buf + 1);
            imgsel[plysel[inr - 1] - 1] = NULL;
          }
        }

        /* check if another player has chosen my selection */
        for (inr = 1; inr <= anzply; inr++) {
          if (inr != clnr && plysel[inr - 1] == plysel[clnr - 1]) { break; }
        }

        if (inr <= anzply) {  /* yes, go back to selection menu */
          if (!firstloop) {  /* selection collision */
            imgsel[plysel[inr - 1] - 1] = imgload[plysel[inr - 1] - 1];
            plysel[inr - 1] = 0;
            print_text(mhelp, VG3_multilang_get(mlang, "sel_imglist-collision"));
          } else {  /* another player has already selected my selection */
            print_text(mhelp, VG3_multilang_get(mlang, "sel_imglist-gone"));
          }
          plysel[clnr - 1] = 0;
          sleep(3);
          goto il_redo;
        }
      }

      if (firstloop) {  /* inform other players of my selection */
        snprintf(buf, sizeof(buf), "1%d", plysel[clnr - 1]);
        VG3_nw_send_data(mhelp->nwptr, buf);
        datada = 1;
      }

      if (datada) {
        /* check if at least one other player has not taken a selection */
        for (inr = 1; inr <= anzply; inr++) {
          if (plysel[inr - 1] == 0) { break; }
        }
        if (inr > anzply) { break; }  /* all done */
      }

      VG3_window_update(mhelp->init.wstruct, 0, 0);
      VG3_wait_time(50);
    }
    VG3_discard_input(mhelp->init.wstruct);
  }

  /* unload player-bitmaps */
  for (inr = 0; inr < maxitem; inr++) {
    VG3_image_unload(mhelp->init.wstruct, imgload[inr]);
  }

  /* set -1 to 0 */
  for (inr = 0; inr < anzply; inr++) {
    if (plysel[inr] < 0) { plysel[inr] = 0; }
  }

  draw_bg(mhelp);

  VG3_multilang_free(mlang);
  return 1;

il_exit:
  if (mhelp->nwptr != NULL) { VG3_nw_send_data(mhelp->nwptr, "0"); }
  for (inr = 0; inr < maxitem; inr++) {
    if (imgload[inr] != NULL) { VG3_image_unload(mhelp->init.wstruct, imgload[inr]); }
  }
  VG3_multilang_free(mlang);
  return erg;
}
