#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"

void activate_text(void *, const char *, int, int, int);
int check_long_distance(void *, struct vg3_ofunc_object *, struct vg3_rect *);
void draw_health(void *, int, int, int);
void level_ending(void *);
int show_zwseq(void *);
int show_failed(void *, const char *);


/* activate_text:
 * show text on window using OID_NAME_TEXT
 * @param vmain      main game struct (struct g_main)
 * @param text       text
 * @param showcount  max_showcount or 0 = default
 * @param loopcount  max_loopcount or 0 = default
 * @param force      whether overwrite existing text
 */
void
activate_text(void *vmain, const char *text, int showcount, int loopcount, int force)
{
  struct g_main *gmain = (struct g_main *)vmain;
  const struct vg3_ofunc_objfunc *ofc;
  struct vg3_ofunc_object *objp;

  if (gmain == NULL || text == NULL || *text == '\0') { return; }

  ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_TEXT));
  if (ofc != NULL && ofc->f_data != NULL) {
    struct fdata_number fdnumber;
    memset(&fdnumber, 0, sizeof(fdnumber));
    fdnumber.flag = FDATA_TEXT;
    fdnumber.number1 = showcount;
    fdnumber.number2 = loopcount;
    fdnumber.number3 = force;
    fdnumber.text = text;
    objp = VG3_ofunc_objlist_isvalid(gmain->ofstruct, gmain->game.instanceid.text);
    if (objp != NULL) { ofc->f_data(gmain, objp, &fdnumber); }
  }
}


/* check_long_distance:
 * check for long-distance-collision and call collision-functions
 * @param vmain   main game struct (struct g_main)
 * @param objp    moved object-instance
 * @param rectz   new position of object-instance
 * @return        0 = OK or -1 = error
 */
int
check_long_distance(void *vmain, struct vg3_ofunc_object *objp, struct vg3_rect *rectz)
{
  struct g_main *gmain = (struct g_main *)vmain;
  struct vg3_coll *collp;
  int anz_coll, icoll, collretw;
  const struct vg3_ofunc_objobjfunc *oofc;

  if (gmain == NULL || objp == NULL || rectz == NULL) { return 0; }
  if (gmain->game.qdtr_long == NULL) { return 0; }

  anz_coll = VG3_coll_q_find(gmain->game.qdtr_long, rectz, NULL, &collp);
  if (anz_coll < 0) { return -1; }

  /* for each collision */
  for (icoll = 0; icoll < anz_coll; icoll++) {

    /* find collision-function */
    oofc = VG3_ofunc_get_objobjfunc(gmain->ofstruct, objp->oid, collp[icoll].oid);
    if (oofc == NULL || oofc->f_collision == NULL) { continue; }

    /* call collision-function, return value irrelevant except error */
    collretw = oofc->f_collision(vmain, objp, collp[icoll].optr, &collp[icoll].ret);
    if (collretw < 0) { return -1; }
  }

  if (collp != NULL) { free(collp); }
  return 0;
}


/* draw a health-bar */
void
draw_health(void *vmain, int max_health, int health, int is_player)
{
  const int minwsize = 30;
  int framesize = 1, rectw, recth = 3;
  struct g_main *gmain = vmain;
  struct vg3_image *img;
  struct vg3_rect rect;
  struct vg3_image_attributes iattr;
  int imgw, imgh;

  if (gmain == NULL || max_health <= 0) { return; }

  rectw = (minwsize + max_health) / max_health;

  imgw = framesize * 2 + rectw * max_health;
  imgh = framesize * 2 + recth;
  img = VG3_image_create(gmain->wstruct, imgw, imgh);
  if (img == NULL) { return; }

  rect.x = rect.y = 0;
  rect.w = imgw;
  rect.h = imgh;
  VG3_draw_rect(gmain->wstruct, img, &rect, 0, VGAG3_COLOR_WHITE);

  if (health > 0) {
    int color;
    rect.x = rect.y = framesize;
    rect.w = rectw * health;
    rect.h = recth;
    if (health > max_health * 2 / 3) {
      color = VGAG3_COLOR_GREEN;
    } else if (health > max_health * 1 / 3) {
      color = VGAG3_COLOR_YELLOW;
    } else {
      color = VGAG3_COLOR_RED;
    }
    VG3_draw_rect(gmain->wstruct, img, &rect, 1, color);
  }

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
  iattr.alpha = 150;
  if (is_player) {
    VG3_image_copy(gmain->wstruct, NULL, img, 5 + imgw / 2, gmain->winh - recth - imgh / 2, &iattr, 0);
  } else {
    VG3_image_copy(gmain->wstruct, NULL, img, gmain->winw - (5 + imgw / 2), gmain->winh - recth - imgh / 2, &iattr, 0);
  }

  VG3_image_unload(gmain->wstruct, img);
}


/* after level ended */
void
level_ending(void *vmain)
{
  struct g_main *gmain = vmain;
  struct vg3_image *img;
  struct vg3_image_attributes iattr;

  if (gmain == NULL) { return; }

  VG3_wait_time(1000);
  img = VG3_image_clone(gmain->wstruct, NULL, NULL, NULL, NULL);
  if (img == NULL) { return; }

  VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
  for (iattr.zoom = 100; iattr.zoom > 0; iattr.zoom -= 2) {
    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    VG3_image_copy(gmain->wstruct, NULL, img, gmain->winw / 2, gmain->winh / 2, &iattr, 0);
    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(20);
  }
  VG3_image_unload(gmain->wstruct, img);

  VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
  VG3_window_update(gmain->wstruct, 0, 0);
}


/* sequence between levels */
int
show_zwseq(void *vmain)
{
  struct g_main *gmain = vmain;
  struct vg3_sprite *sprt_cmd;
  struct vg3_image *hgimg, *img;
  struct vg3_image_attributes iattr;
  char buf[128];
  const char *text;
  size_t tpos, tlen;
  struct vg3_rect rect;
  struct vg3_text stxt;
  int retw, imgw, imgh, tsnd;

  if (gmain == NULL) { return 0; }
  if (level_act(gmain, LEVELACT_TEST) != 1) { return 0; }

  sprt_cmd = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/zwseq/commander.sprite");
  if (sprt_cmd == NULL) { fprintf(stderr, "%s\n", VG3_error()); return 0; }

  VG3_sprite_imagesize(sprt_cmd, "max", 100, &imgw, &imgh);
  rect.w = gmain->winw - imgw - 20;
  rect.h = gmain->winh;
  rect.x = 10; rect.w -= 20;
  rect.y = 10; rect.h -= 20;

  snprintf(buf, sizeof(buf), "zwseq-%d", gmain->gamelevel);
  text = VG3_multilang_get(gmain->mlang, buf);
  tlen = strlen(text);
  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, text);

  rect = VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 1);
  if (rect.w == 0 || rect.h == 0) { fprintf(stderr, "Text %s exceeds target-rectangle\n", buf); VG3_sprite_free(sprt_cmd); return 0; }

  hgimg = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/zwseq/hg.bmp", 1);

  tsnd = VG3_audio_load(gmain->wstruct, FILES_DIR "/sound/type.wav", 100, VGAG3_AUDIO_VOLUME_SOUND);
  if (tsnd > 0) { VG3_audio_play(gmain->wstruct, tsnd, 2, 0); }

  retw = 0;
  VG3_discard_input(gmain->wstruct);
  for (tpos = 0;;) {
    if (VG3_inputevent_update(gmain->wstruct)) { retw = 1; break; }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) {
      if (tpos < tlen) { tpos = tlen - 1; } else { break; }
    }

    if (!VG3_sprite_get(sprt_cmd, &img, &iattr, NULL)) {
      VG3_sprite_rewind(sprt_cmd);
      VG3_sprite_get(sprt_cmd, &img, &iattr, NULL);
    }

    VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
    if (hgimg != NULL) { VG3_image_copy(gmain->wstruct, NULL, hgimg, gmain->winw / 2, gmain->winh / 2, NULL, 0); }
    VG3_image_copy(gmain->wstruct, NULL, img, gmain->winw - imgw / 2 - 10, imgh / 2 + 10, &iattr, 0);
    if (tpos < tlen) {
      tpos++;
      stxt.tsize = tpos;
      if (tpos == tlen && tsnd > 0) { VG3_audio_stop(gmain->wstruct, tsnd, 0); }
    }
    VG3_draw_text(gmain->wstruct, NULL, &rect, ' ', &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT, 0);
    VG3_text_simpledraw(gmain->wstruct, NULL, NULL,
                        gmain->winw - imgw / 2 - 10, gmain->winh - 10,
                        VG3_multilang_get(gmain->mlang, "press-space"),
                        VGAG3_COLOR_VIOLET, VGAG3_COLOR_TRANSPARENT, 0);

    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(50);
  }
  VG3_discard_input(gmain->wstruct);

  if (tsnd > 0) { VG3_audio_unload(gmain->wstruct, tsnd); }
  VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
  VG3_window_update(gmain->wstruct, 0, 0);
  if (hgimg != NULL) { VG3_image_unload(gmain->wstruct, hgimg); }
  VG3_sprite_free(sprt_cmd);

  return retw;
}


/* level failed */
int
show_failed(void *vmain, const char *text)
{
  struct g_main *gmain = vmain;
  char buf[256];
  struct vg3_text stxt;
  int retw, wtime;
  size_t slen;

  if (gmain == NULL) { return 0; }
  if (text != NULL && *text == '\0') { text = NULL; }

  snprintf(buf, sizeof(buf), "%s\n\n\n\n", VG3_multilang_get(gmain->mlang, "level-failed"));
  slen = strlen(buf);
  if (text != NULL) {
    snprintf(buf + slen, sizeof(buf) - slen, "%s\n\n\n", VG3_multilang_get(gmain->mlang, text));
    slen = strlen(buf);
  }
  snprintf(buf + slen, sizeof(buf) - slen, "%s", VG3_multilang_get(gmain->mlang, "level-again"));
  VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, buf);

  VG3_draw_clear(gmain->wstruct, NULL, VGAG3_COLOR_BLACK);
  VG3_text_simpledraw(gmain->wstruct, NULL, NULL,
                      gmain->winw / 2, gmain->winh / 2,
                      buf,
                      VGAG3_COLOR_RED, VGAG3_COLOR_TRANSPARENT, 1);

  retw = 0;
  VG3_discard_input(gmain->wstruct);
  for (wtime = 7000; wtime > 0; wtime -= 100) {
    if (VG3_inputevent_update(gmain->wstruct)) { retw = 1; break; }

    if (VG3_key_ispressed(gmain->wstruct, VGAG3_KEY_SPACE, VGAG3_IS_NEW_PRESSED)) { break; }

    VG3_window_update(gmain->wstruct, 0, 0);
    VG3_wait_time(100);
  }
  VG3_discard_input(gmain->wstruct);

  return retw;
}
