#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-bg.h"

void getofc_bg(struct vg3_ofunc_objfunc *);

static void set_new_bg(struct g_main *, struct g_obj_bg *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_bg(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_BG));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ static functions +++ */

static void
set_new_bg(struct g_main *gmain, struct g_obj_bg *gobj)
{
  if (gmain == NULL || gobj == NULL) { return; }

  gobj->rect1 = gobj->rect2;

  gobj->rect2.x = gmain->winw;
  gobj->rect2.w = gmain->winw;
  gobj->rect2.y = 0;
  gobj->rect2.h = gmain->winh;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameters: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_bg *gobj;
  va_list ap;
  char buf[128];

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* load image */
  snprintf(buf, sizeof(buf), "%s/bmp/bg/bg%02d.bmp", FILES_DIR, gmain->gamelevel);
  gobj->img = VG3_image_load(gmain->wstruct, buf, 1);
  if (gobj->img == NULL) { return NULL; }

  /* set indexes and positions */
  set_new_bg(gmain, gobj);
  set_new_bg(gmain, gobj);
  gobj->rect1.x = 0;

  /* create and fill bg-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_BG));
  objp->subid = 0;
  objp->drawlevel = 1;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert bg-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_bg *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_bg *)objp->ostruct;

  /* remove bg-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free bg-instance */
  if (gobj->img != NULL) { VG3_image_unload(gmain->wstruct, gobj->img); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_bg *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_bg *)objp->ostruct;

  gobj->rect1.x--;
  gobj->rect2.x--;
  if (gobj->rect1.x <= -gmain->winw) {
    set_new_bg(gmain, gobj);
  }
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_bg *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_bg *)objp->ostruct;

  /* draw bg-instance */
  VG3_image_copy(gmain->wstruct, NULL, gobj->img, gobj->rect1.x + gobj->rect1.w / 2, gobj->rect1.y + gobj->rect1.h / 2, NULL, 0);
  VG3_image_copy(gmain->wstruct, NULL, gobj->img, gobj->rect2.x + gobj->rect2.w / 2, gobj->rect2.y + gobj->rect2.h / 2, NULL, 0);

  /* draw text for remaining time */
  if (*gmain->game.btim != '\0') {
    struct vg3_image *img;
    struct vg3_text stxt;

    VGAG3_TEXT_ATTRIBUTES_SET(&stxt, NULL, '\n', 0, gmain->game.btim);
    img = VG3_image_text(gmain->wstruct, &stxt, VGAG3_COLOR_YELLOW, VGAG3_COLOR_TRANSPARENT);
    if (img != NULL) {
      struct vg3_image_attributes iattr;
      VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
      iattr.alpha = 150;
      VG3_image_copy(gmain->wstruct, NULL, img, gmain->winw / 2, gmain->winh - 6, &iattr, 0);
      VG3_image_unload(gmain->wstruct, img);
    }
  }
}
