#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-escort.h"

void getofc_escort(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);


/* +++ get-function +++ */

void
getofc_escort(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_ESCORT));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameters: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_escort *gobj;
  struct vg3_coll coll;
  va_list ap;
  int xm, ym;
  unsigned int *seed;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  seed = VG3_ofunc_get_seed(gmain->ofstruct, get_oid_name(OID_NAME_ESCORT));

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* load images and sprites */
  gobj->images.img = VG3_image_load(gmain->wstruct, FILES_DIR "/bmp/escort/escort.bmp", 1);
  if (gobj->images.img == NULL) { return NULL; }
  gobj->images.expl = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/expl/expl1.sprite");
  if (gobj->images.expl == NULL) { return NULL; }

  /* set position */
  VG3_image_getsize(gmain->wstruct, gobj->images.img, NULL, &gobj->rect.w, &gobj->rect.h);
  xm = gmain->winw + gobj->rect.w / 2 + 50;
  ym = gmain->winh / 2 + (int)VG3_nw_random_getnext(1, 13, seed) - 7;
  gobj->rect.x = xm - gobj->rect.w / 2;
  gobj->rect.y = ym - gobj->rect.h / 2;
  /* set collision rectangle with 80% of width and height of image-rectangle */
  gobj->crect = gobj->rect;
  gobj->crect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w * 80 / 100, gobj->rect.h * 80 / 100);

  /* set speed */
  gobj->xdelta = -20;
  gobj->ydelta = 0;

  /* set health */
  gobj->health = 1;

  /* create and fill escort-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_ESCORT));
  objp->subid = 0;
  objp->drawlevel = 2;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert escort-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* insert escort-instance (collision-rectangle) into quadtree */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_escort *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_escort *)objp->ostruct;

  /* remove escort-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove escort-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* redo level if destroyed */
  if (gobj->health <= 0) {
    gmain->game.level_retval = 2;
  }

  /* free escort-instance */
  if (gobj->images.img != NULL) { VG3_image_unload(gmain->wstruct, gobj->images.img); }
  if (gobj->images.expl != NULL) { VG3_sprite_free(gobj->images.expl); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 100;
  struct g_main *gmain = vmain;
  struct g_obj_escort *gobj;
  struct vg3_coll coll;
  int erg;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_escort *)objp->ostruct;

  /* remove escort-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* check if escort is exploding */
  if (gobj->health <= 0) { return; }

  /* check if escort is on its place */
  if (gobj->xdelta < 0 && gobj->rect.x <= gobj->rect.w) { gobj->xdelta = 2; gobj->xremainder = 0; }
  if (gobj->xdelta > 0 && gobj->rect.x >= gobj->rect.w * 3 / 2) { gobj->xdelta = -2; gobj->xremainder = 0; }

  /* move escort-instance and check for collisions with gobj->crect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->crect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: destroy escort-instance */
    fprintf(stderr, "moving escort: %s\n", VG3_error());
    f_free(gmain, objp);
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }  /* instance has been freed */
  if (erg == VGAG3_COLL_RETURN_CATCHED) { f_free(gmain, objp); return; }

  /* x/y-value of gobj->crect could be modified, correct it also in gobj->rect */
  gobj->rect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w, gobj->rect.h);

  /* insert escort-instance (collision-rectangle) into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_escort *gobj;
  struct vg3_image *img;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_escort *)objp->ostruct;

  /* draw escort-instance */
  if (gobj->health <= 0) {  /* exploding */
    if (!VG3_sprite_get(gobj->images.expl, &img, &iattr, NULL)) { f_free(gmain, objp); return; }
  } else {
    img = gobj->images.img;
    VGAG3_IMAGE_ATTRIBUTES_DEFAULT(&iattr);
  }
  if (img != NULL) {
    VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
  }
}
