#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-fighter.h"

void init_dest_fighter(int);
void getofc_fighter(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);

static struct {
  unsigned int *seed;
  struct random_check *rdchk;
} objvars;


/* +++ get-function +++ */

void
init_dest_fighter(int is_init)
{
  if (is_init) {
    memset(&objvars, 0, sizeof(objvars));
  } else {
    random_check_free(objvars.rdchk);
    memset(&objvars, 0, sizeof(objvars));
  }
}


void
getofc_fighter(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_FIGHTER));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameters:
 * - int: x-position (centered)
 * - int: y-position (centered)
 * - int: one from FIGHTER_*
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_fighter *gobj;
  struct vg3_coll coll;
  va_list ap;
  int xm, ym, subid;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  if (objvars.seed == NULL) {
    objvars.seed = VG3_ofunc_get_seed(gmain->ofstruct, get_oid_name(OID_NAME_FIGHTER));
  }

  if (objvars.rdchk == NULL) {
    objvars.rdchk = random_check_new(objvars.seed, 1);
    if (objvars.rdchk == NULL) { return NULL; }
  }

  /* get arguments */
  va_start(ap, iparent);
  xm = va_arg(ap, int);
  ym = va_arg(ap, int);
  subid = va_arg(ap, int);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* load sprites */
  if (subid == FIGHTER_STEALTH) {
    gobj->images.sprt = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/fighter-stealth.sprite");
    gobj->images.exhaust = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/exhaust-stealth.sprite");
  } else if (subid == FIGHTER_RED) {
    gobj->images.sprt = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/fighter3.sprite");
    gobj->images.exhaust = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/exhaust.sprite");
  } else if (subid == FIGHTER_YELLOW) {
    gobj->images.sprt = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/fighter2.sprite");
    gobj->images.exhaust = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/exhaust.sprite");
  } else {
    subid = FIGHTER_GREEN;
    gobj->images.sprt = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/fighter1.sprite");
    gobj->images.exhaust = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/exhaust.sprite");
  }
  if (gobj->images.sprt == NULL) { return NULL; }
  if (gobj->images.exhaust == NULL) { return NULL; }

  gobj->images.hit = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/hit/hit.sprite");
  if (gobj->images.hit == NULL) { return NULL; }
  gobj->images.expl = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/fighter/expl.sprite");
  if (gobj->images.expl == NULL) { return NULL; }

  /* set position */
  VG3_sprite_imagesize(gobj->images.sprt, "avg", 100, &gobj->rect.w, &gobj->rect.h);
  gobj->rect.x = xm - gobj->rect.w / 2;
  gobj->rect.y = ym - gobj->rect.h / 2;
  /* set collision rectangle with 80% of width and height of image-rectangle */
  gobj->crect = gobj->rect;
  gobj->crect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w * 80 / 100, gobj->rect.h * 80 / 100);

  /* set speed */
  if (ym < gmain->winh / 2) {
    gobj->ydelta = (int)VG3_nw_random_getnext(0, 20, objvars.seed);
  } else {
    gobj->ydelta = -(int)VG3_nw_random_getnext(0, 20, objvars.seed);
  }
  if (subid == FIGHTER_RED) {
    gobj->xdelta = -(int)VG3_nw_random_getnext(50, 80, objvars.seed);
  } else if (subid == FIGHTER_YELLOW) {
    gobj->xdelta = -(int)VG3_nw_random_getnext(40, 70, objvars.seed);
  } else {
    gobj->xdelta = -(int)VG3_nw_random_getnext(30, 60, objvars.seed);
  }
  gobj->xchange = (int)VG3_nw_random_getnext(gmain->winw / 5, gmain->winw * 4 / 5, objvars.seed);

  /* set allowed shots */
  if (subid == FIGHTER_RED) {
    gobj->shot.allowed = 3;
  } else if (subid == FIGHTER_YELLOW) {
    gobj->shot.allowed = 2;
  } else {
    gobj->shot.allowed = 1;
  }

  /* set health */
  if (subid == FIGHTER_RED) {
    gobj->health = 2;
  } else if (subid == FIGHTER_YELLOW) {
    gobj->health = 1;
  } else {
    gobj->health = 1;
  }

  /* create and fill fighter-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_FIGHTER));
  objp->subid = subid;
  objp->drawlevel = 3;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert fighter-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* insert fighter-instance (collision-rectangle) into quadtree */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_fighter *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_fighter *)objp->ostruct;

  /* remove fighter-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove fighter-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* inform object-management if destroyed */
  if (gobj->health <= 0) {
    struct fdata_number fdnumber;
    memset(&fdnumber, 0, sizeof(fdnumber));
    fdnumber.flag = FDATA_HIT;
    fdnumber.number1 = objp->subid;
    VG3_ofunc_mgmt_data(gmain->ofstruct, gmain, get_oid_name(OID_NAME_FIGHTER), &fdnumber);
  }

  /* free fighter-instance */
  if (gobj->images.sprt != NULL) { VG3_sprite_free(gobj->images.sprt); }
  if (gobj->images.exhaust != NULL) { VG3_sprite_free(gobj->images.exhaust); }
  if (gobj->images.hit != NULL) { VG3_sprite_free(gobj->images.hit); }
  if (gobj->images.expl != NULL) { VG3_sprite_free(gobj->images.expl); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 50;
  struct g_main *gmain = vmain;
  struct g_obj_fighter *gobj;
  struct vg3_coll coll;
  int erg;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_fighter *)objp->ostruct;

  /* remove fighter-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* check if fighter is exploding */
  if (gobj->health <= 0) { return; }

  /* check if fighter is gone */
  if (gobj->rect.x < -gobj->rect.w) { f_free(gmain, objp); return; }

  /* change y-direction? */
  if (gobj->rect.y <= gobj->rect.h) {
    gobj->ydelta = (int)VG3_nw_random_getnext(0, 30, objvars.seed);
  } else if (gobj->rect.y >= gmain->winh - gobj->rect.h) {
    gobj->ydelta = -(int)VG3_nw_random_getnext(0, 30, objvars.seed);
  } else if (gobj->xchange > 0 && gobj->rect.x <= gobj->xchange) {
    if (gobj->ydelta < 0) {
      gobj->ydelta = (int)VG3_nw_random_getnext(0, 30, objvars.seed);
    } else {
      gobj->ydelta = -(int)VG3_nw_random_getnext(0, 30, objvars.seed);
    }
    gobj->xchange = 0;
  }

  /* move fighter-instance and check for collisions with gobj->crect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->crect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: destroy fighter-instance */
    fprintf(stderr, "moving fighter: %s\n", VG3_error());
    f_free(gmain, objp);
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }  /* instance has been freed */
  if (erg == VGAG3_COLL_RETURN_CATCHED) { f_free(gmain, objp); return; }

  /* x/y-value of gobj->crect could be modified, correct it also in gobj->rect */
  gobj->rect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w, gobj->rect.h);

  /* release shot? */
  if (gobj->shot.pause > 0) {
    gobj->shot.pause--;
  } else {
    if (gobj->shot.running < gobj->shot.allowed) {
      if (random_check_exec(objvars.rdchk, 50 - gobj->shot.running * 10)) {
        const struct vg3_ofunc_objfunc *ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_FIGHTERSHOT));
        if (ofc != NULL) {
          int speedx, speedy;
          if (objp->subid == FIGHTER_RED) {
            speedx = -1200;
            speedy = 0;
          } else if (objp->subid == FIGHTER_YELLOW) {
            speedx = -800;
            speedy = (int)VG3_nw_random_getnext(1, 10, objvars.seed);
            if (gobj->shot.running % 2) { speedy = -10 - speedy; } else { speedy = 10 + speedy; }
          } else {
            speedx = gobj->xdelta * moving_factor * 15 / 10 / 10;
            speedy = (int)VG3_nw_random_getnext(1, 49, objvars.seed) - 25;
          }
          if (ofc->f_new(gmain, objp->instanceid, gobj->rect.x, gobj->rect.y + gobj->rect.h / 2, speedx, speedy, objp->subid) != NULL) {
            gobj->shot.running++;
            gobj->shot.pause = (int)VG3_nw_random_getnext(4, 8, objvars.seed);
          }
        }
      }
    }
  }

  /* insert fighter-instance (collision-rectangle) into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_fighter *gobj;
  struct vg3_image *img;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_fighter *)objp->ostruct;

  /* draw fighter-instance */
  if (gobj->health <= 0) {  /* exploding */
    if (!VG3_sprite_get(gobj->images.expl, &img, &iattr, NULL)) { f_free(gmain, objp); return; }
  } else {
    if (!VG3_sprite_get(gobj->images.sprt, &img, &iattr, NULL)) {
      VG3_sprite_rewind(gobj->images.sprt);
      VG3_sprite_get(gobj->images.sprt, &img, &iattr, NULL);
    }
  }
  if (img != NULL) {
    VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
    if (gobj->health > 0 && VG3_sprite_get(gobj->images.exhaust, &img, &iattr, NULL) && img != NULL) {
      VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.h + 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
    }
    if (gobj->hit) {  /* being hit */
      if (!VG3_sprite_get(gobj->images.hit, &img, &iattr, NULL)) {
        VG3_sprite_rewind(gobj->images.hit);
        gobj->hit = 0;
      } else if (img != NULL) {
        VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
      }
    }
  }
}
