#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames3.h>
#include "main.h"
#include "obj-mothership.h"

#define MAX_HEALTH 250
#define MAX_SHOTS  10

void getofc_mothership(struct vg3_ofunc_objfunc *);

static struct vg3_ofunc_object * f_new(void *, unsigned int, ...);
static void f_free(void *, struct vg3_ofunc_object *);
static void f_run(void *, struct vg3_ofunc_object *);
static void f_draw(void *, struct vg3_ofunc_object *);

static struct {
  unsigned int *seed;
  struct random_check *rdchk;
} objvars;


/* +++ get-function +++ */

void
init_dest_mothership(int is_init)
{
  if (is_init) {
    memset(&objvars, 0, sizeof(objvars));
  } else {
    random_check_free(objvars.rdchk);
    memset(&objvars, 0, sizeof(objvars));
  }
}

void
getofc_mothership(struct vg3_ofunc_objfunc *ofc)
{
  if (ofc == NULL) { return; }

  snprintf(ofc->oid, sizeof(ofc->oid), "%s", get_oid_name(OID_NAME_MOTHERSHIP));
  ofc->f_new = f_new;
  ofc->f_free = f_free;
  ofc->f_run = f_run;
  ofc->f_draw = f_draw;
}


/* +++ object-instance-functions +++ */

/* new-function
 * variable parameters: none
 */
static struct vg3_ofunc_object *
f_new(void *vmain, unsigned int iparent, ...)
{
  struct g_main *gmain = vmain;
  struct vg3_ofunc_object *objp;
  struct g_obj_mothership *gobj;
  struct vg3_coll coll;
  va_list ap;

  if (gmain == NULL) { VG3_seterror(EINVAL, strerror(EINVAL)); return NULL; }

  if (objvars.seed == NULL) {
    objvars.seed = VG3_ofunc_get_seed(gmain->ofstruct, get_oid_name(OID_NAME_MOTHERSHIP));
  }

  if (objvars.rdchk == NULL) {
    objvars.rdchk = random_check_new(objvars.seed, 1);
    if (objvars.rdchk == NULL) { return NULL; }
  }

  /* get arguments */
  va_start(ap, iparent);
  va_end(ap);

  /* create private struct for object */
  gobj = calloc(1, sizeof(*gobj));
  if (gobj == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }

  /* load sprites */
  gobj->images.sprt = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/mothership/mothership.sprite");
  if (gobj->images.sprt == NULL) { return NULL; }
  gobj->images.hit = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/hit/hit.sprite");
  if (gobj->images.hit == NULL) { return NULL; }
  gobj->images.expl = VG3_sprite_load(gmain->wstruct, FILES_DIR "/bmp/expl/expl3.sprite");
  if (gobj->images.expl == NULL) { return NULL; }

  /* set position */
  VG3_sprite_imagesize(gobj->images.sprt, "avg", 100, &gobj->rect.w, &gobj->rect.h);
  gobj->rect.x = gmain->winw + gobj->rect.w / 2 + 150;
  gobj->rect.y = gmain->winh / 2 - gobj->rect.h / 2;
  /* set collision rectangle with 80% of width and height of image-rectangle */
  gobj->crect = gobj->rect;
  gobj->crect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w * 80 / 100, gobj->rect.h * 80 / 100);

  /* set speed */
  gobj->xdelta = -20;
  gobj->ydelta = 0;

  /* set health */
  gobj->health = MAX_HEALTH;

  /* create and fill mothership-instance */
  objp = calloc(1, sizeof(*objp));
  if (objp == NULL) { VG3_seterror(ENOMEM, strerror(errno)); return NULL; }
  snprintf(objp->oid, sizeof(objp->oid), "%s", get_oid_name(OID_NAME_MOTHERSHIP));
  objp->subid = 0;
  objp->drawlevel = 2;
  objp->instanceid = 0;  /* will be set in VG3_ofunc_objlist_insert() */
  objp->ostruct = gobj;

  /* insert mothership-instance into list of object-instances */
  VG3_ofunc_objlist_insert(gmain->ofstruct, objp);

  /* insert mothership-instance (collision-rectangle) into quadtree */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);

  return objp;
}


/* free-function */
static void
f_free(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_mothership *gobj;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_mothership *)objp->ostruct;

  /* remove mothership-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* remove mothership-instance from list of object-instances */
  VG3_ofunc_objlist_remove(gmain->ofstruct, objp);

  /* free mothership-instance */
  if (gobj->images.sprt != NULL) { VG3_sprite_free(gobj->images.sprt); }
  if (gobj->images.hit != NULL) { VG3_sprite_free(gobj->images.hit); }
  if (gobj->images.expl != NULL) { VG3_sprite_free(gobj->images.expl); }
  free(gobj);
  free(objp);
}


/* run-function */
static void
f_run(void *vmain, struct vg3_ofunc_object *objp)
{
  const int moving_factor = 100;
  struct g_main *gmain = vmain;
  struct g_obj_mothership *gobj;
  struct vg3_coll coll;
  int erg;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_mothership *)objp->ostruct;

  /* remove mothership-instance from quadtree */
  VG3_coll_q_remove(gmain->qdtr, objp);

  /* check if mothership is exploding */
  if (gobj->health <= 0) { return; }

  /* check moving */
  if (gobj->ydelta == 0) {
    if (gobj->rect.x + gobj->rect.w / 2 < gmain->winw * 3 / 4) {
      gobj->ydelta = -7;
      gobj->xdelta = 0;
    }
  } else {
    if (gobj->rect.y <= 0) { gobj->ydelta = 7; }
    if (gobj->rect.y >= gmain->winh - gobj->rect.h) { gobj->ydelta = -7; }
  }

  /* move mothership-instance and check for collisions with gobj->crect */
  erg = VG3_move_object_check_collision(gmain, gmain->ofstruct, gmain->qdtr, objp, &gobj->crect, moving_factor, &gobj->xdelta, &gobj->ydelta, &gobj->xremainder, &gobj->yremainder);
  if (erg < 0) {  /* error: destroy mothership-instance */
    fprintf(stderr, "moving mothership: %s\n", VG3_error());
    f_free(gmain, objp);
    return;
  }
  if (erg == VGAG3_COLL_RETURN_DEAD) { return; }  /* instance has been freed */
  if (erg == VGAG3_COLL_RETURN_CATCHED) { f_free(gmain, objp); return; }

  /* x/y-value of gobj->crect could be modified, correct it also in gobj->rect */
  gobj->rect = VG3_correct_imageposition(&gobj->crect, gobj->rect.w, gobj->rect.h);

  /* release shot? */
  if (gobj->shot.pause > 0) {
    gobj->shot.pause--;
  } else if (gobj->ydelta != 0) {
    if (gobj->shot.running < MAX_SHOTS) {
      if (random_check_exec(objvars.rdchk, 500 - gobj->shot.running * 10)) {
        const struct vg3_ofunc_objfunc *ofc;
        ofc = VG3_ofunc_get_objfunc(gmain->ofstruct, get_oid_name(OID_NAME_MOTHERSHOT));
        if (ofc != NULL) {
          int xm, ym, speedx;
          speedx = -800;
          xm = gobj->rect.x + gobj->rect.w / 4;
          ym = gobj->rect.y + (int)VG3_nw_random_getnext(1, gobj->rect.h - 1, objvars.seed);
          if (ofc->f_new(gmain, objp->instanceid, xm, ym, speedx, 0) != NULL) {
            gobj->shot.running++;
            gobj->shot.pause = (int)VG3_nw_random_getnext(1, 2, objvars.seed);
          }
        }
      }
    }
  }

  /* insert mothership-instance (collision-rectangle) into quadtree again */
  memset(&coll, 0, sizeof(coll));
  coll.rect = gobj->crect;
  snprintf(coll.oid, sizeof(coll.oid), "%s", objp->oid);
  coll.optr = objp;
  VG3_coll_q_insert(gmain->qdtr, &coll);
}


/* draw-function */
static void
f_draw(void *vmain, struct vg3_ofunc_object *objp)
{
  struct g_main *gmain = vmain;
  struct g_obj_mothership *gobj;
  struct vg3_image *img;
  struct vg3_image_attributes iattr;

  if (gmain == NULL || objp == NULL) { return; }

  gobj = (struct g_obj_mothership *)objp->ostruct;

  /* draw mothership-instance */
  if (gobj->health <= 0) {  /* exploding */
    if (!VG3_sprite_get(gobj->images.expl, &img, &iattr, NULL)) { f_free(gmain, objp); return; }
  } else {
    if (!VG3_sprite_get(gobj->images.sprt, &img, &iattr, NULL)) {
      VG3_sprite_rewind(gobj->images.sprt);
      VG3_sprite_get(gobj->images.sprt, &img, &iattr, NULL);
    }
  }
  if (img != NULL) {
    VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->rect.h / 2, &iattr, 0);
    if (gobj->health > 0 && gobj->hit_y > 0) {  /* being hit */
      if (!VG3_sprite_get(gobj->images.hit, &img, &iattr, NULL)) {
        VG3_sprite_rewind(gobj->images.hit);
        gobj->hit_y = 0;
      } else if (img != NULL) {
        VG3_image_copy(gmain->wstruct, NULL, img, gobj->rect.x + gobj->rect.w / 2, gobj->rect.y + gobj->hit_y, &iattr, 0);
        draw_health(vmain, MAX_HEALTH, gobj->health, 0);
      }
    }
  }
}
